/**
 * Created by zengjun on 2017.
 */


import Class from '../base/Class.js';
import Color from './Color.js';
import Rectangle from './Rectangle.js';
import Util from '../util/Util.js';

import ALIGN from '../core/ALIGNMENT.js';
import Graphics from "./Graphics.js";


let Graphics2D = Graphics.extend({


    properties:
        {
            "invalidateRect":
                {
                    get: function () { return this.InvalidateRect;},
                    set: function (rc) { this.setInvalidateRect(rc);}
                }
        },
    constructor: function (canvas) {
        this.ctx = $(canvas)[0].getContext('2d');
        this.canvas = canvas;
        this.InvalidateRect = null;
        this.saveTimes = 0;
        this.currentClipRect = null;
    },


    reInit: function () {
        this.ctx = $(this.canvas)[0].getContext('2d');
    },
    save: function () {
        this.ctx.save();
        this.saveTimes++;
    },

    restore: function () {
        this.ctx.restore();


        if (this.saveTimes > 0) this.saveTimes--;
    },


    setInvalidateRect: function (rc) {
        if (rc == null)
        {
            this.InvalidateRect = null;
            return;
        }

        this.InvalidateRect = rc.clone();
    },

    clearRect: function (x, y, width, height) {
        this.ctx.clearRect(x, y, width, height);
    },


    color: function (c) {
        if (Util.isClas$(c))
        {
            if (c.instanceOf(Color)) return c;
        }
        return new Color(c);
    },


    setColor: function (color) {
        color = this.color(color);
        this.setStrokeColor(color);
        this.setFillColor(color);
    },

    setStrokeColor: function (color) {
        color = this.color(color);
        this.ctx.strokeStyle = Color.stringify(color.color, 'rgba');

    },


    setFillColor: function (color) {
        color = this.color(color);
        this.ctx.fillStyle = Color.stringify(color.color, 'rgba');
    },

    setLineWidth: function (n) {
        this.ctx.lineWidth = n;
    },

    setLineCap: function (cap) {
        this.ctx.lineCap = cap;
    },

    setLineDash: function (dash) {
        this.ctx.setLineDash([dash, dash  * this.ctx.ratio]);

    },

    drawRect: function () {


        if (arguments.length === 1)
        {
            let rect = arguments[0];
            this.ctx.strokeRect(rect.x + 0.5, rect.y + 0.5, rect.width, rect.height);
        }
        if (arguments.length === 4)
        {
            this.ctx.strokeRect(arguments[0] + 0.5, arguments[1] + 0.5, arguments[2], arguments[3]);
        }
        return this;
    },

    draw: function ()//等同于drawRect
    {

        return this.drawRect();
    },

    moveTo: function (x, y) {
        this.ctx.moveTo(x + 0.5, y + 0.5);
    },

    lineTo: function (x, y) {
        this.ctx.lineTo(x + 0.5, y + 0.5);

    },

    arc: function (x, y, radius, startRad, endRad, anticlockwise) {
        this.ctx.arc(x + 0.5, y + 0.5, radius, startRad, endRad, anticlockwise);
    },

    drawLine: function (x1, y1, x2, y2) {

        this.moveTo(x1, y1);
        this.lineTo(x2, y2);


    },

    beginPath: function () {
        return this.ctx.beginPath();
    }
    ,
    closePath: function () {
        return this.ctx.closePath();
    }
    ,

    scale: function (xScale, yScale) {

        this.ctx.scale( xScale/100 , yScale/100);
    }
    ,

    clip: function (rc) {


        if (rc == null) return;
        this.currentClipRect = rc.clone();

        if( rc.x==50 && rc.y==50 && rc.width==100 && rc.height==25)
        {
            var a;
            a=1;
        }
        //beginPath()的调用非常重要
        /* 在调用clip()前，多次的rect后，缺省情况下，裁剪区域是这些rect的并集，而不是交集，除非修改
        ctx.globalCompositeOperation属性。但是现在需要的不是并集，也不是交集，而是清除之前的裁剪区，使用新的裁剪区
        所以通过调用beginPath() 来重置当前的路径,现在的裁剪区就是rc，不会受旧裁剪区的影响
        *
        * */

        this.ctx.beginPath();
        this.ctx.rect(rc.x, rc.y, rc.width, rc.height);
       // console.info("裁剪区："+ rc.toString());
        this.ctx.clip();

    }
    ,


    fillRect: function () {
        if (arguments.length === 1)
        {
            let rect = arguments[0];
            this.ctx.fillRect(rect.x, rect.y, rect.width, rect.height);
        }
        if (arguments.length === 4)
        {
            this.ctx.fillRect(arguments[0], arguments[1], arguments[2], arguments[3]);
        }
        return this;
    }
    ,


    stroke: function () {
        this.ctx.stroke();
    }
    ,


    /**该方法用来绘制一个有填充色的圆角矩形
     *@param cxt:canvas的上下文环境
     *@param x:左上角x轴坐标
     *@param y:左上角y轴坐标
     *@param width:矩形的宽度
     *@param height:矩形的高度
     *@param radius:圆的半径
     *@param fillColor:填充颜色
     **/
    fillRoundRect(x, y, width, height, radius)
    {
        //圆的直径必然要小于矩形的宽高
        if (2 * radius > width || 2 * radius > height)
        { radius = Math.floor(Math.min(width / 2, height / 2)); }

        this.drawRoundRectPath(x, y, width, height, radius);

        this.ctx.fill();

    },


    /**该方法用来绘制圆角矩形
     *@param cxt:canvas的上下文环境
     *@param x:左上角x轴坐标
     *@param y:左上角y轴坐标
     *@param width:矩形的宽度
     *@param height:矩形的高度
     *@param radius:圆的半径
     *@param lineWidth:线条粗细
     *@param strokeColor:线条颜色
     **/
    strokeRoundRect: function (x, y, width, height, radius) {
        //圆的直径必然要小于矩形的宽高
        if (2 * radius > width || 2 * radius > height)
        { radius = Math.floor(Math.min(width / 2, height / 2)); }


        //绘制圆角矩形的各个边
        this.drawRoundRectPath(x, y, width, height, radius);

        this.stroke();

    },

    drawRoundRectPath: function (x, y, width, height, radius) {
        this.beginPath();
        //从右下角顺时针绘制，弧度从0到1/2PI
        this.arc(x + width - radius, y + height - radius, radius, 0, Math.PI / 2);

        //矩形下边线
        this.lineTo(x + radius, y + height);

        //左下角圆弧，弧度从1/2PI到PI
        this.arc(x + radius, y + height - radius, radius, Math.PI / 2, Math.PI);

        //矩形左边线
        this.lineTo(x, y + radius);

        //左上角圆弧，弧度从PI到3/2PI
        this.arc(x + radius, y + radius, radius, Math.PI, Math.PI * 3 / 2);

        //上边线
        this.lineTo(x + width - radius, y);

        //右上角圆弧
        this.arc(x + width - radius, y + radius, radius, Math.PI * 3 / 2, Math.PI * 2);

        //右边线
        this.lineTo(x + width, y + height - radius);
        this.closePath();
    },


    /**
     * 绘制图象
     * @param img
     * @param x
     * @param y
     */
    drawImage: function (img, x, y, w, h, x2, y2, w2, h2) {
        this.ctx.drawImage(img, x, y, w, h, x2, y2, w2, h2);

    }
    ,

    /**
     * 多行文字的绘制
     * @param textRect
     * @param str
     * @param bkMode
     * @param bkcolor
     * @param textColor
     * @param FontName
     * @param FontSize
     * @param FontBold
     * @param FontItalic
     * @param FontUnderline
     * @param hAlign
     * @param vAlign
     * @param lineSpacing
     */
    drawStringWordwrap(textRect, str, bkMode, bkcolor, textColor, FontName, FontSize, FontBold,
                       FontItalic, FontUnderline, hAlign, vAlign, lineSpacing)
    {
        /*首先是要把内容拆分成多行
        基于如下原则
        1  先计算一个汉字的宽度，一个字母的宽度，假设是等宽字体（主要是为了速度）
        2 然后逐个字符累加宽度，直到超出 textRect的宽度。这里不考虑英文单词不拆分的问题，
      */

        if (!str) return;

       // console.info("文本区："+ textRect.toString()+"   " + str);

        if (lineSpacing == undefined) lineSpacing = 1;
        //把强制回车，换成回车符
        str = str.replace(/\\n/g, '\n');

        //参数初始化及检测


        this.save();
        this.clip(textRect);

        FontName = FontName || '';
        if (FontName == '') FontName = 'STHeitiSC-Light'; // mac osX上

        if (['FontAwesome', 'physiology'].contains(FontName))
        {
            //TODO 如果str中有单引号，可能需要特别处理一下
            //如果str 是上述字体，那么把字面值  \ue004 这样的字符串，转成对应的字符
            //这里要搞清楚， 字面字符串，与unicode字符串。  '\ue004' 与 '\\ue004'区别
            //在设计面板中直接输入的\ue004 实际上
            str = eval("'" + str + "'");
        }


        FontBold = FontBold || false;
        FontItalic = FontItalic || false;
        FontUnderline = FontUnderline || false;

        //字体设置
        let font = (FontItalic ? "italic" : "normal") + " normal " +
            (FontBold ? "bold" : "normal") + " " + FontSize + "px  " + FontName;
        this.ctx.font = font;
        let strWidth = 0;
        let lx1, lx2, ly;


        let zh_w = this.ctx.measureText("国", font).width;
        let min_zs_one_row = Math.max(1, Math.floor(textRect.width / zh_w)); //最少一个字


        let splitedStr = [];


        //先按回车拆分
        let strs = str.split('\n');
        for (let ti = 0; ti < strs.length; ti++)
        {
            str = strs[ti];
            while (str.length > 0)
            {

                let p = min_zs_one_row;
                //取一段字符，并计算它的实际宽度
                let one = str.substring(0, p);
                let oneWidth = this.ctx.measureText(one, font).width;

                while (true)
                {
                    if (p >= str.length)
                    {
                        splitedStr.push({str: str, width: oneWidth});
                        str = "";
                        break;
                    }

                    let ch = str.substring(p, p + 1);
                    let ch_width = this.ctx.measureText(ch).width;
                    //console.info(ch + "  width: " + ch_width);
                    oneWidth += ch_width;

                    if (oneWidth > textRect.width)
                    {
                        //到p这里，已经超过宽度了，所以要少一个字符进行换行
                        splitedStr.push({str: str.substring(0, p), width: oneWidth - ch_width});
                        str = str.substring(p);
                        break;
                    } else
                    {
                        p++;
                    }
                }
            }

        }


        // if (bkMode == 1)
        // {
        //     this.setFillColor(bkcolor);
        //     this.fillRect(textRect);
        // }
        this.setColor(textColor);
        this.setFillColor(textColor);


        let lineHeight = FontSize; //通常字号就是文字的高度

        let underlineSpacing = 1;
        let yOffset = textRect.y;
        if (vAlign == ALIGN.Center)
        {
            yOffset = textRect.y +
                (  textRect.height - (lineHeight + lineSpacing) * splitedStr.length) / 2;
        }
        if (vAlign == ALIGN.Bottom)
        {
            yOffset = textRect.y +
                ( textRect.height - (lineHeight + lineSpacing) * splitedStr.length);
        }


        for (let i = 0; i < splitedStr.length; i++)
        {

            let x, y;
            let lx1, lx2, ly;

            str = splitedStr[i].str;
            let strWidth = splitedStr[i].width;


            switch (hAlign)
            {
                case ALIGN.Left:
                    this.ctx.textAlign = 'left';
                    x = textRect.x;
                    if (FontUnderline)
                    {
                        lx1 = x;
                        lx2 = x + strWidth;
                    }
                    break;
                case ALIGN.Center:
                    this.ctx.textAlign = 'center';
                    x = textRect.x + textRect.width / 2;

                    if (FontUnderline)
                    {
                        lx1 = x - strWidth / 2;
                        lx2 = x + strWidth / 2;
                    }

                    break;
                case ALIGN.Right:
                    this.ctx.textAlign = 'right';
                    x = textRect.x + textRect.width;

                    if (FontUnderline)
                    {
                        lx1 = x - strWidth;
                        lx2 = x;
                    }
                    break;
            }


            this.ctx.textBaseline = 'top';
            y = yOffset;

            if (FontUnderline)
            {
                ly = y + lineHeight + underlineSpacing;
            }


            // console.info(str + "   width:" + strWidth + " x:" + x + " y:" + y);
            this.ctx.fillText(str, x, y);

            if (FontUnderline)
            {
                this.moveTo(lx1, ly);
                this.lineTo(lx2, ly);
                this.stroke();
            }

            yOffset += (lineHeight + lineSpacing);

        }

        this.restore();

    }
    ,


    /**
     * 单行显示 ，不进制坐标的计算，这样速度会快些
     * @param textRect
     * @param str
     * @param bkMode
     * @param bkcolor
     * @param textColor
     * @param FontName
     * @param FontSize
     * @param FontBold
     * @param FontItalic
     * @param FontUnderline
     * @param hAlign
     * @param vAlign
     */
    drawStringSingleRow(textRect, str, bkMode, bkcolor, textColor, FontName, FontSize, FontBold,
                        FontItalic, FontUnderline, hAlign, vAlign)
    {
        if (!str) return;

        //参数初始化及检测

        //console.info(textRect.toString());
        this.save();
        this.clip(textRect);

        FontName = FontName || '';
        if (FontName == '') FontName = 'STHeitiSC-Light'; // mac osX上

        FontBold = FontBold || false;
        FontItalic = FontItalic || false;
        FontUnderline = FontUnderline || false;

        //字体设置
        let font = (FontItalic ? "italic" : "normal") + " normal " +
            (FontBold ? "bold" : "normal") + " " + FontSize + "px  " + FontName;
        if (this.ctx.font != font) this.ctx.font = font;
        let strWidth = 0;
        let lx1, lx2, ly;

        if (FontUnderline)
        {
            strWidth = this.ctx.measureText(str, font).width;
        }

        let x = textRect.x;
        let y = textRect.y;
        let w = textRect.width;
        let h = textRect.height;

        switch (hAlign)
        {
            case ALIGN.Left:
                this.ctx.textAlign = 'left';
                x = textRect.x;
                if (FontUnderline)
                {
                    lx1 = x;
                    lx2 = x + strWidth;
                }
                break;
            case ALIGN.Center:
                this.ctx.textAlign = 'center';
                x = textRect.x + textRect.width / 2;

                if (FontUnderline)
                {
                    lx1 = x - strWidth / 2;
                    lx2 = x + strWidth / 2;
                }

                break;
            case ALIGN.Right:
                this.ctx.textAlign = 'right';
                x = textRect.x + textRect.width;

                if (FontUnderline)
                {
                    lx1 = x - strWidth;
                    lx2 = x;
                }
                break;
        }

        let lineHeight = FontSize; //通常字号就是文字的高度

        let underlineSpacing = 2;

        switch (vAlign)
        {
            case ALIGN.Top:
                this.ctx.textBaseline = 'top';
                y = textRect.y;

                if (FontUnderline)
                {
                    ly = y + lineHeight + underlineSpacing;
                }

                break;
            case ALIGN.Center:
                this.ctx.textBaseline = 'top';
                y = textRect.y + (h - FontSize) / 2;
                if (FontUnderline)
                {
                    ly = y + underlineSpacing + lineHeight;
                }
                break;
            case ALIGN.Bottom:
                this.ctx.textBaseline = 'bottom';
                y = textRect.y + textRect.height;
                if (FontUnderline)
                {
                    ly = y - 1;
                }
                break;
        }


        /*  if (bkMode == 1)
          {
              this.setFillColor(bkcolor);
              this.fillRect(textRect);
          }

          */

        this.setColor(textColor);
        this.setFillColor(textColor);
        this.ctx.fillText(str, x, y);

        if (FontUnderline)
        {
            this.beginPath();
            this.moveTo(lx1, ly);

            this.lineTo(lx2, ly);
            this.stroke();
        }

        this.restore();

    }


});

export default Graphics2D;
