/**
 * Created by 三宝爹 on 2017/8.
 * 评估表达式的值
 *
 * 本类提供了表达式的解析，主要用于计算列的解析，以及表单公式的解析
 *
 * 直接使用浏览器的 eval函数来评估表达式的值，要解决如下几个问题
 *
 * 1  表达式中的变量的值，如何提供
 * 表达式中的变量，要避免变量的污染，所以使用了一个local  var   obj ，动态把变量加到obj中，
 * obj 本身是一个局部变量，它不会污染外部变量及对象
 *
 * 2  通过拼脚本的方式，用   var  变量名=  obj['变量名'] ，在函数内部通过这一方式解决了不用加 obj. 前缀来访问数据。
 * obj对象做为这个动态函数之外的局部变量，在动态函数内部是可以访问的
 *
 * 3 obj 是指向this.userDefineFunction对象的， 这样，表达式中可以使用自定义函数，而这些自定义函数也不会污染全局函数
 *
 * 4 表单公式有个特点，它可能是对多个单元格进行计算，比如a1:b10 ,  而且一个单元格的数据，它也可能是一个数组（比如绑定一个多行数据源字段），
 * 所以表单公式不能直接用浏览器的eval来评估，因为js语法中，不能对数组进行计算， 比如一个单元格叫 sl ,一个单元格叫dj
 * ,现在要计算 sl * dj  , 而 sl 及dj 都是数组，比如[2 ,3,4] * [3,4,5] 需要对索引号相同的两个数进行* ， 结果为[6,12,20 ]
 * 因此 ，对表单公式的运算，需要自行处理。 难道要自行写一个表达式引擎？不需要。 解决办法是，把表达式转换成函数调用，
 *  比如 上面的 [2 ,3,4] * [3,4,5] 转换成   mul ( [2 ,3,4] ,  [3,4,5] )  ， mul函数就是自定义的乘法计算，当两个参数是简
 *   单数据时，直接相乘，如果是数组，则相同索引下的数据两两相乘，
 *   再拼成一个数组返回。 值得庆幸的是，运算符并不多，最常用的 就是>,<,==,>=,<=,!=,&&,||,+,_.*,/,%,^ 这几个
 *   如果不够用，可以换用自定义函数处理。比如按位与运算，可以自定义一个 bitAnd( a , b )
 *   （ a+b ）* c  转换成  mul ( add (a , b) , c)
 *5 当实现第4步所述之后，必须面对一个问题：函数必须针对简单参数 及数组参数做出相应的处理。即同一个功能函数，必须适应参数是简单
 * 值，以及参数是数组两种情况。比如    plus( 1,2) ,  plus( [1,2,3],[4,5,6]) 这给实现这些功能函数带来成倍的工作量。所以必须有一种机制
 * 来解决这个问题。 $call( func , p1,p2....) 应运而生，它的作用就是当p1,p2..等都是简单数据时，返回func(p1,p2...) 如果 p1,p2....中任意一个
 * 是数组，那么就是   取出 p1,p2 中的相同序号的数据 ，调用  func( p1[i], p2[i]...)  结果拼成一个数组 返回
 *
 * 这样，只需实现针对简单参数的 func即可。这其中有一个例外，就是对于聚合函数而言，无论什么样的参数 ， 最终它都是返回一个简单数据结果。
 */


import Class from '../base/Class.js';
import Util from '../util/Util.js';
import Esprima from '../base/esprima.js';
import FuncUtil from './function/FuncUtil.js';
import OperatorFunc from './function/OperatorFunc.js';
import DateFunc from './function/DateFunc.js';
import DataTransferFunc from './function/DataTransferFunc.js';
import AggregateFunc from './function/AggregateFunc.js';
import MathFunc from './function/MathFunc.js';
import DataStoreFunc from './function/DataStoreFunc.js';
import DBFunc from './function/DBFunc.js';
import StringFunc from './function/StringFunc.js';
import MD5Func from './function/MD5Func.js';
import SysFunc from './function/SysFunc.js';
import PluginFunc from './function/PluginFunc.js';
import WorkSheetFunc from './function/WorkSheetFunc.js';


var Evaluate = Class.extend({
    static: {
        //
        datastore_internal_thisrow: 'datastore_internal_thisrow',
        datastore_internal_rowcount: 'datastore_internal_rowcount',
        datastore_internal_absolute: 'datastore_internal_absolute',

        //运算符对应的函数名
        '>': '_gt_',
        '<': '_lt_',
        '==': '_eq_',
        '>=': '_ge_',
        '<=': '_le_',
        '!=': '_ne_',
        '&&': '_and_',
        '||': '_or_',
        '+': "_plus_",
        '-': "_minus_",
        '*': '_mul_',
        '/': '_div_',
        '%': '_mod_',
        '!': '_not_',
        '^': '_power_'


        //其它静态功能函数


    },
    constructor: function () {
        var timestamp = (new Date()).valueOf();
        this.tempFunctionName = "tempFunctionForEvaluate" + timestamp;
        this.context = {}; //将用于评估的上下文环境
        this._addDefaultFunction();//给上下文环境添加内置的功能函数
    },

    /**
     * 当用在WorkSheet中时，下面两个函数有用
     * @param cell
     */
    attachCell: function (cell) {
        this.clientCell = cell;
    },

    getClientCell: function () {
        return this.clientCell;
    },
    /**
     * 把一个含有四则运算的表达式。转换成纯函数表达式
     * @param expression
     * @param moreVars 传入的更多的变量数据，
     * @returns {string}
     */
    convertFormulaToFunction: function (expression, moreVars) {
        if (expression == null) return "";
        expression = expression.trim();
        if (expression.startsWith("=")) expression = expression.substring(1); // 去掉等号

        var options = {attachComment: false, range: false, loc: false, sourceType: 'script', tokens: false};
        try
        {
            var ast = Esprima.parse(expression, options);
            var root = ast.body[0].expression; //如果是一个合法的表达式，那么肯定就是这样的结构
            var ret = this.ast2string(root, moreVars || {});
            return ret;
        } catch (err)
        {
            return JSON.stringify(err);
        }
    },

    /**
     * 一个语法结点。转换成纯函数表示,比如  1+2 转换成  _add(1,2)
     *
     *  表达式中只允许使用： 1 常数，2变量，3函数， 4一元运算（!  - ）,5二元运算，6逻辑运算 这几种情况
     *  其它的语法超出了表达式的范畴，不需要处理
     * @param node
     */
    ast2string: function (node, moreVars) {
        if (node.type == 'BinaryExpression' || node.type == 'LogicalExpression') //二元运算或逻辑运算
        {
            return '$call( ' + this.operator2Function(node.operator) + '  , ' +
                this.ast2string(node.left, moreVars) + " , " + this.ast2string(node.right, moreVars) + ' ) ';
        }

        if (node.type == 'UnaryExpression') //一元运算，
        {
            return this.operator2Function(node.operator) + '  ( ' + this.ast2string(node.argument, moreVars) + " ) ";
        }


        if (node.type == 'CallExpression')  //函数调用
        {
            var funcName = node.callee.name;
            var args = node.arguments;
            var arg2string = [];
            //如果不是聚合函数，那么把函数也做为参数压入，后面$call 将用第一个参数也就是funcName,来回调
            //但是如果是聚合函数，那么就直接调用它，不需要用$call处理
            if (AggregateFunc[funcName] == undefined) arg2string.push(funcName);
            for (var i = 0; i < args.length; i++)
            {
                arg2string.push(this.ast2string(args[i], moreVars));
            }

            //如果是聚合函数，那么就不需要区分 简单参数还是数组参数 情况， 直接聚合函数返回即可
            if (AggregateFunc[funcName] != undefined) return funcName + ' ( ' + arg2string.join(' , ') + ' ) ';

            //外包一层，$call 会根据参数是不是数组而对 funcName进行一次调用或循环多次调用
            return '$call( ' + arg2string.join(' , ') + ' ) ';


        }


        //变量
        //本函数是针对worksheet.evaluate的，所以公式中的变量，只能是单元格名称或单元格区域
        //此时，在上下文中，必须提供
        // $sheet 表明当前的worksheet ,
        // $innerRow 表明当前内部行号
        //#clientCell 表明公式所在的单元格（用于做循环引用校验）

        if (node.type == 'Identifier')
        {
            if (moreVars[node.name] != undefined) return node.name; //变量优先，如果传入的变量中不包含node.name，那么node.name就是单元格别名或名称

            return "$sheet.getValueForJEP( $clientCell,'" + node.name + "' , $innerRow ) ";
        }

        //立即数
        if (node.type == 'Literal') return node.raw; //如果是字符串，它自带单引号


    },

    /**
     * 运算符转换成相应的函数名
     * @param operator
     */
    operator2Function: function (operator) {
        var ret = this.Clas$[operator];  //注意，运行算映射到函数名， 是静态变量
        if (ret == undefined) ret = operator;
        return ret;

    },

    _addDefaultFunction: function () {

        Util.merge(this.context, OperatorFunc, true);
        Util.merge(this.context, DateFunc, true);
        Util.merge(this.context, AggregateFunc, true);
        Util.merge(this.context, DataTransferFunc, true);
        Util.merge(this.context, MathFunc, true);
        Util.merge(this.context, DataStoreFunc, true);
        Util.merge(this.context, DBFunc, true);
        Util.merge(this.context, StringFunc, true);
        Util.merge(this.context, MD5Func, true);
        Util.merge(this.context, SysFunc, true);
        Util.merge(this.context, PluginFunc, true);
        Util.merge(this.context, WorkSheetFunc, true);


    },

    addUserDefineFunction(funcName, func)
    {
        this.context[funcName] = func;
    },
    /**
     *   根据提供的数据，评估一个表达式的值
     * @param vars        包含属性及值的JSON对象，示例{ a:1 , b:'abc' }
     * @param expression         表达式， 其中用到的变量，必须在 vars中给出数据
     * @returns {Object}
     */
    evaluate: function (vars, expression) {

        //console.info( expression);

        var obj = this.context;


        //Util.merge(obj, vars, true ,false); //复制时不需要克隆，直接指向即可
        //上面的复制，太复杂了，而且第二次，会出问题，算了，简单处理一下
        for (var key in vars)
        {
            obj[key] = vars[key];
        }


        var $call = FuncUtil.$call;


        var script = [];

        script.push("function " + this.tempFunctionName + "() ");
        script.push("{");
        //此处通过局部变量指向数据对象obj 的属性，避免了使用字符串化数据值带来的引号问题
        // 比如   var  name='abc'  ;  就需要在abc外用引号括起来。但是通过 var name= obj['name'];
        // 就避免了这样的处理， 特别在处理Date型时，无法通过字符量方式定义一个Date
        for (var col in obj)
        {
            script.push(" var " + col + " =obj['" + col + "'];"); //不仅定义了变量，也引入了自定义函数及上下文的函数
        }
        // script.push("console.dir(this); ");
        script.push("return " + expression + ";");
        script.push("} ");
        script.push(this.tempFunctionName + "() ");


        try
        {
            return eval(script.join('\n'));
        } catch (err)
        {
            console.dir(err);
            return null;
        }
    }

});

export default Evaluate;
