/**
 * Created by zengjun on 2017/5/4.
 */


import Class from '../base/Class.js';
import ObjectPool from '../util/ObjectPool.js';
import Evaluate from '../eval/Evaluate.js';
import EventManager from '../core/EventManage.js';
import Util from '../util/Util.js';
import Tools from '../util/Tools.js';
import EditStyle from './EditStyle.js';
import ClipBoard from './ClipBoard.js';
import WorkSheet from './WorkSheet.js';
import WorkSheetView from './WorkSheetView.js';
import JD from '../base/jdetects.js';
import WorkBookView from './WorkBookView.js';
import DataSourceConfig from './DataSourceConfig.js';
import DataStoreEventAdapter from './DataStoreEventAdapter.js';
import Property from "./Property.js";
import ajax from '../util/ajax.js';
import ISetEditStyle from './IBook/ISetEditStyle.js';
import IExcel from "./IBook/IExcel.js";
import ICSS from "./IBook/ICSS.js";
import Color from "../gdi/Color.js"

import DataStore from "../db/DataStore.js";
import Physiology from '../plugin/physiology/Physiology.js';
import Map from '../util/Map.js';


var WorkBook = Class.extend({

    components: [
        IExcel, //excel文件读写支持
        ICSS,  //全局样式表支持
        ISetEditStyle  //设置编辑格式，带缓存
    ],

    properties: {
        /**
         * @api {workSheetCount} 只读属性 [属性]workSheetCount
         * @apiName  workSheetCount
         * @apiDescription 工作表的个数
         * @apiGroup WorkBook
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){int} - 工作表的个数
         * @apiExample {js}(示例：)
         *  for( var i=0;i<book.workSheetCount;i++)
         *  {
             *   alert(  book.getWorkSheet(i).getName());
             *  }
         */
        "workSheetCount": {
            get: function () { return this.sheetList.length;}
        },

        /**
         * @api {activeSheet} 只读属性 [属性]activeSheet
         * @apiName  activeSheet
         * @apiDescription 当前活动的工作表对象。一组工作表，通常以tab页方式展示，当前显示的tab页对应的工作表就是活动工作表
         * @apiGroup WorkBook
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){WorkSheet} - 活动的工作表
         * @apiExample {js}(示例：)
         * alert(  book.activeSheet.name );
         */
        "activeSheet": {
            get: function () { return this.ActiveSheet;}
        },

        //最多支持直接20个sheet访问
        /**
         * @api {sheet0} 只读属性 [属性]sheet0
         * @apiName  sheet0
         * @apiDescription 第1个工作表对象，依此类推 sheet1,sheet2,sheet3......sheet19 系统内置了20个工作表对象。
         * 是最多内置了20个工作表对象，并不表示一定有20个。
         * @apiGroup WorkBook
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){WorkSheet} - 工作表
         * @apiExample {js}(示例：)
         * //示例
         * book.sheet0.cells("a1").setValue('abc');
         * //下面语句与上面等效
         * book.getWorkSheet(0).cells("a1").setValue('abc');
         *
         */
        'sheet0': {get: function () {return this.getWorkSheet(0)}},
        'sheet1': {get: function () {return this.getWorkSheet(1)}},
        'sheet2': {get: function () {return this.getWorkSheet(2)}},
        'sheet3': {get: function () {return this.getWorkSheet(3)}},
        'sheet4': {get: function () {return this.getWorkSheet(4)}},
        'sheet5': {get: function () {return this.getWorkSheet(5)}},
        'sheet6': {get: function () {return this.getWorkSheet(6)}},
        'sheet7': {get: function () {return this.getWorkSheet(7)}},
        'sheet8': {get: function () {return this.getWorkSheet(8)}},
        'sheet9': {get: function () {return this.getWorkSheet(9)}},
        'sheet10': {get: function () {return this.getWorkSheet(10)}},
        'sheet11': {get: function () {return this.getWorkSheet(11)}},
        'sheet12': {get: function () {return this.getWorkSheet(12)}},
        'sheet13': {get: function () {return this.getWorkSheet(13)}},
        'sheet14': {get: function () {return this.getWorkSheet(14)}},
        'sheet15': {get: function () {return this.getWorkSheet(15)}},
        'sheet16': {get: function () {return this.getWorkSheet(16)}},
        'sheet17': {get: function () {return this.getWorkSheet(17)}},
        'sheet18': {get: function () {return this.getWorkSheet(18)}},
        'sheet19': {get: function () {return this.getWorkSheet(19)}},


        /**
         * 是否显示字段绑定信息
         */
        "cellShowBind": {
            get: function () { return this.isCellShowBind();},
            set: function (val) { return this.setCellShowBind(val);}
        },

        "cellShowAlias": {
            get: function () { return this.isCellShowAlias();},
            set: function (val) { return this.setCellShowAlias(val);}
        },
        "cellShowFormula": {
            get: function () { return this.isCellShowFormula();},
            set: function (val) { return this.setCellShowFormula(val);}
        },

        "undoEnabled": {
            get: function () { return this.m_undoEnabled;},
            set: function (val) { this.m_undoEnabled = val;}
        },

        "initOK": {
            get: function () {return this.m_initOK;},
            set: function (val) { this.setInitOK(val);}
        },

        /**
         * @api {uploadingCount} 可读可写属性 [属性]uploadingCount
         * @apiName  uploadingCount
         * @apiDescription 正在上传中的文件数
         * @apiGroup WorkBook
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){int} - 正在上传中的文件数
         * @apiExample {js}(示例：)
         * //示例
         *  alert( book.uploadingCount);
         */
        "uploadingCount": {
            get: function () {return this.m_uploadingCount;},
            set: function (val) {this.m_uploadingCount=val;}
        },


        /**
         * @api {showTabOnWhere} 可读可写属性 [属性]showTabOnWhere
         * @apiName  showTabOnWhere
         * @apiDescription 工作表名称tab显示在什么位置，默认是"bottom",可以是如下：
         * <br><br>
         * <ul>
         * <li>'bottom':底部</li>
         * <li>'top':顶部</li>
         * <li>'none':不显示</li>
         * </ul>
         * @apiGroup WorkBook
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){String} - tab的显示位置
         * @apiExample {js}(示例：)
         *  book.showTabOnWhere='top';
         */
        "showTabOnWhere":
            {
                get: function () { return this.m_showTabOnWhere;},
                set: function (val) { this.setShowTabOnWhere(val);}
            },

        /**
         * @api {paintPermit} 只写属性 [属性]paintPermit
         * @apiName  paintPermit
         * @apiDescription 设置所有工作表是否允许绘制
         * @apiGroup WorkBook
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){int} - 设置所有的工作表是否允许绘制。如果是false,表示所有工作表都暂停绘制
         * @apiExample {js}(示例：)
         * book.paintPermit=false;
         */
        "paintPermit":
            {

                set: function (v) { this.setAllSheetsPaintPermit(v);}
            },
        /**
         * 焦点区域亮显
         */
        "focusRangeVisible": {
            get: function () {return this.m_focusRangeVisible;},
            set: function (val) { this.setFocusRangeVisible(val);}
        }


    },


    /**
     config:   示例
     {
         withGUI: true,
         containerID:'xxx',
         sheets:  [ {guid: 'abafse' , code:'' , rowCount:10, columnCount:10,name:'Sheet1' },
                        {guid: 'sfsfsfse' , code:'' , rowCount:10, columnCount:10,name:'Sheet2' }
                      ]
     }
     *
     * @param config
     */
    constructor: function (config) {

        Util.merge(config, {withGUI: false, sheets: []}, false);

        this.evalPool = new ObjectPool(Evaluate, 5);//用来评估公式的值

        this.containerID = config.containerID;

        this.designMode = false; //如果是设计模式，那么拖动tab后，sheet顺序要变，否则只是tab显示顺序改变，但sheet的顺序不变
        /*WorkSheet*/
        this.sheetList = [];
        /*WorkSheet*/
        this.ActiveSheet = null;
        /*WorkBookView*/
        this.View = null;
        this.withGUI = false;
        this.m_undoEnabled = true; //
        this.tabVisible = true;
        /*WorkBookEventManage*/
        this.EM = new EventManager();
        /*PropertyManage*/
        this.PM = []; // 属性管理
        this.propertyCache = {}; //保存及调取单元格属性时用到的缓存
        this.propertyIdList = [];

        this.formDesignLastBorderColor="gray";

        var dp = new Property(true);
        dp.refCount = 999999;//确保它不会被释放掉
        this.PM.push(dp); //缺省的属性对象，
        /*HashMap/*<String, DataSourceConfig>*/
        this.DataSource = {}; //new HashMap/*<String, DataSourceConfig>*/();
        /*HashMap/*<String, EditStyle>*/
        this.EditStyleMap = {}; // new HashMap/*<String, EditStyle>*/();
        /*boolean*/
        this.CellShowBind = true;
        this.CellShowAlias = false;
        this.CellShowFormula = false;

        this.cssMap={ }; // 样式表
        /*ClipBoard*/
        this.clipBoard = null;
        /*LinkedHashMap<String, ContextMenuConfig>*/
        this.contextMenuMap = {}; //= new LinkedHashMap<String, ContextMenuConfig>();

        this.highlightDataSourceCurrentRow = true;
        this.showDefineIfInDesignMode = false; //用在设计中
        this.showTabOnBottom = true;
        this.theme = "";

        this.m_focusRangeVisible = false;
        this.m_showTabOnWhere = 'bottom';
        this.m_uploadingCount=0; //正在上传中的文件数 在 UploadBrick中有用到

        var nd = config.noDebugging || window['noDebugging'];  //允许全局变量设置

        this.options={

            editableCellHighlightBackground:true, //单据可编辑单元格是否亮显     ok
            editableCellHighlightBackgroundColor:new Color('#FFFF001A'), //   ok
            gridAlternantBackgroundColor:new Color('#D8D8D81A'),//多行数据交替背景   ok
            gridFocusRowBackgroundColor:new Color('#336C3340'),//焦点行背景颜色  ok
            gridFocusRowUseUnifiedTextColor:false, // 焦点行文字统一使用指定颜色
            gridFocusRowTextColor:new Color('#FFFFFF'), // 焦点行文字颜色
            gridFocusRowSign:'\uF0DA', // 焦点行首列焦点标记,为空表示不显示 如果要指定，现在只能是FontAwesome css字体中的unicode
            ddlbSign:'\uF107', // 下拉列表标记符号， 现在只能是FontAwesome css字体中的unicode     ok
            dblClickToSortSignColor:new Color('orange')
        };


        if (nd)
        {
            /*
            JD.create({
                once: false,
                onchange: function (status) {
                    if (status == 'on')
                    {
                        window.location.replace( $.base64.atob('anMvc3ByZWFkc2hlZXQvYmFzZS9ub2RlYnVnLmh0bWw='));
                    }

                }
            });
            */

            var checkStatus;

            var element = new Image();
            Object.defineProperty(element, 'id', {
                get: function () {
                    checkStatus = 'on';
                    window.location.replace($.base64.atob('anMvc3ByZWFkc2hlZXQvYmFzZS9ub2RlYnVnLmh0bWw='));
                }
            });

            setTimeout(function check() {
                checkStatus = 'off';
                //下面一句不能注释掉，
                console.dir(element);
                setTimeout(check, 1000);
            }, 1000);


        }

        this.property = new Map(); //通用属性对象，用来盛放用户自定义的一些属性，它是且仅是一个容器

        //  重要修改
        // 设置单元格编辑模式是否需要触发依赖它的单元格进行重新计算
        //缺省是需要的， 比如  b1= $A1  , 那么当A1 设置编辑格式为DDLB后， b1的数据可能是需要变化的
        //便是在初始时，如果因设置编辑格式，导致通知依赖它的单元格重新计算，那么会引起公式计算的值覆盖数据库的值的情况
        //比如：  je= sl*dj  其中 je , sl , dj  是三个单元格， 并且它们都绑定到数据库， 并且 je 设置了公式 （不是计算列回填）=sl*dj
        // 比如当录入 sl =10 , dj =10后， je 计算出现是100，之后手工改成 200, 保存后,当再次打开表单， 当设置sl 编辑格式时，
        // 它触发 je 重新计算，于是 je 又被改成100 ,  从理论上讲100是合理的。 但是实际应用中可能需要这样的功能 ：缺省是按公式进行计算
        // 并且允许手工修改， 当再次打开时， 不能覆盖手工修改的数据，但是当进行数据修改（比如修改sl ）引起重新计算时，又需要把新的je
        // 计算结果填到 je 上。
        // 因此：可以在初始化时，把 setInitOK设置成 false

        // 当单元格即绑定到数据库， 又用公式进行了定义，当在初始化workbook 时
        // 数据检索后，再用公式定义计算关系时，如果公式的结果与数据库中的数据不一至，会用公式的数据
        // 覆盖数据库的数据。 而在初始化时，有时不的希望这样覆盖。可以在初始化时，用setInitOK(false)表明 一下
        // 再在合适的时候setInitOK(true);

        this.m_initOK = true; //缺省是true的，这样做表示缺省情况下，应该保持数据库中的数据与公式的计算结果保持一至

        this.m_allBrickDisabled = false; //　如果设置成true ,则禁止点击所有的小件，通常用来控制初始化时，禁止点击brick


        // 如果是创建可视的，那么创建用于显示的Component
        this.withGUI = config.withGUI;

        if (config.withGUI && config.containerID)
        {
            this.View = new WorkBookView(this);

        }


        var paintPermit = config.paintPermit;
        if (paintPermit == undefined) paintPermit = true;

        for (var i = 0; i < config.sheets.length; i++)
        {

            let sheetConfig = config.sheets[i];
            let guid = sheetConfig.guid;
            if (!guid) guid = Tools.newGUID();
            let code = sheetConfig.code || guid;
            let rowCount = sheetConfig.rowCount || 100;
            let columnCount = sheetConfig.columnCount || 10;
            let name = sheetConfig.name || ('Sheet' + (i + 1));


            this.newWorkSheet(name, rowCount, columnCount, guid, code, paintPermit);

        }


        this.setActiveSheet(0);

    },

    setInitOK: function (val) {
        this.m_initOK = val;

        /*
        if (val)
        {
            var n = this.getWorkSheetCount();
            for (let i = 0; i < n; i++)
            {
                var sheet = this.sheetList[i];

                var RC = sheet.rowCount;
                var CC = sheet.columnCount;
                for (var row = 0; row < RC; row++)
                {
                    for (var col = 0; col < CC; col++)
                    {
                        var cell = sheet.$cells(row, col);
                        if (cell == null) continue;
                        var bricks = cell.getBricks();
                        for (var bi = 0; bi < bricks.length; bi++)
                        {
                            var brick = bricks[i];
                            brick.workBookInitOK();
                        }
                    }
                }


            }
        }*/


    },

    /**
     * @api {getWorkSheetCount} 函数   getWorkSheetCount
     *
     * @apiDescription getWorkSheetCount()
     * <br><br>返回工作表的个数，本函数是属性workSheetCount的getter函数
     *
     * @apiName  getWorkSheetCount
     * @apiGroup WorkBook
     * @apiVersion 1.0.0
     *
     *
     * @apiSuccess (返回值){int} - 返回工作表的个数
     *
     * @apiExample   {js}示例：
     *
     *  for(let i=0;i<book.getWorkSheetCount();i++)
     *  {
     *      alert( book.getWorkSheet(i).name);
     *  }
     */
    getWorkSheetCount: function () {

        return this.sheetList.length;
    },

    /**
     * @api {getSheetIndex} 函数   getSheetIndex
     *
     * @apiDescription getSheetIndex(nameOrCodeOrGuidOrIndex)
     * <br><br> 根据名称，代码，或GUID查找工作表，然后返回它的序号
     *
     * @apiName  getSheetIndex
     * @apiGroup WorkBook
     * @apiVersion 1.0.0
     *
     * @apiParam {int/String} nameOrCodeOrGuidOrIndex 工作表的序号，或GUID或代码，或名称
     *
     *
     * @apiSuccess (返回值){int} - 返回对应工作表的序号
     *
     * @apiExample   {js}示例：
     *
     *  var i= book.getSheetIndex("Sheet1");
     */
    getSheetIndex: function (nameOrCodeOrGuidOrIndex ) {
        if (Util.isInteger(nameOrCodeOrGuidOrIndex))
        {
            if (nameOrCodeOrGuidOrIndex >= 0 && nameOrCodeOrGuidOrIndex < this.sheetList.length) return nameOrCodeOrGuidOrIndex;
            return -1;
        }

        for (var i = 0; i < this.sheetList.length; i++)
        {
            if (this.sheetList[i].name.equalsIgnoreCase(nameOrCodeOrGuidOrIndex)) return i;
            if (this.sheetList[i].guid.equalsIgnoreCase(nameOrCodeOrGuidOrIndex)) return i;
            if (this.sheetList[i].code.equalsIgnoreCase(nameOrCodeOrGuidOrIndex)) return i;
        }
        return -1;

    },

    /**
     *  在指定的forDevice 上查询 nameOrCodeOrGuidOrIndex 是否已经存在，
     *  本函数只运行在设计状态。运行状态下，使用上面的函数
     * @param nameOrCodeOrGuidOrIndex
     * @param device
     * @returns {number|*}
     */
    getSheetIndexForDevice: function (nameOrCodeOrGuidOrIndex , device ) {
        if (Util.isInteger(nameOrCodeOrGuidOrIndex))
        {
            if (nameOrCodeOrGuidOrIndex >= 0 && nameOrCodeOrGuidOrIndex < this.sheetList.length) return nameOrCodeOrGuidOrIndex;
            return -1;
        }

        for (var i = 0; i < this.sheetList.length; i++)
        {
            var sheetDevice= this.sheetList[i].m_forDevice;
            //如果sheetDevice与device 有重叠，那么才继续判断
            //
            if( sheetDevice=='all' || device=='all' || sheetDevice==device )
            {

                if (this.sheetList[i].name.equalsIgnoreCase(nameOrCodeOrGuidOrIndex)) return i;
                if (this.sheetList[i].guid.equalsIgnoreCase(nameOrCodeOrGuidOrIndex)) return i;
                if (this.sheetList[i].code.equalsIgnoreCase(nameOrCodeOrGuidOrIndex)) return i;
            }
        }
        return -1;

    },

    /**
     * @api {getWorkSheet} 函数   getWorkSheet
     *
     * @apiDescription getWorkSheet (index)
     * <br><br> 根据序号返回对应的工作表对象
     *
     * @apiName  getWorkSheet
     * @apiGroup WorkBook
     * @apiVersion 1.0.0
     *
     * @apiParam {int} index 序号，从0开始，0表示第一个工作表
     *
     *
     * @apiSuccess (返回值){WorkSheet} - 根据序号返回对应的工作表对象
     *
     * @apiExample   {js}示例：
     *
     *  var sheet= book.getWorkSheet(0);
     *
     */
    /*WorkSheet*/getWorkSheet: function (index , forDevice) {


        //2021。06.20 运行时，sheet的Name肯定是不允许重复的，但是在设计时， 只需要控制在相同的设备上，name必须唯一即可
        if( forDevice===undefined)
        {
            index = this.getSheetIndex(index); //通常是在运行时
        }else {
            index= this.getSheetIndexForDevice((index , forDevice)); //通常是在设计时
        }

        if (index < 0) return null;
        return this.sheetList[index];
    },

    /**
     * @deprecated  仅为了兼容
     * @param guid
     */
    getWorkSheetByGuid: function (guid) {
        return getWorkSheet(guid);
    },


    /**
     * @api {getActiveSheet} 函数   getActiveSheet
     *
     * @apiDescription getActiveSheet()
     * <br><br> 得到当前活动的工作表，本函数是属性activeSheet的getter函数
     *
     * @apiName  getActiveSheet
     * @apiGroup WorkBook
     * @apiVersion 1.0.0
     *
     *
     *
     * @apiSuccess (返回值){WorkSheet} - 当前活动工作表
     *
     * @apiExample   {js}示例：
     *
     *  alert( book.getActiveSheet().name);
     */
    getActiveSheet: function () {
        return this.ActiveSheet;
    },

    /**
     * 切换激活的sheet
     * @param nameOrCodeOrGuidOrIndex
     */


    /**
     * @api {setActiveSheet} 函数   setActiveSheet
     *
     * @apiDescription setActiveSheet(nameOrCodeOrGuidOrIndex)
     * <br><br> 把指定的工作表设置为激活状态，原激活的工作表转成非激活状态
     *
     * @apiName  setActiveSheet
     * @apiGroup WorkBook
     * @apiVersion 1.0.0
     *
     * @apiParam {int/String} nameOrCodeOrGuidOrIndex 工作表的名称，或代码或GUID或序号
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *  book.setActiveSheet(2);
     *  book.setActiveSheet('Sheet1');
     */
    setActiveSheet: function (nameOrCodeOrGuidOrIndex) {

        var index = this.getSheetIndex(nameOrCodeOrGuidOrIndex);
        if (index < 0) return;

        if (this.ActiveSheet != null) // 如果不是最初的第一次设置
        { // 如果要设置的就是当前活动Sheet，那么不需要做什么
            // 因为当多次点击同一个Sheet的耳朵时，第一次之后的就应该做无实际效果的点击
            if (index == this.ActiveSheet.getIndex()) return;
        }


        var /*WorkSheet*/deActiveSheet = this.ActiveSheet;

        if (this.ActiveSheet != null)
        {
            var r = this.ActiveSheet.getSelection();
            this.View.forceCurrentEditControlGiveUpFocus();
            this.EM.fire("cellFocusLost", [this.ActiveSheet, r.startRow, r.startCol]);
        }

        if (deActiveSheet != null) deActiveSheet.setActive(false);
        this.ActiveSheet = this.sheetList[index];
        this.ActiveSheet.setActive(true);

        //sheet中的单元格中可能会集成一些小件，它们可能是独立的DOM，而非在view 上直接绘制的，需要对它们的可见性做一些处理
        if (deActiveSheet != null) deActiveSheet.bricksShowOrHideWithWorkSheet();
        this.ActiveSheet.bricksShowOrHideWithWorkSheet();

        //触发激活状态改变事件
        r = this.ActiveSheet.getSelection();
        this.EM.fire("cellFocusGained", [this.ActiveSheet, r.startRow, r.startCol]);
        this.EM.fire("sheetActiveStateChanged", [this.ActiveSheet, deActiveSheet]);
    },

    /**
     * @api {setShowTabOnWhere} 函数   setShowTabOnWhere
     *
     * @apiDescription setShowTabOnWhere (val)
     * <br><br> 设置工作表名称tab显示在什么位置，本函数是属性 showTabWhere的setter函数，详情请参看属性 showTabWhere
     *
     * @apiName  setShowTabOnWhere
     * @apiGroup WorkBook
     * @apiVersion 1.0.0
     *
     * @apiParam {String} val 显示位置
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *  book.setShowTabOnWhere('top');
     *  //下面是等效的写法
     *  book.showTabOnWhere='top';
     *
     */
    setShowTabOnWhere: function (val) {
        if (this.m_showTabOnWhere == val) return;
        if (!['top', 'bottom', 'none'].contains(val)) return;
        this.m_showTabOnWhere = val;
        if (this.View != null) this.View.setShowTabOnWhere(val);



    },

    /**
     * @api {setAllSheetsPaintPermit} 函数   setAllSheetsPaintPermit
     *
     * @apiDescription setAllSheetsPaintPermit(b)
     * <br><br> 设置所有工作表是否允许绘制。本函数是属性allSheetsPaintPermit的setter函数。
     * <br><br>当对工作表做大量UI上的调整，或它绑定的数据做大量修改时，如果允许绘制，则UI上的处理会消耗
     * 大量CPU时间，此时可以关闭UI绘制，在最后，再打开UI绘制，刷新UI。
     *
     * @apiName  setAllSheetsPaintPermit
     * @apiGroup WorkBook
     * @apiVersion 1.0.0
     *
     *
     * @apiParam {boolean} b true表示允许绘制，false表示不允许绘制
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *  book.setAllSheetsPaintPermit(false);
     */
    setAllSheetsPaintPermit: function (b) {
        var n = this.getWorkSheetCount();
        for (let i = 0; i < n; i++)
        {
            var sheet = this.sheetList[i];
            sheet.paintPermit = b;
        }
    },


    /**
     * @api {newWorkSheet} 函数   newWorkSheet
     *
     * @apiDescription newWorkSheet
     * <br><br> 插入一个新的工作表
     *
     * @apiName  newWorkSheet ( [name, rowCount, colCount, guid, code, paintPermit])
     * @apiGroup WorkBook
     * @apiVersion 1.0.0
     *
     * @apiParam {String} name 名称，可以由字母，数字，中文，下划线组成 ， 不允许有!符号。可选参数，如是不提供，则系统自动命名
     * @apiParam {int} rowCount 行数。可选参数，默认是10
     * @apiParam {int} colCount 列数。可选参数，默认是10
     * @apiParam {String} guid 唯一标识，可以是任意字符串。  可选参数，默认是随机生成的一个GUID
     * @apiParam {String} code 代码，可以是任意字符串 。可选参数，默认与guid相同
     * @apiParam {boolean} paintPermit 插入后，是否允许绘制。可选参数，默认是true
     *
     * @apiSuccess (返回值){WorkSheet} - 返回新插入的工作表对象
     *
     * @apiExample   {js}示例：
     *
     *  var sheet= book.newWorkSheet();
     *
     */
    newWorkSheet(name, rowCount, colCount, guid, code, paintPermit)
    {

        name = name || ('Sheet' + (this.sheetList.length + 1));
        rowCount = rowCount || 10;
        colCount = colCount || 10;
        if (!guid) guid = Tools.newGUID();
        if (!code) code = guid;

        if (paintPermit == undefined) paintPermit = true;

        var /*WorkSheet*/sheet = new WorkSheet(this, name, guid, code, true, rowCount, colCount, paintPermit);


        this.sheetList.push(sheet);
        this.EM.fire("afterNewWorkSheet", [sheet]);
        return sheet;
    },


    /**
     * @api {deleteWorkSheet} 函数   deleteWorkSheet
     *
     * @apiDescription deleteWorkSheet(nameOrIndex)
     * <br><br>
     *
     * @apiName  deleteWorkSheet
     * @apiGroup WorkBook
     * @apiVersion 1.0.0
     *
     * @apiParam {int/String} nameOrIndex 序号或工作表名称
     *
     * @apiSuccess (返回值){boolean} - true表示删除成功
     *
     * @apiExample   {js}示例：
     *
     *  book.deleteWorkSheet(0);
     *
     */
    /*boolean*/  deleteWorkSheet: function (nameOrIndex) {
        //至少必须存在一个WorkSheet
        if (this.getWorkSheetCount() <= 1) return false;

        var /*WorkSheet*/sheet = this.getWorkSheet(nameOrIndex);
        if (sheet == null) return false;

        if (!this.EM.fire("workSheetDeletePermit", [sheet], true)) return false;

        var index = sheet.getIndex();

        var view = this.getWorkBookView();
        if (view != null)
        {
            view.removeTab(sheet.guid);
            view.removeWorkSheetView(sheet.getWorkSheetView());
        }

        this.sheetList.remove(sheet);

        //检查公式定义
        //TODO this.RebuildAllDefine(sheet, new RedefineBecause(RedefineBecause.RebuildDefineBecauseSheetDeleted, 0, 0, 0, 0));

        //激活一个新的WorkSheet
        this.ActiveSheet = null; //当前活动SHeet设置成NULL
        if (index >= this.sheetList.length) index--;
        this.setActiveSheet(index);

        return true;
    },

    /*WorkBookView*/  getWorkBookView: function () {
        return this.View;
    },

    /**
     * @api {getWorkSheetIndex} 函数   getWorkSheetIndex
     *
     * @apiDescription getWorkSheetIndex(sheet)
     * <br><br> 得到指定sheet的顺序号（从0开始计数）
     *
     * @apiName  getWorkSheetIndex
     * @apiGroup WorkBook
     * @apiVersion 1.0.0
     *
     * @apiParam {WorkSheet} sheet 工作表对象
     *
     * @apiSuccess (返回值){int} - sheet的序号
     *
     * @apiExample   {js}示例：
     *
     *  let  i= book.getWorkSheetIndex( book.sheet0 );
     *
     */
    getWorkSheetIndex: function (sheet) {

        return this.sheetList.indexOf(sheet);
    },


    /**
     * @api {setWorkSheetIndex} 函数   setWorkSheetIndex
     *
     * @apiDescription setWorkSheetIndex(sheet, newIndex)
     * <br><br> 此函数用在调整Sheet的顺序
     *
     * @apiName  setWorkSheetIndex
     * @apiGroup WorkBook
     * @apiVersion 1.0.0
     *
     * @apiParam {WorkSheet} sheet 工作表对象
     * @apiParam {int} newIndex 新的序号
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *  book.setWorkSheetIndex( book.sheet0 , 2);
     *
     */
    setWorkSheetIndex: function (sheet, newIndex) {
        if (this.getWorkSheetIndex(sheet) == newIndex) return;
        if (newIndex < 0) return;
        if (newIndex >= this.getWorkSheetCount()) return;

        this.sheetList.remove(sheet);

        this.sheetList.splice(newIndex, 0, sheet);


    },


    /**
     * @api {addDataSource} 函数   addDataSource
     *
     * @apiDescription addDataSource(name, ds, dsType)
     * <br><br> 添加一个数据结果集
     *
     * @apiName  addDataSource
     * @apiGroup WorkBook
     * @apiVersion 1.0.0
     *
     * @apiParam {String} name 结果集的名称
     * @apiParam {DataStore} ds 一个结果集对象
     * @apiParam {int} dsType  结果集对象绑定类型
     * <ul>
     *     <li>1:单行结果集</li>
     *     <li>2:多行结果集</li>
     *     <li>4:多行结果集的当前行</li>
     * </ul>
     *
     * @apiSuccess (返回值){int} - 返回本次检索出的数据条数
     *
     * @apiExample   {js}示例：
     *
     * //将结果集绑定到UI上
     *  var ds=newDataStore("","select id, name,code from t1");
     *  book.addDataSource("ds1",ds , 2);
     *  book.sheet0.cells("a1").setBind("ds1", "id"); //绑定到ID字段
     *  ds.retrieve();
     *
     *
     */
    /*boolean*/addDataSource(name, ds, dsType)
    {
        name = name.trim().toLowerCase();
        if (this.DataSource[name] != undefined) return false;
        var /*DataSourceConfig*/  dsc = new DataSourceConfig(name, ds, dsType);

        this.DataSource[name] = dsc;
        // 新建一个对该ds的事件接收器
        var /*DataStoreEventAdapter*/ dsa = new DataStoreEventAdapter(this, dsc);
        // 注意，，此事件的优先级应该小一点
        //因为 ds 自已会有事件侦听，它自已的优先级应该高一些 ,解决ds在inerrt后，应该先触发自已的 afterInsert ,再触发界面响应事件
        var eventLevel = 10;

        ds.EM.addListener(dsa, eventLevel, name); // 指定一个名字，方便注销 ,

        return true;

    },

    /**
     * @api {removeDataSource} 函数   removeDataSource
     *
     * @apiDescription removeDataSource(name)
     * <br><br>移除数据源。移除数据源，所有绑定在其上的单元格都会自动取消绑定
     *
     * @apiName  removeDataSource
     * @apiGroup WorkBook
     * @apiVersion 1.0.0
     *
     * @apiParam {String} name 数据源名称
     *
     *
     * @apiSuccess (返回值){boolean} - 是否移除成功
     *
     * @apiExample   {js}示例：
     *
     *  book.removeDataSource('query_1');
     */

    removeDataSource: function (name) {
        name = name.trim().toLowerCase();
        if (this.DataSource[name] == undefined) return false;
        var /*DataSourceConfig*/  dsc = this.DataSource[name];

        var /*DataStoreEventAdapter*/ dsa = dsc.getDataStore().EM.findListener(name);
        // 取消所有单元格的绑定设置
        dsa.UnRegisterAllCells();
        // 注销事件接收
        dsc.getDataStore().EM.removeListener(name);
        // 从数据源中也删除它
        //需要验证下面的删除
        delete this.DataSource[name];
        return true;
    },


    /*boolean*/addEditStyle(/*EditStyle*/   editStyle)
    {
        var name = editStyle.getName().toLowerCase().trim();
        if (this.EditStyleMap[name]) return false;

        this.EditStyleMap[name] = editStyle;
        return true;

    },

    /*    EditStyle*/
    getEditStyle: function (editStyleName) {
        editStyleName = editStyleName.toLowerCase().trim();
        if (this.EditStyleMap[editStyleName] == undefined) return null;
        return this.EditStyleMap[editStyleName];
    },

    RebuildAllDefine: function (/*WorkSheet*/trigger, /*RedefineBecause*/ because) {
        for (var i = 0; i < this.getWorkSheetCount(); i++)
        {
            var /*WorkSheet*/  sheet = this.sheetList[i];
            sheet.rebuildCellsDefine(trigger, because);
        }
    },

    /*ClipBoard*/
    getClipBoard: function () {

        return this.clipBoard;
    },


    setClipBoard: function (cb) {
        this.clipBoard = cb;

    },


    /**
     * @api {getDataSources} 函数   getDataSources
     *
     * @apiDescription getDataSources()
     * <br><br> 得到所有数据源
     *
     * @apiName  getDataSources
     * @apiGroup WorkBook
     * @apiVersion 1.0.0
     *
     *
     *
     * @apiSuccess (返回值){Array} - 返回一个数据源数组
     *
     * @apiExample   {js}示例：
     *
     *  var  dss= book.getDataSources();
     *  for( var i=0;i<dss.length;i++)
     *  {
         *  alert(  dss[i].getName());
         *  }
     */
    getDataSources: function () {
        var /*DataSourceConfig[]*/  ret = [];
        var c = this.getDataSourceCount();
        if (c == 0) return ret;

        for (var name in this.DataSource)
        {
            var dsc = this.DataSource[name];
            ret.push(dsc);
        }

        return ret;
    },


    /**

     /**
     * @api {getDataSourceCount} 函数   getDataSourceCount
     *
     * @apiDescription getDataSourceCount()
     * <br><br>  得到数据源对象的个数
     *
     * @apiName  getDataSourceCount
     * @apiGroup WorkBook
     * @apiVersion 1.0.0
     *
     *
     *
     * @apiSuccess (返回值){int} - 数据源个数
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     */
    getDataSourceCount: function () {
        return Object.getOwnPropertyNames(this.DataSource).length;
    },


    /**
     * @api {getDataSourceName} 函数   getDataSourceName
     *
     * @apiDescription getDataSourceName(i)
     * <br><br> 得到第i个数据源的名称
     *
     * @apiName  getDataSourceName
     * @apiGroup WorkBook
     * @apiVersion 1.0.0
     *
     * @apiParam {int} i 序号
     *
     * @apiSuccess (返回值){String} - 得到第i个数据源的名称
     *
     * @apiExample   {js}示例：
     *
     *  var name= book.getDataSourceName(0);
     *
     */
    getDataSourceName: function (i) {
        var count = 0;
        for (var name in this.DataSource)
        {
            if (this.DataSource.hasOwnProperty(name))
            {

                if (count == i) return name;
                count++;
            }
        }
        return '';
    },


    /**
     * @api {getDataSource} 函数   getDataSource
     *
     * @apiDescription getDataSource(i)
     * <br><br>  得到某个个数据源对象
     *
     * @apiName  getDataSource
     * @apiGroup WorkBook
     * @apiVersion 1.0.0
     *
     * @apiParam {int/String} i  序号或数据源名称
     *
     * @apiSuccess (返回值){DataSourceConfig} - 得到某个数据源对象
     *
     * @apiExample   {js}示例：
     *
     * var dsc= book.getDataSoure("ds1");
     * dsc= book.getDataSource(0);
     *
     */
    getDataSource: function (i) {

        if (Util.isString(i))
        {
            var name = i;
            name = name.trim().toLowerCase();
            if (!this.DataSource[name]) return null;
            return this.DataSource[name];
        }

        var count = 0;
        for (var name in this.DataSource)
        {
            if (this.DataSource.hasOwnProperty(name))
            {

                if (count == i) return this.DataSource[name];
                count++;
            }
        }
        return null;
    },

    setCellShowBind: function (show) {
        this.CellShowBind = show;
        if (this.View != null) this.getActiveSheet().View.repaint();
    },


    isCellShowBind: function () {
        return this.CellShowBind;
    },

    setCellShowAlias: function (show) {
        this.CellShowAlias = show;
        if (this.View != null) this.getActiveSheet().View.repaint();
    },


    isCellShowAlias: function () {
        return this.CellShowAlias;
    },
    setCellShowFormula: function (show) {
        this.CellShowFormula = show;
        if (this.View != null) this.getActiveSheet().View.repaint();
    },


    isCellShowFormula: function () {
        return this.CellShowFormula;
    },


    getContextMenuMap: function () {
        return this.contextMenuMap;
    },


    isHighlightDataSourceCurrentRow: function () {

        return this.highlightDataSourceCurrentRow;
    },

    setHighlightDataSourceCurrentRow: function (hightlight) {
        this.highlightDataSourceCurrentRow = hightlight;
        this.getActiveSheet().View.repaint();
    },

    isShowDefineIfInDesignMode: function () {

        return this.showDefineIfInDesignMode;
    },

    setShowDefineIfInDesignMode: function (yesOrNot) {
        this.showDefineIfInDesignMode = yesOrNot;
        if (this.withGUI)
        {
            for (var i = 0; i < this.getWorkSheetCount(); i++)
            {
                this.getWorkSheet(i).getWorkSheetView().repaint();
            }
        }

    },

    loadImageToDataSource: function (dsn, row, col) {
        //TODO

    },


    /**
     * @api {newEditStyle} 函数   newEditStyle
     *
     * @apiDescription newEditStyle (name, config)
     * <br><br>  新建一个编辑格式
     *
     * @apiName  newEditStyle
     * @apiGroup WorkBook
     * @apiVersion 1.0.0
     *
     * @apiParam {String} name 编辑格式名称
     * @apiParam {Object} config 编辑格式的配置
     *
     * @apiSuccess (返回值){Object} - 编辑格式对象
     *
     * @apiExample   {js}示例：
     *
     *   var es5=book.newEditStyle('radio&checkbox',
     {
         type:8,
         valueMustInDDLB:true,
         dropDownListEditable :true,
         dropDownData : {dataValue:vvv,showText:availableTags},
         columnCount:3,
         widthPercent:200

     });



     book.sheet0.cells('a3').setEditStyle(es5);
     book.sheet0.cells('a3').editable=true;

     */
    newEditStyle: function (name, config) {
        if (this.EditStyleMap[name] != undefined) return this.EditStyleMap[name];

        var /*EditStyle*/   es = new EditStyle(this, name, config);
        this.EditStyleMap[name] = es;
        return es;
    },


    /*ContextMenuConfig*/
    findContextMenu: function (map, name) {
        if (map == null) return null;
        if (map[name] != undefined) return map[name];
        for (var s in map)
        {
            var cmc = map[s];
            var ret = this.findContextMenu(cmc.subMenu, name);
            if (ret != null) return ret;
        }
        return null;
    },

    /**
     * @api {addContextMenu} 函数   addContextMenu
     *
     * @apiDescription  addContextMenu(name, caption, parentMenuName)
     * <br><br> 增加右键菜单项。当点击该菜单项时，触发   {name}_onClick事件
     *
     * @apiName  addContextMenu
     * @apiGroup WorkBook
     * @apiVersion 1.0.0
     *
     * @apiParam {String} name 菜单名称 ，通常英文字母，数字，下划线
     * @apiParam {String} caption 菜单文字
     * @apiParam {String} parentMenuName 上级菜单名称。如果为空字符串'',表示它就是一级菜单，没有上级
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *  book.addContextMenu("menu1" , "我的菜单","");
     *
     *
     */
    addContextMenu: function (name, caption, parentMenuName) {

        if (this.contextMenuMap == null) this.contextMenuMap = {};

        if (findContextMenu(this.contextMenuMap, name) != null)
        {
            console.log(name + "菜单已经存在 ");
            return;
        }

        var parentMap = null;
        var /*ContextMenuConfig*/ pmc = null;
        if (parentMenuName == "")
        {
            parentMap = this.contextMenuMap;
            pmc = null;
        } else
        {
            pmc = this.findContextMenu(this.contextMenuMap, parentMenuName);
            if (pmc == null)
            {
                console.log(parentMenuName + "菜单不存在，无法添加子菜单 ");
                return;
            }
            if (pmc['subMenu'] == undefined) pmc.subMenu = {};
            parentMap = pmc.subMenu;

        }

        var cmc = {
            ame: name,
            caption: caption,
            enabled: true,
            parentMenu: pmc
        }

        parentMap[name] = cmc;

    },


    /**
     * @api {removeContextMenu} 函数   removeContextMenu
     *
     * @apiDescription removeContextMenu(name)
     * <br><br>移除右键菜单
     *
     * @apiName  removeContextMenu
     * @apiGroup WorkBook
     * @apiVersion 1.0.0
     *
     * @apiParam {String} name 要移除的菜单名称
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     */
    removeContextMenu: function (name) {
        var pmc = this.findContextMenu(this.contextMenuMap, name);

        if (pmc == null) return;

        delete pmc.parentMenu.subMenu[name];
        pmc.parentMenu = null;  //断开父指针
        pmc.subMenu = null;//断开子指针

    },

    /**
     * @api {setContextMenuEnabled} 函数   setContextMenuEnabled
     *
     * @apiDescription setContextMenuEnabled(name, enabled)
     * <br><br>设置右键菜单是否可以点击
     *
     * @apiName  setContextMenuEnabled
     * @apiGroup WorkBook
     * @apiVersion 1.0.0
     *
     * @apiParam {String} name 菜单名称
     * @apiParam {boolean} enabled 是否可以点击
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     */
    setContextMenuEnabled: function (name, enabled) {
        var cmc = this.findContextMenu(this.contextMenuMap, name);
        cmc.enabled = enabled;

    },

    /**
     * @api {isContextMenuEnabled} 函数   isContextMenuEnabled
     *
     * @apiDescription isContextMenuEnabled(name)
     * <br><br> name右键菜单是否可以点击
     *
     * @apiName  isContextMenuEnabled
     * @apiGroup WorkBook
     * @apiVersion 1.0.0
     *
     * @apiParam {String} name 菜单名称
     *
     * @apiSuccess (返回值){boolean} - 返回name右键菜单是否可以点击
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     */
    isContextMenuEnabled: function (name) {
        var cmc = this.findContextMenu(this.contextMenuMap, name);
        if (cmc == null) return false;
        return cmc.enabled;

    },

    setFocusRangeVisible: function (v) {
        if (this.m_focusRangeVisible == v) return;
        this.m_focusRangeVisible = v;
        this.getActiveSheet().getWorkSheetView().repaint();

    },

    isFocusRangeVisible: function () {

        return this.m_focusRangeVisible;
    },


    isAllBrickDisabled()
    {
        return this.m_allBrickDisabled;
    },

    /**
     * @api {setAllBrickDisabled} 函数   setAllBrickDisabled
     *
     * @apiDescription setAllBrickDisabled( allBrickDisabled)
     * <br><br>所有控件禁用吗
     *
     * @apiName  setAllBrickDisabled
     * @apiGroup WorkBook
     * @apiVersion 1.0.0
     *
     *
     * @apiParam {boolean} allBrickDisabled true表示禁用工作本上的所有控件
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     */
    setAllBrickDisabled: function (allBrickDisabled) {
        this.m_allBrickDisabled = allBrickDisabled;

        this.getActiveSheet().getWorkSheetView().repaint();


        /*
         //如果有内嵌sheet,也要重绘
         if (sheet.cellPanelList != null)
         {
         {
         for (int i = 0;
         i < sheet.cellPanelList.size();
         i++
         )
         {
         WorkSheetView
         view = sheet.cellPanelList.get(i).innerSheetView;
         view.doLayout();
         view.repaint();
         }
         }

         }
         */

    },


    newPhysiology: function (name, rsss, mzks, ssks, ssjs, mzjs, csss) {
        var physiology = new Physiology(name, rsss, mzks, ssks, ssjs, mzjs, csss);
        this.property.put(name, physiology);


    },


    runTask: function (taskCode, param, success) {
        var ret = '';
        var asyncCall = false;//缺省是同步调用
        if (!success)
        {
            success = function (data) {ret = data;};
        } else
        {
            asyncCall = true; //异步
        }


        // runTask 就应该移出本控件 ，临时先这么用着
        //2021.01.09 把当前表单上的一些信息也附带上 ，这个偶合得有点过了，暂时先这样，
        var header={};
        try
        {
            header.templateid = form.configInfo.templateid;
            header.currentGUID= $('meta[name="currentGUID"]').attr('content'); //这个是密文，防止前端非法注入数据
            header.moduleDefineKey= $('meta[name="moduleDefine"]').attr('content');
            header.authorizationURL= $('meta[name="AuthorizationURL"]').attr('content');
            header.authorizationTemplate= $('meta[name="AuthorizationTemplate"]').attr('content');
        }catch(err)
        {

        }



    ajax.post('runTask?task=' + taskCode, param, success, 'text', asyncCall ,header);
        return ret;
    },

    newDataStore: function (connection, sql, dataProvider, dialect, columnInfo) {

        //客户端创建dataStore时，需要进行安全的校验 , 需要templateId 中的安全配置信息。
        // newQueryId与 templateId都做了加密处理，防止客户端伪造。
        var ds = new DataStore(connection, sql, dataProvider, dialect, columnInfo, this.newQueryId, this.templateId);

        return ds;
    },

    fire: function (eventName, args, returnValueUnEqualToBreak) {

        return this.EM.fire(eventName, args, returnValueUnEqualToBreak);
    },

    setVisible:function(v)
    {
        this.View.setVisible(v);
    }


});

export default WorkBook ;
