/**
 * 单元格核心对象
 * Created by zengjun on 2017/4/26.
 */

import Color from '../gdi/Color.js';
import Class from '../base/Class.js';
import Range from './Range.js';
import ActionTool from '../mouse/ActionTool.js';
import UniformDataType from './UniformDataType.js';
import Util from '../util/Util.js';
import ObjectTool from '../db/ObjectTool.js';
import ALIGNMENT from './ALIGNMENT.js';
import EditStyle from './EditStyle.js';
import DataSourceConfig from './DataSourceConfig.js';
import DBBindConfig from './DBBindConfig.js';
import Rectangle from '../gdi/Rectangle.js';
import Point from '../gdi/Point.js';
import MathFunc from '../eval/function/MathFunc.js';
import Affect from "./Affect.js";
import Property from "./Property.js";
import RowPropertyManage from './RowPropertyManage.js';
import Tools from "../util/Tools.js";

import EChartBrick from '../brick/EChartBrick.js';
import HTMLBrick from "../brick/HTMLBrick.js";
import DBPageBrick from "../brick/DBPageBrick.js";
import Button from '../brick/Button.js';
import TreeBrick from '../brick/TreeBrick.js';
import SheetContainer from '../brick/SheetContainer.js';
import TabBrick from '../brick/TabBrick.js';
import CKEditorBrick from '../brick/CKEditorBrick.js';
import UploadBrick from "../brick/UploadBrick.js";
import ImageUtil from "../util/ImageUtil.js";
import NumericEdit from '../edit/NumericEdit.js';


//
import ItemStatus from '../db/ItemStatus.js';

import CMD_SetValue from '../cmd/CMD_SetValue.js';
import CMD_SetProperty from '../cmd/CMD_SetProperty.js';
import ObjectType from "../db/ObjectType.js";

let Cell = Class.extend({

    static:
        {
            //都以 $#{开头，这样在draw时，可以先统一预判断，避免每个单元格都针对每个特别类型都做判断
            $RichData_Image: "$#{image}#$",
            $RichData_URLImage: "$#{urlimage}#$",

            $RichData_URL: "$#{<url>",
            $RichData_BARCODE: "$#{{$zexcel_barcode:true,",


            $InnerAction_URL: "url",

            $Physiology_dataArea: "$#{physiology-dataArea:}#$",

            $Physiology_dateAxis: "$#{physiology-dateAxis}#$", // 生理指标-时间轴
            $Physiology_yAxis_C: "$#{physiology-yAxis-C}#$",  // 生理指标-体温
            $Physiology_yAxis_mmHg: "$#{physiology-yAxis-mmHg}#$",  // 生理指标-血压

            $iframe_html2img: null
        },


    properties:
        {
            /**
             * @api {alias} 可读可写属性 [属性]alias
             * @apiName  alias
             * @apiDescription 单元格的别名
             * @apiGroup Cell
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){int} - 单元格的别名
             * @apiExample {js}(示例：)
             *
             * sheet.cells("a1").alias="sl";
             * alert( sheet.cells("a1").alias);
             */
            "alias": {
                get: function () {
                    return this.getAlias();
                },
                set: function (val) {
                    return this.setAlias(val);
                }
            },


            /**
             * @api {name} 只读属性 [属性]name
             * @apiName  name
             * @apiDescription 单元格的名称
             * @apiGroup Cell
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){String} - 单元格的名称
             * @apiExample {js}(示例：)
             *  alert( sheet.cells( 3,4).name);
             */
            "name":
                {
                    get: function () {
                        return this.getName();
                    }
                },

            /**
             * @api {rowIndex} 只读属性 [属性]rowIndex
             * @apiName  rowIndex
             * @apiDescription 单元格的行号
             * @apiGroup Cell
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){int} - 单元格的行号
             * @apiExample {js}(示例：)
             * let row= sheet.cells("sl").rowIndex;
             *
             */
            "rowIndex":
                {
                    get: function () {
                        return this.getRowIndex();
                    }
                },

            /**
             * @api {columnIndex} 只读属性 [属性]columnIndex
             * @apiName  columnIndex
             * @apiDescription 单元格的列号
             * @apiGroup Cell
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){int} - 单元格的列号
             * @apiExample {js}(示例：)
             * let row= sheet.cells("sl").columnIndex;
             *
             */
            "columnIndex": {
                get: function () {
                    return this.getColumnIndex();
                }
            },

            /**
             * @api {define} 只读属性 [属性]define
             * @apiName  define
             * @apiDescription 单元格的定义，文本内容或公式定义
             * @apiGroup Cell
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){String} - 单元格的定义，文本内容或公式定义
             * @apiExample {js}(示例：)
             * //示例
             */
            "define"://用于设计
                {
                    get: function () {
                        return this.Define == null ? '' : this.Define;
                    }
                },

            /**
             * @api {bind} 只读属性 [属性]bind
             * @apiName  bind
             * @apiDescription 单元格的绑定信息
             * @apiGroup Cell
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){Object} - 单元格的绑定信息
             * @apiExample {js}(示例：)
             * //示例
             */
            "bind":
                {
                    get: function () {
                        return this.Bind;
                    }
                },

            /**
             * @api {decorate} 可读可写属性 [属性]decorate
             * @apiName  decorate
             * @apiDescription 单元格的装饰
             * @apiGroup Cell
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){Object} - 单元格的边框装饰配置，是一组CSS属性的组合用来描述边框装饰
             * @apiExample {js}(示例：)
             *
             * sheet.cells("a1").decorate={ "border":"1px solid red"};
             *
             */
            "decorate": {
                get: function () {
                    return this.m_decorate == null ? {} : this.m_decorate;
                },
                set: function (val) {
                    return this.m_decorate = val;
                }
            },


            /**
             * @api {merged} 只读属性 [属性]merged
             * @apiName  merged
             * @apiDescription 单元格是否有合并
             * @apiGroup Cell
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){boolean} - 单元格是否有合并
             * @apiExample {js}(示例：)
             * //示例
             */
            "merged":
                {
                    get: function () {
                        return this.isMerged()
                    }
                },

            /**
             * @api {leftTopCorner} 可读可写属性 [属性]leftTopCorner
             * @apiName  leftTopCorner
             * @apiDescription 单元格的合并区域左上角的单元格对象
             * @apiGroup Cell
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){Cell} - 单元格的合并区域左上角的单元格对象
             * @apiExample {js}(示例：)
             * //示例
             */
            "leftTopCorner":
                {
                    get: function () {
                        return this.getLeftTopCorner();
                    }
                },

            /**
             * @api {rightBottomCorner} 只读属性 [属性]rightBottomCorner
             * @apiName  rightBottomCorner
             * @apiDescription 单元格的合并区域右下角的单元格对象
             * @apiGroup Cell
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){Cell} - 单元格的合并区域右下角的单元格对象
             * @apiExample {js}(示例：)
             * //示例
             */
            "rightBottomCorner":
                {
                    get: function () {
                        return this.getRightBottomCorner();
                    }
                },

            /**
             * @api {visible} 可读可写属性 [属性]visible
             * @apiName  visible
             * @apiDescription 单元格是否可见
             * @apiGroup Cell
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){boolean} - 单元格是否可见
             * @apiExample {js}(示例：)
             * //示例
             */
            "visible":
                {
                    get: function () {
                        return this.Visible;
                    },
                    set: function (val) {
                        this.Visible = val, this.repaint();
                    }

                },
            /**
             * @api {css} 可读可写属性 [属性]css
             * @apiName  css
             * @apiDescription 单元格使用全局的css样式表
             * @apiGroup Cell
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){boolean} - 单元格是否可见
             * @apiExample {js}(示例：)
             * //示例
             */
            "css":
                {
                    get: function () {
                        return this.cssName;
                    },
                    set: function (val) {
                        this.cssName = val;
                        this.repaint();
                    }

                },

            /**
             * @api {clickable} 可读可写属性 [属性]clickable
             * @apiName  clickable
             * @apiDescription 单元格是否允许触发左键点击事件,左键双击事件，右键单击事件 。 当本属性为false时，
             * 此3种事件不触发。注意仅仅是不触发用户事件，组件本身的功能不受影响
             * @apiGroup Cell
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){boolean} - 单元格是否允许触发点击事件
             * @apiExample {js}(示例：)
             * //示例
             *   cell.clickable=false;
             */
            "clickable":
                {
                    get: function () {
                        return this.Clickable;
                    },
                    set: function (val) {
                        this.Clickable = val;
                    }
                },

            /**
             * @api {editable} 可读可写属性 [属性]editable
             * @apiName  editable
             * @apiDescription 单元格是否允许编辑
             * @apiGroup Cell
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){boolean} - 单元格是否允许编辑
             * @apiExample {js}(示例：)
             * //示例
             */
            "editable":
                {
                    get: function () {
                        return this.isEditable();
                    },
                    set: function (val) {
                        this.setEditable(val, false);
                    }
                },

            /**
             * @api {dblClickSort} 可读可写属性 [属性]dblClickSort
             * @apiName  dblClickSort
             * @apiDescription 双击单元格时，重新检索指定结果集的数据，并按指定字段排序升序或降序排序
             * @apiGroup Cell
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){String} - 结果集.字段名
             * @apiExample {js}(示例：)
             * //示例
             *
             *  cell.dblClickSort="query1.col1";
             *
             */
            "dblClickSort":
                {
                    get: function () {
                        return this.getDblClickSortDefine();
                    },
                    set: function (val) {
                        this.setDblClickSortDefine(val);
                    }
                },
            /**
             * @api {viewAs} 可读可写属性 [属性]viewAs
             * @apiName  viewAs
             * @apiDescription 单元格内容显示风格
             * @apiGroup Cell
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){String} - 单元格显示风格
             * @apiExample {js}(示例：)
             * //示例
             */
            "viewAs":
                {
                    get: function () {
                        return this.ViewAs;
                    },
                    set: function (val) {
                        this.ViewAs = val;
                        this.repaint();
                    }
                },


            /**
             * @api {prefix} 可读可写属性 [属性]prefix
             * @apiName  prefix
             * @apiDescription 单元格内容显示时增加前缀
             * @apiGroup Cell
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){String} - 单元格内容显示时增加前缀
             * @apiExample {js}(示例：)
             * //示例
             */
            "prefix":
                {
                    get: function () {
                        return this.Prefix;
                    },
                    set: function (val) {
                        this.Prefix = val;
                        this.repaint();
                    }
                },

            /**
             * @api {suffix} 可读可写属性 [属性]suffix
             * @apiName  suffix
             * @apiDescription 单元格内容显示时增加后缀
             * @apiGroup Cell
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){String} - 单元格内容显示时增加后缀
             * @apiExample {js}(示例：)
             * //示例
             */
            "suffix":
                {
                    get: function () {
                        return this.Suffix;
                    },
                    set: function (val) {
                        this.Suffix = val;
                        this.repaint();
                    }
                },

            /**
             * @api {drawInScript} 可读可写属性 [属性]drawInScript
             * @apiName  drawInScript
             * @apiDescription 单元格是否允许在脚本中进行自绘.当单元格允许自绘时，
             * 单元格在绘制前及绘制后，都会触发一次 cellPaint: function( sheet, cell, dbrow, g, borderRect, data, s ,before)
             * 事件。其中参数 before 表示是在单元格由系统绘制前触发事件，还是绘制后触发事件。
             * @apiGroup Cell
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){boolean} - 单元格是否允许在脚本中进行自绘
             * @apiExample {js}(示例：)
             * //示例
             *
             * function cellPaint ( sheet, cell, dbrow, g, borderRect, data, s ,before)
             * {
             *     if( cell.name=='A1' && before)
             *     {
             *          g.setColor("red");
             *          g.moveTo( borderRect.x , borderRect.y);
             *          g.lineTo( borderRect.x+ borderRect.width , borderRect.y + borderRect.height);
             *     }
             * }
             */
            "drawInScript":
                {
                    get: function () {
                        return this.DrawInScript;
                    },
                    set: function (val) {
                        this.DrawInScript = val;
                        this.repaint();
                    }
                },
            /**
             * @api {passwordMask} 可读可写属性 [属性]passwordMask
             * @apiName  passwordMask
             * @apiDescription 显示为密码时的掩码，只能是如下4类格式
             * <ul>
             *     <li>*: 表示显示为***</li>
             *     <li>n*:表示左边n位字符显示，其它的显示为***</li>
             *     <li>*n:表示右边n位字符显示，其它的显示为***</li>
             *     <li> n1*n2:表示左边n1位字符显示,右边n2位字符显示，中间显示为***</li>
             *     </ul>
             *     <font color=red> 注意：无论实际数据有多长，显示为*时，统一显示为***</font>
             * @apiGroup Cell
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){String} - 掩码
             * @apiExample {js}(示例：)
             * //示例
             *
             * sheet.cells("a1").setValue( '13512345678');
             * sheet.cells("a1").passwordMask="3*4" // 显示为 135***5678
             *
             */
            "passwordMask":
                {
                    get: function () {
                        return this.viewAsPasswordMask;
                    },
                    set: function (val) {
                        this.viewAsPasswordMask = ('' + val).trim();
                        this.repaint();
                    }
                },

            /**
             * @api {dataStore} 可读可写属性 [属性]dataStore
             * @apiName  dataStore
             * @apiDescription 单元格如果绑定到数据源，则返回它绑定的结果集
             * @apiGroup Cell
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){DataStore} - 单元格绑定到的结果集对象
             * @apiExample {js}(示例：)
             * //示例
             */
            "dataStore":
                {
                    get: function () {
                        return this.getDataStore();
                    }
                },


            /**
             * @api {serialize} 可读可写属性 [属性]serialize
             * @apiName  serialize
             * @apiDescription 单元格属性序列到（从）字符串
             * @apiGroup Cell
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){String} -  单元格属性序列到（从）字符串
             * @apiExample {js}(示例：)
             *
             * let s= sheet.cells("a1").serialize;
             * sheet.cells("c3").serialize =s ;
             *
             */
            "serialize":
                {
                    get: function () {
                        return this.serializeTo();
                    },
                    set: function (val) {
                        this.serializeFrom(val)
                    }
                },

            /**
             * @api {workSheet} 只读属性 [属性]workSheet
             * @apiName  workSheet
             * @apiDescription 单元格所在的工作表，如果是一个虚拟单元格，则返回 null
             * @apiGroup Cell
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){WorkSheet} - 单元格所在的工作表
             * @apiExample {js}(示例：)
             *
             *
             * let sheet= cell.workSheet;
             *
             */
            "workSheet":
                {
                    get: function () {
                        return this.Sheet;
                    }
                },

            /**
             * @api {workBook} 只读属性 [属性]workBook
             * @apiName  workBook
             * @apiDescription 单元格所在的工作本
             * @apiGroup Cell
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){WorkBook} - 单元格所在的工作本
             * @apiExample {js}(示例：)
             * //示例
             */
            "workBook":
                {
                    get: function () {
                        return this.Book;
                    }
                },


            "barcodeType":

                {
                    get: function () {
                        return this.m_barcodeType || "";
                    },
                    set: function (val) {
                        this.m_barcodeType = val;
                        this.clearInternalData();
                    }
                },


            "barcodeWidth":

                {
                    get: function () {
                        return this.m_barcodeWidth || 2;
                    },
                    set: function (val) {
                        this.m_barcodeWidth = val;
                        this.clearInternalData();
                    }
                },


            "barcodeHeight":

                {
                    get: function () {
                        return this.m_barcodeHeight || 30;
                    },
                    set: function (val) {
                        this.m_barcodeHeight = val;
                        this.clearInternalData();
                    }
                },

            "barcodeTextMargin":

                {
                    get: function () {
                        return this.m_barcodeTextMargin || 10;
                    },
                    set: function (val) {
                        this.m_barcodeTextMargin = val;
                        this.clearInternalData();
                    }
                },
            "barcodeTextVisible":

                {
                    get: function () {
                        return this.m_barcodeTextVisible == null ? true : this.m_barcodeTextVisible;
                    },
                    set: function (val) {
                        this.m_barcodeTextVisible = val;
                        this.clearInternalData();
                    }
                },
            /**
             * @api {printable} 可读可写属性 [属性]printable
             * @apiName  printable
             * @apiDescription 单元格是否允许打印
             * @apiGroup Cell
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){boolean} - 单元格是否允许打印
             * @apiExample {js}(示例：)
             * //示例
             */
            "printable":
                {
                    get: function () {
                        return this.Printable;
                    },
                    set: function (val) {
                        this.Printable = val;
                    }
                }


        },
    constructor: function (sheet, row) {
        this.Sheet = sheet;// 所从属的Sheet
        this.Book = sheet.workBook;
        this.pRow = row;// 所从属的行

        this.Define = null;
        this.isValid = false; // 当是公式时，值是不是有效的
        this.formulaValue = null;
        //0表示自动判断，1表示公式优先, 2 表示数据库优先
        this.formulaIsPriority = 0;//当单元格有数据库绑定，也有公式时，是不是公式更有优先权

        this.property = null; // 属性
        this.VT = UniformDataType.$Auto; // 数据类型
        this.DependList = null; // 依赖列表
        this.AffectList = null; // 影响列表
        this.Bind = null;
        this.EditingFlag = 0;
        this.Editable = false;
        this.ES = null;
        this.ViewAs = '';
        this.viewAsPasswordMask = "*";
        this.BrickMap = null;
        this.ToolTip = null;

        this.Alias = null;
        this.Tag = null;
        this.formulaNeedCache = true; //公式需要缓存结果吗，缺省是需要的


        //校验相关
        this.validateRule = null;
        this.validateTip = null;

        this.subscribeList = null;
        this.cursorName = "";

        this.Visible = true;
        this.Printable = true;

        this.MouseAdapterList = null;
        this.innerImage = null;
        this.innerImnageData = null;

        this.drawProxy = null;

        this.dblClickSortDefine = ""; //双击本单元格时排序
        this.dblClickSortSignChar = "";

        //单击
        this.clickJumpTo = "";
        this.clickJumpToWhere = "_blank";
        this.clickPopupDialogWidth = 800;
        this.clickPopupDialogHeight = 600;
        this.clickPopupDialogTitle = "";

        //悬停提示 2021.12.14增加
        this.popoverType = "html"; //可以是 html ， iframe ， script
        this.popoverContent = "";
        this.popoverDialogWidth = "";
        this.popoverDialogHeight = "";
        this.popoverDialogTitle = "";
        this.popoverAutoClose = true;


        this.placeholder = ''; //占位提示
        this.placeholderOnlyViewWhenEditing = true; //仅在编辑时显示占位提示

        this.cssName = ""; // 使用全局的css属性对象

        this.Clickable = true; //是否允许点击
        this.Suffix = '';
        this.Prefix = '';

    },

    destructor: function () {
        this.clearInternalData();

        this.Define = null;
        this.formulaValue = null;
        this.pRow = null;
        this.Sheet = null;
        this.property = null; // 属性
        this.DependList = null; // 依赖列表
        this.AffectList = null;// 影响列表
        this.Bind = null;
        this.ES = null;
        this.BrickMap = null;
        this.Tag = null;
        this.MouseAdapterList = null;
        this.m_decorate = null; //边框装饰


    },

    addMouseAdapter: function (name, adapter) {
        if (this.MouseAdapterList == null) this.MouseAdapterList = {};
        this.MouseAdapterList[name] = adapter;
    },

    removeMouseAdapter: function (name) {
        if (this.MouseAdapterList == null) return;
        if (this.MouseAdapterList[name] == undefined) return;

        delete this.MouseAdapterList[name];
    },

    getMouseAdapter: function (name) {
        if (this.MouseAdapterList == null) return undefined;
        return this.MouseAdapterList[name];
    },

    setDblClickSortDefine: function (q_c) {
        if (q_c == null) q_c = "";
        q_c = q_c.trim();
        this.dblClickSortDefine = q_c;
    },

    getDblClickSortDefine: function () {
        return this.dblClickSortDefine;
    },

    GetValueInsteadOfNull: function () {
        let m_InsteadOfNull = null;
        switch (this.VT)
        {
            case UniformDataType.$Integer:
                m_InsteadOfNull = 0;
                break;
            case UniformDataType.$Numeric:
                m_InsteadOfNull = 0;
                break;
            case UniformDataType.$String:
                m_InsteadOfNull = '';
                break;
            case UniformDataType.$Datetime:
                m_InsteadOfNull = new Date();
                break;
            case UniformDataType.$Binary:
                m_InsteadOfNull = '';
                break;
            case UniformDataType.$Other:
                m_InsteadOfNull = '';
                break;
            case UniformDataType.$Auto:
                m_InsteadOfNull = '';
                break;

        }
        return m_InsteadOfNull;
    },


    /**
     * 得到本单元格中数据的个数，如果是vector ,或者绑定，那么
     * @return
     */
    getValueCount: function () {
        if (this.isMerged())
        {
            let /*Cell*/    leftTop = this.getLeftTopCorner();
            if (this != leftTop) return leftTop.getValueCount();
        }

        // 如果同时有公式定义和数据据绑定时
        // 如果 dbrow==-1 （通常是用this.getValue() 函数引起的） 那么公式的结果优先返回
        // 这是 为了实现将公式定义回填到字段中而必须的

        if (this.Bind != null)
        {
            return this.Book.getDataSource(this.Bind.dataSource).dataStore.rowCount;
        }

        //如果是公式，如果是数组，如果 dbrow=-1则返回整个vector,还，否则返回指定行中的数据
        if (this.isFormula())
        {
            if (!this.isValid)
            {
                // 如果值不是合法的，那么重新计算它
                let formula = this.Define.toString().trim(); // 上面的 this.isFormula()保证了这里的合法性
                // 下面的这个检测不是很必要
                if (formula.startsWith("=")) formula = formula.substring(1, formula.length); // 去掉等号
                if (formula == "") return 0;
                // 重新计算
                this.formulaValue = this.Calculate(formula, -1);

                // 设置标志
                this.isValid = true;
            }

            //如果是数组
            if (Util.isArray(this.formulaValue))
            {
                return this.formulaValue.length;
            } else
            {
                return 1;
            }

        }

        return 1;
    }

    ,

    //  增加,避免编辑后，直接点击其它按钮，此时编辑的值没有生效，所以需要强制一下
    //有问题， 当一个单元格需要编辑时，它需要先从DataStore中取数，而取数又强制编辑框取消编辑，结果就是无法编辑

    //if ( this.isEditing() ) ((WorkSheetView_)this.Sheet.View).forceCurrentEditControlGiveUpFocus();

    //在 this.Calculate 中， 没有强制设置 JEP的innerRow ,因为缺省它是 -99
    // 这表示当在表达式中引用单元格时，请自动判断dbrow ,原则是如果是绑定到单行数据源， 那么
    // 在表达式中使用单元格引用数据时，自动将dbrow=0 ,表示取数据库第０行的数据。如果没有数据，那么
    // 使dbrow=-1 表示返回公式的解析结果优先　。如果是多行数据源，那么自动将dbrow=-1 表示返回一个vector
    //这里的关键是当使用单元格引用绑定到单行数据源的字段时，应该使用数据库字段的数据
    //在没有做此处理前，　由于缺省的dbrow==-1 ,所以会返回公式的数据，而不是数据库的数据


    /**
     * @api {getValue} 函数   getValue
     *
     * @apiDescription getValue([dbRow])
     * <br><br> 得到单元格中的数据
     * <br>根据单元单元格的绑定情况，返回单元格的数据。细节如下：<br>
     * <ul>
     * <li>如果单元格没有绑定数据库。也没有定义公式，那么直接返回单元格的内容</li>
     * <li>如果单元格绑定到单行数据库，无论dbRow为何值，都返回绑定的字段的值</li>
     * <li>如果绑定到多行数据源，如果dbRow参数未提供，则视dbRow为-1 ,此时返回绑定字段整列的数据（数组）</li>
     * <li>如果绑定到多行数据源，如果dbRow是合法的行号（0 到 行数-1），那么返回绑定字段在dbRow行上的数据 </li>
     * <li>如果没有绑定到数据源，但有定义公式。且公式的结果是单个数据，那么直接返回它</li>
     * <li>如果没有绑定到数据源，但有定义公式。且公式的结果是数组，如果dbRow是合法的序号，那么返回第dbRow行处的数据，否则返回整个数组</li>
     *
     *
     * </ul>
     *
     * 注意：如果绑定到数据源，建议使用 cell.dataStore.getValue(row,col)的方式获取数据
     *
     * @apiName  getValue
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     * @apiParam {int} dbRow 子行号，可选参数，默认是-1
     *
     *
     * @apiSuccess (返回值){variant} - 返回单元格的值
     *
     *
     * @apiExample   {js}示例：
     *
     *  cell.getValue();
     *  cell.getValue(-1);
     *  cell.getValue(2);
     *
     */
    getValue: function (dbrow, debug) {

        if (debug) console.info(" getvlaue ");

        if (dbrow == undefined) dbrow = -1;
        if (dbrow == -99)
        {
            if (this.Bind != null)
            {
                let /*DataSourceConfig*/ dsc = this.Book.getDataSource(this.Bind.dataSource);
                if (this.Bind.dataSourceType == DataSourceConfig.SingleRowDataSource)
                {
                    if (dsc.dataStore.rowCount > 0)
                    {
                        dbrow = 0;
                    } else
                    {
                        dbrow = -1;
                    }
                } else if (this.Bind.dataSourceType == DataSourceConfig.AsSingleRowDataSource) //2015.08支持
                {
                    dbrow = dsc.currentBindRow;
                } else
                {
                    dbrow = -1;
                }

            } else
            {
                dbrow = -1;
            }
        }

        //  修改，如果本单元格被合并，并且自已不是最左上角单元格，那么返回最左上角单元格的数据
        if (this.isMerged())
        {
            let leftTop = this.getLeftTopCorner();
            if (this != leftTop) return leftTop.getValue(dbrow);
        }

        // 如果同时有公式定义和数据据绑定时
        // 如果 dbrow==-1 （通常是用this.getValue() 函数引起的） 那么公式的结果优先返回
        // 这是 为了实现将公式定义回填到字段中而必须的

        if (this.Bind != null)
        {
            //如果是公式，并且dbrow<0那么就不要取数据值，
            //如果包含公式，并且dbrow<0 那么就不要取数据值
            if (!((this.isFormula() && dbrow < 0) || (this.isFormulaContained() && dbrow < 0)))
            {
                let dsc = this.Book.getDataSource(this.Bind.dataSource);
                let dbcol = this.Bind.DBCol;
                if (dsc != null)
                {
                    let ds = dsc.dataStore;
                    // 如果是绑定到单行DataStore上
                    if (this.Bind.dataSourceType == DataSourceConfig.SingleRowDataSource)
                    {
                        if (ds.rowCount == 1) // 如果有一行，那么就返回它
                        {
                            let v = ds.getValue(0, dbcol);
                            return v;
                        } else
                        {
                            return null;
                        }

                    } else if (this.Bind.dataSourceType == DataSourceConfig.AsSingleRowDataSource) //2015.08 如果是视为单行，那么
                    {
                        dbrow = dsc.currentBindRow;
                        //取当前绑定行   修正，因为 this.getValue() 被转向 this.getValue(-1) ,所以此时 dbrow=-1 ，要重新取一下当前行号
                        if (dbrow >= 0)
                        {
                            let v = ds.getValue(dbrow, dbcol);
                            return v;
                        } else
                        {
                            return null;
                        }
                    } else
                        // 如果是绑定到多行数据源上
                    {
                        // 如果指定了行号
                        if (dbrow >= 0)
                        {
                            if (dbrow < ds.rowCount)
                            {
                                return ds.getValue(dbrow, dbcol);
                            } else
                            {
                                return null;
                            }

                        } else if (!this.Bind.quickFindByCol == "")
                        {
                            dbrow = ds.fastFind(this.Bind.quickFindByCol, this.Bind.quickFindByValue);
                            if (dbrow < 0) return null;
                            let v = ds.getValue(dbrow, dbcol);
                            return v;
                        } else
                        {
                            // 没有指定行号就返回所有
                            let ret = [];
                            for (let row = 0; row < ds.rowCount; row++
                            )
                            {
                                ret.push(ds.getValue(row, dbcol));
                            }
                            return ret;
                        }
                    }
                }
            }
        }

        //如果是公式，如果是数组，如果 dbrow=-1则返回整个vector,还，否则返回指定行中的数据
        if (this.isFormula())
        {
            if (!this.isValid)
            {
                // 如果值不是合法的，那么重新计算它
                let formula = this.Define.toString().trim(); // 上面的 this.isFormula()保证了这里的合法性
                // 下面的这个检测不是很必要
                if (formula.startsWith("=")) formula = formula.substring(1, formula.length); // 去掉等号
                if (formula == "") return null;
                // 重新计算
                /*
                         *  dbrow参数可能是在8月加上的，参看this.Calculate　，直接代入dbrow 是可能存在问题的，
                         * 因为formula的结果可能是一个数组，但是指定dbrow后，　this.formulaValue就只是一个值了，导致一个数组变成了一个值
                         * 常见场景是　visible 属性表达式
                         * */
                /*
                         * 修改
                         *
                         * 比如  a1   绑定到字段 rq , 且定义了公式 = newDate() ;
                         *   b1= getYear(a1)
                         *   当修改a1值后，  b1 被通知失效，在重 绘 时，它被要求重新计算
                         *   如果在this.Calculate中的innerRow用 dbrow,而且dbrow=-1时，那么JEP中的inerRow就是-1
                         *   此时 JEP.getValueFromSheet 时，就用innerRow=-1，如果单元格是bind+formula ,那么就会优先返回formula的值
                         *   而不是当前db的值。这个就有问题了
                         *   上面b1 始终返回的是  getYear( newDate())的值，而不是   getYear( a.bind的值)
                         *   所以下面改成了 -99 ,有没有什么负作用，尚不可知
                         *   初步测试：上面的公式正常，  多行绑定中  je=sl*dj 也正常。进一步的影响 还有待测试
                         * */

                this.formulaValue = this.Calculate(formula, -99); //dbrow );

                if (this.DependList == null && !this.formulaNeedCache)
                {
                    //重大改变：对于不依赖任何其它单元格的公式，让它没有缓存，即每次都需要重新计算
                    //重大改变，增加 了this.formulaNeedCache标记进行控制。如果公式是类似getStringFromSQL这样的语句，
                    // 如果不加缓存的话，反复执行十分影响性能
                } else
                {
                    //2019.04 增加如下判断，如果book被设置成没有初始化完成，那么单元格公式虽然计算了，也不要设置成有效的
                    //
                    if (this.Sheet.Book.initOK)
                    {
                        // 设置标志
                        this.isValid = true;
                    }
                }
            }

            //如果是数组
            if (Util.isArray(this.formulaValue))
            {

                if (dbrow >= this.formulaValue.length) return null;
                if (dbrow >= 0) return this.formulaValue[dbrow];
                //注意当dbrow=-1时，就是返回整个vector ,而不是返回超界
                return this.formulaValue;

            } else
            {
                return this.formulaValue;
            }

        }

        // 不是公式或者是包含公式，那么直接返回定义

        if (!this.isValid)
        {
            if (this.Define == '') return this.Define;

            let vt = this.VT;
            if (this.VT == UniformDataType.$Auto)
            {

                vt = this.guessType();

                if (!this.Sheet.designMode) this.VT = vt;
            }

            switch (vt)
            {
                case UniformDataType.$Auto:

                    if (Util.isInteger(this.Define))
                    {
                        this.formulaValue = ObjectTool.changeType(this.Define, UniformDataType.$Integer);
                    } else if (!isNaN(this.Define))
                    {
                        this.formulaValue = ObjectTool.changeType(this.Define, UniformDataType.$Numeric);
                    } else if (Util.isDate(this.Define))
                    {
                        this.formulaValue = ObjectTool.changeType(this.Define, UniformDataType.$Datetime);

                    } else if (Util.isString(this.Define))
                    {
                        this.formulaValue = ObjectTool.changeType(this.Define, UniformDataType.$String);

                    }


                    break;
                default:
                    this.formulaValue = ObjectTool.changeType(this.Define, vt);

            }
            this.isValud = true;

        }

        return this.formulaValue;


    },


    guessType: function () {
        if (this.VT != UniformDataType.$Auto) return this.VT;

        if (this.ES == null) return this.VT;

        switch (this.ES.ET)
        {


            case EditStyle.$MultiLine:
            case EditStyle.$DDLB:
            case EditStyle.$RadioButton:
            case EditStyle.$RichText:
            case EditStyle.$MultiCheckBox:
            case EditStyle.$Tree:
                return UniformDataType.$String;
            case EditStyle.$Datetime:
                return UniformDataType.$Datetime;
            case EditStyle.$Numeric:
                if (this.ES.decimalCount == 0) return UniformDataType.$Integer;
                return UniformDataType.$Numeric;
            default:
                return this.VT;

        }

    },
    /**
     * @api {getValueString} 函数   getValueString
     *
     * @apiDescription getValueString()
     * <br><br> 得到单元格的数据，并转换成字符串，如果单元格的数据为null，则返回空字符串。当单元格数据为数组时，本函数不适用。
     *
     * @apiName  getValueString
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     * @apiParam {int} dbRow 可选  行号
     *
     * @apiSuccess (返回值){String} - 转换成字段串的单元格的值
     *
     *
     * @apiExample   {js}示例：
     *
     *  let v= cell.getValueString();
     *
     */
    getValueString: function (dbRow) {
        let v = this.getValue(dbRow);
        if (v == null) return "";

        return ObjectTool.changeType(v, UniformDataType.$String);

    },

    /**
     * @api {getValueInt} 函数   getValueInt
     *
     * @apiDescription getValueInt()
     * <br><br> 得到单元格的数据，并转换成整数，如果单元格的数据为null或无法转换成整数，则返回0。当单元格数据为数组时，本函数不适用。
     *
     * @apiName  getValueInt
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     * @apiParam {int} dbRow 可选  行号
     *
     * @apiSuccess (返回值){int} - 转换成整数的单元格的值
     *
     *
     * @apiExample   {js}示例：
     *
     *  let v= cell.getValueInt();
     *
     */
    getValueInt: function (dbRow) {
        let v = this.getValue(dbRow);
        if (v == null) return 0;
        return ObjectTool.changeType(v, UniformDataType.$Integer);
    },


    /**
     * @api {getValueBoolean} 函数   getValueBoolean
     *
     * @apiDescription getValueBoolean()
     * <br><br> 得到单元格的数据，并转换成boolean型，如果单元格的数据为null或无法转换成则返回false。当单元格数据为数组时，本函数不适用。
     * 数据为  true , t ,y ,TRUE , T ,Y ,1 这些数据时，本函数返回true,其它值则返回false
     * @apiName  getValueBoolean
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     * @apiParam {int} dbRow 可选  行号
     *
     * @apiSuccess (返回值){boolean} - 转换成boolean的单元格的值
     *
     *
     * @apiExample   {js}示例：
     *
     *  let v= cell.getValueInt();
     *
     */
    getValueBoolean: function (dbRow) {
        let v = this.getValue(dbRow);
        if (v == null) return 0;
        v = v + '';
        return ['true', 't', 'y', 'TRUE', 'T', 'Y', '1'].contains(v);
    },

    /**
     * @api {getValueDouble} 函数   getValueDouble
     *
     * @apiDescription getValueDouble()
     * <br><br> 得到单元格的数据，并转换成小数，如果单元格的数据为null或无法转换成，则返回0。当单元格数据为数组时，本函数不适用。
     *
     * @apiName  getValueDouble
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     * @apiParam {int} dbRow 可选  行号
     *
     * @apiSuccess (返回值){numeric} - 转换成小数的单元格的值
     *
     *
     * @apiExample   {js}示例：
     *
     *  let v= cell.getValueInt();
     *
     */
    getValueDouble: function (dbRow) {
        let v = this.getValue(dbRow);
        if (v == null) return 0;
        return ObjectTool.changeType(v, UniformDataType.$Numeric);
    },
    /**
     * @api {getValueDate} 函数   getValueDate
     *
     * @apiDescription getValueDate()
     * <br><br> 得到单元格的数据，并转换成日期，如果单元格的数据为null或无法转换成日期，则返回null。当单元格数据为数组时，本函数不适用。
     * 当单元格的数据是类似于 "2018.01.01"这样的字符串，或  1576075710这样，或1576075710000这样的时间戳时，可以转换成日期型。
     *
     * @apiName  getValueDouble
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     *@apiParam {int} dbRow 可选  行号
     *
     * @apiSuccess (返回值){numeric} - 转换成小数的单元格的值
     *
     *
     * @apiExample   {js}示例：
     *
     *  let v= cell.getValueInt();
     *
     */
    getValueDate: function (dbRow) {
        let v = this.getValue(dbRow);
        if (v == null) return null;
        return ObjectTool.changeType(v, UniformDataType.$Datetime);
    },


    /**
     * @api {getValueNumber} 函数   getValueNumber
     *
     * @apiDescription getValueNumber()
     * <br><br> 得到单元格的数据，并转换成小数，如果单元格的数据为null或无法转换成，则返回0。当单元格数据为数组时，本函数不适用。
     * 本函数等同于  getValueDouble
     *
     * @apiName  getValueNumber
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     * @apiParam {String} where 检索条件
     *
     *
     * @apiSuccess (返回值){numeric} - 转换成小数的单元格的值
     *
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     */
    getValueNumber: function () {
        return this.getValueDouble();
    },


    getName: function () {
        //  如果这个单元格是独立存在的，而不是加入到行中的，那么它只有别名
        if (this.pRow == null) return this.alias;
        let row = this.rowIndex;
        let col = this.columnIndex;
        return this.Sheet.CPM.getColumnName(col) + (row + 1);
    },


    /**
     * 得到显示的文本
     */

    /**
     * @api {getShowText} 函数   getShowText
     *
     * @apiDescription getShowText(dbRow)
     * <br><br> 先用 getValue( dbRow)得到数据，然后根据编辑格式，转换成用于显示的字符串。
     *
     * @apiName  getShowText
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     * @apiParam {int} dbRow 子行号
     *
     *
     *
     *
     * @apiSuccess (返回值){String} - 返回显示值
     *
     *
     * @apiExample   {js}示例：
     *
     *  cell.setValue( 12345.678);  //单元格编辑格式是小数编辑格式，显示千分位，保留2位小数
     *  let s= cell.getShowText( ); // s=='12,345.68'
     *
     */
    getShowText: function (row) {


        if (row == undefined) row = -1;


        //如果是设计状态，并且是公式定义，那么也显示出公式定义
        if (this.Sheet.designMode && this.Book.showDefineIfInDesignMode && (this.isFormula() || this.isFormulaContained()))
        {
            return this.Define;
        }

        let v = this.getValue(row); // 得到了数据值，
        // 如果数据值中包含有公式，那么还要解析其中的公式

        //   显然只对单行绑定有效。因为如果是多行绑定，那么无法用一个this.isValid, this.formulaValue
        // 来保存数据
        if (this.isFormulaContained(v))
        {
            if (this.isValid)
            {
                return this.toFormatString(this.formulaValue);
            }

            // 如果值不是合法的，那么重新计算它
            let formula = v.toString().trim(); // 上面的 this.isFormulaContained()保证了这里的合法性
            if (formula == "") return null;
            // 重新计算
            //TODO
        }

        return this.toFormatString(v);

    },

    getTextForExport: function (dbrow) {

        let s = this.getShowText(dbrow);


        if (this.viewAs == 'image')
        {
            let str = Tools.normalizeImageSrc(s);

            let /*Property*/ prop = this.getPropertyObject();// 不要直接用成员变量this.property,因为它可能为空
            let /*int*/ hAlign = prop.get(Property.text_align, ALIGNMENT.Left);
            let /*int*/ vAlign = prop.get(Property.vertical_align, ALIGNMENT.Center);
            let rc = this.Sheet.View.getShowRectangleOfCell(false, this.rowIndex, this.columnIndex);
            rc.x = 0;
            rc.y = 0;


            // 在导出excel前，一定是先确保图片都已经加载了，
            let img = window['spreadSheetImageCache'][str]; //str 即是img.src ,同时也是缓存的key
            if (img)
            {
                //此时的 g 已经正确设置了裁剪区，不需要再设置了
                let t = s.toLowerCase();
                let w = img.width;
                let h = img.height;
                console.info("图片原始大小：w:" + w + " h:" + h);

                let scale = 0;
                //只是一个图片地址，那么 ，默认是自动调整大小
                if (t.indexOf(" ") < 0 && t.indexOf("width" < 0) && t.indexOf("height") < 0)
                {
                    let wr = w * 1.0 / rc.width; //图宽与容器宽的比率
                    let hr = h * 1.0 / rc.height; //
                    if (wr > hr) //说明要
                    {

                        w = rc.width;
                        h = h / wr;
                        scale = wr;
                    } else
                    {
                        h = rc.height;
                        w = w / hr;
                        scale = hr;
                    }

                    console.info("自动调整大小 适应单元格  w:" + w + " h:" + h);

                } else
                {

                    try
                    {
                        let sizeDefined = false;
                        w = t.match(/width\s*=\s*[\'\"]([^\'\"]*)[\'\"]/);
                        if (w.length == 2)
                        {
                            w = parseInt(w[1]);
                            sizeDefined = true;

                        }

                        h = t.match(/height\s*=\s*[\'\"]([^\'\"]*)[\'\"]/);
                        if (h.length == 2)
                        {
                            h = parseInt(h[1]);
                            sizeDefined = true;

                        }
                        //如果没有指定尺寸，则按单元格进行自适应缩放
                        if (!sizeDefined)
                        {
                            let wr = w * 1.0 / rc.width; //图宽与容器宽的比率
                            let hr = h * 1.0 / rc.height; //
                            if (wr > hr) //说明要
                            {

                                w = rc.width;
                                h = h / wr;
                                scale = wr;
                            } else
                            {
                                h = rc.height;
                                w = w / hr;
                                scale = hr;
                            }
                            console.info("自动调整大小 适应单元格  w:" + w + " h:" + h);
                        }

                    } catch (e)
                    {
                        w = img.width;
                        h = img.height;
                    }
                }
                //通过缩放，重设显示尺寸

                let x, y;
                if (hAlign == ALIGNMENT.Left) x = rc.x;
                if (hAlign == ALIGNMENT.Center) x = rc.x + (rc.width - w) / 2;
                if (hAlign == ALIGNMENT.Right) x = rc.x + rc.width - w;

                if (vAlign == ALIGNMENT.Top) y = rc.y;
                if (vAlign == ALIGNMENT.Center) y = rc.y + (rc.height - h) / 2;
                if (vAlign == ALIGNMENT.Bottom) y = rc.y + rc.height - h;

                let ret = {};
                ret.viewAs = "image";
                ret.url = str;
                ret.vAlign = vAlign;
                ret.hAlign = hAlign;
                ret.x = x;
                ret.y = y;
                ret.width = w;
                ret.height = h;
                ret.x2 = rc.width - w - x;
                ret.y2 = rc.height - h - y;
                ret.scale = scale;
                // 因为跨域问题，所以 getBase64Image不一定成功，所以放弃
                //ret.img = ImageUtil.getBase64Image(img, w,h);

                return '$#{urlimage}#$' + JSON.stringify(ret);
            }
        }

        if (this.ES != null)
        {
            let /*int*/ et = this.ES.ET;

            let ss=[];
            //  增加下拉列表显示为Radio,checkbox样式
            if (et == EditStyle.$RadioButton || et == EditStyle.$MultiCheckBox)
            {

                if( dbrow==undefined) dbrow=-1;

                let dropDownData = this.ES.getDropDownDataForEdit(this, dbrow);

                let checked = [];
                let vs = (s == null ? '' : ('' + s)).split(',');
                for (let i = 0; i < vs.length; i++)
                {
                    let v = vs[i];
                    let index = dropDownData.dataValue.indexOf(v);
                    if (index >= 0) checked.push(index);
                }


                let listCount = dropDownData.dataValue.length;
                //需要显示几行
                //当实际的数据不足列数时，原先是以实际为准，现在改在以设置为准，这样方便对齐
                let colCount = Math.max(1, this.ES.columnCount);
                let rowCount = Math.max(1, Math.floor((listCount - 1) / colCount) + 1); //肯定不为0


                for (let ri = 0; ri < rowCount; ri++)
                {
                    let lineStr=[];
                    for (let ci = 0; ci < colCount; ci++)
                    {

                        let index = ri * colCount + ci;
                        if (index > listCount - 1) break;

                        let str = '';

                       // 输出时， 转成宋体中可以打印的符号

                            str = checked.contains(index) ? "[√]︎" : "[ ]";
                            str = str + dropDownData.showText[index];

                        lineStr.push( str);
                    }
                    ss.push( lineStr.join(" \t"));
                }

                return ss.join("\n");

            }
        }

        return s;

    },


    // 根据编辑模式自动调整水平对齐方式


    $AutoHalign: function (align) {
        if (align != ALIGNMENT.Auto) return align;
        if (this.ES == null) return ALIGNMENT.Left;
        switch (this.ES.ET)
        {
            case EditStyle.$Numeric:
                return ALIGNMENT.Right;
            case EditStyle.$Datetime:
                return ALIGNMENT.Left;
            default:
                return ALIGNMENT.Left;
        }

    },

    // 根据编辑格式把值格式化

    /**
     * @api {toFormatString} 函数   toFormatString
     *
     * @apiDescription 把数据值转换成显示值，但是要注意，当单元格是列表时，列表如果尚未初始化完成，可以通过async参数指定是否等待列表列表
     * 初始化完成（true表示不等待）
     * <br><br> 单元格
     *
     * @apiName  toFormatString
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     * @apiParam {Object} v 要转换成显示值的数据
     *
     *
     *
     * @apiSuccess (返回值){String}    - 返回转换成显示值
     *
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     */
    toFormatString: function (v) {

        //2021.10.10 当 v是数组时，也要返回数组
        //比如一个单元格绑定到多行结果集字段 ， 编辑格式是列表，另一个单元格用 $别名方式引用它的显示值，
        // 就需要把多行数据都转换成显示值，并且仍保持数组返回
        if (v == null) return this.$toFormatString(v);

        if (Util.isArray(v))
        {
            let ret = [];
            for (let i = 0; i < v.length; i++)
            {
                ret.push(this.$toFormatString(v[i]));
            }
            return ret;
        } else
        {
            return this.$toFormatString(v);
        }
    },

    $toFormatString: function (v) {


        // console.info("toFormatString");

        if (this.viewAs == 'locked')
        {
            return "※※※";
        }

        if (this.viewAs == 'password')
        {
            if (this.viewAsPasswordMask == '*')
            {
                if (v == null) return "";
                if (v == '') return '';  //对于多行数据源， 数据条数为0时，它也显示*** 就有点怪
                return "***";
            }

            if (v == undefined) return '***';
            if (v == null) return '***';

            v = '' + v;

            let mask = this.viewAsPasswordMask;
            if (mask.startsWith('*'))
            {
                mask = mask.substring(1);
                if (isNaN(mask)) return '***';
                return '***' + v.substring(v.length - parseInt(mask), v.length);
            }

            if (mask.endsWith('*'))
            {
                mask = mask.substring(0, mask.length - 1);
                if (isNaN(mask)) return '***';
                return v.substring(0, parseInt(mask)) + '***';
            }

            mask = mask.split(/\*/);
            if (mask.length == 2)
            {
                let m1 = mask[0];
                let m2 = mask[1];
                if (isNaN(m1) || isNaN(m2)) return '***';
                let l1 = parseInt(m1);
                let l2 = parseInt(m2);
                let l = v.length;
                if (l <= l1) return '***';
                if (l <= l2) return '***';

                return v.substring(0, l1) + '***' + v.substring(v.length - parseInt(m2));
            }

            return '***';


        }

        if (v == undefined) v = '';
        if (v == null) v = '';

        //如果值是一个JSON对象

        //注意 Date ， Number也是  object特别注意
        if (typeof (v) == 'object' && v.hasOwnProperty('spreadsheetInnerAction'))
        {
            //2021.09.05 为了支持 innerAction函数
            //如果它包含有text属性，那么直接返回它
            return v.text;

        }

        // if (v == null) return ""; 不能直接返回，假如是DDLB，并对Null作了映射，这不是正确的显示值了

        if (this.ES == null)
        {
            if (Util.isDate(v)) return v.Format("yyyy.MM.dd");
            return v.toString();
        }

        switch (this.ES.ET)
        {
            case EditStyle.$Normal:


                if (Util.isDate()) return v.Format(EditStyle.DefaultDatetimeFormat);
                if (Util.isInteger(v)) return '' + v;
                if (!isNaN(v)) Number(v).toFixed(4);
                return v.toString();

            case EditStyle.$MultiLine:

                return v.toString();

            case EditStyle.$Numeric:
            {

                if (v == '') v = '0';
                v = parseFloat(v);
                if (isNaN(v)) return '';
                let t = MathFunc.round(v, this.ES.decimalCount);
                //if (!this.ES.showComma) return ''+t;


                return NumericEdit.format(t, this.ES.decimalCount,
                    this.ES.showComma,
                    '.',
                    ',');


            }

            case EditStyle.$Datetime:
            {
                let t = ObjectTool.changeType(v, UniformDataType.$Datetime);
                if (t != null) return t.Format(this.ES.datetimeFormat);
                return '';
            }
            case EditStyle.$CheckBox:


                return '' + v; //FontAwesome字体中的toggle-on 与toggle-off


            case EditStyle.$DDLB:
                //2010.12 增加如果显示真实数据，那么就不要转换成显示值

                if (this.ES.showRealData) return v.toString();
                let showText = this.ES.ddlb_data2text(v);
                return showText;

            //2019.07.31 增加树编辑方式下的数据转译显示
            case EditStyle.$Tree:

                //  if( this.$treeShowText!=null) return this.$treeShowText;
                let ts = ('' + v).split(',');
                let ret = [];
                let nodes = this.ES.treeNodes;

                for (let i = 0; i < nodes.length; i++)
                {
                    let node = nodes[i];
                    if (node.hasOwnProperty('data') && node.data == '') continue;  //控制节点不允许选中
                    if (node.hasOwnProperty('id') && node.id == '') continue;  //控制节点不允许选中

                    if (ts.contains(node.id) || ts.contains(node.data))
                    {

                        if (node.caption != undefined)
                        {
                            ret.push(node.caption);
                        } else
                        {
                            ret.push(node.name);
                        }

                    }

                }
                this.$treeShowText = ret.join(',');
                return this.$treeShowText;


            case EditStyle.$RichText:

                return v.toString();

            default:

                return v.toString();
        }

    },

    /**
     * @api {getBind} 函数   getBind
     *
     * @apiDescription
     * <br><br> 单元格
     *
     * @apiName  getBind
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     * @apiSuccess (返回值){Bind} - 返回本单元格的绑定信息
     *
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     *  let bind= cell.getBind();
     *
     */
    getBind: function () {
        return this.Bind;
    },

    /**
     * 得到编辑的文本 因为对于公式，编辑的是公式定义，对于数据库绑定，编辑的是数据库的值 注意与 getText()的区别
     */
    getEditText: function (row) {

        // 如果是数据库绑定
        if (this.Bind != null)
        {
            let v = this.getValue(row);
            //2019.07.31 增加： 对于 tree编辑方式，它不需要用显示值来构建显示，但它却需要数据值来构建选中状态，所以返回数据值，而不是显示值
            if (this.ES != null && this.ES.ET == EditStyle.$Tree) return v;

            return this.toFormatString(v);
        }

        //2019.07.31 增加： 对于 tree编辑方式，它不需要用显示值来构建显示，但它却需要数据值来构建选中状态，所以返回数据值，而不是显示值
        if (this.ES != null && this.ES.ET == EditStyle.$Tree) return this.Sheet.designMode ? this.Define : this.getValue();

        //2019.07.31 增加结束

        return this.toFormatString(this.Sheet.designMode ? this.Define : this.getValue());

    },

    /**
     * 计算公式
     * @param exp   公式
     * @param innerRow    如果有绑定到多行数据源，那么只取其中的innerRow
     * @returns {*}
     * @constructor
     */
    Calculate: function (exp, innerRow) {

        return this.Sheet.evaluate(exp, innerRow, false, null);

    },

    /**
     * 是不是公式定义， 如果是以＝开头，那么就是公式
     *
     * @param s
     * @return
     */


    isFormula: function (s) {
        if (s == undefined) s = this.Define;
        if (s == null) return false;
        let t = s.toString().trim();
        //出现设置值为 =========== 这样的字符串，结果被认为是公式，其实不是
        if (t.startsWith("=") && !t.startsWith("==")) return true;
        return false;

    },

    /**
     * 是不是包含有公式
     * @param s
     * @return
     */
    isFormulaContained: function (s) {
        if (s == undefined) s = this.Define;
        if (s == null) return false;
        let t = s.toString().trim();
        if (t.indexOf("{=") >= 0 && t.indexOf("}") > 0) return true;
        return false;
    },


    /**
     * 修改为合适的类型
     *
     * @param value
     * @param vt
     * @return
     */


    changeType(value, vt)
    {
        return ObjectTool.changeType(value, vt);
    },

    // 在调用本方法前，一定要用changeType将二者改成一样的类型


    objectEqual: function (obj1, obj2) {
        // 都为空，什么都不发生
        if (obj1 == null && obj2 == null) return true;
        // 都不为空，如果两者相等，那么，什么都不发生
        if (obj1 != null && obj2 != null)
        {
            if (obj1 == (obj2)) return true;
        }
        return false;
    },

    /**
     * 单元格没有绑定到数据库时，使用该函数设置单元格的值
     * 或者在单元格绑定到数据库时，且这个字段需要本单元格的值回填时，可以使用本函数 不带行号
     * 设置单元格的公式


     * 当单元格绑定到数据库时，使用本函数设置dbRow 数据行中的字段的值
     * 当 dbRow=-1时或value不是一个Vector时，它表示是用来设置单元格的公式
     */

    /**
     * @api {setValue} 函数   setValue
     *
     * @apiDescription setValue(value, dbRow)
     * <br><br> 设置单元格的值
     * <br><h3>细节</h3>
     * <ul>
     * <li>value被转换成与单元格数据类型一至，如果无法转换，则value被置成null</li>
     * <li>如果value与单元格中的数据是相等的，那么直接返回</li>
     * <li>触发事件cellValueChangeAccept ，如果该事件返回false表示本操作被拒绝</li>
     * <li>将单元格的数据设置成value</li>
     * <li>通知所有引用本单元格的公式重新计算</li>
     * <li>触发事件cellValueChanged</li>
     * </ul>
     *
     * @apiName  retrieve
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     * @apiParam {variant} value 数据
     * @apiParam {int} dbRow 子数据行号，可选参数，默认为-1. 当单元格没有绑定数据源时，本参数无意义
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *  cell.setValue( 12334);
     *  cell.setValue( 1234,4);
     *  cell.dataStore.setValue( 4, cell.bind.dbCol , 1234); //与上国的cell.setValue(1234,4)等效
     */
    setValue: function (value, dbRow, repaint) {

        if (repaint == undefined) repaint = true;

        if (dbRow == undefined) dbRow = -1;

        // 如果不是公式，也不包含公式，也不是数组时，那么修改类型为合适的类型
        if (!this.isFormula(value) && !this.isFormulaContained(value) && !Util.isArray(value))
        {
            value = this.changeType(value, this.VT);
        }
        // 如果值相等，那就不需要做什么

        //  dbRow<0时，有两情况符合：1是设置单元格的公式时，2是不分行号设置整列数据时
        if (dbRow < 0) // 如果是设置公式定义（一个单元格可以是公式定义和绑定共存，但不会是单元格值与绑定共存
        {
            if (this.objectEqual(this.Define, value)) return;
        } else
            // 如果是数据库绑定，那么需要和相对应的进行比较
        {
            let oldValue = this.getValue(dbRow);
            if (this.objectEqual(oldValue, value)) return;
        }

        let scene = new Property();
        scene.put("row", this.rowIndex);
        scene.put("col", this.columnIndex);
        scene.put("cell", this);
        scene.put("value", value);
        scene.put("dbrow", dbRow);
        scene.put("repaint", repaint);


        let cmd = new CMD_SetValue(this.Sheet, scene);
        return cmd.execute();

    },

    // 可能是虚拟单元格


    getColumnIndex: function () {
        if (this.pRow == null)
        {
            if (this.Sheet.ExpressionList == null) return -1;
            return this.Sheet.ExpressionList.indexOf(this);
        }
        return this.pRow.cells.indexOf(this);
    },


    getRowIndex: function () {
        if (this.pRow == null) return -9999;
        return this.pRow.rowIndex;//
    },


    permitBackFill: function (/*DataStore*/ds, /*int*/row, /*String*/col) {
        // 如果是不允许回填，那么返回false
        if (!this.Sheet.isFormulaBackFillBindEnabled()) return false;

        // 增加，如果设置成公式优先，那么允许回填
        //(缺省是公式优先。比如金额合计回填到一个字段上。再有一些用公式来填入缺省值的，可以设置成数据库优先)　

        if (this.isFormulaPriority()) return true;

        // 如果不允许回填 ，且数据状态是检索出来的，那么不要回填
        let /*int*/ state = ds.getPrimaryBuffer().getItemStatus(row, col);
        //如果是检索出来的来数据，那么在禁止回填时，就不要回填，而以检索出来的数据为准
        if (state == ItemStatus.isDataModified || state == ItemStatus.isNotModified)
        {
            // 如果数据库里为null , 那么允许公式回填，出现这种情况比如：当第一次打开某个表单时
            //数据库中记录已经有了，但某些字段是null ,且昌第一次被公式回填，那么应该允许回填
            if (ds.getValue(row, col) == null) return true;
            if (ds.getString(row, col) == "") return true;
            //如果是有数据的，那么就不要回填了
            return false;
        }
        // 如果是新行，那么是允许回填的
        return true;
    },


    /**
     *
     * @param value
     * @param dbRow
     * @param isBackFill
     * @param needRepaint 是否触发重绘
     */
    setValueToDS: function (value, dbRow, /*boolean*/    isBackFill, needRepaint) {
        if (this.Bind == null) return;
        if (needRepaint == undefined) needRepaint = true;

        let /*DataSourceConfig*/   dsc = this.Book.getDataSource(this.Bind.dataSource);
        let workBookInitOK = this.Book.initOK;
        if (dsc != null)
        {
            let dbcol = this.Bind.DBCol;
            let ds = dsc.dataStore;
            // 如果是绑定到单行DataStore上
            if (this.Bind.dataSourceType == DataSourceConfig.SingleRowDataSource)
            {
                if (ds.rowCount == 1) // 如果有一行，那么就
                {

                    //如果workbook 尚未初始化完成，那么当数据是从数据库中检索出来的，此时，不需要把数据回填
                    // 这个主要是为了解决：当单元格即绑定到数据库， 又用公式进行了定义，当在初始化workbook 时
                    // 数据检索后，再用公式定义计算关系时，如果公式的结果与数据库中的数据不一至，会用公式的数据
                    // 覆盖数据库的数据。 而在初始化时，有时不的希望这样覆盖。可以在初始化时，用setInitOK(false)表明 一下
                    // 再在合适的时候setInitOK(true);
                    if (!workBookInitOK)
                    {
                        if (ds.getPrimaryBuffer().getItemStatus(0, dbcol) == ItemStatus.isNotModified) return;
                    }

                    if (isBackFill && !this.permitBackFill(ds, 0, dbcol)) return;

                    if (Util.isArray(value))
                    {

                        if (vec.length > 0) ds.setValue(0, dbcol, vec[0]);

                    } else
                    {
                        ds.setValue(0, dbcol, value);

                    }
                } else
                {
                    return;
                }

            } else if (this.Bind.dataSourceType == DataSourceConfig.AsSingleRowDataSource) //2015.08对多行视为单行的支持
            {
                dbRow = dsc.currentBindRow; //  修正，如果单元格定义了公式，有回填到绑定的字段，此时 dbrow=-1 ，需要重新取一下当前行
                if (dbRow >= 0) // 如果有数据行，那么就
                {

                    //如果workbook 尚未初始化完成，那么当数据是从数据库中检索出来的，此时，不需要把数据回填
                    // 这个主要是为了解决：当单元格即绑定到数据库， 又用公式进行了定义，当在初始化workbook 时
                    // 数据检索后，再用公式定义计算关系时，如果公式的结果与数据库中的数据不一至，会用公式的数据
                    // 覆盖数据库的数据。 而在初始化时，有时不的希望这样覆盖。可以在初始化时，用setInitOK(false)表明 一下
                    // 再在合适的时候setInitOK(true);
                    if (!workBookInitOK)
                    {
                        if (ds.getPrimaryBuffer().getItemStatus(dbRow, dbcol) == ItemStatus.isNotModified) return;
                    }

                    if (isBackFill && !this.permitBackFill(ds, dbRow, dbcol)) return;

                    if (Util.isArray(value))
                    {

                        if (value.length > 0) ds.setValue(dbRow, dbcol, value[dbRow]);

                    } else
                    {
                        ds.setValue(dbRow, dbcol, value);

                    }
                } else
                {
                    return;
                }

            } else
                // 如果是绑定到多行数据源上
            {
                let row = dbRow;
                //如果dbRow 是一个合法的行号
                if (row >= 0 && row < ds.rowCount)
                {
                    //如果workbook 尚未初始化完成，那么当数据是从数据库中检索出来的，此时，不需要把数据回填
                    // 这个主要是为了解决：当单元格即绑定到数据库， 又用公式进行了定义，当在初始化workbook 时
                    // 数据检索后，再用公式定义计算关系时，如果公式的结果与数据库中的数据不一至，会用公式的数据
                    // 覆盖数据库的数据。 而在初始化时，有时不的希望这样覆盖。可以在初始化时，用setInitOK(false)表明 一下
                    // 再在合适的时候setInitOK(true);
                    if (!workBookInitOK)
                    {
                        if (ds.getPrimaryBuffer().getItemStatus(row, dbcol) == ItemStatus.isNotModified) return;
                    }

                    if (isBackFill && !permitBackFill(ds, row, dbcol)) return;

                    if (Util.isArray(value)) // 如果value 是一个数组，那么取第一个值
                    {

                        if (value.length > 0) ds.setValue(dbRow, dbcol, value[dbRow]);

                    } else
                        // 不是数组，那么直接回填
                    {
                        ds.setValue(row, dbcol, value);

                    }

                } else
                    // dbRow==-1
                {

                    //到这里， ds是一个多行结果集，且 dbRow==-1  表示全部行都要回填
                    //如果 value 不是数组，那么就需要转成数组，如果个数不够，那么就可补齐个数
                    let v = value;
                    if (!Util.isArray(v)) v = [v];


                    let RC = ds.rowCount;
                    let lastV = v[v.length - 1];
                    //如果v的个数不足填充RC行，那么用最后一个数据补齐少的
                    for (let i = v.length; i < RC; i++)
                    {
                        v[i] = lastV;
                    }

                    for (let i = 0; i < RC; i++)
                    {
                        //如果workbook 尚未初始化完成，那么当数据是从数据库中检索出来的，此时，不需要把数据回填
                        // 这个主要是为了解决：当单元格即绑定到数据库， 又用公式进行了定义，当在初始化workbook 时
                        // 数据检索后，再用公式定义计算关系时，如果公式的结果与数据库中的数据不一至，会用公式的数据
                        // 覆盖数据库的数据。 而在初始化时，有时不的希望这样覆盖。可以在初始化时，用setInitOK(false)表明 一下
                        // 再在合适的时候setInitOK(true);
                        if (!workBookInitOK)
                        {
                            if (ds.getPrimaryBuffer().getItemStatus(i, dbcol) == ItemStatus.isNotModified) return;
                        }

                        if (isBackFill && !this.permitBackFill(ds, i, dbcol)) continue;
                        ds.setValue(i, dbcol, v[i]);
                    }


                }

            }
        } else
        {
            return;
        }
    },


    /**
     * 当单元格是图片，条码时， 渲染所需要的数据，并非单元格的数据能直接提供，通常是需要转换后才能渲染，
     * 比如数据是图片地址，但是渲染时需要用到对应的图片数据 ，比如数据是一个条码，渲染时需要用到条码的条空信息
     * 当单元格的数据变化后， 这些内部使用的渲染数据需要清除并重建，此处负责清除，在需要时再重建
     */
    clearInternalData: function () {
        //图片数据
        if (this['innerImage']) delete this['innerImage'];
        //条码数据 2019.11.02增加
        if (this['barcodeEncodings'])
        {
            delete this['barcodeEncodings'];
            console.info("清除条码缓存");
        }

        //2021.09.10 增加树结点显示信息的缓存的处理，参看 toFormatString
        if (this['$treeShowText']) delete this['$treeShowText'];

    },

    // 供内部使用的函数，不对外提供使用
    // 所以不需要判断value和当前Value是否相等等，因为所有的判断在setValue中已经判断
    // 且类型也是经过转换成合适的

    /*boolean*/
    $setValue: function (value, dbRow, fireEvent, repaint) {

        if (fireEvent == null) fireEvent = true;
        if (repaint == null) repaint = true;


        if (fireEvent)
        {

            if (this.validateRule != null)
            {
                if (this.validateRule.trim() != '')
                {
                    let pass = this.Sheet.evaluate(this.validateRule, dbRow, true, true, {$$$: value});
                    if (!pass)
                    {
                        let info = this.validateTip;
                        if (!info || info.trim() == '') info = '数据不合法';
                        this.Book.EM.fire("error", [info]);
                        return false;
                    }
                }
            }


            // 触发是否接受改变事件
            let accept = this.Book.EM.fire("cellValueChangeAccept", [this.Sheet, this, this.Define, value, dbRow], true);
            // 如果不接受值的改变，那么直接返回失败
            if (!accept)
            {
                if (this.isEditing())
                {
                    let view = this.Sheet.View;
                    let edit = view.$getCurrentEditingEdit();
                    if (edit != null) edit.reloadEdit();

                }
                return false;
            }
        }

        this.clearInternalData(); // 把图象清空

        let needResetRowHeight = false; //2021.06.21增加，根据内容自动调整行高

        // 临时保存旧的值
        let oldValue = this.Define;
        // 设置新的值


        //注意这个位置，它后面必须保存把this.needRepaing删除，它是一个session变量，即过了这个函数，它就消失
        // 之所以加这么一个变量， 是因为 当绑定数据库时， 进入到 [标记20200502] ，向结果集设置数据
        // 然后在结果集的值发生变化事件中， 会重绘单元格，但是 repaint 参数无法传递过去
        // 所以加了这么一个变量来控制。 这个变量在本函数结束后就需要删除掉
        this.temp_needRepaint = repaint;

        // 如果是绑定到数据时， 要么 dbRow是一个合法的行，要么 value 是一个数组时
        // 才表示是向数据源中填数据，此时同步把数据回填到数据源中
        if (this.Bind != null && (dbRow >= 0 || Util.isArray(value)))
        {

            //★★★★★★★  重大修改，去掉了下面的一行
            //this.Define = value;
            //去掉上面的一行的原因是：当单元格是绑定数据库，并且有公式，并且单元格允许编辑时
            //当直接输入数据，回车， 数据需要回填到datastore中，此时绝对不要this.Define=value , 如果这样做，反而把公式给丢掉了
            // [标记20200502]
            this.setValueToDS(value, dbRow, false, repaint);
            // 因为回填到DataStore后,value 可能会被作数据转换
            //
            //   注意，把数据设置到DS中后，并不需要再显示地使用下面else中的 标记20051214后的代码
            // 做一些动作，因为在 ds 事件的 ItemChanged事件中显示地用了FireSomeOnethis.BindCellsValueChanged
            // 来触发相关的事件

            if (this.Bind.dataSourceType == DataSourceConfig.SingleRowDataSource || this.Bind.dataSourceType == DataSourceConfig.AsSingleRowDataSource)
            {
                needResetRowHeight = true;
            }

        } else
            // 运行到这里来，要么是本单元格没有绑定到数据库，要么是数据库的值是用本单元格的公式值的结果回填的
            //
        {
            // 不是用数组数据直接向数据库灌值时，才是设置公式。
            if (!Util.isArray(value) && dbRow < 0)
            {
                //2020.11.12 增加类型转换，当单元格是日期编辑时，方便用字符串给它设置数据，且又能得到正确的日期
                // 2020.11.21 重大bug，如果是公式，并且没有设置类型，那么公式可能丢失
                let vt = this.guessType();//得到数据类型
                if (vt != UniformDataType.$Auto)
                {
                    let t = ('' + value).trim(); // 2020.11.21 重大bug，如果是公式，并且没有设置类型，那么公式可能丢失
                    if (!t.startsWith("=")) value = ObjectTool.changeType(value, vt);
                }

                this.Define = value;
                this.isValid = false;

                needResetRowHeight = true;
            }
            // 分析依赖影响关系( 先清除旧的依赖关系)
            this.ParseDependAffect();
            // 根据自己的依赖关系，更新自己所依赖的单元格的影响列表，让它们加上自己
            this.RebuildAffactOfMyDepend();

            // 标记20051214
            // 设置自己的标记,如果是公式，那么标记为值无效，强制重新计算
            this.setInvalid();

            //通知影响的单元格重新计算
            this.NotifyAffectCellInvalid();


            let newValue = this.Define;
            //2020.01.17 增加，如果单元格有补全，且有数据回填，那么清除数据

            //到这里，通常是单元格没有绑定的情况
            if (newValue == null || newValue == '') this.clearMoreValueOnAutoComplateIfNeeded(dbRow);

            // 触发相关事件
            //  改成把触发事件放到this.NotifyAffectCellInvalid之后


            if (fireEvent)
            {
                this.Book.EM.fire("cellValueChanged", [this.Sheet, this, oldValue, newValue, dbRow]);
            }


            //在serializeFrom时，其实是不需要触发这些的
            if (fireEvent)
            {
                //如果正在编辑，那么要通知编辑框重新加载数据
                if (this.isEditing())
                {
                    let view = this.Sheet.View;
                    let edit = view.$getCurrentEditingEdit();
                    if (edit != null) edit.reloadEdit();

                }


            }

            if (repaint) this.repaint();

        }

        //  增加通知订阅者，单元格改变了
        this.notifyChangedToSubscriber(dbRow);

        //标记20200502-01
        delete this.temp_needRepaint;

        //2021.06.21  如果高度是需要根据内容自适应，那么重新计算高度
        if (needResetRowHeight)
        {
            let col = this.columnIndex;
            if (this.Sheet.CPM.isRowHeightAutoFit(col))
            {
                let /*Property*/ prop = this.getPropertyObject();// 不要直接用成员变量this.property,因为它可能为空
                let /*boolean*/ word_wrap = prop.get(Property.word_wrap, false);
                if (word_wrap)
                {
                    let row = this.rowIndex;
                    let leftTopCell = this.leftTopCorner;
                    setTimeout(function () {
                        this.Sheet.resetRowHeightWhenColumnWidthChanged(row, col, this, leftTopCell);
                    }.bind(this), 100);
                }
            }
        }


        return true;
    },

    /**
     * 通知订阅者，本单元格改变了，
     */

    notifyChangedToSubscriber: function (dbRow, notifiedList) {
        // 　

        //console.info("通知自已的观察者，它们依赖的对象变化了");
        if (this.subscribeList != null)
        {
            if (notifiedList == null) notifiedList = []; //创建一个已经通知过的对象列表，用于检测循环通知
            let n = this.subscribeList.length;
            for (let i = 0; i < n; i++)
            {
                let subscribe = this.subscribeList[i];
                if (notifiedList.contains(subscribe)) continue;
                if (subscribe['observableChanged'] != undefined)
                {
                    notifiedList.push(subscribe);
                    // subscribe 可能并不是一个单元格，比如可能是一个TreeBrick，它没有$fireSelfValueChanged函数
                    if (subscribe['$fireSelfValueChanged'] != undefined) subscribe.$fireSelfValueChanged(dbRow, notifiedList); //递归
                    subscribe.observableChanged(this, dbRow);
                }
            }
        }

    },


    /**
     *  被观察者变化了
     * @param observable  被观察者
     */
    observableChanged: function (observable, dbRow) {
        this.repaint();
    },

    // 本函数只在DataSourceAction中被调用，当单元格绑定到数据源，且数据源发生如
    // retrieveend 等事件时，需要触发所有绑定到这个数据源的单元格发生变化的事件
    // 且需要通知affectedCells 重新计算自己
    //  由于本函数并不由界面对象触发，因此需要自己处理reloadEdit

    $fireSelfValueChanged: function (innerRow, notifiedList) {


        if (notifiedList == null) notifiedList = [];//防止死循环

        this.clearInternalData(); // 把图象清空
        //  由于本函数仅在DataSourceAction中被调用，所以当绑定变化后
        // 再取数据时，也是会从数据源中取，所以公式是不需要重新计算的，且一定不能重新计算
        //比如先设置公式，再创建数据源对象，然后绑定到某个字段，如果此时触发重新计算，那么
        //会把数据库中取出的数据用公式结果覆盖掉
        //  如果完全不计算，那么当是新插入的数据，然后绑定时，公式也将不计算，显然
        //应该是这样：检索出来的数据不应该被公式替换，而新插入的行，应该用公式的数据替换

        // 当插入一行，或者是在服务器端插入一行后，再传回到FlowForm中，此时触发的是
        //
        if (this.Bind != null)
        {
            let /*DataSourceConfig*/dsc = this.Book.getDataSource(this.Bind.dataSource);
            let dbcol = this.Bind.DBCol;
            if (dsc != null)
            {
                let /*DataStore*/  ds = dsc.dataStore;
                let /*DataBuffer*/primaryBuffer = ds.getPrimaryBuffer();
                let hadSetInvalid = false;
                let rc = ds.rowCount;
                for (let i = rc - 1; i >= 0; i--)
                {
                    let state = primaryBuffer.getItemStatus(i, dbcol);
                    if (state == ItemStatus.isNew)
                    {
                        this.setInvalid();
                        hadSetInvalid = true;
                        break;
                    }
                }
                // 增加如下处理：
                //如果数据不是　isNew,而是从数据库中检索出来的，　那么上面就没有执行 this.setInvalid
                //因此需要继续判断：如果所有数据都是null, 那么强制用 this.setInvalid使用计算公式再次计算一下，并回填　
                if (!hadSetInvalid)
                {
                    let allIsNull = true;
                    for (let i = 0; i < ds.rowCount; i++)
                    {
                        if (ds.getValue(i, dbcol) != null)
                        {
                            if (!ds.getString(i, dbcol) == "") allIsNull = false;
                        }
                    }

                    if (allIsNull)
                    {
                        this.setInvalid();
                        hadSetInvalid = true;

                    }

                }
            }

        }


        //通知影响的单元格重新计算
        this.NotifyAffectCellInvalid(innerRow, notifiedList);

        // 触发相关事件
        //  改成把触发事件放到this.NotifyAffectCellInvalid之后

        /*   触发CellValueChanged事件时，带上了当前单元格的数据，如果单元格是绑定到多行数据源的，并且数据量巨大
                 * 将是极大的影响性能。但由于可能存在一些用户，在它们的脚本中使用了CellValueChanged事件的newvalue参数来访问单元格
                 * 的数据，因此出于兼容，还不能完全去掉newvalue参数。但是可以理解的是如果单元格是多行，或者计算列，那么通常不会直接用
                 * newvalue来取数据。所以才做了如下的妥协：即不是计算列，不是绑定到多行的数据源，还是提供newvalue参数，否则不提供
                 */

        let needResetRowHeight = true;
        let newValue = null;
        if (this.Bind != null)
        {
            let /*DataSourceConfig*/     dsc = this.Book.getDataSource(this.Bind.dataSource);
            if (this.Bind.dataSourceType == DataSourceConfig.MultiRowDataSource)
            {
                newValue = this.getValue(dsc.currentBindRow);
                needResetRowHeight = false; //多行绑定不支持行高自动，以后可能会支持
            } else if (this.Bind.dataSourceType == DataSourceConfig.AsSingleRowDataSource)
            {

                newValue = this.getValue(dsc.currentBindRow);

            } else
            {
                newValue = this.getValue();

            }
        } else
        {
            if (this.isFormula() || this.isFormulaContained())
            {
                newValue = null;
            } else
            {
                newValue = this.getValue();
            }
        }

        //2020.01.17 增加，如果单元格有补全，且有数据回填，那么清除数据

        //到这里，通常是单元格有绑定的情况
        if (newValue == null || newValue == '') this.clearMoreValueOnAutoComplateIfNeeded(innerRow);

        this.Book.EM.fire("cellValueChanged", [this.Sheet, this, null, newValue, innerRow]);


        // 如果正在编辑，那么重新调取编辑内容
        if (this.isEditing())
        {
            let view = this.Sheet.View;
            let edit = view.$getCurrentEditingEdit();
            if (edit != null) edit.reloadEdit();

        }

        // 增加通知订阅者，单元格改变了
        this.notifyChangedToSubscriber(innerRow, notifiedList);

        //2021.06.21  如果高度是需要根据内容自适应，那么重新计算高度
        if (needResetRowHeight)
        {
            let col = this.columnIndex;
            if (this.Sheet.CPM.isRowHeightAutoFit(col))
            {
                let /*Property*/ prop = this.getPropertyObject();// 不要直接用成员变量this.property,因为它可能为空
                let /*boolean*/ word_wrap = prop.get(Property.word_wrap, false);
                if (word_wrap)
                {
                    let row = this.rowIndex;
                    let leftTopCell = this.leftTopCorner;
                    setTimeout(function () {
                        this.Sheet.resetRowHeightWhenColumnWidthChanged(row, col, this, leftTopCell);
                    }.bind(this), 100);
                }
            }
        }
    },


    //2020.01.17 增加，如果单元格有补全，且有数据回填，那么清除数据
    clearMoreValueOnAutoComplateIfNeeded: function (innerRow) {
        let appData = this.appData;

        if (!appData) return;  // RETURN


        let sourceType = appData.valueChangedToAutoCompleteType || '';

        //字段回填对应关系
        let colMaps = appData.valueChangedToAutoCompleteCopyMap || '';

        colMaps = colMaps.trim();

        if (!sourceType) return;


        let colMap = Tools.buildColMap(colMaps);
        for (let ci = 0; ci < colMap.length; ci++)
        {
            let cp = colMap[ci];

            if (cp.target.alias)
            {
                let toCell = this.Sheet.cells(cp.target.alias);
                if (toCell)
                {
                    toCell.setValue(null);
                } else
                {
                    toastr.error(" 补全定义,不存在" + cp.target.alias + "别名的单元格");
                }
            } else
            {
                let dsc = this.Sheet.Book.getDataSource(cp.target.ds);
                if (dsc == null)
                {
                    console.error(cp.target.ds + "不是合法的结果集名称")
                } else
                {
                    let ds = dsc.dataStore;
                    ds.setValue(innerRow, cp.target.col, null);
                }
            }
        }

    },

    /**
     * 分析依赖影响关系
     *
     * @param exp
     */

    ParseDependAffect: function () {
        let exp = this.Define || "";

        // 如果它不是公式，那么它就不需要依赖其它单元格，所以Depend列表置空
        // 如果是公式，那么先清除旧的依赖关系，再分析新的依赖关系
        this.ClearDepend();

        let isF = this.isFormula(exp);
        let isFC = this.isFormulaContained(exp);
        if (!isF && !isFC) return;

        let formulaList = [];
        if (isF)
        {
            if (exp.startsWith("=")) exp = exp.substring(1); // 去掉等号, 不要去空格
            formulaList.push(exp);
        }

        if (isFC)
        {
            //TODO 公式包含
        }

        console.info("ParseDependAffect");

        //再从属性中找是不是的公式
        //
        let prop = this.getPropertyObject(); //不要直接访问this.property它缺省是空的
        let font_color_expression = prop.get(Property.font_color_expression, "").trim();
        let background_color_expression = prop.get(Property.background_color_expression, "").trim();
        let display_expression = prop.get(Property.display_expression, "").trim();

        //	if (font_color_expression.length > 0) formulaList.add(font_color_expression);
        //	if (background_color_expression.length > 0) formulaList.add(background_color_expression);
        //	if (display_expression.length > 0) formulaList.add(display_expression);

        // 得到语法树

        for (let fi = 0; fi < formulaList.length; fi++)
        {
            let formula = formulaList[fi];
            //
            let dl = this.Sheet.parseDependList(formula);
            //追加到依赖列表中
            if (this.DependList == null) this.DependList = [];
            this.DependList = this.DependList.concat(dl);

        }
    },


    /**
     * 清除旧的依赖关系
     *
     */

    ClearDepend: function () {
        if (this.DependList == null) return;
        for (let di = 0; di < this.DependList.length; di++)
        {
            let da = this.DependList[di];
            let /*      WorkSheet*/sheet = this.Book.getWorkSheet(da.sheetGuid);
            if (sheet == null) // 当删除Sheet后有可能会出现这样的情况
            {
                continue;
            }

            let /*Range*/  range = da.mergedRange.getMergedRange();
            let thisSheetGuid = this.Sheet.guid;
            for (let row = range.startRow; row <= range.endRow; row++)
            {
                for (let col = range.startCol; col <= range.endCol; col++)
                {
                    let /*Cell*/            cell = sheet.cells(row, col);
                    if (cell == null) continue; // 基本上不应该出现，因为在设置影响关系时，确保它已经被创建
                    cell.RemoveAffect(new Affect(thisSheetGuid, cell, da.dependShowText));

                }
            }

        }
        this.DependList.clear();
        this.DependList = null;
    },

    // 根据自己的依赖关系，更新自己所依赖的单元格的影响列表，让它们加上自己

    RebuildAffactOfMyDepend: function () {
        if (this.DependList == null) return;
        for (let di = 0; di < this.DependList.length; di++)
        {
            let /*Depend*/da = this.DependList[di];
            let sheet = this.Book.getWorkSheet(da.sheetGuid);
            if (sheet == null) // 通常是不会出现的
            {
                console.error("重建影响关系时指定的[" + da.sheetGuid + "] 不存在");
                continue;
            }

            let thisSheetGuid = this.Sheet.GUID;

            let /*Range*/    range = da.mergedRange.getMergedRange();
            if (range != null)
            {
                for (let row = range.startRow; row <= range.endRow; row++)
                {
                    for (let col = range.startCol; col <= range.endCol; col++)
                    {
                        let /*Cell*/ cell = sheet.cells(row, col);
                        if (cell == null) continue;
                        // 确保单元格已经被创建，注意是
                        cell.AddAffect(new Affect(thisSheetGuid, this, da.dependShowText));
                    }
                }
            }
        }
    },

    // 增加一个对另一个单元格的影响设置

    AddAffect: function (/*Affect*/   affect) {
        if (this.AffectList == null) this.AffectList = [];
        if (this.AffectList.contains(affect)) return;
        this.AffectList.push(affect);
    },

    // 移除对另一个单元格的影响

    RemoveAffect: function (/*Affect*/affect) {
        if (this.AffectList == null) return;

        this.AffectList.remove(affect);
    },

//=============
    /**
     * 仅当行号在 afterRow之后时，才对公式做rowOffset距离的移位操作
     * 仅当列号在 afterCol之后时，才对公式做rowOffset距离的移位操作
     */

    redefine: function (rowOffset, colOffset, afterRow, afterCol) {
        afterRow = afterRow || -1;
        afterCol = afterCol || -1;

        //TODO

    },


    redefineReplace: function (cellName, replaceToName) {
        //TODO

    },

    /**
     * 重新自动修改公式
     *
     */

    RebuildDefine: function (/*WorkSheet*/   trigger, /* RedefineBecause*/because) {
        //TODO

    },


//==16:40 ============================================


    AutochangeType: function (v) {
        //TODO
        return v;
    },


    /**
     * @api {setPropertyValue} 函数   setPropertyValue
     *
     * @apiDescription setPropertyValue
     * <br><br> 设置单元格的属性
     *
     * @apiName  setPropertyValue(propertyName, propertyValue)
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     * @apiParam {String} propertyName 属性名
     * 可以是如下属性：
     * <ul>
     * <li><b>border-left-style</b></li>
     * <li><b>border-left-width</b></li>
     * <li><b>border-left-color</b></li>
     * <li><b>border-top-style</b></li>
     * <li><b>border-top-width</b></li>
     * <li><b>border-top-color</b></li>
     * <li><b>border-right-style</b></li>
     * <li><b>border-right-width</b></li>
     * <li><b>border-right-color</b></li>
     * <li><b>border-bottom-style</b></li>
     * <li><b>border-bottom-width</b></li>
     * <li><b>border-bottom-color</b></li>
     * <li><b>padding-left</b></li>
     * <li><b>padding-top</b></li>
     * <li><b>padding-right</b></li>
     * <li><b>padding-bottom</b></li>
     * <li><b>background-color</b></li>
     * <li><b>background-image</b></li>
     * <li><b>font-family</b></li>
     * <li><b>font-size</b></li>
     * <li><b>font-italic</b></li>
     * <li><b>font-underline</b></li>
     * <li><b>font-bold</b></li>
     * <li><b>color</b></li>
     * <li><b>word-wrap</b></li>
     * <li><b>text-align</b></li>
     * <li><b>vertical-align</b></li>
     * <li><b>display</b></li>
     *
     * </ul>
     *
     * @apiParam {variant} propertyValue 属性值
     *
     * @apiSuccess (返回值){Cell} - 返回单元格自身
     *
     *
     * @apiExample   {js}示例：
     *
     *  cell.setPropertyValue('font-family','黑体');
     *  cell.setPropertyValue('font-size',14);
     *  cell.setPropertyValue('font-bold',true);
     *  cell.setPropertyValue('background-color','#ff0000');
     *
     */
    setPropertyValue: function (propertyName, propertyValue) {


        // 为空，什么都不发生
        if (propertyName == null) return this;

        var
            oldPropertyValue = null;
        if (this.property != null)
        {
            oldPropertyValue = this.property.get(propertyName, null);
        }

        if (propertyValue == null && oldPropertyValue == null) return this;
        // 新旧属性相等，那么什么也不做

        if (propertyValue != null && oldPropertyValue != null)
        {
            if (propertyValue == oldPropertyValue) return this;
        }

        let scene = new Property();
        scene.put("row", this.rowIndex);
        scene.put("col", this.columnIndex);
        scene.put("cell", this);
        scene.put("propertyname", propertyName);
        scene.put("propertyvalue", propertyValue);
        scene.put("oldpropertyvalue", oldPropertyValue);

        let /*CMD_SetProperty*/  cmd = new CMD_SetProperty(this.Sheet, scene);
        cmd.execute();

        this.repaint();
        return this;
    },


    setPropertyObject: function (newProperty) {
        if (this.property == newProperty) return;

        if (this.property != null)
        {

            this.property.releaseRef();
            // 如果这个属性对象的引用计数为０了，那么删除它
            if (this.property.refCount == 0) this.Book.PM.remove(this.property);
        }

        newProperty.addRef();
        this.property = newProperty;

        // 触发事件
        //this.Book.EM.FireCellPropertyChanged(Sheet, this, propertyName, oldPropertyValue, propertyValue);

    },


    /**
     *
     * @param propertyName
     * @param propertyValue
     * @returns {boolean}
     */
    $setProperty: function (propertyName, propertyValue, fireEvent) {


        if (fireEvent === null) fireEvent = true;

        let PM = this.Book.PM;


        let /*Property*/   tempProp;
        let oldPropertyValue = null;

        // 如果本单元格已经有属性对象，那么把它克隆一份。

        if (this.property != null)
        {
            oldPropertyValue = this.property.get(propertyName, null);

            if (propertyValue == null && oldPropertyValue == null) return false;
            if (propertyValue != null && oldPropertyValue != null)
            {
                if (propertyValue == oldPropertyValue) return false;
            }

            tempProp = new Property();

            this.property.copyTo(tempProp);
            // 因为要设置为新的属性对象 ，所以，老的属性对象的引用计数要减１
            this.property.releaseRef();
            // 如果这个属性对象的引用计数为０了，那么删除它
            if (this.property.refCount == 0) PM.remove(this.property);
        } else
            // 如果本单元格还没有属性对象,那么就取缺省的
        {
            tempProp = new Property();
            this.Book.PM[0].copyTo(tempProp);
        }
        // 把指定的属性设置成新的值
        tempProp.put(propertyName, propertyValue);
        // 在属性集中查看看新的属性对象是否已经存在
        // 如果没有找到相同的属性对象，表明tempProp 是个新属性集合
        let pi = PM.indexOf(tempProp);

        if (pi < 0)
        {
            // 那么把它加到属性集合中，并把本单元格的属性指向它，并addRef
            PM.push(tempProp);
            this.property = tempProp;
            this.property.addRef();
        } else
        { // 表明已经存在一个相同的属性对象 ，把本单元格的属性指向它，并addRef
            this.property = PM[pi];
            this.property.addRef();
        }
        // 触发事件
        if (fireEvent)
        {
            this.Book.EM.fire("cellPropertyChanged", [this.Sheet, this, propertyName, oldPropertyValue, propertyValue]);
        }
        //不能在这里触发重绘 改放在 setPropertyValue中
        //
        // this.repaint();

        // 一些表达式属性，自适应高度属性的特别处理

        //2021.06.21 增加对行高自动适应的处理
        if (propertyName === Property.word_wrap && propertyValue == 'normal')
        {
            let col = this.columnIndex;
            if (this.Sheet.CPM.isRowHeightAutoFit(col)) //如果列在宽度改变后，需要重新设置各行高度，那么
            {

                let row = this.rowIndex;
                let leftTopCell = this.leftTopCorner;
                setTimeout(function () {
                    this.Sheet.resetRowHeightWhenColumnWidthChanged(row, col, this, leftTopCell);
                }.bind(this), 100);
            }
        }


        // 增加，
        // 当是设置属性表达式时，需要把自已加入到那个表达式的影响列表中，当表达式的值发生变化后，会通知本单元格重绘
        //详情参看 ChangeHistory.txt中　 　修改
        if (propertyName == Property.display_expression ||
            propertyName == Property.font_color_expression ||
            propertyName == Property.background_color_expression)
        {

            let t = '' + propertyValue;

            let ce = this.Sheet.getExpression(t);
            if (ce != null)
            {
                //此时的ce并不是一个真实的单元格，仅是一个表达式缓存，所以不存在 显示值，所以第3个参数用false
                ce.AddAffect(new Affect(this.Sheet.guid, this, false));
            }

        }

        return true;

    },

    /**
     * 当设置属性时，不要用本函数来得到属性对象，而是直接使用property变量
     */
    getPropertyObject: function () {

        //2020.09.30 css优先
        if (this.cssName != '')
        {
            let ret = this.Sheet.Book.getCSSProperty(this.cssName);
            if (ret != null) return ret;
        }

        if (this.property == null)
        {
            this.property = this.Book.PM[0];
            this.property.addRef();
        }

        return this.property;
    },


    /**
     * 如果是存在缺省属性，那么必须保证返回类型与缺省属性一致，即使给了defaultValue,也要保证它与
     *  _defaultValue 类型一致。 除非不存在对应的缺省属性。
     */


    /**
     * @api {getPropertyValue} 函数   getPropertyValue
     *
     * @apiDescription getPropertyValue(propertyName, defaultValue)
     * <br><br> 得到单元格指定的属性
     *
     * @apiName  getPropertyValue
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     * @apiParam {String} propertyName 属性名称，参看 setPropertyValue
     * @apiParam {String} defaultValue 缺省值
     *
     *
     *
     * @apiSuccess (返回值){variant} - 返回属性值
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *  cell.getPropertyValue('font-size',12);
     */
    getPropertyValue: function (propertyName, defaultValue) {

        return this.getPropertyObject().get(propertyName, defaultValue);


    },


    isPropertyDefault: function (propertyName) {
        let defaultValue = this.property = this.Book.PM[0].get(propertyName);
        let v = this.getPropertyValue(propertyName, defaultValue);
        if (v == null) return true; //没有值
        return v == defaultValue;

    },


    /**
     * @api {getCellValueType} 函数   getCellValueType
     *
     * @apiDescription getCellValueType()
     * <br><br> 得到单元格数据的类型
     *
     * @apiName  getCellValueType
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     *
     *
     *
     * @apiSuccess (返回值){int} - 值的类型
     *
     * 可以是：
     * <ul>
     * <li>1:整数</li>
     * <li>2：小数</li>
     *<li>3：字符串</li>
     *<li>4：日期</li>
     *<li>99：自动</li>
     * </ul>
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     */
    getCellValueType: function () {

        return this.VT;
    }
    ,

    /**
     * @api {setCellValueType} 函数   setCellValueType
     *
     * @apiDescription setCellValueType(vt)
     * <br><br> 设置单元格值的类型
     *
     * @apiName  setCellValueType
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     * @apiParam {int} vt 类型
     *
     *可以是：
     * <ul>
     * <li>1:整数</li>
     * <li>2：小数</li>
     *<li>3：字符串</li>
     *<li>4：日期</li>
     *<li>99：自动</li>
     * </ul>
     *
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     */
    setCellValueType: function (vt) {
        if (isNaN(vt)) return;
        vt = parseInt(vt);
        if (this.VT == vt) return;

        this.VT = vt;
        //触发数据改变事件


        // 设置自己的标记,如果是公式，那么标记为值无效，强制重新计算
        this.setInvalid();

        //通知影响的单元格重新计算
        this.NotifyAffectCellInvalid();


        this.Book.EM.fire("cellValueChanged", [this.Sheet, this, this.Define, this.Define, -1]);

        this.repaint();


    },


    /*boolean*/

    isValid: function () {
        return this.isValid;
    },


    // 强制设置为无效
    // 如果仅仅设置状态，那么不要通知重算


    /**
     * @api {setInvalid} 函数   setInvalid
     *
     * @apiDescription setInvalid()
     * <br><br> 如果单元格有定义公式，本函数是通知单元格重新计算公式的值
     *
     * @apiName  setInvalid
     * @apiGroup Cell
     * @apiVersion 1.0.0
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     */
    setInvalid: function (onlySetState) {

        if (onlySetState == null) onlySetState = false;

        if (!this.isFormula() && !this.isFormulaContained())
        {
            this.repaint(); //　[标记：可能是属性中有公式，被通知需要重绘，详情参看 ChangeHistory.txt中　
            return;
        }

        this.isValid = false;
        //  清除图片缓存

        this.clearInternalData();

        if (!onlySetState)
        {

            this.Book.EM.fire("cellNeedRecalculate", [this.Sheet, this]);

            // 如果单元格是公式定义，也定义了绑定，那么当单元格需要重新计算时，就需要
            // 对单元格进行强制计算并回填
            if (this.Bind != null)
            {
                this.backFill();
            }

            this.repaint();

            //  想增加这个,但后来没加,主要是考虑会不会影响性能
            // 现在单元格如果是公式,那么它的值 变了,是不会有cellvaluechanged事件触发的
            //this.Book.EM.fireCellValueChanged(Sheet, this, null,null, -1);
            //log( this.getName()+" "+this.alias+" 重新计算");
        }

    },

//iioi
    NotifyAffectCellInvalidWhenMyEditStyleChanged: function () {
        this.NotifyAffectCellInvalid(-1, [], true);
    },


    NotifyAffectCellInvalid: function (innerRow, already, /*boolean*/  becauseOfEditStyleChanged) {

        // 搜索自己的影响列表，通知相关单元格重新计算

        if (already == null) already = [];

        //[ 备注]
        //  修改：当单元格值变化后，首先要通知所有它影响到的单元格将
        //自已的值设置成无效
        //然后再通知所以它影响到的单元格重新计算
        // 为什么要搞两遍呢：比如　a3=sum(a1,a2) , b1=a1/a3 , b2=a2/a3
        // 当a1变化时，它将影响b1,a3的数据，因此当a1变化时，它通知b1需要需要重新计算
        //此时b1=a1/a3 ,但是a3还没有来得及重新计算，这样，b1的数据就是不对的
        //所以需要先通知b1,a3的值无效，再通知它们重新计算，这样当计算b1时需要用到a3的值
        //而a3被通知当前值已经无效了，因此a3会先重新计算后，再计算b1
        // 不良后果是：当计算b2时，a3已经重新计算过了，但最后，a3还将被通知重新计算一次

        //注意， 第一次调用时， 并不代入already ,而是新建一个数组代入。因为后面$NotifyAffectCellInvalid还会调用 一次
        // 避免第二次调用 时，already中已包含了相关的cell而不再执行。
        this.$NotifyAffectCellInvalid(innerRow, [], true, becauseOfEditStyleChanged);

        this.$NotifyAffectCellInvalid(innerRow, already, false, becauseOfEditStyleChanged);

    },


    $NotifyAffectCellInvalid: function (innerRow, already, onlySetState, becauseOfEditStyleChanged) {

        if (this.AffectList != null)
        {

            for (let ai = 0; ai < this.AffectList.length; ai++)
            {
                let affect = this.AffectList[ai];
                //当是编辑格式变化而触发时，需要判断被自已影响的单元格是不是依赖于自已的显示值
                //如果触发原因不是因为编辑格式变化，那么继续
                //TODO
                /*
                        if (becauseOfEditStyleChanged)
                        {
                            if (!affect.editStyleChangeWillAffect)
                            {
                                //System.out.println(affect.cell.getName() + "并不是依赖" + this.getName() + "的显示值，所以" +
                                  this.getName() + "编辑格式的改变不影响" + affect.cell.getName());
                                continue;
                            }
                        }
                        */

                if (already.contains(affect)) continue;
                already.push(affect);

                let /*WorkSheet*/
                    sheet = this.Sheet;

                sheet = this.Book.getWorkSheet(affect.sheetGuid);
                if (sheet == null)
                {
                    console.log("在this.NotifyAffectCellInvalid指定的[" + affect.sheetGuid + "] 不存在");
                    continue;
                }

                let cell = affect.cell;
                if (cell == null) continue;
                cell.setInvalid(onlySetState);

                //  增加通知订阅者，单元格改变了


                if (!onlySetState) cell.notifyChangedToSubscriber(innerRow, already);

                // 递归调用
                cell.$NotifyAffectCellInvalid(innerRow, already, onlySetState, becauseOfEditStyleChanged);

            }
        }


    },


    /**
     * 如果单元格是公式定义，也定义了绑定，那么当单元格需要重新计算时，就需要
     对单元格进行强制计算并回填
     *  本函数在 WorkBookAdapterIns.CellNeedRecalculate中调用
     */

    backFill: function () {
        //   增加了一个开关，这样当打开表单，并从数据库中检索出数据
        // 这时，公式会自动计算，并回填。这样导致数据一检索出来又被修改（当然，多数情况
        // 下与修改前是一样的，但当公式中含有不确定返回值函数时，导致每次检索出来的数据时
        // 它又被公式值修改掉。所以增加了开关，在检索数据前不要回填，在检索结束后，再打开开关

        // 只有是公式时，才允许回填，
        // 比如当是包含公式时，比如RichEdit中先定义一个模板，模板中包含公式，
        // 如果有RichEdit对数据作修改后，数据填到ds 中，并触发单元格需要重新计算，
        // 此函数被调用，如果再用this.getValue()取得数据回填，那么其中的包含的公式就
        // 被取代成了实际的值，结果是其中的公式丢失了
        if (this.isFormula())
        {
            // 把innerRow由-1改成-99
            //这样修改是发现一个问题
            /* 比如  a1= b1+c1;且a1绑定了数据库
                     * b1,c1 也都绑定了数据库
                     * 并且b1上用 =1 定义初始值，
                     * 当修改c1后， a1 需要重装计算并回填，如果此时innerRow用 -1 那么，b1.getValue(-1) 它会返回公式 =1的结果
                     * 而不是当前数据库字段的值，
                     * =====
                     * 马上发现问题，公式定义不生效了。所以又改回来
                     * 当用立即数定义缺省值时，不要加＝号，比如不要用=1,直接用1即可，这样就不会引起上面的问题
                     * 而且用=1定义也是不合理的。因为它的确不是公式，而是想设置一个缺省值
                     * */
            let innerRow = -1;
            let value = this.getValue(innerRow); //注意一定要加下行号为-1 这样才会是从公式优先，因为 this.getValue()是数据库优先的
            this.setValueToDS(value, -1, true);
        }

    },

    /**
     *
     * Range
     */

    /**
     * @api {getMergedRange} 函数   getMergedRange
     *
     * @apiDescription getMergedRange()
     * <br><br> 返回合并单元格的区域
     *
     * @apiName  getMergedRange
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     *
     *
     * @apiSuccess (返回值){Range} - 返回合并单元格的区域
     *
     *
     * @apiExample   {js}示例：
     *
     *  let r= cell.getMergedRange();
     *  alert( JSON.toString(r));
     *
     */
    getMergedRange: function () {
        if (!this.isMerged()) return new Range(this.rowIndex, this.columnIndex, this.rowIndex, this.columnIndex);

        return this.Sheet.getMergeMap().get(this).getMergedRange();
    },


    /**
     * @api {mergedBy} 函数   mergedBy
     *
     * @apiDescription mergedBy()
     * <br><br> 返回本单元格被哪个单元格合并
     *
     * @apiName  mergedBy
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     *
     *
     * @apiSuccess (返回值){Cell} - 返回本单元格被哪个单元格合并
     *
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     */
    mergedBy: function () {
        if (!this.isMerged()) return null;
        let r = this.getMergedRange();
        return this.Sheet.cells(r.startRow, r.startCol);
    },


    /**
     * @api {isMerged} 函数   isMerged
     *
     * @apiDescription isMerged()
     * <br><br> 单元格是否有合并
     *
     * @apiName  retrieve
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     *
     *
     *
     * @apiSuccess (返回值){boolean} - 单元格是否有合并
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     */
    isMerged: function () {
        return this.Sheet.getMergeMap().containsKey(this)

    },


    /**
     * @api {repaint} 函数   repaint
     *
     * @apiDescription repaint()
     * <br><br> 单元格强制重绘
     *
     * @apiName  repaint
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     */
    repaint: function () {


        let view = this.Sheet.View;
        if (view == null) return;
        if (!view.Sheet.paintPermit) return; //不允许绘制，那么下面就不要做了， getShowRectangleOfCell也是很浪费时间的


        let /*Rectangle*/   rc = view.getShowRectangleOfCell(false, this.rowIndex, this.columnIndex);
        //向外扩展一些，因为要画边框，而这个边框，不仅是画自己的边框，而是相临单元格的
        //边框者要考虑到。比如：自己是无边框的，相临的是有边框的。如果只画自己，就会出现
        // 相临的边框被擦掉了。

        // 转换一个思路，如果相临的单元格共有的边的边框是一样的属性，这个问题不就解决了吗，所以不要单独设置某个单元格的边框，
        // 而是用设置一个区域边框的方式

        // console.info("cell重绘 " + this.name + "   ( " + rc.toString() + ")");
        let lastSameBindRow = this.rowIndex;

        let mrdsn = this.Sheet.RPM.getMultiRowDataSourceName(this.rowIndex);
        if (mrdsn != '')
        {
            lastSameBindRow = this.Sheet.RPM.findLastRowWhichBindToMultiRowDataSource(mrdsn);
        }
        //得到下一行的Y，整个区域都可重绘，因为本cell可能是绑定多行数据源的
        let y = this.Sheet.CPM.getColumnHeadHeight() + this.Sheet.RPM.getRowY(lastSameBindRow + 1);

        //2021.12.06修正bug
        //当单元格有合并 ，并且它没有绑定时，上面的y可能就是本单元格下一行的y ，假设本单元格合并了多行，用  y - rc.y 的高度就是单单 this.rowIndex这一行的高度
        // 会造成单元格画不全 , 改成 Math.max( rc.height,   y - rc.y) ;

        rc.height = Math.max(rc.height, y - rc.y);

        //如果是多行结果集，那么当行数很多时， height可能是很大的，所以还要限制在屏幕幕之内
        rc.height = Math.min(rc.height, view.getHeight() - rc.y);

        view.repaint(rc);

    },

    // 重绘本单元格对应的行头

    repaintRowHead: function () {
        let view = this.Sheet.View;
        if (view == null) return;
        let /*Rectangle*/rc = view.getShowRectangleOfCell(false, this.rowIndex, this.columnIndex);
        rc.x = 0;
        rc.width = this.Sheet.RPM.getRowHeadWidth();
        view.repaint(rc);
    },


    repaintAffect: function (/*[ ]*/ already) {
        if (this.AffectList == null) return;

        if (already == null) already = [];

        for (let ai = 0; ai < this.AffectList.length; ai++)
        {
            let affect = this.AffectList [ai];
            let cell = affect.cell;
            cell.repaint();
            //可能会循环通知，比如一个单元格的颜色依赖自已的值，这样，自已的值 变了，通知自己重绘，又通知
            //颜色表达式重绘（虽然它不需要重绘），颜色表达式又通知引用它的单元格（就是自已）重绘 ，这样就死循环了
            // 增加防止循环通知重绘
            if (already.contains(cell)) continue;
            already.add(cell);
            cell.repaintAffect(already);
            cell.repaintRowHead();
        }
    }
    ,


    //如果是网页中调用可能传入的参数为null


    /**
     * @api {setBind} 函数   setBind
     *
     * @apiDescription setBind(DataSource, dbCol, dsType)
     * <br><br> 单元格
     *
     * @apiName  setBind
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     * @apiParam {String} DataSource 数据源名称
     * @apiParam {String} dbCol 字段
     * @apiParam {int} dsType 绑定类型
     *
     * 可以是：
     * <ul>
     * <li>1:单行结果集</li>
     * <li>2:多行结果集</li>
     * <li>4:多行结果集的当前行</li>
     * </ul>
     *
     *
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     */
    setBind: function (DataSource, dbCol, dsType) {

        if (dsType == undefined) dsType = 0;

        if (DataSource == null)  DataSource="";
        if (dbCol == null) dbCol="";


        if( this.Bind!=null &&  DataSource!=""   &&  dbCol!= "")
        {
            let tb=new DBBindConfig(book, DataSource, dbCol, dsType);

            if( this.Bind.dataSource.equalsIgnoreCase( tb.dataSource ) &&
                this.Bind.DBCol.equalsIgnoreCase(tb.DBCol ) &&
                this.Bind.dsType == tb.dsType)
            {
                return;
            }
        }

        // 如果事先已经绑定到其它字段，那么先注销它
        this.clearBind();

        //System.out.println(" after clearbind" );
        if (DataSource == ""  ) return false;
        if (dbCol == "" ) return false;


        let dsc = this.Book.getDataSource(DataSource);
        if (dsc == null)
        {
            this.Bind = null;
            return false;
        }

        // 找到数据源配置

        DataSource = DataSource.toLowerCase().trim();
        this.Bind = new DBBindConfig(book, DataSource, dbCol, dsType);


        //不能直接用dbCol ,因为它可能是一个复合设置，
        let /*columnProperty*/  CP = dsc.dataStore.getColumnProperty(this.Bind.DBCol);
        if (CP == null)
        {
            this.Bind = null;
            return false;
        }

        // 保数据类型设置成与字段类型一致
        this.VT = CP.getUniformDataType();

        // 向该数据源的事件接收器注册本单元格

        let /*DataStoreEventAdapter*/ dsa = dsc.dataStore.EM.findListener(DataSource);
        dsa.registerCell(this);
        // 表示本行的绑定到字段的列数增加了一个

        this.Sheet.checkRowIsMultiRowDataSource(this.getMergedRange());


        return true;
    },

    /**
     * 得到合适的字体名称　,数字用"Times New Roman，其它用宋体
     * @return
     */


    getAppropriateFontName: function () {

        let type = this.getCellValueType();
        if (type == $Auto)
        {
            if (this.Bind != null)
            {
                let ds = this.Book.getDataSource(this.Bind.dataSource).dataStore;
                type = ds.getColumnProperty(this.Bind.DBCol).getUniformDataType();
            }

        }

        if (this.ES != null)
        {
            if (this.ES.ET != $Numeric) type = $Auto;

        }

        switch (type)
        {
            case $Integer:
            case $Numeric:
                return "Times New Roman";

            default:
                return "";
        }

    },


    /**
     * @api {clearBind} 函数   clearBind
     *
     * @apiDescription clearBind()
     * <br><br> 单元格取消绑定
     *
     * @apiName  clearBind
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     * @apiParam {String} where 检索条件
     *
     *
     * @apiSuccess (返回值){int} - 返回本次检索出的数据条数
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     */
    clearBind: function () {
        if (this.Bind == null) return;

        // 找到数据源配置


        let dsc = book.getDataSource(this.Bind.dataSource);
        if (dsc == null)
        {
            this.Bind = null;
            return;
        }

        // 向该数据源的事件接收器注销本
        let /*DataStoreEventAdapter*/ dsa = dsc.dataStore.getEvent().findListener(this.Bind.dataSource);
        dsa.unRegisterCell(this);
        // 表示本行的绑定到字段的列数减少了一个
        this.Sheet.RPM.ReleaseBindCount(this.rowIndex);
        this.Bind = null;
    },

    // 本函数不放到接口中
    // 当用 setthis.Bind绑定到某个数据源后，本函数被回调，用来设置容纳本Cell的行的高度


    $ResetContainerColumnWidth: function () {

    },

    /**
     * 计算单元格显示值占用的最大高度
     * @return  Point.x 是DBRowHeight，　point.y是RowHeight
     */

    /*Point*/
    getShowTextMaxHeight: function () {
        let /*Point*/ ret = new Point();

        let row = this.rowIndex;
        let RPM = this.Sheet.RPM;
        let mrc = RPM.getMaxDBRowCount(row);

        let dbrh = RPM.getDBRowHeight(row);

        ret.x = dbrh;
        ret.y = RPM.getRowHeight(row);

        let c = this.getValueCount();
        let /*Property*/prop = this.getPropertyObject();// 不要直接用成员变量this.property,因为它可能为空

        let view = this.Sheet.View;
        if (view == null) return ret;

        let /*Rectangle*/  textRect = view.getShowRectangleOfCell(false, this.rowIndex, this.columnIndex);

        //  边框可能占用的１个点,再减去边界
        textRect.width = textRect.width - 2 - prop.get(Property.padding_left, 1) - prop.get(Property.padding_right, 1);

        let padding = 4 + prop.get(Property.padding_top, 1) + prop.get(Property.padding_bottom, 1);

        let font_family = prop.get(Property.font_family, "宋体");
        if (font_family.equals("自动")) font_family = this.getAppropriateFontName();
        let font_size = parseInt(prop.get(Property.font_size, 12)) + (Tools.isMobile() ? 2 : 0); //2021.06.24 移动设备上字体偏小，所以自动放大一点
        let font_bold = prop.get(Property.font_bold, false);
        let font_italic = prop.get(Property.font_italic, false);
        let font_underline = prop.get(Property.font_underline, false);


        let /*Graphics2D*/   g = view.getGraphics();

        let textHeight = RowPropertyManage.defaultRowHeight;

        for (let i = 0; i < c; i++)
        {
            let t = this.getShowText(i);
            if (this.Prefix != '') t = this.Prefix + t;
            if (this.Suffix != '') t = t + this.Suffix;

            let th = Tools.MultiLineTextHeight(g, textRect.width, t, font_family, font_size,
                font_bold, font_italic, font_underline);
            textHeight = Math.max(textHeight, th + padding);

        }

        ret.x = textHeight;
        ret.y = mrc * textHeight;


        return ret;

    },

    //==========22:20====================================

    /**
     *  本函数不放到接口中
     * 在如下情况下，本函数本调用
     *
     *  1当用 setthis.Bind绑定到某个数据源后，本函数被回调，用来设置容纳本Cell的行的高度
     *  2 当绑定到数据库的数据行数发生变化（主要是由于 insert , delete , retrieve , filter 等操作引起 ）时被调用 ，
     *  参看 DataStoreEventAdapter中的诸多事件
     *    最终还是由$FireSelfValueChanged引发的 WorkBookAdapterIns中的CellValueChanged 来实际调用
     *  3 当没有绑定数据库，但Value发生变化时，并且 属性中auto_fit_height 为TRUE时，参看WorkBookAdapterIns中的CellValueChanged
     *  4 当属性发生改变时， 参看 WorkBookAdapterIns.CellPropertyChanged

     */


    $ResetContainerRowHeight: function () {

        //TODO

    },


    drawBackground: function (g, /*Rectangle*/cellRect, row, col) {

        let view = this.Sheet.View;

        let viewRect = view.getBounds();//整个view的尺寸

        let expandRect = cellRect.clone();
        let rb = this.getRightBottomCorner();


        //如果单元格位于多行绑定区域中，并且数据数据，并且是运行状态，并且是在移动设备上，那么不显示背景，只在有数据是才显示
        let mrdsn = view.RPM.getMultiRowDataSourceName(this.rowIndex);
        if (mrdsn != '')
        {
            let dsc = this.Sheet.Book.getDataSource(mrdsn);
            if (dsc.dataStore.rowCount == 0 && !this.Sheet.designMode && Tools.isMobile()) return;
        }

        //如果是最后一行，那么扩展区域到底边，主要是画背景
        if (rb.rowIndex == this.Sheet.rowCount - 1)
        {
            expandRect.height = viewRect.height - expandRect.y;
        } else
        {

            //如果它不是最后一行，
            //但是它是最后一列，那么要看看，是不是一个多行绑定的行
            //如果是，要把整个多行的背景都要画，而不仅仅是一个数据行
            if (rb.columnIndex == this.Sheet.columnCount - 1)
            {
                let mrdsn = view.RPM.getMultiRowDataSourceName(this.rowIndex);
                if (mrdsn != '' && this.Bind == null)
                {
                    let y2 = view.RPM.getRowY(rb.rowIndex + 1);

                    //如果这个多行源有很多数据，那么可能高度超出了可视区域。所以要限定一下
                    //TODO  后面的  + cellRect.height 理应不需要，但是实际效果是需要， 没时间调试了，先这样吧
                    expandRect.height = Math.min(viewRect.height - expandRect.y,
                        y2 - expandRect.y + cellRect.height);
                }
            }
        }

        if (rb.columnIndex == this.Sheet.columnCount - 1)
        {
            expandRect.width = viewRect.width - expandRect.x;


        }

        // 能到这里 cell 是不为空的


        // 背景色

        let background_color = this.getPropertyValue(Property.background_color, '');
        if (background_color != '' && background_color != 'transparent')
        {
            g.setColor(background_color);
            g.fillRect(expandRect);
        }

    },


    /**
     * 画边框：
     *   1 如果没有边框，或者边框宽度是0，那么需要看一下其它相临的边框
     *   2 如果是双线框，那么要注意角上的处理
     * @param g
     * @param cellRect
     */


    drawBorder: function (/*Graphics2D*/        g, /*Rectangle*/ cellRect) {

        if (!this.Sheet.designMode)
        {
            let RPM = this.Sheet.RPM;
            let mrdsn = RPM.getMultiRowDataSourceName(this.rowIndex);


            if (mrdsn != '')
            {

                let dsc = this.Book.getDataSource(mrdsn);
                let ds = dsc.dataStore;
                //绑定到多行的行，如果没有数据，并且是在mobile设备上， 那么网格了不要画了 ，这里建议加一个开关属性来控制
                if (ds.rowCount == 0 && Tools.isMobile()) return;
            }
        }

        let thisRow = this.rowIndex;
        let thisCol = this.columnIndex;

        let view = this.Sheet.View;
        // 能到这里 cell 是不为空的
        let property = this.getPropertyObject();

        // 边框
        // 先画网格线
        if (this.Sheet.gridLineVisible)
        {
            g.setColor(this.Sheet.gridLineColor);
            g.drawRect(cellRect.x, cellRect.y, cellRect.width, cellRect.height);
        }

        // 再画边框
        // 左
        let border_left_style = property.get(Property.border_left_style, 0);// 缺省是无
        let border_left_width = property.get(Property.border_left_width, 0);// 缺省是无
        let /*Color*/    border_left_color = property.get(Property.border_left_color, Color.BLACK);

        //如果左边框没有，那么看看自已左边的Cell的右边框是什么
        if (border_left_style == 0 || border_left_width == 0)
        {
            if (thisCol > 0)
            {
                let cell = this.Sheet.$cells(thisRow, thisCol - 1);
                if (cell != null)
                {
                    let /*Property*/p = cell.getPropertyObject();
                    border_left_style = p.get(Property.border_right_style, 0);
                    border_left_width = 1; // 宽度设成1就行了，这1个单位即显示了边框，也不会把边框加粗
                    border_left_color = p.get(Property.border_right_color, Color.BLACK);
                }
            }
        }

        view.addDrawCellBorderLine(view.drawBorderMap, g, border_left_style, border_left_width, border_left_color,
            cellRect.x, cellRect.y, cellRect.x, cellRect.y + cellRect.height, this.name + " -left");
        // 上
        let border_top_style = property.get(Property.border_top_style, 0);// 缺省是无
        let border_top_width = property.get(Property.border_top_width, 0);// 缺省是无
        let /*Color*/    border_top_color = property.get(Property.border_top_color, Color.BLACK);

        //		如果上边框没有，那么看看自已上边的Cell的下边框是什么
        if (border_top_style == 0 || border_top_width == 0)
        {
            if (thisRow > 0)
            {
                let cell = this.Sheet.$cells(thisRow - 1, thisCol);
                if (cell != null)
                {
                    let /*Property*/p = cell.getPropertyObject();
                    border_top_style = p.get(Property.border_bottom_style, 0);
                    border_top_width = 1; // 宽度设成1就行了，这1个单位即显示了边框，也不会把边框加粗
                    border_top_color = p.get(Property.border_bottom_color, Color.BLACK);
                }
            }
        }

        view.addDrawCellBorderLine(view.drawBorderMap, g, border_top_style, border_top_width, border_top_color,
            cellRect.x, cellRect.y, cellRect.x + cellRect.width, cellRect.y, this.name + " -top");
        // 右
        let border_right_style = property.get(Property.border_right_style, 0);// 缺省是无
        let border_right_width = property.get(Property.border_right_width, 0);// 缺省是无
        let /*Color*/    border_right_color = property.get(Property.border_right_color, Color.BLACK);

        //		如果右边框没有，那么看看自已右边的Cell的左边框是什么
        if (border_right_style == 0 || border_right_width == 0)
        {
            if (thisCol < (this.Sheet.columnCount - 1))
            {
                let cell = this.Sheet.$cells(thisRow, thisCol + 1);
                if (cell != null)
                {
                    let /*Property*/ p = cell.getPropertyObject();

                    border_right_style = p.get(Property.border_left_style, 0);
                    border_right_width = 1; // 宽度设成1就行了，这1个单位即显示了边框，也不会把边框加粗
                    border_right_color = p.get(Property.border_left_color, Color.BLACK);
                }
            }
        }
        view.addDrawCellBorderLine(view.drawBorderMap, g, border_right_style, border_right_width, border_right_color,
            cellRect.x + cellRect.width, cellRect.y, cellRect.x + cellRect.width, cellRect.y + cellRect.height, this.name + " -right");
        // 下
        let border_bottom_style = property.get(Property.border_bottom_style, 0);// 缺省是无
        let border_bottom_width = property.get(Property.border_bottom_width, 0);// 缺省是无
        let /*Color*/    border_bottom_color = property.get(Property.border_bottom_color, Color.BLACK);

        //		如果下边框没有，那么看看自已下边的Cell的上边框是什么
        if (border_bottom_style == 0 || border_bottom_width == 0)
        {
            if (thisRow < (this.Sheet.rowCount - 1))
            {
                let cell = this.Sheet.$cells(thisRow + 1, thisCol);
                if (cell != null)
                {
                    let /*Property*/p = cell.getPropertyObject();
                    border_bottom_style = p.get(Property.border_top_style, 0);
                    border_bottom_width = 1; // 宽度设成1就行了，这1个单位即显示了边框，也不会把边框加粗
                    border_bottom_color = p.get(Property.border_top_color, Color.BLACK);
                }
            }
        }

        view.addDrawCellBorderLine(view.drawBorderMap, g, border_bottom_style, border_bottom_width, border_bottom_color,
            cellRect.x, cellRect.y + cellRect.height, cellRect.x + cellRect.width, cellRect.y + cellRect.height, this.name + " -bottom");

    },


    getSplitLineColor: function () {
        let /*Property*/prop = this.getPropertyObject();// 不要直接用成员变量this.property,因为它可能为空
        let /*Color*/  splitLineColor = null;
        //如果有边框，那么找到边框的颜色，似边框颜色都是一样的，所以找到一个就结束
        if (splitLineColor == null && prop.get(Property.border_left_style, 0) > 0) return prop.get(Property.border_left_color, Color.BLACK);
        if (splitLineColor == null && prop.get(Property.border_top_style, 0) > 0) return prop.get(Property.border_top_color, Color.BLACK);
        if (splitLineColor == null && prop.get(Property.border_right_style, 0) > 0) return prop.get(Property.border_right_color, Color.BLACK);
        if (splitLineColor == null && prop.get(Property.border_bottom_style, 0) > 0) return prop.get(Property.border_bottom_color, Color.BLACK);
        if (splitLineColor == null) return this.Sheet.getGridLineColor();

    },

    // 用来显示文本的区域，去提边界,padding , 以及装饰的留边
    /**
     *
     * @param cellRect
     * @returns {Rectangle}
     */
    getTextRectangleOfCell: function (/*Rectangle*/ cellRect) {
        let /*Property*/    prop = this.getPropertyObject();// 不要直接用成员变量this.property,因为它可能为空

        let /*Rectangle*/    bordertRect = new Rectangle();

        // 除去边框可能占用的１个点,再减去边界
        let /*Rectangle*/ textRect = new Rectangle();

        //如果有装饰，装饰的边距也要去掉
        let /*int*/ ph = 0;
        let /*int*/ pv = 0;

        textRect.x = cellRect.x + 1 + prop.get(Property.padding_left, 1) + ph;
        textRect.y = cellRect.y + 1 + prop.get(Property.padding_top, 1) + pv;
        //除去留边后，的尺寸最小为1，如果为0 ，那么在cell的draw中可能就不会被绘制了
        textRect.width = Math.max(1, cellRect.width - 2 - prop.get(Property.padding_left, 1)
            - prop.get(Property.padding_right, 1) - ph * 2);
        textRect.height = Math.max(1, cellRect.height - 2 - prop.get(Property.padding_top, 1)
            - prop.get(Property.padding_bottom, 1) - pv * 2);


        return textRect;
    },


    getExpressionValue: function (defaultValue, font_color_expression, dbRow, vt) {
        if (font_color_expression == "") return defaultValue;
        let expressionName = font_color_expression;
        let /*Cell*/ expression = this.Sheet.getExpression(expressionName);
        if (expression == null)
        {
            this.Sheet.addExpression(expressionName, font_color_expression);
            expression = this.Sheet.getExpression(expressionName);
            expression.addSubscriber(this);
        }

        if (expression == null) return defaultValue;

        let v = null;

        if (dbRow == -1)
        {
            v = expression.getValue();
        } else
        {
            v = expression.getValue(dbRow);
        }


        // 对于公式  = if( id>0, rgb(255,0,0) , null )  当id>0时，返回的不是null ，而是　""
        if (v != null)
        {
            if (v.toString() == "") v = defaultValue;
        }

        try
        {
            //当出现绑定到多行数据源的单元格取消绑定后，就会现出如下v 是一个数组的情况
            if (Util.isArray(v)) return defaultValue;

            v = ObjectTool.changeType(v, vt);
        } catch (e)
        {
            v = defaultValue;
        }

        if (v == null) return defaultValue;

        return v;

    },

    /**
     * @api {ensureOnScreen} 函数   ensureOnScreen
     *
     * @apiDescription  ensureOnScreen(dbRow)
     * <br><br> 确保单元整处于显示屏上的可见区域。如果是绑定到多行结果集，则确保其中的 dbRow处于可见状态
     *
     * @apiName  ensureOnScreen
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     * @apiParam {int} dbRow 数据行号，当没有绑定数据库或绑定单行结果集时，本参数无意义
     *
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *   book.sheet0.cells("d15").ensureOnScreen(3);
     *
     */
    ensureOnScreen: function (dbRow) {
        let row = this.getRowIndex();
        let col = this.getColumnIndex();
        this.Sheet.setSelection(row, col, row, col);
        if (this.Bind != null)
        {
            let dsc = this.Sheet.Book.getDataSource(this.Bind.dataSource);
            let rc = dsc.dataStore.rowCount;
            if (dbRow < rc) dsc.setCurrentBindRow(dbRow);
        }
        this.Sheet.View.insureSelectionOnScreen(null);

    },


    /**
     * @api {getBounds} 函数   getBounds
     *
     * @apiDescription  getBounds(dbRow)
     * <br><br> 返回当前状态下，单元格dbRow行所占据的矩形区域
     *
     * @apiName  getBounds
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     * @apiParam {int} dbRow 数据行号，当没有绑定数据库或绑定单行结果集时，本参数无意义
     *
     *
     *
     * @apiSuccess (返回值){Rectangle} - 返回单元格dbRow行所占据的矩形区域 ， Rectangle是这样结构的对象
     *  {x,y,width,height}
     *
     *
     * @apiExample   {js}示例：
     *
     *  let rc= book.sheet0.cells("b3").getBounds(2);
     *  alert( rc.x+"  "+rc.y+"  "+ rc.width+"    "+ rc.height);
     *
     */
    getBounds: function (dbRow) {


        let rc = this.Sheet.View.getShowRectangleOfCell(false, this.getRowIndex(), this.getColumnIndex());

        if (this.Bind != null)
        {

            let dsn = this.Bind.dataSource;
            //2019.06.25取消了限制
            if (this.Bind.dataSourceType == DataSourceConfig.MultiRowDataSource)
            {
                let oneDataRowHeight = this.Sheet.RPM.getOneDataRowHeightOfDataSource(dsn);

                let dsc = this.Sheet.Book.getDataSource(dsn);

                rc.y = rc.y + dbRow * oneDataRowHeight;
                rc.height = oneDataRowHeight;
            }
        }
        return rc;

    },

    /**
     * @api {grumble} 函数   grumble
     *
     * @apiDescription grumble (info, showAfter, hideAfter, dbRow)
     * <br><br> 单元格位置处显示一个提示信息
     *
     * @apiName  grumble
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     * @apiParam {String} info 提示信息
     * @apiParam {int} showAfter 延时显示
     * @apiParam {int} hideAfter 延时关闭
     * @apiParam {int} dbRow 子行号
     *
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     */
    grumble: function (info, showAfter, hideAfter, dbRow) {
        let rc = this.getBounds(dbRow);
        let x = rc.x;
        let y = rc.y;
        let cx = rc.x + rc.width / 2;
        let cy = rc.y + rc.height / 2;

        let viewDom = $('#' + this.Sheet.Book.View.containerID);
        let bookWidth = viewDom.width();
        let bookHeight = viewDom.height();

        let tx, ty;
        let angle = 0;
        let alignX = 'left';

        //在book的4个不同象限，使用不同的角度
        if (cx > bookWidth / 2 && cy < bookHeight / 2)
        {
            alignX = 'left';
            tx = x;
            ty = cy;
            angle = 225;
        }
        if (cx < bookWidth / 2 && cy < bookHeight / 2)
        {
            alignX = 'right';
            tx = x + rc.width;
            ty = cy;
            angle = 135;

        }
        if (cx < bookWidth / 2 && cy > bookHeight / 2)
        {
            alignX = 'right';
            tx = x + rc.width;
            ty = cy;
            angle = 45;
        }
        if (cx > bookWidth / 2 && cy > bookHeight / 2)
        {
            alignX = 'left';
            tx = x;
            ty = cy;
            angle = 315;
        }

        //在 bookView中放了一个basePoint  DOM对象，用它来定位气泡提示的坐标

        let hasHideButton = false;
        if (hideAfter < 500) hideAfter = 500;
        if (hideAfter > 60000) hideAfter = 60000;
        if (hideAfter == undefined)
        {
            hideAfter = false;
            hasHideButton = true;
        }

        let tipBasePoint = this.Book.View.tooltipBasePoint();
        tipBasePoint.css("left", tx + 'px');
        tipBasePoint.css("top", ty + 'px');


        tipBasePoint.poshytip({
            content: info,
            showOn: 'none',
            alignTo: 'target',
            alignX: alignX,
            alignY: 'center',
            offsetX: 0,
            offsetY: 0,
            bgImageFrameSize: 10,
            className: 'tip-yellowsimple',
            timeOnScreen: hideAfter
        });
        tipBasePoint.poshytip('show');

    },


    popover: function (title, type, contentOrURL, innerRow, moreConfig) {
        if (innerRow == null) innerRow = 0;
        if (moreConfig == null) moreConfig = {};
        let rc = this.getBounds(innerRow);
        let x = rc.x;
        let y = rc.y;
        let cx = rc.x + rc.width / 2;
        let cy = rc.y + rc.height / 2;

        let viewDom = $('#' + this.Sheet.Book.View.containerID);
        let bookWidth = viewDom.width();
        let bookHeight = viewDom.height();

        let tx, ty;

        let alignX = 'left';

        let placement = '';

        //在book的4个不同象限，使用不同的角度
        if (cx > bookWidth / 2 && cy < bookHeight / 2)
        {
            placement = 'bottom-left';
            tx = x + 10;
            ty = cy;

        }
        if (cx < bookWidth / 2 && cy < bookHeight / 2)
        {
            placement = 'bottom-right';
            tx = x + rc.width - 10;
            ty = cy;


        }
        if (cx < bookWidth / 2 && cy > bookHeight / 2)
        {
            placement = 'top-right';
            tx = x + rc.width - 10;
            ty = cy;

        }
        if (cx > bookWidth / 2 && cy > bookHeight / 2)
        {
            placement = 'top-left';
            tx = x + 10;
            ty = cy - 30;

        }


        placement = 'auto';
        //坐标修正
        //可能在嵌入Sheet中
        if (this.Sheet.pSheetContainer)
        {
            var p = this.Sheet.View.viewContainer.getBoundingClientRect();

            tx +=   p.x;
            ty +=   p.y;
        }

        //在 bookView中放了一个basePoint  DOM对象，用它来定位气泡提示的坐标


        let tipBasePoint = this.Book.View.tooltipBasePoint();
        tipBasePoint.css("left", tx + 'px');
        tipBasePoint.css("top", ty + 'px');

        let config = moreConfig;
        config.placement = placement;
        config.title = title;
        config.type = type;
        config.animation = 'fade';
        config.trigger = 'manual';
        config.multi = true;
        // config.cache=false;//不要缓存，每次都重新创建，不然始终只显示第一次的内容

        if (type == 'html') config.content = contentOrURL;
        if (type == 'iframe') config.url = contentOrURL;


        tipBasePoint.webuiPopover(config);
        tipBasePoint.webuiPopover('show');


    },

    popoverHide: function () {
        this.Sheet.popoverHide();

    },

    //当需要显示单元格提示信息时，本函数被调用
    //本函数在 SelectTool.js 中启用
    onTooltipBegin: function (innerRow) {
        if (this.popoverContent == '') return;
        if (this.Sheet.designMode) return;

        let url = this.popoverContent;
        if (url.startsWith("=")) url = this.Sheet.evaluate(url, innerRow);
        if (url == '') return;
        if (url == null) return;

        if (this.popoverType == 'script') return; //上面已经执行了

        let moreConfig = {};
        if (this.popoverDialogWidth != '') moreConfig.width = parseInt(this.popoverDialogWidth);
        if (this.popoverDialogHeight != '') moreConfig.width = parseInt(this.popoverDialogHeight);
        if (!this.popoverAutoClose) moreConfig.closeable = true; //表示需要手工关闭
        let title = this.popoverDialogTitle;
        if (title.startsWith("=")) title = this.Sheet.evaluate(title, innerRow);


        this.popover(title, this.popoverType, url, innerRow, moreConfig);


    },

    //当需要隐藏提示信息时，本函数被调用
    //本函数在 SelectTool.js 中启用
    // 当新的单元格需要显示提示信息，时，旧的正在显示的提示，无论是否自动关闭，都会被强行关闭，参看SelectTools.js标记20211214
    onTooltipEnd: function (force) {
        if (force == null) force = false;
        if (force)
        {
            this.popoverHide();
            return;
        }
        //避免无意义的操作
        if (this.popoverContent == '') return;
        if (this.Sheet.designMode) return;
        //仅在需要自动关闭时， 才关闭，否则需要手工点击关闭
        if (this.popoverAutoClose || force) this.popoverHide();
    },


    /**
     *
     * @param isPrint
     * @param g
     * @param cellRect
     */
    draw: function (/*{boolean}*/isPrint, /*{Graphics2D}*/ g, /*{Rectangle}*/cellRect) {


        // 如果这个单元格是独立存在的，而不是加入到行中的，那么它不需要绘画 , 也不一定， 比如页眉页脚就是用一个虚拟单元格来定义
        //if (this.pRow == null) return;

        //		如果单元格正在编辑中，那么不要绘制自己
        // 比如当是RichEdit编辑方式时，如果选择字体，这时，字体列表框在消失后
        //会导致屏幕重绘，如果此时重绘，且这个Cell绑定到数据库，那么会出现单元格绘制与编辑
        // 框重叠显示的效果. 但是不能直接在这里进行判断。因为当是多行绑定时，只是对当前编辑的
        // 行对应的数据不需要重绘，所以要在下面进行判断，参看 [标记2005122501] [标记2005122502] [标记2005122503]

        if (cellRect.width == 0 || cellRect.height == 0) return;

        let /*Property*/ prop = this.getPropertyObject();// 不要直接用成员变量this.property,因为它可能为空

        let /*Rectangle*/ borderRect = new Rectangle();

        // 用来显示文本的区域，去提边界,padding , 以及装饰的留边
        let /*Rectangle*/ textRect = this.getTextRectangleOfCell(cellRect);
        // 文本属性


        let firstRowTextRect = textRect.clone();

        let /*Color*/ font_color_default = prop.get(Property.color, Color.BLACK);
        let /*String*/ font_color_expression = prop.get(Property.color_expression, "");
        let /*String*/ background_color_expression = prop.get(Property.background_color_expression, "");

        // 背景色

        let background_color = prop.get(Property.background_color, Color.WHITE);

        //设计模式下，都可见，不然不好设计
        let cellVisible = (isPrint && this.printable && this.visible) ||
            (!isPrint && this.visible) ||
            this.Sheet.designMode;


        //运行时不可见，直接返回,不画了

        let /*String*/ display_expression = prop.get(Property.display_expression, "").trim();//是否显示属性的定义

        let /*Color*/ groupRowBackground = new Color(0x0000ff);
        let /*String*/ font_family = prop.get(Property.font_family, "宋体");
        if (font_family.equals("自动")) font_family = this.getAppropriateFontName();
        let isDateEditStyle = false;
        if (this.ES != null)
        {
            if (this.ES.ET == EditStyle.$CheckBox && !isPrint) font_family = "FontAwesome";
            if (this.ES.ET == EditStyle.$Datetime) isDateEditStyle = true;
        }

        let /*int*/ font_size = parseInt(prop.get(Property.font_size, 12)) + (Tools.isMobile() ? 2 : 0); //2021.06.24 移动设备上字体偏小，所以自动放大一点

        let /*boolean*/ font_bold = prop.get(Property.font_bold, false);
        let /*boolean*/ font_italic = prop.get(Property.font_italic, false);
        let /*boolean*/ font_underline = prop.get(Property.font_underline, false);
        let /*int*/ align_h = prop.get(Property.text_align, ALIGNMENT.Left);
        let /*int*/ align_v = prop.get(Property.vertical_align, ALIGNMENT.Center);

        //绑定多行的单元格，需要计算每个单元格坐标，y会需要重新计算，
        let /*int*/ padding_top = prop.get(Property.padding_top, 1);

        // 上面统一处理了背景，所以这里不要再处理背景
        let /*boolean*/ word_wrap = prop.get(Property.word_wrap, false);
        // 如果字号大于16那么做柔化处理
        let /*boolean*/ ANTIALIAS_ON = false;
        if (font_size > 16 || !font_family != '' || this.Sheet.viewScale != 100)
        {
            ANTIALIAS_ON = true;
        }


        let /*int*/ rowIndex = this.rowIndex;


        let /*int*/ yOffset = 0; //  单元格内在垂直方向显示的偏移量

        //  增加对多行数据的分隔线的处理
        let /*Color*/ splitLineColor = this.getSplitLineColor();

        // 如果没有绑定， 但这个值可能是Vector

        let /*Rectangle*/ clipRect = g.invalidateRect; //用来在下面做conatains测试


        let sheet = this.Sheet;
        let RPM = this.Sheet.RPM;

        let CC = this.Sheet.columnCount; //列数
        let gridFocusRowSign = this.Sheet.Book.options.gridFocusRowSign;

        try
        {
            // 限制在单元格区域内，不要画出去
            //clip后，新的裁剪区是当前裁剪区和指定矩形的交集,clip 只能使当前裁剪区域更小，如果希望增大裁剪区域，则使用setClip.
            g.save();


            let mrdsn = RPM.getMultiRowDataSourceName(this.rowIndex);

            let startRow = 0;
            let endRow;
            let dataRC = 0;
            let currentRow;
            //是不是需要画亮显的行
            let needDrawLightBar;
            let isFirstColBindToDSN = false; //是不是绑定到该结果集的第一个列

            //下面计算这三个变量 分别是 一个完整明细行的高度， 绑定该结果集的最小列号及最大列号

            let oneDataRowHeight = cellRect.height;
            let minColumnIndexBinded = 999;
            let maxColumnIndexlBinded = 0;

            if (mrdsn == '')  //不是绑定到多行的行，那么只需要显示 一个数据，甭管它是数据绑定，还是公式，还是静态值
            {

                endRow = 0;
                currentRow = 0;
                needDrawLightBar = false;

            } else
            {

                let dsc = this.Book.getDataSource(mrdsn);
                let ds = dsc.dataStore;

                dataRC = ds.rowCount;


                //如果当前已经不在多行绑定区了，那么如果一行一行循环计算判断，就慢了，所以先直接全高比较，确定y是否落在多行绑定区之内
                //数据行数，默认需要有一行数据的高度

                currentRow = dsc.currentBindRow;
                endRow = dataRC - 1; //注意下面是<=


                //下面的其实可以在外面一次性计算好，每个单元格就不要计算了，节省时间

                //2020.05.27 实现从缓存中取这些信息，不需要每次都算
                let foundInCache = false;
                if (!sheet.designMode) //设计时，强制重新计算，运行时才从缓存取
                {
                    let cache = sheet.multiRowDataSourceSomeConfigCache[mrdsn];
                    if (cache != undefined)
                    {
                        minColumnIndexBinded = cache.minColumnIndexBinded;
                        maxColumnIndexlBinded = cache.maxColumnIndexlBinded;
                        oneDataRowHeight = cache.oneDataRowHeight;
                        foundInCache = true;
                    }

                }

                if (!foundInCache)
                {
                    //如果绑定到多行数据源，那么看看整个多行绑定区的总高度，
                    let firstSameBindRow = RPM.findFirstRowWhichBindToMultiRowDataSource(mrdsn);
                    let lastSameBindRow = RPM.findLastRowWhichBindToMultiRowDataSource(mrdsn);
                    //如果当前已经不在多行绑定区了，那么如果一行一行循环计算判断，就慢了，所以先直接全高比较，确定y是否落在多行绑定区之内
                    //数据行数，默认需要有一行数据的高度

                    //计算最大及最小列号，绑定到mrdsn上的


                    for (let ci = 0; ci < CC; ci++)
                    {
                        for (let ri = firstSameBindRow; ri <= lastSameBindRow; ri++)
                        {
                            let cell = this.Sheet.$cells(ri, ci);

                            if (cell == null) continue;
                            let b = cell.Bind;
                            if (b == null) continue;
                            if (b.dataSource != mrdsn) continue;

                            let colIndex = cell.columnIndex;
                            if (colIndex < minColumnIndexBinded) minColumnIndexBinded = colIndex;
                            if (colIndex > maxColumnIndexlBinded) maxColumnIndexlBinded = colIndex;


                            //最大的列，要看本单元格合并区域的右下角所在的列
                            let rb = cell.getRightBottomCorner();
                            if (rb != cell)
                            {

                                let colIndex = rb.columnIndex;
                                if (colIndex > maxColumnIndexlBinded) maxColumnIndexlBinded = colIndex;

                            }
                        }
                    }


                    oneDataRowHeight = 0;//多行数据源区，一行数据的高度
                    for (let di = firstSameBindRow; di <= lastSameBindRow; di++)
                    {
                        oneDataRowHeight += RPM.getRowHeight(di);
                    }

                    //计算好了，缓存起来
                    let cache = {};
                    cache.minColumnIndexBinded = minColumnIndexBinded;
                    cache.maxColumnIndexlBinded = maxColumnIndexlBinded;
                    cache.oneDataRowHeight = oneDataRowHeight;
                    sheet.multiRowDataSourceSomeConfigCache[mrdsn] = cache;

                }

            }


            //在设计状态下，静态文字，上面的 dataRC -1 == -1 ,会导致不显示，这样不方便设计
            //但是在运行时，如果没有数据行， 明细行的静态文字也不需要显示
            if (this.Sheet.designMode) endRow = Math.max(startRow, endRow);

            let freezedLineY = -1;
            for (let i = startRow; i <= endRow; i++)
            {

                //y坐标需加再加上分类汇总行所占据的调度
                //2019。11.04 修正， 需要加上 padding_top
                // 2020.05.02 将旧的计算方法 cellRect.y + i * oneDataRowHeight + 1 - yOffset + padding_top; 作废，
                // 改成直接由最初始计算的文本坐标的y +偏移就好
                textRect.y = firstRowTextRect.y + i * oneDataRowHeight - yOffset;


                //如果在显示区的上面，并且不用管汇总行
                if (textRect.y + textRect.height < clipRect.y)
                {

                    continue;
                }

                //2021.12.03 增加优化
                //如果是一个绑定多行数据源的单元格，并且当前sheet有行冻结，那么这个多行结果集通常不太会绑定到冻结行之上
                //所以当子数据行在冻结线之上时，通常是不需要显示的

                if (mrdsn != '')
                {
                    // freezedLineY<0 用来表示冻结线Y坐标还没有计算，它只需要计算一次，不要循环中每次都计算
                    //更好的是做为一个全局变量存在，避免每个单元格都去算它，但是这又带来一个问题，在调整行高后，这个值要重新计算，这样程序又
                    //更复杂了，折中一下， 在需要的单元格内临时计算一次
                    if (freezedLineY < 0) //需要计算冻结线的Y坐标
                    {
                        freezedLineY = this.Sheet.CPM.getColumnHeadHeight() + this.Sheet.RPM.getFixedRowHeight();
                    }
                    if (textRect.y + textRect.height < freezedLineY)
                    {
                        continue;
                    }
                }
                //2021.12.03增加结束


                let ignoreThisRow = false;


                if (clipRect.intersects(textRect) && !ignoreThisRow) // 如果在裁剪区内，这画这个数据
                {
                    //画分隔框
                    //System.out.println( "drawCell "+this.getName()+" innerRow:"+i);

                    borderRect.x = cellRect.x;
                    borderRect.y = cellRect.y + i * oneDataRowHeight - yOffset;
                    borderRect.width = cellRect.width;
                    borderRect.height = cellRect.height;


                    g.setColor(splitLineColor);


                    this.drawBorder(g, borderRect);

                    let visible = true;


                    if (!isPrint && this.isEditing() && i == currentRow)
                    {

                    } else
                    {


                        let font_color = this.getExpressionValue(font_color_default, font_color_expression, i,
                            UniformDataType.$Auto);


                        if (!background_color_expression == "")
                        {
                            background_color = this.getExpressionValue(null, background_color_expression, i,
                                UniformDataType.$Auto);
                        }

                        if (background_color != null && background_color != '' && background_color != 'transparent')
                        {
                            g.setColor(background_color);
                            g.fillRect(borderRect);
                        }

                        if (this.Bind != null && this.Editable && this.Sheet.Editable)
                        {

                            if (false) // drawFocus
                            {
                                let focusRC = borderRect.clone();


                                focusRC.x += 1;
                                focusRC.y += 1;
                                focusRC.width -= 2;
                                focusRC.height -= 2;
                                g.setLineDash(1);
                                g.setColor(new Color(255, 0, 0, 0.9));
                                g.drawRect(focusRC);
                            }

                            //可编辑的单元格是否
                            let dsc = this.Sheet.Book.getDataSource(this.Bind.dataSource);
                            //2020.10.05 增加了全局设置项
                            let hc = dsc.highlightEditableColumn && this.Sheet.Book.options.editableCellHighlightBackground;
                            if (hc)
                            {

                                g.setColor(this.Sheet.Book.options.editableCellHighlightBackgroundColor);
                                g.fillRect(borderRect);
                            }

                        }


                        //是不是需要画亮显的行
                        needDrawLightBar = mrdsn != '' && endRow >= 0 && currentRow >= 0 && dataRC > 0 && !this.Sheet.designMode;
                        //如果本行可能需要画亮行，那么进一步判断 ，

                        if (needDrawLightBar && !isPrint)
                        {
                            //如果本单元格不在最小绑定及最大绑定列之间，那么就要看它是不是公式，是不是有绑定，是则画，㫘则不画亮条
                            if (this.columnIndex < minColumnIndexBinded || this.columnIndex > maxColumnIndexlBinded)
                            { //如果有绑定，或是公式，那么才需要，如果是空的，没内容或静态内容，那么不要显示亮行
                                needDrawLightBar = needDrawLightBar && (this.Bind != null || this.isFormula());
                            }

                            isFirstColBindToDSN = this.columnIndex == minColumnIndexBinded;
                        }


                        if (i % 2 == 1 && !isPrint && this.Sheet.alternantBackground)
                        {
                            //有内容，且可能与数据源有关，才需要
                            if (this.Bind != null || this.isFormula() ||
                                (this.columnIndex >= minColumnIndexBinded && this.columnIndex <= maxColumnIndexlBinded))
                            {

                                g.setColor(this.Sheet.Book.options.gridAlternantBackgroundColor);
                                g.fillRect(borderRect);
                            }
                        }

                        // 亮显当前行
                        if (needDrawLightBar && currentRow == i && !isPrint
                            && this.Sheet.Book.highlightDataSourceCurrentRow
                            && this.Sheet.highlightDataSourceCurrentRow
                        )
                        {

                            g.setColor(this.Sheet.Book.options.gridFocusRowBackgroundColor);
                            g.fillRect(borderRect);


                            if (this.Sheet.Book.options.gridFocusRowUseUnifiedTextColor && font_color_expression == '')
                            {
                                let c = this.Sheet.Book.options.gridFocusRowTextColor;
                                font_color = c;
                            }

                        }

                        visible = true;
                        if (!display_expression == "")
                        {
                            visible = this.getExpressionValue(Boolean.TRUE, display_expression, i, UniformDataType.$Auto);
                        }


                        if (visible && cellVisible)
                        {
                            let v = this.getValue(i);
                            let st = "";
                            if (this.ES !== null && this.ES.ET == EditStyle.$CheckBox)
                            {
                                if (isPrint)
                                {
                                    st = (v == this.ES.checkOnValue) ? '[√]' : '[ ]';
                                } else
                                {
                                    st = (v == this.ES.checkOnValue) ? this.ES.checkOnChar : this.ES.checkOffChar;
                                }
                                let t = this.ES.checkCaption || '';
                                if (t != '') t = ' ' + t;
                                st = st + t;

                            } else
                            {
                                st = this.getShowText(i);
                                if (this.Prefix != '') st = this.Prefix + st;
                                if (this.Suffix != '') st = st + this.Suffix;


                            }
                            //2020.04.29 增加边框装饰的处理

                            if (this.m_decorate != null && Object.getOwnPropertyNames(this.m_decorate).length > 0)
                            {
                                this.drawDecorate(g, borderRect);
                            }


                            if (needDrawLightBar && !isPrint && isFirstColBindToDSN && gridFocusRowSign != '')
                            {
                                //如果是第一个绑定到该多行结果集的列，并且是需要显示焦点行标记时，那么要空出位置给焦点标记
                                //避免标记与单元格内容重叠显示
                                let trc = textRect.clone();
                                trc.x += font_size + 2;
                                trc.width -= font_size + 2;

                                this.drawString(isPrint, i, g, cellRect, borderRect, trc, v, st, 0, null, font_color,
                                    font_family, font_size, font_bold, font_italic, font_underline,
                                    this.$AutoHalign(align_h), align_v, word_wrap);

                            } else
                            {
                                this.drawString(isPrint, i, g, cellRect, borderRect, textRect, v, st, 0, null, font_color,
                                    font_family, font_size, font_bold, font_italic, font_underline,
                                    this.$AutoHalign(align_h), align_v, word_wrap);
                            }
                            //2019.11.02 增加 如果是下拉编辑格式，并且需要显示下拉按钮，则显示一个下拉标记
                            if (!isPrint && this.isEditable() && this.ES !== null
                                && this.ES.ET == EditStyle.$DDLB && this.ES.dropDownButtonVisible)
                            {
                                let tmpRC = textRect.clone();
                                tmpRC.x = tmpRC.x + tmpRC.width - font_size;
                                //"\uf078"
                                this.drawString(isPrint, i, g, cellRect, borderRect, textRect, "",
                                    this.Sheet.Book.options.ddlbSign, 0, null, font_color,
                                    "FontAwesome", font_size, false, false, false,
                                    ALIGNMENT.Right, ALIGNMENT.Center, false);

                            }

                            //如果是绑定到结果集的第一个列，那么显示一个当前行标记

                            if (needDrawLightBar && currentRow == i && !isPrint
                                && this.Sheet.Book.highlightDataSourceCurrentRow
                                && this.Sheet.highlightDataSourceCurrentRow
                                && isFirstColBindToDSN
                                && gridFocusRowSign != ''
                                && !this.Sheet.designMode
                                && textRect.width > 20
                            )
                            {
                                let trc = textRect.clone();
                                trc.x += 2;

                                this.drawString(isPrint, i, g, cellRect, borderRect, trc, "", gridFocusRowSign, 0, null, font_color,
                                    "FontAwesome", font_size, false, false, false,
                                    ALIGNMENT.Left, ALIGNMENT.Center, false);


                            }

                        }

                    }


                } else
                {
                    if (textRect.y > clipRect.y + clipRect.height) break;
                }

            }


            // 如果有小物件，那么画它们
            if (this.BrickMap != null)
            {

                for (let brickName in this.BrickMap)
                {
                    let brick = this.BrickMap[brickName];

                    brick.paint(isPrint, g, cellRect);
                }
            }

            // 最后处理一些标记性信息
            // 如果是设计模式，并且是显示绑定信息，那么
            if (this.Bind != null && this.Sheet.designMode && this.Sheet.Book.cellShowBind)
            {
                let t = this.Bind.DBCol + "@" + this.Bind.dataSource;
                g.drawStringSingleRow(firstRowTextRect, t, 0, null, Color.green, "宋体", 12, false, false, false,
                    ALIGNMENT.Left, ALIGNMENT.Top);

            }

            if (this.alias != '' && this.Sheet.designMode && this.Sheet.Book.cellShowAlias)
            {
                let t = this.alias;
                g.drawStringSingleRow(firstRowTextRect, t, 0, null, Color.red, "宋体", 12, false, false, false,
                    ALIGNMENT.Left, ALIGNMENT.Bottom);

            }

            if (this.Define != null && this.Define != '' && this.Sheet.designMode && this.Sheet.Book.cellShowFormula)
            {
                let t = '' + this.Define;
                if (t.startsWith('='))
                {
                    g.drawStringSingleRow(firstRowTextRect, t, 0, null, Color.blue, "宋体", 12, false, false, false,
                        ALIGNMENT.Left, ALIGNMENT.Bottom);
                }

            }

            //如果是日期，且可编辑，那么它就显示 一个日历图标
            if (this.isEditable() && this.Sheet.Editable && this.visible)
            {
                if (isDateEditStyle)
                {
                    g.drawStringSingleRow(firstRowTextRect, "\uf073", 0, null, new Color(15, 186, 223, 1),
                        "FontAwesome", font_size,
                        false, false, false, ALIGNMENT.Right, ALIGNMENT.Center);
                }
            }


            if (this.dblClickSortSignChar != '') //&& !this.Sheet.designMode) 设计状态下也显示排序标记便于了解它被设置了双击排序
            {


                g.drawStringSingleRow(firstRowTextRect, this.dblClickSortSignChar, 0, null,
                    this.Sheet.Book.options.dblClickToSortSignColor, "FontAwesome", 12,
                    false, false, false, ALIGNMENT.Right, ALIGNMENT.Bottom);


            }

            //设计模式下显示一些标记

            let flag = "";
            //显示一个单元格是可编辑的标记
            if (this.Sheet.designMode)
            {

                if (this.isEditable()) flag += "\uf246";  //可编辑属性
                if (this.appData)
                {
                    if (this.appData.clickControlEnabled) flag += "\uf0a7";  //点击需控制权限

                    if (this.clickJumpTo || '' != '') flag += "\uf1d9"; //单元格点击动作
                    if (this.dblClickSort || '' != '') flag += "\uf15d";

                    //取数
                    if (this.appData['valueChangedToQuery1'] != undefined ||
                        this.appData['valueChangedToQuery2'] != undefined ||
                        this.appData['valueChangedToQuery3'] != undefined ||
                        this.appData['valueChangedToQuery4'] != undefined)
                    {
                        flag += "\uf12e";
                    }

                    //补全
                    if (this.appData['valueChangedToAutoCompleteType'] != undefined) flag += "\uf035";

                }


                //校验规则
                if (this.validateRule || '' != '')
                {
                    flag += "\uf0e3";
                }

                if (this.dblClickSort != '') flag += "\uf0d7"; //双击排序


            }
            //悬停提示
            if (this.popoverContent != '') flag += "\uf27b";

            if (flag != '')
            {
                g.drawStringSingleRow(firstRowTextRect, flag, 0, null,
                    this.Sheet.designMode ? Color.red : "#52B8DD",
                    "FontAwesome", 9,
                    false, false, false,
                    ALIGNMENT.Right,
                    ALIGNMENT.Top);
            }


        } catch (e)
        {
            console.error(e);
        } finally
        {


            // 如果是设计状态，不可见， 那么加一个图标标记
            if (!this.visible && this.Sheet.designMode)
            {

                g.drawStringSingleRow(firstRowTextRect, "\uf070", 0, null, Color.red, "FontAwesome", 12,
                    false, false, false, ALIGNMENT.Right, ALIGNMENT.Center);
            }

            g.restore();

        }


    },


    drawDecorate: function (g, borderRect) {
        if (this.m_decorate == null) return;

        //计算尺寸
        let rc = this.getDecorateBounds(borderRect);
        g.save();

        let b = this.m_decorate['border'];
        b = b.split('\s'); //空格分隔

        let borderWidth = parseInt(b[0].replace(/px/g, ''));
        let borderColor = new Color(b[2]);
        let borderRadius = parseInt((this.m_decorate['border-radius'] || '0').replace(/px/g, ''));

        let backgroundColor = null;
        if (this.m_decorate['background-color']) backgroundColor = new Color(this.m_decorate['background-color']);


        g.drawRoundRectPath(rc.x, rc.y, rc.width, rc.height, borderRadius);

        g.stroke();

        g.restore();


    },

    /**
     * 得到装饰框的区域
     * @param pRect
     * @returns {Rectangle}
     */
    getDecorateBounds: function (/*Rectangle*/ pRect) {
        if (this.m_decorate == null) return pRect;

        let /*int*/ t = 0;
        let /*Rectangle*/ tc = new Rectangle();


        let /*int*/ left = pRect.x;
        let /*int*/ right = pRect.x + pRect.width;
        let /*int*/ top = pRect.y;
        let /*int*/ bottom = pRect.y + pRect.height;

        let /*int*/ viewWidth = pRect.width;
        let /*int*/ viewHeight = pRect.height;


        t = parseInt((this.m_decorate.left || '0').replace(/px/g, ''));


        if (t >= 0)
        {
            tc.x = Math.min(left + t, right);
        } else
        {
            tc.x = Math.max(left, Math.max(right + t, 0));
        }


        t = parseInt((this.m_decorate.top || '0').replace(/px/g, ''));
        if (t >= 0)
        {
            tc.y = Math.min(top + t, bottom);
        } else
        {
            tc.y = Math.max(top, Math.max(bottom + t, 0));
        }

        t = parseInt((this.m_decorate.width || '0').replace(/px/g, ''));
        if (t == 0) tc.width = right - tc.x;
        if (t > 0) tc.width = t;
        if (t < 0) tc.width = Math.max(0, right + t - tc.x);
        //下面一句，是一直延伸到页面最大宽度
        if (t <= -10000) tc.width = Math.max(tc.x, Math.max(viewWidth + (t + 10000), 0)) - tc.x;


        t = parseInt((this.m_decorate.height || '0').replace(/px/g, ''));
        if (t == 0) tc.height = bottom - tc.y;
        if (t > 0) tc.height = t;
        if (t < 0) tc.height = Math.max(0, bottom + t - tc.y);
        //下面一句， 是一直延伸到页面最大高度
        if (t <= -10000) tc.height = Math.max(tc.y, Math.max(viewHeight + (t + 10000), 0)) - tc.y;

        if (tc.height < 0) tc.height = 0;
        if (tc.width < 0) tc.width = 0;
        return tc;
    },

    destroyBalloonTipControl: function (row) {
        //TODO

    },

    /**
     * 返回一个Promise对象，该对象用来加载一个页面，并用它把html转换成canvas
     * 然后，把此canvas的数据创建一个img标签，在img标签准备好之后，把它做参数
     * 调用本Promise的 resole(img),随即执行流程进入到 promise的 then 中做后续的调用
     *
     * 细节：用一个Iframe来处理 html2canas .默认情况下，这个 iframe并不存在，所以是需要时，临时创建
     * 在创建后，还要加载页面，这时就可能存在多种状态： 1 iframe 不存在， 2 iframe已存在，但是contentWindow还没有初始化
     * 3  其中的 html2Img 脚本还没有初始化完成。
     * 出现
     * 所以出现上述情况时，直接 resolve('waiting') 让Promise 结束，并通知单元格再等等
     * 仅当上面都初始化完成后，才调用 html2img 并把结果返回给单元格让它绘制。 这个过程 是异步 的，如果多个单元格都需要drawAsHTML
     * ,那么其中一个单元格中创建 了iframe ，并等待时，另一个单元格又需要 drawAsHTML，此时iframe 已创建，但不一定初始化完成
     *
     *
     * 所以注意如下几个要点： 1 单元格的常规绘制是同步，即一个单元格没有画完，另一个单元格不可能进入绘制状态
     *      2  当出现要绘制图片 或 HTML时，需要异步获取图片数据 ，等图片数据准备好之后，再绘制，此时就出现了异步情况
     *      3  虽然是异步，但是绘制过程，仍是单线程的， 即由Sheet循环对cell做绘制，虽然一些单元格当时无法绘制，会在
     *      setTimeout里重新申请绘制，这些重绘申请会在正常的绘制完成后，再进行，总体上仍是单线程。 因此不用担心 graphics的
     *      上下文问题。这也要求， 单元格在绘制时，需要根据自已的需要重设置 上下文信息（即颜色，笔型，填充模式，裁剪区的信息）
     *
     *
     * @param str
     * @returns {*}
     */
    promiseInnerImageValid: function () {

        if (this.viewAs != 'html') return null;
        if (this.innerImage != null) return null;

        let that = this;


        let ret = function (str, rc) {


            let promiseGo = function (that, str, resolve) {
                let iframe = that.Clas$.$iframe_html2img;
                if (!iframe.contentWindow)
                {
                    resolve('waiting');

                    return;
                }

                if (!iframe.contentWindow.html2Img)
                {
                    resolve('waiting');
                    return;
                }

                iframe.contentWindow.html2Img(str, function (canvas) {

                    let s = canvas.toDataURL();
                    let img = document.createElement("img");
                    img.src = s;

                    img.onload = function () { //图片下载完毕时异步调用callback函数。

                        resolve(img);

                    };


                });
            };

            return new Promise(function (resolve, reject) {

                str = `<div style="width:${rc.width}px; height:${rc.height}px;overflow: hidden;">
                                        ${str}
                                        </div>
                                 `;

                if (that.Clas$.$iframe_html2img == null)
                {

                    let iframe = document.createElement("iframe");
                    that.Clas$.$iframe_html2img = iframe;


                    $(iframe).css({
                        position: "absolute", "left": "-600px", "top": "0px",
                        'width': '600px', 'height': '300px'
                    });

                    $(iframe).on('load', function () {
                        promiseGo(that, str, resolve);

                    });

                    iframe.src = "js/spreadsheet/base/html2canvas.html";
                    document.body.appendChild(iframe);

                } else
                {
                    promiseGo(that, str, resolve);

                }


            });
        };
        return ret;

    },


    drawAsHTML: function (/*Graphics2D*/ g, /*Rectangle*/rc, /*String*/s) {
        if (s.trim() == "") return;

        //如果内容有缓存，直接画

        if (window['spreadSheetImageCache'] == undefined) window['spreadSheetImageCache'] = {};
        let img = window['spreadSheetImageCache'][s]; //,s 是缓存的key
        if (img)
        {

            g.drawImage(img, 0, 0, rc.width, rc.height, rc.x, rc.y, rc.width, rc.height);
            return;
        }

        //得到一个Promise对象，
        let promise = this.promiseInnerImageValid();


        if (promise)
        {
            let that = this;
            promise(s, rc).then(function (img) {

                if (Util.isString(img))
                {
                    if (img == 'waiting')
                    {
                        setTimeout(function () {
                            that.repaint();
                        }, 100);
                        return;
                    }
                }

                //有可能img 就是不存在
                window['spreadSheetImageCache'][s] = img;
                that.repaint();

            });
        }


    },


    /**
     * 返回一个Promise对象，该对象用来加载一个img标签，在img标签准备好之后，把它做参数
     * 调用本Promise的 resolve(img),随即执行流程进入到 promise的 then 中做后续的调用
     * @returns {ret}
     */
    imagePromise: function () {


        let ret = function (str) {


            return new Promise(function (resolve, reject) {

                //如果str仅仅就是src的内容，而不是完整的img标签内容，那么拼上
                str = Tools.normalizeImageSrc(str);//图片地址规整一下
                //2019.07.30 低级错误引发异常
                // 下面原先是  img=$(str)[0]; 少了 var 或let 导致 img成了一个隐形定义的全局变量
                // 导致所有图片都变成最后一张图片
                let img = $(str)[0];
                //2020.06.04 加上crossOrigin解决canvas使用 getImageData 引发的同源限制异常
                //2020.06.05 这个参数有点让人看不明白， 在手机上，似乎正常，而在电脑上，反而出问题了
                // 算了，先去掉吧
                // img.crossOrigin='anonymous';


                console.info("需要加载" + str);
                //document.body.appendChild(img[0]);
                img.onload = function () { //图片下载完毕时异步调用callback函数。
                    resolve(img);
                };

            });
        };
        return ret;

    },


    /**
     *
     * @param g 图形设备对象
     * @param rc 绘制区域
     * @param s  图片的src地址
     */
    drawAsImage: function (/*Graphics2D*/  g, /*Rectangle*/rc, /*String*/s, hAlign, vAlign) {

        if (s.trim() == "") return;
        let str = Tools.normalizeImageSrc(s);

        console.info(this.getName() + " image " + str);

        //图片缓存
        if (window['spreadSheetImageCache'] == undefined) window['spreadSheetImageCache'] = {};
        //图片缓存请求中标记，避免重复请求
        if (window['spreadSheetImageCacheIng'] == undefined) window['spreadSheetImageCacheIng'] = {};

        let img = window['spreadSheetImageCache'][str]; //str即是img.src ,同时也是缓存的key
        if (img)
        {
            console.info("from cache");

            //此时的 g 已经正确设置了裁剪区，不需要再设置了
            let t = s.toLowerCase();
            let w = img.width;
            let h = img.height;
            console.info("图片原始大小：w:" + w + " h:" + h);

            //只是一个图片地址，那么 ，默认是自动调整大小
            if (t.indexOf(" ") < 0 && t.indexOf("width" < 0) && t.indexOf("height") < 0)
            {
                let wr = w * 1.0 / rc.width; //图宽与容器宽的比率
                let hr = h * 1.0 / rc.height; //
                if (wr > hr) //说明要
                {

                    w = rc.width;
                    h = h / wr;
                } else
                {
                    h = rc.height;
                    w = w / hr;
                }

                console.info("自动调整大小 适应单元格  w:" + w + " h:" + h);

            } else
            {

                try
                {
                    let sizeDefined = false;
                    w = t.match(/width\s*=\s*[\'\"]([^\'\"]*)[\'\"]/);
                    if (w.length == 2)
                    {
                        w = parseInt(w[1]);
                        sizeDefined = true;

                    }

                    h = t.match(/height\s*=\s*[\'\"]([^\'\"]*)[\'\"]/);
                    if (h.length == 2)
                    {
                        h = parseInt(h[1]);
                        sizeDefined = true;

                    }
                    //如果没有指定尺寸，则按单元格进行自适应缩放
                    if (!sizeDefined)
                    {
                        let wr = w * 1.0 / rc.width; //图宽与容器宽的比率
                        let hr = h * 1.0 / rc.height; //
                        if (wr > hr) //说明要
                        {

                            w = rc.width;
                            h = h / wr;
                        } else
                        {
                            h = rc.height;
                            w = w / hr;
                        }

                        console.info("自动调整大小 适应单元格  w:" + w + " h:" + h);

                    }

                } catch (e)
                {
                    w = img.width;
                    h = img.height;
                }
            }

            //高分辨率显示器要放大


            //通过缩放，重设显示尺寸

            let x, y;
            if (hAlign == ALIGNMENT.Left) x = rc.x;
            if (hAlign == ALIGNMENT.Center) x = rc.x + (rc.width - w) / 2;
            if (hAlign == ALIGNMENT.Right) x = rc.x + rc.width - w;

            if (vAlign == ALIGNMENT.Top) y = rc.y;
            if (vAlign == ALIGNMENT.Center) y = rc.y + (rc.height - h) / 2;
            if (vAlign == ALIGNMENT.Bottom) y = rc.y + rc.height - h;

            //
            let ratio = g.ratio; //打印时，没有这个属性，
            if (ratio == undefined) ratio = 1;
            w = w * ratio;
            h = h * ratio;

            g.save();
            //2021.07.24 去掉了下面的clip
            // 原因是在windows上打印时，因为此句，会导致此后的所有绘制输出都没了，主要体现在单元格的边框线都没有了
            // 但是在mac上打印是正常的， 原因不明。
            // 去掉后，windows上打印也正常了，
            //通常图片不存在裁剪，所以去掉也没问题。

            //g.clip(rc);

            g.drawImage(img, 0, 0, img.width, img.height, x, y, w, h);

            g.restore();
            return;
        } else
        {

            //
            if (window['spreadSheetImageCacheIng'][str] != undefined)
            {
                console.info("image is caching , needn't  redo cache");
                return;
            }
            //表示正在缓存请求中
            window['spreadSheetImageCacheIng'][str] = true;

            console.info("build image cache ")
            //得到一个Promise对象，
            let promise = this.imagePromise();

            if (promise)
            {
                let that = this;
                promise(str).then(function (img) {
                    window['spreadSheetImageCache'][str] = img;
                    console.info(str + " cache  ok " + that.getName() + " need repaint");
                    //移除标记
                    delete window['spreadSheetImageCacheIng'][str];
                    that.repaint();
                });
            }

        }

    },


    /**
     *
     * @param dbrow
     * @param g
     * @param cellRect   单元格的坐标
     * @param rc   去除边框，留白后的显示文本的坐标
     * @param data
     * @param s
     * @param bkMode
     * @param bkcolor
     * @param textcolor
     * @param FontName
     * @param FontSize
     * @param FontBold
     * @param FontItalic
     * @param FontUnderline
     * @param hAlign
     * @param vAlign
     * @param MultiLine
     * @param noES   true 表示不管编辑格式，画文字即可 , false　需要根据编辑格式画文字
     */
    drawString: function (isPrint, dbrow, g, cellRect, bordertRect, rc, data, s, bkMode,
                          /*Color*/bkcolor, /*Color*/textcolor, /*String*/   FontName, /* int*/FontSize,
                          /*boolean*/FontBold, /*boolean*/FontItalic, /*boolean*/FontUnderline,
                          /*int*/hAlign, /*int*/vAlign, /*boolean*/MultiLine, /*boolean*/ noES) {


        try
        {

            if (this.DrawInScript)
            {
                this.Book.EM.fire("cellPaint", [this.Sheet, this, dbrow, g, bordertRect, data, s, true]);

            }


            // 因为下面会对rc 做一些改动，但又不能影响rc 原来的值，所以要Clone一份
            let /*Rectangle*/ textRect = rc.clone();

            if (data != null)
            {
                if (Util.isObject(data))
                {
                    if (data.plugin != undefined)
                    {
                        // g.save();
                        //  g.clip(cellRect);

                        data.plugin.draw(this, dbrow, g, bordertRect, data, s, bkMode, bkcolor, textcolor, FontName, FontSize,
                            FontBold, FontItalic, FontUnderline, hAlign, vAlign, MultiLine, noES);
                        //   g.restore();
                        return;
                    }
                }
            }

            if (this.viewAs == 'html')
            {
                this.drawAsHTML(g, bordertRect, s);
                return;
            }

            if (this.viewAs == 'image')
            {
                this.drawAsImage(g, bordertRect, s, hAlign, vAlign);
                return;
            }


            if (this.viewAs == 'barcode')
            {
                this.drawAsBarcode(dbrow, g, rc, data, s, bkMode, bkcolor, textcolor, FontName, FontSize,
                    FontBold, FontItalic, FontUnderline, hAlign, vAlign);
                return;
            }

            s = '' + s;


            //如果有增加更多的RichData，那么同步处理this.toFormatString

            if (this.ES != null && !noES)
            {
                let /*int*/ et = this.ES.ET;


                //  增加下拉列表显示为Radio,checkbox样式
                if (et == EditStyle.$RadioButton || et == EditStyle.$MultiCheckBox)
                {

                    let dropDownData = this.ES.getDropDownDataForEdit(this, dbrow);

                    let checked = [];
                    let vs = (s == null ? '' : ('' + s)).split(',');
                    for (let i = 0; i < vs.length; i++)
                    {
                        let v = vs[i];
                        let index = dropDownData.dataValue.indexOf(v);
                        if (index >= 0) checked.push(index);
                    }


                    let listCount = dropDownData.dataValue.length;
                    //需要显示几行
                    //当实际的数据不足列数时，原先是以实际为准，现在改在以设置为准，这样方便对齐
                    let colCount = Math.max(1, this.ES.columnCount);
                    let rowCount = Math.max(1, Math.floor((listCount - 1) / colCount) + 1); //肯定不为0

                    let itemRC = new Rectangle();
                    //为什么下面计算width时要+1 ， 因为在编辑及显示时，可能在处理坐标时，有+- padding 及微调，
                    // 出现了显示与编辑时，坐标有细微差异， 通过这么调整后，基本削除差异。这不是治本的办法。先这么处理吧
                    itemRC.width = Math.floor((rc.width) / colCount);
                    itemRC.height = Math.floor(rc.height / rowCount);

                    for (let ri = 0; ri < rowCount; ri++)
                    {
                        itemRC.y = rc.y + ri * itemRC.height;
                        for (let ci = 0; ci < colCount; ci++)
                        {
                            itemRC.x = rc.x + ci * itemRC.width;
                            let index = ri * colCount + ci;
                            if (index > listCount - 1) break;

                            let str = '';

                            if (isPrint)   // 打印时， 转成宋体中可以打印的符号
                            {
                                str = checked.contains(index) ? "[√]︎" : "[ ]";
                                str = str + dropDownData.showText[index];
                            } else
                            {
                                str = checked.contains(index) ?
                                    (this.ES.ET == EditStyle.$RadioButton ? this.ES.radioCheckOnChar : this.ES.multiCheckOnChar) :
                                    (this.ES.ET == EditStyle.$RadioButton ? this.ES.radioCheckOffChar : this.ES.multiCheckOffChar);
                                str = ' ' + str + ' ' + dropDownData.showText[index];
                            }


                            g.drawStringWordwrap(itemRC, str, bkMode, bkcolor, textcolor,
                                isPrint ? FontName : "FontAwesome", FontSize, FontBold, FontItalic, FontUnderline, hAlign, vAlign, MultiLine);
                        }
                    }

                    return;

                }

                if (et == EditStyle.$MultiLine)
                {
                    g.drawStringWordwrap(textRect, s, bkMode, bkcolor, textcolor, FontName, FontSize,
                        FontBold, FontItalic, FontUnderline, hAlign, vAlign);
                    return;
                }

                if (et == EditStyle.$RichText)
                {
                    drawAsHTML(g, cellRect, s);//用单元格原始坐标
                    return;
                }

                if (et == EditStyle.$Numeric)
                {
                    if (data == 0 || Math.abs(data) < 0.000000001)
                    {
                        if (!this.ES.zeroVisible) s = '';
                    }
                }

            }

            //console.dir( "====");
            //console.dir( textRect);

            if (s == '' && !this.placeholderOnlyViewWhenEditing && this.placeholder != '')
            {
                s = this.placeholder;
                textcolor = this.Sheet.placeholderColor;
            }

            if (MultiLine)
            {
                g.drawStringWordwrap(textRect, s, bkMode, bkcolor, textcolor, FontName, FontSize,
                    FontBold, FontItalic, FontUnderline, hAlign, vAlign);
            } else
            {
                g.drawStringSingleRow(textRect, s, bkMode, bkcolor, textcolor, FontName, FontSize,
                    FontBold, FontItalic, FontUnderline, hAlign, vAlign);
            }

        } catch (err)
        {
            console.dir(err);
        } finally
        {
            if (this.DrawInScript)
            {
                this.Book.EM.fire("cellPaint", [this.Sheet, this, dbrow, g, bordertRect, data, s, false]);

            }
        }
    },


    /**
     * 画条码
     *
     * @param dbrow
     * @param g
     * @param bordertRect
     * @param data
     * @param s
     * @param bkMode
     * @param bkcolor
     * @param textcolor
     * @param FontName
     * @param FontSize
     * @param FontBold
     * @param FontItalic
     * @param FontUnderline
     * @param hAlign
     * @param vAlign
     */
    drawAsBarcode: function (dbrow, g, rect, data, s, bkMode,
                             /*Color*/bkcolor, /*Color*/textcolor, /*String*/   FontName, /* int*/FontSize,
                             /*boolean*/FontBold, /*boolean*/FontItalic, /*boolean*/FontUnderline,
                             /*int*/hAlign, /*int*/vAlign) {

        if (data == null) return;

        //二维码
        if (this.barcodeType == 'QRCode')
        {

            this.drawAsQRCode(dbrow, g, rect, data, s, bkMode, bkcolor, textcolor, FontName, FontSize,
                FontBold, FontItalic, FontUnderline, hAlign, vAlign);
            return;
        }


        //下面是一维码
        let barcodeEncodings = this['barcodeEncodings'];
        if (barcodeEncodings == undefined)
        {
            this['barcodeEncodings'] = {};
            barcodeEncodings = this['barcodeEncodings'];
        }
        let thisBarcodeEncodings = barcodeEncodings[s];
        if (thisBarcodeEncodings == null)
        {
            thisBarcodeEncodings = {};
            let option = {
                format: this.barcodeType,
                width: this.barcodeWidth
            };

            try
            {
                JsBarcode(thisBarcodeEncodings, data, option);
            } catch (err)
            {
                console.dir(err);
                return;
            }
            barcodeEncodings[s] = thisBarcodeEncodings;

        }

        let encodings = thisBarcodeEncodings.encodings;
        let barRC = new Rectangle();


        g.setFillColor(textcolor);

        let left = rect.x; //已经是加上了padding的坐标
        let blockX;
        let top = rect.y;
        let lineWidth = this.barcodeWidth; //线宽


        let barPointCount = 0; //整个条码的模组的点数
        for (let i = 0; i < encodings.length; i++)
        {

            let encoding = encodings[i];
            let binary = encoding.data;

            barPointCount += binary.length;
        }


        //水平 ，按模组的点占据的宽度，计算x偏移坐标
        if (hAlign == ALIGNMENT.Center) left = rect.x + Math.floor((rect.width - barPointCount * lineWidth) / 2);
        if (hAlign == ALIGNMENT.Right) left = rect.x + rect.width - barPointCount * lineWidth;

        //垂直方向，按条码高度及文字是否显示来加上图文间隔加上文字高度，来计算y向的偏移
        let barAndTextHeight = this.barcodeHeight; //条码高度
        if (this.barcodeTextVisible)
        {
            barAndTextHeight += this.barcodeTextMargin + FontSize; //加上间隔,加上文字高度

        }

        if (vAlign == ALIGNMENT.Center) top = rect.y + Math.floor((rect.height - barAndTextHeight) / 2);
        if (vAlign == ALIGNMENT.Bottom) top = rect.y + rect.height - barAndTextHeight;


        let x = left;
        let y = top;

        for (let i = 0; i < encodings.length; i++)
        {

            let encoding = encodings[i];
            let binary = encoding.data;

            //不使用jsBarcode的padding , rc已经扣掉了padding
            let xPadding = 0;
            let yPadding = 0;

            blockX = x; //记录下条码分段的超始X坐标 ，比如 EAN13 它就分成6段
            let h = this.barcodeHeight; //条码高度


            if (this.barcodeType == 'EAN13' && [1, 3, 5].contains(i)) h = h + 10 + this.barcodeTextMargin;

            for (let b = 0; b < binary.length; b++)
            {

                if (binary[b] === "1")
                {
                    g.fillRect(x, y, lineWidth, h);
                }

                x += lineWidth;

            }
            //


            //条码统一使用padding-left , padding-top 来控制条码的显示， 对齐方式，只是控制文字的对齐，且垂直方向上的对齐方式不需要

            if (this.barcodeTextVisible)
            {
                let trc = new Rectangle();
                trc.x = blockX;
                trc.width = binary.length * lineWidth;
                trc.y = y + h + this.barcodeTextMargin;
                trc.height = FontSize;
                //画条码文字

                //EAN13 第一个块的文字，都是左对齐
                let ha = (this.barcodeType == 'EAN13' && i == 0) ? ALIGNMENT.Left : hAlign;

                g.drawStringSingleRow(trc, encoding.text, bkMode, bkcolor, textcolor, FontName, FontSize,
                    FontBold, FontItalic, FontUnderline, ha, vAlign);
            }


        }


        this.barcodeRect = new Rectangle(left, top, barAndTextHeight, this.barcodeHeight);


    },


    drawAsQRCode: function (dbrow, g, rect, data, s, bkMode,
                            /*Color*/bkcolor, /*Color*/textcolor, /*String*/   FontName, /* int*/FontSize,
                            /*boolean*/FontBold, /*boolean*/FontItalic, /*boolean*/FontUnderline,
                            /*int*/hAlign, /*int*/vAlign) {

        if (data == null) return;

        if (this.barcodeType != 'QRCode') return;


        let barcodeEncodings = this['barcodeEncodings'];
        if (barcodeEncodings == undefined)
        {
            this['barcodeEncodings'] = {};
            barcodeEncodings = this['barcodeEncodings'];
        }

        let thisBarcodeEncodings = barcodeEncodings[s];
        if (thisBarcodeEncodings == null)
        {

            let qrcode = new QRCode(s);
            thisBarcodeEncodings = qrcode._oQRCode;
            barcodeEncodings[s] = thisBarcodeEncodings;

        }

        let oQRCode = thisBarcodeEncodings;


        let nCount = oQRCode.getModuleCount();
        let nWidth = this.barcodeWidth;
        let nHeight = this.barcodeWidth;

        g.setFillColor(textcolor);

        //根据不同的对齐方式，计算左，顶点的坐标
        let left = rect.x;
        let top = rect.y;

        if (hAlign == ALIGNMENT.Center) left = rect.x + Math.floor((rect.width - nCount * nWidth) / 2);
        if (hAlign == ALIGNMENT.Right) left = rect.x + rect.width - nCount * nWidth;

        if (vAlign == ALIGNMENT.Center) top = rect.y + Math.floor((rect.height - nCount * nWidth) / 2);
        if (vAlign == ALIGNMENT.Bottom) left = rect.y + rect.height - nCount * nWidth;

        //开始画二维码的各个模块点
        for (let row = 0; row < nCount; row++)
        {
            for (let col = 0; col < nCount; col++)
            {
                let bIsDark = oQRCode.isDark(row, col);
                if (!bIsDark) continue;

                let nLeft = left + col * nWidth;
                let nTop = top + row * nHeight;
                g.fillRect(nLeft, nTop, nWidth, nHeight);

            }
        }

        this.barcodeRect = new Rectangle(left, top, nCount * nWidth, nCount * nHeight);

        if (this.barcodeCenterImageURL || '' != '')
        {
            //整个二维码的坐标，做为图片绘制的区域

            //居中绘制
            this.drawAsImage(g, this.barcodeRect, this.barcodeCenterImageURL, ALIGNMENT.Center, ALIGNMENT.Center);
        }


    },

    /**
     * 判断第dbrow行数据是不是CheckOn
     *
     * @param dbrow
     * @return
     */


    isCheckOn: function (dbrow) {
        let v = this.getValue(dbrow);
        if (this.ES == null) return false;

        if (this.ES.ET != EditStyle.$CheckBox) return false;

        let onv = this.ES.checkOnValue;

        // 为空表示checkoff
        if (v != null && onv != null)
        {
            if (v.toString().equals(onv.toString())) return true;
        }

        return false;
    },


    isEditing: function () {

        return this.EditingFlag > 0;
    },

    setIsEditing: function (b) {
        this.EditingFlag += b ? 1 : -1;

    },


    isEditable: function () {

        return this.Editable;
    },


    setEditable: function (editable, /*boolean*/addBorder) {
        addBorder = addBorder || false;

        let /*String*/ editHightLight = "EditableHightLight";

        this.Editable = editable;

        if (addBorder && this.decorate == null) //如果没有装饰，那么加上一个红线框
        {


        }


        this.repaint();

    },

    getEditStyle: function () {
        return this.ES;
    },
    /**
     *
     *  @param editStyle 可以是名称，也可以是对象
     */
    setEditStyle: function (editStyle) {
        if (this.ES == editStyle) return;


        if (Util.isString(editStyle)) editStyle = this.Book.getEditStyle(editStyle);


        this.ES = editStyle;

        if (this.ES != null)
        {
            if (this.ES.ET == EditStyle.$DDLB)
            {
                let map = this.ES.dropDownDataMap;

                if (this.Bind != null)
                {
                    let /*DataSourceConfig*/ dsc = this.Book.getDataSource(this.Bind.dataSource);
                    if (dsc != null)
                    {
                        let dbcol = this.Bind.DBCol;
                        let ds = dsc.dataStore;

                        let CP = ds.getColumnProperty(dbcol);
                        if (CP != null) CP.setData2ViewMap(map);
                    }
                }
            }
        }

        // ，由于增加了对 $别名或 $单元格名　来取单元格的显示值，所以当编辑格式发生变化后，显示值也会受影响
        // 所以需要通知引用本单元那些单元格需要重新计算

        //  重要修改
        // 设置单元格编辑模式是否需要触发依赖它的单元格进行重新计算
        //缺省是需要的， 比如  b1= $A1  , 那么当A1 设置编辑格式为DDLB后， b1的数据可能是需要变化的
        //便是在初始时，如果因设置编辑格式，导致通知依赖它的单元格重新计算，那么会引起公式计算的值覆盖数据库的值的情况
        //比如：  je= sl*dj  其中 je , sl , dj  是三个单元格， 并且它们都绑定到数据库， 并且 je 设置了公式 （不是计算列回填）=sl*dj
        // 比如当录入 sl =10 , dj =10后， je 计算出现是100，之后手工改成 200, 保存后,当再次打开表单， 当设置sl 编辑格式时，
        // 它触发 je 重新计算，于是 je 又被改成100 ,  从理论上讲100是合理的。 但是实际应用中可能需要这样的功能 ：缺省是按公式进行计算
        // 并且允许手工修改， 当再次打开时， 不能覆盖手工修改的数据，但是当进行数据修改（比如修改sl ）引起重新计算时，又需要把新的je
        // 计算结果填到 je 上。

        this.NotifyAffectCellInvalidWhenMyEditStyleChanged();

    },


    getRow: function () {
        return this.pRow;
    },


    getWorkSheet: function () {
        return this.Sheet;
    },


    //	这个函数在copy,paste时被用到，参看　CMD_Paste , ClipBoard_


    clear: function () {
        this.property = null;
        this.ES = null;
        this.ViewAs = '';
        if (this.Bind != null) this.setBind(null);
        if (this.BrickMap != null)
        {

            this.BrickMap = null;
        }

        this.appData = null;
        this.setEditable(false);
        this.Visible = true;
        this.setValue(null);
        this.setAlias(null);
        this.validateRule = null;

        //取消合并
        if (this.isMerged()) this.Sheet.unMerge(this.getMergedRange());

        this.ES = null;
        this.ViewAs = '';
        this.viewAsPasswordMask = "*";
        this.BrickMap = null;
        this.ToolTip = null;

        this.Alias = null;
        this.Tag = null;
        this.formulaNeedCache = true; //公式需要缓存结果吗，缺省是需要的


        //校验相关
        this.validateRule = null;
        this.validateTip = null;

        this.subscribeList = null;
        this.cursorName = "";

        this.Visible = true;
        this.Printable = true;

        this.MouseAdapterList = null;
        this.innerImage = null;
        this.innerImnageData = null;

        this.drawProxy = null;

        this.dblClickSortDefine = ""; //双击本单元格时排序
        this.dblClickSortSignChar = "";

        //单击
        this.clickJumpTo = "";
        this.clickJumpToWhere = "_blank";
        this.clickPopupDialogWidth = 800;
        this.clickPopupDialogHeight = 600;
        this.clickPopupDialogTitle = "";

        this.placeholder = ''; //占位提示
        this.placeholderOnlyViewWhenEditing = true; //仅在编辑时显示占位提示


    },


    /**
     * @api {addBrick} 函数   addBrick
     *
     * @apiDescription addBrick
     * <br><br> 单元格中添加控件
     *
     * @apiName  addBrick
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     * @apiParam {String} where 检索条件
     *
     *
     * @apiSuccess (返回值){int} - 返回本次检索出的数据条数
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}插入一张图片：
     *
     *   let img = 'http://pic.58pic.com/58pic/15/32/41/86t58PICXGP_1024.jpg';
     *  book.sheet0.cells('logo').addBrick('html',
     *        'logoImg',
     *        { html: "<div><img src='" + img + "' height='60'></div>" },
     *        0, 0, 0, 0);
     *
     * @apiExample   {js}插入一张图片：
     *
     *   let img = 'http://pic.58pic.com/58pic/15/32/41/86t58PICXGP_1024.jpg';
     *  book.sheet0.cells('logo').addBrick('html',
     *        'logoImg',
     *        { html: "<div><img src='" + img + "' height='60'></div>" },
     *        0, 0, 0, 0);
     *
     */
    addBrick: function () {

        console.info("add brick");

        let brick = null;
        if (arguments.length == 1)
        {
            brick = arguments[0];
        }


        if (arguments.length == 7)
        {
            let type = arguments[0];
            let name = arguments[1];
            let config = arguments[2];
            let x = arguments[3];
            let y = arguments[4];
            let w = arguments[5];
            let h = arguments[6];

            switch (type)
            {
                case 'echart':
                    brick = new EChartBrick(name, config, x, y, w, h);
                    break;
                case "html":
                    brick = new HTMLBrick(name, config, x, y, w, h);
                    break;
                case "dbpage":
                    brick = new DBPageBrick(name, config, x, y, w, h);
                    break;
                case "button":
                    brick = new Button(name, config, x, y, w, h);
                    break;
                case "tree":
                    brick = new TreeBrick(name, config, x, y, w, h);
                    break;
                case "SheetContainer":
                    brick = new SheetContainer(name, config, x, y, w, h);
                    break;
                case "Tab":
                    brick = new TabBrick(name, config, x, y, w, h);
                    break;

                case "ckeditor":
                    brick = new CKEditorBrick(name, config, x, y, w, h);
                    break;

                case "uploadImage":
                    brick = new UploadBrick(name, config, x, y, w, h);
                    break;
            }

        }
        if (!brick) return;

        let name = brick.name;
        name = name.toLowerCase().trim();
        if (this.BrickMap == null) this.BrickMap = {};

        brick.setParentCell(this);

        this.BrickMap[name] = brick;
        brick.bind(); //v 通知brick做一些初化
        this.repaint();
        return this;
    },


    /**
     * 移除单元格的中小物件
     */


    /**
     * @api {removeBrick} 函数   removeBrick
     *
     * @apiDescription removeBrick(brickName)
     * <br><br> 单元格中移除指定的控件
     *
     * @apiName  removeBrick
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     * @apiParam {String} brickName 控件名称
     *
     *
     * @apiSuccess (返回值){Cell} - 返回单元格自已
     *
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     */
    removeBrick: function (/*String*/      brickName) {

        if (this.BrickMap == null) return this;
        brickName = brickName.toLowerCase().trim();
        let brick = this.BrickMap[brickName];
        if (!brick) return this;
        brick.unBind();
        if (brick.dom) brick.dom.remove();
        delete this.BrickMap[brickName];

        this.repaint();
        return this;
    },


    /**
     * @api {getBrick} 函数   getBrick
     *
     * @apiDescription getBrick(nameOrIndex)
     * <br><br> 单元格
     *
     * @apiName  getBrick
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     * @apiParam {String} nameOrIndex 控件名称
     *
     *
     * @apiSuccess (返回值){Brick} - 返回控件对象
     *
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     */
    getBrick: function (nameOrIndex) {
        if (this.BrickMap == null) return null;
        if (Util.isInteger(nameOrIndex)) nameOrIndex = this.getBrickName(nameOrIndex);

        if (nameOrIndex == null) return null;
        nameOrIndex = nameOrIndex.toLowerCase().trim();
        return this.BrickMap[nameOrIndex];
    },

    /**
     * @api {getBrickCount} 函数   getBrickCount
     *
     * @apiDescription getBrickCount()
     * <br><br> 单元格中控件的个数
     *
     * @apiName  getBrickCount
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     *
     * @apiSuccess (返回值){int} - 返回单元格中控件的个数
     *
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     */
    getBrickCount: function () {
        if (this.BrickMap == null) return 0;
        return Object.getOwnPropertyNames(this.BrickMap).length;
    },


    /**
     * @api {getBrickName} 函数   getBrickName
     *
     * @apiDescription getBrickName(index)
     * <br><br> 单元格中第index个控件的名称
     *
     * @apiName  getBrickName
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     * @apiParam {int} index 序号
     *
     *
     * @apiSuccess (返回值){String} - 返回控件的名称
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     */
    getBrickName: function (index) {
        if (this.BrickMap == null) return null;
        let n = this.getBrickCount();
        if (index >= n) return null;
        if (index < 0) return null;

        let i = 0;
        for (let name in this.BrickMap)
        {
            if (i == index) return name;
            i++;
        }

        return null;
    },


    getBricks()
    {
        let ret = [];
        if (this.BrickMap == null) return ret;

        for (let name in this.BrickMap)
        {
            ret.push(this.BrickMap[name]);
        }
        return ret;

    },

    /**
     * 本函数在 onLbuttonClick 中调用
     * @param innerRow 点击的行是单元格内的第n个子行
     */

    $innerAction: function (innerRow) {

        //2021.04.22 增加仅在可见且允许点击时触发相关的处理

        if (this.clickJumpTo != "" && !this.Sheet.designMode && this.visible)
        {
            if (!this.Clickable)
            {
                toastr.info("抱歉，您无此权限");
                return;
            }

            let url = this.clickJumpTo.trim();
            if (url.startsWith("=")) url = this.Sheet.evaluate(url, innerRow);
            if (url == '') return;
            if (url == null) return;

            let title = this.clickPopupDialogTitle.trim();
            if (title.startsWith("=")) title = this.Sheet.evaluate(title, innerRow);


            if (this.clickJumpToWhere == "_blank")
            {
                let a = $('#openInNewWindow')[0];
                a.href = url;
                a.click();
                return;
            }

            if (this.clickJumpToWhere == "_self")
            {
                window.location.href = url;
                return;
            }

            //上面的evaluate已经执行过了
            if (this.clickJumpToWhere == "javascript") return;

            if (this.clickJumpToWhere == "popupIframe")
            {
                let w = this.clickPopupDialogWidth;
                let h = this.clickPopupDialogHeight;
                $.dialog({
                    boxWidth: w + "px",
                    useBootstrap: false,
                    title: title,
                    content: `<iframe style="width:100%;height:${h}px; border:0 " frameborder="0"  src="${url}"  ></iframe>`,
                });
                return;
            }

            if (this.clickJumpToWhere == "popupContent")
            {
                let w = this.clickPopupDialogWidth;
                let h = this.clickPopupDialogHeight;
                $.dialog({
                    boxWidth: w + "px",
                    useBootstrap: false,
                    title: title,
                    content: url,
                });
                return;
            }


        }


    },


    OnLButtonDown: function (/*WorkSheetView*/  pView, /*MouseEvent*/   e) {

        if (this.MouseAdapterList != null)
        {
            for (let name in this.MouseAdapterList)
            {
                let adapter = this.MouseAdapterList[name];
                adapter.OnLButtonDown(this, pView, e);
            }
        }

    },


    getInnerRowAtPoint: function (/*Point*/p, /*boolean*/setAsCurrentBindRow) {


        return this.Sheet.getInnerRowAtPoint(this, p, setAsCurrentBindRow);

    },

    /**
     * 得到坐标点下的小件
     * @param p
     * @return
     */


    getBrickAt: function (/*Point*/p) {
        if (this.BrickMap == null) return null;

        // 如果有小物件，那么
        let me = p;
        for (let name in this.BrickMap)
        {
            let brick = this.BrickMap[name];

            if (brick.contains(me)) return brick;


        }
        return null;
    },

    // 如果是绑定到一个多行的字段上，那么在鼠标点击后，计算当前数据焦点行，并亮显它
    //如果不是绑定到一个多行上， 但是是一个数组，那么找到该行上的数据源，并计算和设置当前数据焦点行
    // 触发 cellOnClick事件


    OnLButtonClick: function (/*WorkSheetView*/  pView, /*MouseEvent*/   e) {


        let /*int*/ innerRow = this.getInnerRowAtPoint(new Point(e.offsetX, e.offsetY));

        if (!this.Sheet.designMode)
        {
            //不能直接调用this.beginEdit ,需要根据鼠标位置计算编辑哪个数据行
            pView.$beginEdit(this.rowIndex, this.columnIndex, null, e, innerRow);
        }

        this.$innerAction(innerRow);

        // 触发事件
        if (this.Clickable) this.Book.EM.fire("cellClicked", [this.Sheet, this, innerRow, e]);

        //2021.09.05 增加innerAction的支持
        if (!this.Sheet.designMode)
        {
            if (this.Define != null)
            {
                if (this.Define.startsWith("innerAction"))
                {
                    let v = this.getValue(innerRow);
                    let action = v.action;
                    if (action != null)
                    {
                        if (this.Clickable) this.Book.EM.fire("cellInnerAction", [this.Sheet, this, innerRow, e, action]);
                    }
                }
            }

            //如果是绑定到结果集的计算列，且公式是  innerAction
            if (this.Bind != null)
            {
                let cp = this.dataStore.getColumnProperty(this.Bind.DBCol);
                if (cp.getObjType() == ObjectType.isUserComputer)
                {
                    let formula = cp.getFormula();
                    if (formula.match(/\s*={0,1}\s*innerAction\s*\(/))  //如果是 = innerAction或没有=号直接就是 innerAction ，那么
                    {
                        let v = this.getValue(innerRow);
                        let action = v.action;
                        if (action != null)
                        {
                            if (this.Clickable) this.Book.EM.fire("cellInnerAction", [this.Sheet, this, innerRow, e, action]);
                        }
                    }

                }
            }
        }


        if (this.BrickMap != null)
        {

            // 如果有小物件，那么
            let /*Point*/ me = new Point(e.offsetX, e.offsetY);

            for (let name in this.BrickMap)
            {
                let brick = this.BrickMap[name];

                if (brick.contains(me)) brick.OnLButtonClick(pView, e);

            }

        }

        if (this.MouseAdapterList != null)
        {
            for (let name in this.MouseAdapterList)
            {
                let adapter = this.MouseAdapterList[name];
                adapter.OnLButtonClick(this, pView, e);
            }
        }
        //2021.07.01 把 innerRow返回，避免后续再计算它
        return innerRow;
    },

    // 如果是绑定到一个多行的字段上，那么在鼠标点击后，计算当前数据焦点行，并亮显它
    //如果不是绑定到一个多行上， 但是是一个数组，那么找到该行上的数据源，并计算和设置当前数据焦点行
    // 触发 cellOnClick事件


    OnRButtonClick: function (/*WorkSheetView*/  pView, /*MouseEvent*/   e) {

        let /*int*/ innerRow = this.getInnerRowAtPoint(new Point(e.offsetX, e.offsetY));
        //  触发事件
        if (this.Clickable) this.Book.EM.fire("cellRightClicked", [this.Sheet, this, innerRow]);

        for (let name in this.MouseAdapterList)
        {
            let adapter = this.MouseAdapterList[name];
            adapter.OnRButtonClick(this, pView, e);
        }


    }
    ,

    OnLButtonDblClk: function (/*WorkSheetView*/  pView, /*MouseEvent*/   e) {
        if (this.Sheet.designMode)
        {
            pView.$beginEdit(this.rowIndex, this.columnIndex, null, e, -1);
        }

        let /*int*/ innerRow = this.getInnerRowAtPoint(new Point(e.offsetX, e.offsetY));

        //看看是否需要双击排序
        if (this.dblClickSortDefine != '')
        {
            let t = this.dblClickSortDefine;
            t = t.split('.');
            let dsname = t[0];
            let orderByCol = t[1];//按哪个字段排序


            let dsc = this.Book.getDataSource(dsname);
            if (dsc != null)
            {
                let sortBy = dsc.lastDblClickSortBy; //记录上一次双击排序的规则
                if (sortBy == null)//不存在，说明是第一次以双击排序，那么
                {

                    dsc.lastDblClickSortBy = {cell: this, asc: true};
                    dsc.dataStore.orderBy = orderByCol + " asc";
                    this.dblClickSortSignChar = "\uf0d8";
                    this.repaint();


                } else
                {
                    if (dsc.lastDblClickSortBy.cell == this)//如果上一次被双击的单元格就是自已，那么反序一下
                    {
                        dsc.lastDblClickSortBy.asc = !dsc.lastDblClickSortBy.asc;
                        dsc.dataStore.orderBy = orderByCol + (dsc.lastDblClickSortBy.asc ? " asc" : " desc");
                        this.dblClickSortSignChar = (dsc.lastDblClickSortBy.asc ? "\uf0d8" : "\uf0d7");
                        this.repaint();
                    } else
                    {
                        let lastCell = dsc.lastDblClickSortBy.cell;
                        lastCell.dblClickSortSignChar = "";
                        lastCell.repaint();

                        dsc.lastDblClickSortBy = {cell: this, asc: true};
                        dsc.dataStore.orderBy = orderByCol + " asc";
                        this.dblClickSortSignChar = "\uf0d8";
                        this.repaint();
                    }

                }
            }

        }

        //  触发事件
        if (this.Clickable) this.Book.EM.fire("cellDblClicked", [this.Sheet, this, innerRow, e]);

        if (this.BrickMap != null)
        {
            // 如果有小物件，那么
            let /*Point*/ me = new Point(e.offsetX, e.offsetY);

            for (let name in this.BrickMap)
            {
                let brick = this.BrickMap[name];

                if (brick.contains(me)) brick.OnLButtonDblClk(pView, e);


            }
        }

        if (this.MouseAdapterList != null)
        {
            for (let name in this.MouseAdapterList)
            {
                let adapter = this.MouseAdapterList[name];
                adapter.OnLButtonDblClk(this, pView, e);
            }
        }


    },


    OnLButtonUp: function (/*WorkSheetView*/  pView, /*MouseEvent*/   e) {
        if (this.BrickMap != null)
        {
            let /*Point*/ me = new Point(e.offsetX, e.offsetY);

            for (let name in this.BrickMap)
            {
                let brick = this.BrickMap[name];

                if (brick.contains(me)) brick.OnLButtonUp(pView, e);

            }
        }

        for (let name in this.MouseAdapterList)
        {
            let adapter = this.MouseAdapterList[name];
            adapter.OnLButtonUp(this, pView, e);
        }


    },


    OnMouseMove: function (/*WorkSheetView*/  pView, /*MouseEvent*/   e) {
        if (this.editable && this.Sheet.Editable)
        { //如果能编辑
            if (!this.Sheet.designMode) //如果是运行模式（或者说是数据库模式)
            {

                //如果强制设置了光标，或双击本单元格做排序，那么强制设置成排序

                if (this.cursorName.equalsIgnoreCase('hand') && this.visible
                    || this.dblClickSortDefine != '' && this.visible)
                {

                    pView.setCursor(this.Clickable ? ActionTool.IDC_HAND : ActionTool.IDC_STOP);

                } else
                {

                    pView.setCursor(ActionTool.IDC_TEXT);
                }
                //  如果是CheckBox编辑格式，那么把光标设置成手型
                if (this.ES != null)
                {
                    if (this.ES.ET == EditStyle.$CheckBox) pView.setCursor(ActionTool.IDC_HAND);
                    //  增加对radio, checkbox 的支持
                    if (this.ES.ET == EditStyle.$DDLB)
                    {
                        if (this.ES.listShowAsCheckBox || this.ES.listShowAsRadioButton)
                        {
                            pView.setCursor(ActionTool.IDC_HAND);
                        } else
                            // 如果是DDLB
                        {
                            if (this.decorate != null)
                            {
                                pView.setCursor(ActionTool.IDC_HAND);
                            }

                        }

                    }
                    if (this.ES.ET == EditStyle.$Datetime)
                    {
                        pView.setCursor(ActionTool.IDC_HAND);
                    }

                }
                // 如果是绑定到数据库，且数据库没有数据，那么光标设置为[请插入一行]
                let dbc = this.Bind;
                if (dbc != null)
                {
                    if (this.Book.getDataSource(dbc.dataSource).dataStore.rowCount == 0)
                    {

                        pView.setCursor(ActionTool.IDC_EDITTIP);
                    }
                }

            }
        } else
        {
            if (this.cursorName.equalsIgnoreCase('hand') && this.visible)
            {
                pView.setCursor(ActionTool.IDC_HAND);
            }

            if (this.dblClickSortDefine != '' && this.visible)
            {
                pView.setCursor(ActionTool.IDC_SORT);
            }

        }


        this.BrickMouseEnterExitTest(pView, e);

        // 触发事件
        this.Book.EM.fire("onMouseMove", [pView, this, e]);

        for (let name in this.MouseAdapterList)
        {
            let adapter = this.MouseAdapterList[name];
            adapter.OnMouseMove(this, pView, e);
        }


    },


    BrickMouseEnterExitTest: function (/*WorkSheetView*/  pView, /*MouseEvent*/   e) {
        if (this.BrickMap != null)
        {

            let /*Point*/ me = new Point(e.offsetX, e.offsetY);

            for (let name in this.BrickMap)
            {
                let brick = this.BrickMap[name];


                if (brick.contains(me))
                {
                    brick.OnMouseMove(pView, e);
                    if (!brick.focused)
                    {
                        brick.focused = true;
                        brick.OnMouseEntered(pView, e);
                    }
                } else
                {
                    if (brick.focused)
                    {
                        brick.focused = false

                        brick.OnMouseExited(pView, e);
                    }
                }


            }
        }

    },


    OnMouseDrag: function (/*WorkSheetView*/  pView, /*MouseEvent*/   e) {

        this.BrickMouseEnterExitTest(pView, e);

        if (this.BrickMap != null)
        {
            // 如果有小物件，那么


            let /*Point*/ me = new Point(e.offsetX, e.offsetY);

            for (let name in this.BrickMap)
            {
                let brick = this.BrickMap[name];


                //   必须contains 去掉了，比如在拖动的过程中鼠标会拖出去
                //	if (brick.contains(me))
                {
                    brick.OnMouseDrag(pView, e);

                }
            }
        }

        for (let name in this.MouseAdapterList)
        {
            let adapter = this.MouseAdapterList[name];
            adapter.OnMouseDrag(this, pView, e);
        }

    },


    OnRButtonDown: function (/*WorkSheetView*/  pView, /*MouseEvent*/   e) {
        if (this.BrickMap != null)
        {
            // 如果有小物件，那么
            for (let name in this.BrickMap)
            {
                let brick = this.BrickMap[name];
                if (brick.contains(me))
                {
                    brick.OnRButtonDown(pView, e);

                }
            }
        }

    },


    OnMouseEntered: function (/*WorkSheetView*/  pView, /*MouseEvent*/   e) {
        //		不需要交给单元格处理其中的小件，因为Cell会在MouseMove中处理小件们的MouseEntered和MouseExited

    },


    OnMouseExited: function (/*WorkSheetView*/  pView, /*MouseEvent*/   e) {
        //		不需要交给单元格处理其中的小件，因为Cell会在MouseMove中处理小件们的MouseEntered和MouseExited

    },

    keyPressed: function (/*WorkSheetView*/  pView, /*KeyEvent*/   e) {

    },


    keyReleased: function (/*WorkSheetView*/  pView, /*KeyEvent*/   e) {

    },


    getAutoTip: function (/*MouseEvent*/   e) {
        //TODO
        return "";

    },


    getAlias: function () {

        return this.Alias == null ? "" : this.Alias;
    },


    /**
     * 返回错误信息，如果返回""表示成功
     */


    setAlias: function (alias, /*boolean*/  replace) {
        replace = replace || false;
        if (this.Alias == null && alias == null) return "";
        if (this.Alias != null && alias != null) if (this.Alias.equals(alias)) return "";
        if (alias != null) alias = alias.trim();
        let ret = this.Sheet.$setCellAlias(this, alias, replace);
        if (ret.length > 0) return ret;
        this.Alias = alias;
        return "";

    },


    //是不是显示为图片


    isShowAsImage: function () {
        let /*String*/ s = this.getShowText(0);
        return (s.startsWith(Cell.$RichData_Image));

    },


    getDataStore: function () {
        if (this.Bind == null) return null;
        let dsc = this.Book.getDataSource(this.Bind.dataSource);
        if (dsc == null) return null;
        return dsc.dataStore;
    },


    /**
     * @api {beginEdit} 函数   beginEdit
     *
     * @apiDescription beginEdit()
     * <br><br> 让单元格进入编辑状态
     *
     * @apiName  beginEdit
     * @apiGroup Cell
     * @apiVersion 1.0.0
     *
     *
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     */
    beginEdit: function (splash) {

        let /*int*/     row = this.rowIndex;
        let /*int*/     col = this.columnIndex;
        let view = this.Sheet.View;
        if (view != null) view.$beginEdit(row, col, null, null, -1);
        if (splash) view.getCurrentEditingEdit().splash(true);

    },


    getLeftTopCorner: function () {
        if (!this.isMerged()) return this;
        let r = this.getMergedRange();
        return this.Sheet.cells(r.startRow, r.startCol);
    },


    getRightBottomCorner: function () {
        if (!this.isMerged()) return this;

        let r = this.getMergedRange();
        return this.Sheet.cells(r.endRow, r.endCol);
    }
    ,


    addSubscriber: function (/*ChangeListener*/   cl) {
        if (this.subscribeList == null) this.subscribeList = [];
        this.subscribeList.push(cl);
    },


    removeSubscriber: function (/*ChangeListener*/   subscriber) {
        if (this.subscribeList == null) return;
        this.subscribeList.remove(subscriber);
    },


    //得到优先级的原始的定义


    // 得到优先级的结果


    isFormulaPriority: function () {
        if (this.formulaIsPriority == 1) return true; //公式优先
        if (this.formulaIsPriority == 2) return false; //数据库优先
        if (this.formulaIsPriority == 0) //自动判断　
        {
            //当它的公式依赖其它单元格时，公式优先，如果仅仅是函数及运算，但不包含对其它单元格的引用，则数据库优先
            if (this.DependList == null) return false; //不包含对其它单元格的引用，则数据库优先
            if (this.DependList.length > 0) return true; // 包含对其它单元格的引用，则公式优先
        }

        return true;
    },


    setFormulaNeedCache: function (b) {
        if (this.DependList != null)
        {
            Tools.log("本单元格公式中引用了其它单元格，因此它的缓存是自动处理的，不受此参数控制。仅类似 newData() " +
                "这样的不引用其它单元格的公式才受引标记影响 ");
            return;
        }
        this.formulaNeedCache = b;
        if (!b) this.setInvalid();//如果公式被设置成不需要缓存，那么使缓存无效

    },


    isFormulaNeedCache: function () {
        if (this.DependList != null) return true; //对于有引用其它单元格的公式，系统自动处理缓存，不受此标记控制
        return this.formulaNeedCache;
    },

    /**
     *
     *
     * @param holdProperty
     * @returns {null}
     */
    serializeTo: function (holdProperty) {

        if (holdProperty == null) holdProperty = false;

        let ret = {};

        try
        {
            if (this.isMerged())
            {
                let mcell = this.mergedBy();
                if (this != mcell) return null; // 如果是被合并的，且不是最左上角的单元格，那么忽略它
            }

            ret.row = this.rowIndex;
            ret.col = this.columnIndex;

            if (this.Define || '' != '') ret.define = this.Define || '';
            if (!this.visible) ret.visible = this.visible;
            if (this.editable) ret.editable = this.editable;
            if (this.viewAs != '') ret.viewAs = this.viewAs;
            if (this.passwordMask != "*") ret.passwordMask = this.passwordMask;
            if (this.cursorName != '') ret.cursorName = this.cursorName;
            if (this.dblClickSort != '') ret.dblClickSort = this.dblClickSort;
            if (this.cssName != '') ret.cssName = this.cssName;

            if (this.Prefix != '') ret.prefix = this.Prefix;
            if (this.Suffix != '') ret.suffix = this.Suffix;


            if (this.clickJumpTo != '') ret.clickJumpTo = this.clickJumpTo;
            if (this.clickJumpToWhere != '_blank') ret.clickJumpToWhere = this.clickJumpToWhere;
            if (this.clickPopupDialogWidth != 800) ret.clickPopupDialogWidth = this.clickPopupDialogWidth;
            if (this.clickPopupDialogHeight != 600) ret.clickPopupDialogHeight = this.clickPopupDialogHeight;
            if (this.clickPopupDialogTitle != '') ret.clickPopupDialogTitle = this.clickPopupDialogTitle;


            if (this.popoverType != 'html') ret.popoverType = this.popoverType;
            if (this.popoverContent != '') ret.popoverContent = this.popoverContent;
            if (this.popoverDialogWidth != '') ret.popoverDialogWidth = this.popoverDialogWidth;
            if (this.popoverDialogHeight != '') ret.popoverDialogHeight = this.popoverDialogHeight;
            if (this.popoverDialogTitle != '') ret.popoverDialogTitle = this.popoverDialogTitle;
            if (this.popoverAutoClose != true) ret.popoverAutoClose = this.popoverAutoClose;


            if ((this.validateRule || '') != '') ret.validateRule = this.validateRule;
            if ((this.validateTip || '') != '') ret.validateTip = this.validateTip;

            if (this.tag) ret.tag = this.tag;
            if (this.appData) ret.appData = this.appData;

            //2019.11.03 增加条码支持
            if (this.viewAs == 'barcode')
            {
                ret.barcodeType = this.barcodeType;
                ret.barcodeWidth = this.barcodeWidth;
                ret.barcodeHeight = this.barcodeHeight;
                ret.barcodeTextMargin = this.barcodeTextMargin;
                ret.barcodeTextVisible = this.barcodeTextVisible;
                ret.barcodeCenterImageURL = this.barcodeCenterImageURL;
            }

            let vt = this.getCellValueType();

            if (vt > 0 && vt < UniformDataType.$Auto) ret.cellValueType = vt;

            if (this.isMerged())
            {
                let r = this.getMergedRange();
                ret.rowSpan = r.endRow - r.startRow + 1;
                ret.colSpan = r.endCol - r.startCol + 1;
            }

            // 别名
            let alias = this.getAlias();
            if (alias != '') ret.alias = alias;

            if (this.placeholder != '') ret.placeholder = this.placeholder;   //占位提示
            //仅在编辑时显示占位提示
            if (this.placeholderOnlyViewWhenEditing == false)
            {
                ret.placeholderOnlyViewWhenEditing = this.placeholderOnlyViewWhenEditing;
            }

            if (this.cssName == '')//如果有css，那么直接忽略属性设置
            {
                // 属性
                let p = this.getPropertyObject();
                //不是缺省属性
                if (p != this.Book.PM[0] && !holdProperty)
                {
                    let prop = {};
                    for (let pn in p.property)
                    {
                        let v = p.property[pn];
                        if (v == this.Book.PM[0].get(pn)) continue;//如果就是默认值，那么就不需要保存
                        if (v == null) continue;
                        if ('' + v == '') continue;  // 2019.12.17    v=0 时  v=='' 是 true  v===''是false


                        if (Util.isClas$(v))
                        {
                            if (v.instanceOf(Color))
                            {
                                prop[pn] = v.color;
                                continue;
                            }
                        }

                        prop[pn] = v;
                    }

                    let ps = JSON.stringify(prop);
                    let propID = Tools.MD5(ps);

                    if (!this.Sheet.Book.propertyIdList.contains(propID)) this.Sheet.Book.propertyIdList.push(propID);
                    this.Sheet.Book.propertyCache[propID] = prop; // 临时的，

                    ret.PI = this.Sheet.Book.propertyIdList.indexOf(propID);

                }

                // 下面是旧的属性导出代码，常规的不需要这么导出
                //但是当跨模板进行超级复制粘贴时，不可能再用book.propertyCache 和 propcertyIdList来共享，
                //这种情况下， 属性必须直接输出
                //不是缺省属性
                if (holdProperty)
                {
                    let prop = {};
                    ret.property = prop;
                    if (p != this.Book.PM[0])
                    {
                        for (let pn in p.property)
                        {
                            let v = p.property[pn];
                            if (v == this.Book.PM[0].get(pn)) continue;//如果就是默认值，那么就不需要保存
                            if (v == null) continue;
                            if (v == "") continue;


                            if (Util.isClas$(v))
                            {
                                if (v.instanceOf(Color))
                                {
                                    prop[pn] = v.color;
                                    continue;
                                }
                            }

                            prop[pn] = v;
                        }

                    }
                }
            }


            if (this.Bind != null)
            {
                let bind = this.Bind;
                ret.bind = {
                    dataSource: bind.dataSource,
                    DBCol: bind.DBCol,
                    dsType: bind.dsType
                };

            }
            if (this.ES != null)
            {
                let es = this.ES;
                let et = es.ET;
                let one = {
                    editType: et
                };

                ret.editstyle = one;

                switch (et)
                {
                    case EditStyle.$Normal:
                        one.inputAttribute = es.inputAttribute;
                        break;
                    case EditStyle.$MultiLine:
                        break;
                    case EditStyle.$Datetime:
                        one.datetimeFormat = es.datetimeFormat;
                        break;
                    case EditStyle.$Numeric:
                        one.showComma = es.showComma;
                        one.decimalCount = es.decimalCount;
                        one.zeroVisible = es.zeroVisible;
                        break;
                    case EditStyle.$DDLB:
                    case EditStyle.$RadioButton:
                    case EditStyle.$MultiCheckBox:
                        one.ddlbName = es.ddlbName;
                        one.dropDownListEditable = es.dropDownListEditable;
                        one.valueMustInDDLB = es.valueMustInDDLB;
                        one.ddlbFilterBy = es.ddlbFilterBy || '';
                        //注意 ddlbFilterBy_encrypted 与 ddlbFilterBy可能并不匹配
                        //比如当调用配置后，修改了ddlbFilterBy，然后保存 ， 此时ddlbFilterBy_encrypted还是上次的结果
                        //所以保存的ddlbFilterBy与ddlbFilterBy_encrypted并不匹配
                        //当调用配置使用时， ddlbFilterBy_encrypted会再次重新计算并覆盖
                        //所以ddlbFilterBy_encrypted是一个运行时由服务端生成的，并上设计时就存在的
                        //此处也可以不必保存它，只是为了便于查看
                        one.ddlbFilterBy_encrypted = es.ddlbFilterBy_encrypted;

                        one.ddlbFilterContext = es.ddlbFilterContext || '';
                        one.ddlbInnerItems = es.ddlbInnerItems || '';
                        one.showRealData = es.showRealData;
                        one.columnCount = es.columnCount;
                        break;
                    case EditStyle.$CheckBox:
                        one.checkOnValue = es.checkOnValue;
                        one.checkOffValue = es.checkOffValue;
                        one.checkCaption = es.checkCaption;

                        break;
                    case EditStyle.$Tree:
                        one.treeName = es.treeName;
                        one.treeFilterBy = es.treeFilterBy;
                        one.treeFilterBy_encrypted = es.treeFilterBy_encrypted; //

                        one.treeFilterContext = es.treeFilterContext;
                        one.treeWidth = es.treeWidth;
                        one.treeHeight = es.treeHeight;
                        one.treeSelectMulti = es.treeSelectMulti;
                        one.treeExpandOnLoad = es.treeExpandOnLoad;
                        break;
                }

            }

            //控件
            let bricks = this.getBricks();
            if (bricks.length > 0)
            {
                let bs = [];
                ret.bricks = bs;
                for (let i = 0; i < bricks.length; i++)
                {
                    let b = bricks[i];
                    bs.push({
                        type: b.type,
                        name: b.Name,
                        config: b.config,
                        x: b.X,
                        y: b.Y,
                        width: b.Width,
                        height: b.Height
                    });
                }
            }


        } catch
            (e)
        {

            console.error(e);
        }
        return ret;

    }
    ,

    /**
     *
     *
     * @param config
     * @param move  是把 config设置移动到这里(true)，还是复制到这里(false),当复制到这里时，别名不复制
     */
    serializeFrom: function (config, move, sync) {

        if (sync == undefined) sync = true; //同步设置
        //先清空自已
        this.clear();

        if (!config) return;


        if (move == undefined) move = true;
        for (let p in config)
        {
            let v = config[p];

            if (p == 'define')
            {
                this.$setValue(v, -1, false); //  这个速度更快吗?,实测好象没什么太明显的变化
                // this.setValue(v);
                continue;
            }

            if (p == 'rowSpan')
            {
                let rowSpan = config.rowSpan;
                let colSpan = config.colSpan;
                let row = this.rowIndex;  //要用自已的行，列号，因为复制粘贴时， config.row, config.col 是被复制对象的坐标，
                let col = this.columnIndex;
                let sheet = this.Sheet;
                //如果是复制，粘贴， 合并单元格不生效，改成异步，它生效了，原因未明，临时这么解决了先
                if (sync)
                {
                    sheet.merge(row, col, row + rowSpan - 1, col + colSpan - 1);
                } else
                {
                    setTimeout(function () {
                        sheet.merge(row, col, row + rowSpan - 1, col + colSpan - 1);
                    }, 10);
                }
                continue;
            }

            if (p == 'alias')
            {
                if (move)
                {
                    let tc = this.Sheet.cells(v);
                    if (tc != null) tc.alias = ''; //清除已存在的的别名
                    this.alias = v;
                }
                continue;
            }

            if (p == 'cssName')
            {
                this.cssName = v;
                continue;
            }


            if (p == 'visible')
            {
                this.visible = v;
                continue;
            }
            if (p == 'editable')
            {
                this.editable = v;
                continue;

            }
            if (p == 'viewAs')
            {
                this.viewAs = v;
                continue;
            }

            if (p == 'suffix')
            {
                this.Suffix = v;
                continue;
            }

            if (p == 'prefix')
            {
                this.Prefix = v;
                continue;
            }


            //2019.11.03 增加条码支持

            if (p == 'barcodeType')
            {
                this.barcodeType = v || 'CODE128';
                continue;
            }

            if (p == 'barcodeWidth')
            {
                this.barcodeWidth = parseInt(v || 2);
                continue;
            }

            if (p == 'barcodeHeight')
            {
                this.barcodeHeight = parseInt(v || 30);
                continue;
            }


            if (p == 'barcodeCenterImageURL')
            {
                this.barcodeCenterImageURL = v;
                continue;
            }

            if (p == 'barcodeTextMargin')
            {
                this.barcodeTextMargin = parseInt(v || 10);
                continue;
            }

            if (p == 'barcodeTextVisible')
            {
                this.barcodeTextVisible = v;
                if (this.barcodeTextVisible == null) this.barcodeTextVisible = true;
                continue;
            }


            if (p == 'passwordMask')
            {
                this.passwordMask = v;
                continue;
            }

            if (p == 'dblClickSort')
            {
                this.dblClickSort = v;
                continue;
            }

            if (p == 'cursorName')
            {
                this.cursorName = v;
                continue;
            }

            if (p == 'bind')
            {
                this.setBind(v.dataSource, v.DBCol, v.dsType);
                continue;
            }
            if (p == 'cellValueType')
            {
                this.setCellValueType(v);
                continue;
            }

            if (p == 'clickJumpTo')
            {
                this.clickJumpTo = v;
                continue;
            }
            if (p == 'clickJumpToWhere')
            {
                this.clickJumpToWhere = v;
                continue;
            }
            if (p == 'clickPopupDialogWidth')
            {
                this.clickPopupDialogWidth = v;
                continue;
            }
            if (p == 'clickPopupDialogHeight')
            {
                this.clickPopupDialogHeigh = v;
                continue;
            }

            if (p == 'clickPopupDialogTitle')
            {
                this.clickPopupDialogTitle = v;
                continue;
            }

            if (p == 'popoverType')
            {
                this.popoverType = v;
                continue;
            }
            if (p == 'popoverContent')
            {
                this.popoverContent = v;
                continue;
            }
            if (p == 'popoverDialogWidth')
            {
                this.popoverDialogWidth = v;
                continue;
            }
            if (p == 'popoverDialogHeight')
            {
                this.popoverDialogHeight = v;
                continue;
            }
            if (p == 'popoverDialogTitle')
            {
                this.popoverDialogTitle = v;
                continue;
            }
            if (p == 'popoverAutoClose')
            {
                this.popoverAutoClose = v;
                continue;
            }


            if (p == "placeholder") //占位提示
            {
                this.placeholder = v;
                continue;
            }

            if (p == "placeholderOnlyViewWhenEditing")//仅在编辑时显示占位提示
            {
                this.placeholderOnlyViewWhenEditing = v;
                continue;
            }


            if (p == 'appData')//应用定义的单元格更多的数据项，如果应用程序要对单元整扩展更多的属性，请在appData中扩展
            {
                this.appData = v;
                continue;
            }

            if (p == 'validateRule')
            {
                this.validateRule = v;
                continue;
            }

            if (p == 'validateTip')
            {
                this.validateTip = v;
                continue;
            }

            if (p == 'editstyle') continue; //在后面再做，在确保已经绑定后，再做格式的设置


            if (['colSpan', 'row', 'col'].contains(p)) continue;

            //之前的导出，是每个单元格属性独立导出，这个占空间,逐个初始化属性，速度也慢
            if (p == 'property')
            {

                let prop = config.property;
                for (let pn in prop)
                {
                    let pv = prop[pn];
                    this.$setProperty(pn, pv, false); //不要触发事件
                }
                continue;
            }


            //2019.09.04新的属性恢复模式改成，直接用属性ID定位属性对象，而不是每个单元格创建一个属性对象
            // 但是超级复制粘贴，不可能使用book.propertyCache ,所以仍是使用内联输出的属性
            if (p == 'propertyID')
            {
                this.property = this.Sheet.Book.propertyCache[v];
                continue;
            }

            if (p == 'propertyIdIndex' || p == 'PI')
            {

                //这里不是很优雅
                //有时候 book.propertyCache 是一个纯object ,有时候 是一个 Property类
                //所以要判断 ，当它是一个纯Object时，要把它转换成一个Property
                //为什么可能是一个Property呢，是因为在调取配置初始化book时，为了性能
                // 提前把属性对象创建好，放到book.propertyCache中
                // 但是在复制，粘贴时， book.propertyCache中放的是纯Object，而不是Property
                // 所以要根据类型，判断 是不是不转换一下
                let propID = this.Sheet.Book.propertyIdList[v];
                this.property = this.Sheet.Book.propertyCache[propID];

                if (!Util.isClas$(this.property))
                {


                    let prop = this.property;
                    let cellProp = new Property(true);
                    for (let p in prop)
                    {
                        let v = prop[p];
                        cellProp.put(p, v);
                    }
                    cellProp.refCount = 9999;// 引用计数有问题，所以放大一点
                    this.property = cellProp;
                }
                continue;
            }


            //控件
            if (p == 'bricks')
            {

                for (let i = 0; i < v.length; i++)
                {
                    let b = v[i];
                    this.addBrick(b.type, b.name, b.config, b.x, b.y, b.width, b.height);

                }
                continue;
            }


            console.info(p + "没有处理");


        }

        let es = config['editstyle'];
        if (es)
        {

            switch (es.editType)
            {

                case EditStyle.$Normal:
                    this.Book.setEditStyleNormal(this, es.inputAttribute);
                    break;
                case EditStyle.$MultiLine:
                    this.Book.setEditStyleMultiLine(this);
                    break;
                case EditStyle.$Datetime:
                    this.Book.setEditStyleDate(this, es.datetimeFormat);
                    break;
                case EditStyle.$Numeric:
                    this.Book.setEditStyleNumeric(this, es.showComma, es.decimalCount, es.zeroVisible);
                    break;
                case EditStyle.$DDLB:
                    this.Book.setEditStyleDDLB(this, es.ddlbName, es.dropDownListEditable,
                        es.valueMustInDDLB, es.ddlbFilterBy, es.ddlbFilterContext, es.ddlbInnerItems, es.showRealData, es.columnCount);
                    break;
                case EditStyle.$RadioButton:
                    this.Book.setEditStyleRadio(this, es.ddlbName, es.ddlbFilterBy, es.ddlbFilterContext, es.ddlbInnerItems, es.columnCount);
                    break;
                case EditStyle.$MultiCheckBox:
                    this.Book.setEditStyleMultiCheckbox(this, es.ddlbName, es.ddlbFilterBy, es.ddlbFilterContext,
                        es.ddlbInnerItems, es.columnCount);
                    break;
                case EditStyle.$CheckBox:
                    this.Book.setEditStyleCheckbox(this, es.checkOnValue, es.checkOffValue, es.checkCaption);
                    break;
                case EditStyle.$Tree:
                    this.Book.setEditStyleTree(this, es.treeName, es.treeWidth, es.treeHeight,
                        es.treeSelectMulti, es.treeExpandOnLoad, es.treeFilterBy, es.treeFilterContext);
                    break;


                default:
                    console.info("没有处理的编辑方式" + es.editType);
            }
            if (es.ddlbFilterBy_encrypted) this.ES.ddlbFilterBy_encrypted = es.ddlbFilterBy_encrypted;
            if (es.treeFilterBy_encrypted) this.ES.treeFilterBy_encrypted = es.treeFilterBy_encrypted;

        }

    }


});

export default Cell;
