/**
 *
 * 这是所有命令类的基类， 保存当时的坐标偏移，并调用 $SaveScene()来保存当时的场景，
 *
 * 当Execute执行时，首先是恢复坐标偏移（当是第一次执行时，实际上是没有什么实际效果，但 是当做ReDo时，这个恢复就是有必要的），再执行实际的$Execute
 *
 * 当Execute执行时，首先是恢复坐标偏移，再执行实际的$Execute
 *
 * 对Sheet采取的弱引用，在Redo和Undo时要即时获取实际的Sheet，如果它被删除了，那么就不要做了
 *
 * Created by zengjun on 2017/5/4.
 */


import Class from '../base/Class.js';

var Command = Class.extend({


    constructor: function (/*WorkSheet*/sheet, /*Property*/ scene, level) {
        this.Sheet = sheet;
        this.Scene = scene;
        this.Level = level || 0;
        // 如果不需要Undo功能，那么不需要保存场景
        if (sheet.Book.undoEnabled) this.saveScene();
    },


    setLevel: function (level) {
        this.Level = level;
    },


    getLevel: function () {
        return this.Level;
    },


    /**
     *  命令真正的执行实体
     * @returns {boolean}
     */
    $execute: function () {
        return true;
    },


    /**
     * 命令真正的撤销执行的实体
     * @returns {boolean}
     */
    $undo: function () {
        return true;
    },


    /**
     * 命令真正的保存场景的实体
     */
    $saveScene: function () {

    },


    saveScene: function () {

        this.backupXYOffset();
        this.$saveScene();
    },


    execute: function () {
        // 如果不需要Undo功能，那么直奔主题吧
        if (!this.Sheet.workBook.undoEnabled)
        {
            return this.$execute();
        }


        var /*CommandManage*/ CM = this.Sheet.CM;

        this.restoreXYOffset();
        CM.stepIn();// 层次加1 ，
        // 设置命令的层次。初始CM.getLevel==0 ,所以如果命令不是被嵌套调用的话，
        // 在setLevel后，命令的level应该是1
        // 注意：如果一个命令的leve>1 表明它是被嵌套调用的，它将会作为一个操作组中的一环，
        // 参看CommandManage中的说明

        this.setLevel(CM.getLevel());

        // 当在 $execute实体中又调用其它的Command时，其它的Command的层次是当前层次加1
        var ok = this.$execute();
        // 在自己真正的执行过程被执行后，复原CM的层次
        CM.stepOut(); // 层次减1

        // 如果执行成功，那么放到Undo列表里去
        // 可能存在的隐患： 当嵌套的命令被压栈了，但自己却因失败而没有压栈，会导致CM中的 Undo 存在一些问题
        if (ok)
        { // 如果有些命令临时调用，不需要进入Undo栈，那么可以用CM.setNeedUndo(false)来控制
            // 参看 Cell.CopyFrom();
            if (CM.needUndo) CM.pushCommand(this);
        }
        else
        {
            // TODO：对上述隐患的解决方法：pop所有Undo栈中 level>自己level的Command

        }

        return ok;
    },

    redo: function () {
        //
        if (!this.Sheet.workBook.undoEnabled) return true;

        this.restoreXYOffset();
        return this.$execute();
    },


    /**
     * <pre>
     * 注意： Undo中调用的其它Command，不需要进入Redo栈。
     * =================================================
     *
     *
     *   a  设置 c3=c1+c2
     *   b  删除第一行
     *   b.1  由于b中调用了重新设置单元格公式，所以 c3被成了c2 公式变成了=0+c1 ,于是Redefine中调用
     *               setValue 设置 c2的新公式。此时的Cmd_SetValue 的 level==2
     *   c  Undo  删除行
     *   c.1  由于Undo，所以重新设置单元格公式，所以 c2被成了c3 公式变成了=0+c2 ,于是Redefine中调用
     *               setValue 设置 c3的新公式。此时的Cmd_SetValue 的 level==1 ， 于是新的Cmd_SetValue 被
     *               压入Undo栈。 当 Cmd_DeleteMultiRow 的Undo做完后，继续看一下个Undo命令的Level，本来
     *               这个命令应该是 b.1 中的Cmd_SetValue，但是由于刚才又新入栈了一个Level==1的Cmd_SetValue，
     *               于是， CommandManege_ .Undo 结束操作 。这样，Undo框中非但没有把 b.1给Undo，反而又增加
     *               了一个命令。于时再&lt;Ctrol&gt;&lt; Z&gt;两次，才能恢复到b执行前的状态。这是个错误。
     *
     *     正解是：
     *--------------------------------------------------------------------------------------------------
     *             所有$Execute 过程中调用的其它Command都需要进入Undo栈（它们的Level&gt;1) .而在$Undo过程
     *               中调用其它的Command都不能进入Undo栈。因为调用它们目的是为Undo操作，而不
     *               是产生新的Undo。
     * -------------------------------------------------------------------------------------------------
     *
     * </pre>
     *
     */

    //
    undo: function () {
        if (!this.Sheet.workBook.undoEnabled) return true;

        this.restoreXYOffset();

        var b = this.Sheet.CM.getNeedUndo();
        this.Sheet.CM.setNeedUndo(false);
        var ret = this.$undo();
        this.Sheet.CM.setNeedUndo(b);
        return ret;

    },

    backupXYOffset: function () {

        var xoffset = this.Sheet.getXOffset();
        var yoffset = this.Sheet.getYOffset();
        this.Scene.put("xoffset", xoffset);
        this.Scene.put("yoffset", yoffset);

    },

    restoreXYOffset: function () {

        var xoffset = this.Scene.get("xoffset", 0);
        var yoffset = this.Scene.get("yoffset", 0);
        this.Sheet.setXOffset(xoffset);
        this.Sheet.setYOffset(yoffset);

        var /*WorkBookView*/ view = this.Sheet.Book.View;
        if (view != null)
        {
            view.resetHScrollBarScrollRange();
            view.resetVScrollBarScrollRange();
        }
    }


});


export default Command ;
