/**
 * Created by 三宝爹 on 2020/04/07.
 *
 *
 * 这是一个集成的上传图片的组件，用于选择图片上传。用到了gridly 对缩略图做排列显示，以及拖拽排序
 *
 * 通过 config.action = 'new','edit','view' 来控制上传动作。当action=='view'时，只读，不允许上传及删除
 *
 * 删除前触发事件用来控制删除权限。上传所需要的参数配置与 全局的 uploadFile 函数参数一至，需要配合formEngine才能工作
 *
 * 实现细节描述
 *
 * 1 默认的上传文件INPUT太丑，解决办法是，临时创建一个 input ,并模拟触发它的click，弹出选择文件，选择后，它的change
 *   事件被触发 ，因此在change事件中，直接触发上传
 *
 * 2 根据 action ,最所有图片最后面增加一个 [+] 触发新的上传
 *
 * 3 所有图片 ，以及 [+] 组成一组 gridly ,可以拖动排序 ，注意，可能[+]在拖动后不在最后面，解决办法就是先删除这个按钮，再增加这个按钮
 *
 * 4 根据action ，在图片右上角显示 删除按钮
 *
 * 5 点击图片，弹出，浏览该组大图 ， 然而，居然，没有找到方法获取当前的图片顺序列表。最后只好，按坐标进行比较排序
 *
 *
 * 2020.05.08增加自动加载图片的支持，实现原理是传入一个函数名，执行这个函数，返回值就是所需要的图片信息
 * 在 formEngine中增加一个默认的根据配置返回图片清单的函数，这样，基本上0开发就可以实现所有功能
 *
 *
 *  在  初始化之前事件中可以 通过 设置   config.uploadButtonDisabled 禁用上传按钮
 *
 *   config.deleteButtonDisabled 禁用删除按钮
 *
 */


import HTMLBrick from './HTMLBrick.js';
import Util from '../util/Util.js';
import Gridly from '../base/jquery.gridly.js'; //只需要导入，它会自已注入到jquery中
import Tools from '../util/Tools.js';
import ImageUtil from '../util/ImageUtil.js'
import uploadDemoData from '../util/UploadDemoData.js';

const VK_LEFT = 37
const VK_UP = 38;
const VK_RIGHT = 39;
const VK_DOWN = 40;
const VK_SPACE = 32;
const VK_ESC = 27;


//根据坐标来排序
function compare(img1, img2)
{
    if (img1.y > img2.y) return 1;
    if (img1.y < img2.y) return -1;

    if (img1.x > img2.x) return 1;
    return -1;
}


let UploadBrick = HTMLBrick.extend({

    properties:
        {


            /**
             * @api {selectionMode} 可读可写属性 [属性]selectionMode
             * @apiName  visible
             * @apiDescription 图片管理控件的选择模式
             * @apiGroup UploadBrick
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){String} - 选择类型 ,可选
             *  <ul><li>none:默认， 没有选择标记显示</li>
             *  <li>single:单选模式</li>
             *  <li>multi:多选模式，当按下Shift 时，如果当前图片没有选中，则把当前图片选中，并把它它前面的图片也选中，直到碰到一个被选中的图片为止。
             *  如果当前图片是选中状态，则把当前图片取消选中，并把它前面的图片逐级取消选中，直到碰到一个没有选中图片为止</li>
             * </ul>
             *
             *
             * @apiExample {js}(示例：)
             * //示例
             *
             *  alert( book.sheet0.getBrick('Upload1').selectionMode );
             *  book.sheet0.getBrick('Upload1').selectionMode="multi";
             *
             */
            "selectionMode":
                {
                    get: function () { return this.getSelectionMode();},
                    set: function (val) { this.setSelectionMode(val);}
                },


            /**
             * @api {imageCount} 只读属性 [属性]imageCount
             * @apiName  imageCount
             * @apiDescription 图片数
             * @apiGroup UploadBrick
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){int} - 图片数
             *
             * @apiExample {js}(示例：)
             * //示例
             *
             *  alert( book.sheet0.getBrick('Upload1').imageCount;
             *
             */
            "imageCount":
                {
                    get: function () { return this.getImageCount();},
                }
        },


    constructor: function (name, config, x, y, width, height) {

        this.init(name, config, x, y, width, height);

        this.m_selectionMode = 'none';

        console.info("upload init ");
        this.type = 'uploadImage';


        //所有的参数理论上都可以用公式来定义 ， 以=开头即表示是公式，
        //但这样，就需要在使用参数前，每次都 parseConfig()才行， 现在没有这个需求，等需要时再扩展
        this.config = Util.merge(this.config, {
                gutter: 10, //图块间隔
                columns: 3, // 一行显示几个图块
                blockSize: 64, //生成的缩略图大小
                action: '=currentAction()',
                maxCount: 3, //最多允许上传几张图
                homeURL: "",
                imgMaxSize: 1920,
                //上传选项
                serverIndex: 1, //上传到哪个文件服务器
                serverIndexExpression: "",
                maxThumbnailSize: 256,
                dbpool: '', //上传结果保存到哪个连接池
                logTable: 'app_fileupload', //上传结果保存到哪个表
                tableName: '=currentMasterTableName()',   // 数据相关的表名称
                path: '=currentMasterTableName()',
                gguid: '=currentGUID()',
                autoLoad: false,
                showImageIndexNo: true, //显示图片序号
                rightMouseDownToViewLargeImage: true //右键点下显示大图
            },
            false //表示不要用默认值覆盖传入的值，仅当没有设置某参数时，才使用用缺省参数
        );


        //参数规整一下类型
        this.config.gutter = parseInt(this.config.gutter);
        this.config.blockSize = parseInt(this.config.blockSize);
        //保持设置原样 ，下面用到的时候再解析
        this.config.columns = parseInt(this.config.columns);

        this.config.maxCount = parseInt(this.config.maxCount);
        this.config.imgMaxSize = parseInt(this.config.imgMaxSize);
        this.config.maxThumbnailSize = parseInt(this.config.maxThumbnailSize) || parseInt(this.config.blockSize); // 上传缩略图的大小


    },

    getSelectionMode: function () {
        return this.m_selectionMode;
    },

    setSelectionMode: function (val) {
        if (!['none', 'single', 'multi'].contains(val)) return;
        this.m_selectionMode = val;
    },

    //在自已加入到单元格后， 本事件触发
    bind: function () {

        let that = this;

        if (!this.pSheet.designMode)
        {
            console.info("autoLoad uploaded image");
            if (this.config.autoLoad && this.config.autoLoadFunction != '')
            {

                that.reInit = function (callback) {
                    try
                    {
                        //得到数据来源函数，并执行它，
                        //把当前的配置做参数代入，注意，需要临时再解析一下， 因为此时的公式值可能变化了
                        // 比如 单据翻到下一张后， currentGUID()就变化了，所以需要再解析一下
                        // 因为本响应是在flowOpen后执行，所以相关的一些与单据有关的数据，都已经初始化完成了
                        let func = window[that.config.autoLoadFunction];
                        if (func)
                        {
                            func(that.parseConfig(), function (ret) {
                                if (ret.success)
                                {
                                    that.buildGridly(ret.data, callback);
                                    //2021.05.14
                                    that.pSheet.Book.View.resize(true);//强行resize重新计算一下尺寸，不然可能卷滚条不显示
                                }
                            });
                        }

                    } catch (err)
                    {
                        alert("图片上传控件加载图片异常：" + err);
                    }
                }
                // 通过事件簇的方式增加一个 formOpen 响应函数，formOpen之后执行的响应函数
                //2021.09.08 补丁 此前是用  'formOpen$loadImage_' + that.name 做函数名，但是当Sheet中有多个单元格中都有此控件时
                // 由于默认的名称都是upload1，并且通常不会去改它，这样就导致此多个控件的此函数名同名，导致只有最后一个控件加载了图片
                // 解决方案是函数名上再拼上一个guid
                // formOpen$ 是一个事件族，formOpen时此族事件都会被自动调用，它在formOpen事件之后触发
                window['formOpen$loadImage_' + that.name + "_" + Tools.newGUID()] = that.reInit;

            }
        }

    },

    getAction: function () {


        return this.getConfig("action").toLowerCase();

    },

    getConfig: function (propName, defaultValue) {
        let t = this.config[propName];
        if (t == undefined) return "";
        if (typeof (t) != 'string') return t;

        t = t.trim();

        if (t.startsWith("="))
        {
            t = t.substring(1);
            if (this.pSheet == null) return defaultValue; //设计状态可能出现
            let ret = this.pSheet.evaluate(t);

            if (ret == null) ret = "";
            return ret;
        } else
        {
            return t;
        }

    },

    getImageCount: function () {

        let brickImages = $(`#${this.gridlyID} .brickImage`);//获取所有划块
        return brickImages.length;

    },

    /**
     * this.config 中不少属性是公式，本函数就是负责把它们解析成数据
     */
    parseConfig: function () {
        let ret = {};
        for (let p in this.config)
        {
            if (p == 'autoLoadFunction') continue; // 本函数主要用在两个地方，一个是初始时构建配置以参数形式代入全局函数中执行，此时不要递归执行此函数，二是以参数形式代入上传中，也不要再执行此函数
            ret[p] = this.getConfig(p);
        }
        //文件服务器，也可能是表达式，表达式优先
        if (ret.serverIndexExpression != null && !isNaN(ret.serverIndexExpression) && ret.serverIndexExpression != '')
        {
            ret.serverIndex = parseInt(ret.serverIndexExpression);
        }

        delete ret['serverIndexExpression'];

        return ret;
    },

    paint: function (/*boolean */isPrint  /*int*/, /*Graphics2D*/ g, /*Rectangle*/ parentRC) {

        if (!this.Visible) return;

        if (this.dom == null) return;


        let rc = this.getBounds(parentRC);

        this.dom.css("position", "absolute");

        this.dom.css("left", rc.x + "px");
        this.dom.css("top", rc.y + "px");
        this.dom.css("width", rc.width + "px");
        this.dom.css("height", rc.height + "px");
        this.dom.css("overflow", "hidden");
        this.dom.css("tabindex", 1);
        if (this.lastWidth == undefined)
        {
            this.lastWidth = rc.width;
            return;
        }

        //当单元格需要重绘时，echart并不一定需要重绘，仅当size变化时，才需要重绘
        if (this.lastWidth != rc.width)
        {
            this.lastWidth = rc.width;
            this.rebuildGridly();
        }


    },


    createDOM: function () {

        this.domID = this.DOMName + Tools.newGUID();
        this.gridlyID = 'gridly_' + this.domID;

        let html = `<div><div id='${this.domID}' style='width:100%;height:100%;background-color: transparent;'></div></div>`;

        this.config.html = html;//是为了下面授createDOM


        this.$uper.createDOM.call(this, false);//false表示基类中不要触发 brickOnload事件


        this.buildGridly();

        let that = this;
        that.pSheet.Book.EM.fire("brickOnload", [that.pSheet, that]);


    },

    //构建内容
    /**
     * @api {buildGridly} 函数   buildGridly
     *
     * @apiDescription buildGridly(data)
     * <br><br> 初始化或重建图片管理控件的图片清单
     *
     * @apiName  buildGridly
     * @apiGroup UploadBrick
     * @apiVersion 1.0.0
     *
     * @apiParam {Array} data 图片清单数组。每个元素是这样的一个对象：
     * <ul>
     *     <li>id: 图片的唯一标识</li>
     *     <li>url_img:图片的地址</li>
     *     <li>url_thumbnail:图片缩略图地址 </li>
     * </ul>
     *
     *
     *
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *  let data = [
     {
                "id": "5e8d400dae7ad7fe6866b74c",
                "url_img": "http://devcloud.oss-cn-qingdao.aliyuncs.com/a/b/5e8d400dae7ad7fe6866b749.jpg",
                "url_thumbnail": "http://devcloud.oss-cn-qingdao.aliyuncs.com/a/b/5e8d400dae7ad7fe6866b749_thumbnail.jpg"
            },
     {
                "id": "5e8d400dae7ad7fe6866b748",
                "url_img": "http://devcloud.oss-cn-qingdao.aliyuncs.com/a/b/5e8d400dae7ad7fe6866b745.jpg",
                "url_thumbnail": "http://devcloud.oss-cn-qingdao.aliyuncs.com/a/b/5e8d400dae7ad7fe6866b745_thumbnail.jpg"

            },
     {
                "id": "5e90a7e999326d9cbd8c6c3a",
                "url_img": "https://devcloud.oss-cn-qingdao.aliyuncs.com/a/b/5e90a7e999326d9cbd8c6c37.jpg",
                "url_thumbnail": "https://devcloud.oss-cn-qingdao.aliyuncs.com/a/b/5e90a7e999326d9cbd8c6c37_thumbnail.jpg"
            }
     ];
     let brick= book.sheet0.getBrick('Upload1');
     brick.buildGridly(data);

     */
    buildGridly: function (data, callback) {

        console.info("初始化 gridly ");


        if (data != null) this.data = data; //缓存起来 ，便于 rebuild

        if (data == null) data = [];

        this.pSheet.Book.EM.fire("imageGridlyBeforeInit", [this.pSheet, this.Name, this.data]);


        this.justBuilding = true;
        //创建CSS，因为这些小块的尺寸是通过参数传入的，而不是预先定好了
        if (!this.dom) this.createDOM();

        $(`#${this.domID}`).html(''); //清空旧的内容

        let that = this;

        let w = this.pCell.getBounds().width;

        //在手机上，默认是resize图块大小， 在PC上， 默认是 resize 列数
        if (Tools.isMobile())
        {

            //(b )*n + (n-1)*10 = w;
            //  b= (w - (n-1)*10 )/ n  ;
            //gutter 是块间间隔 ，  columns是每行显示块数
            //visualBlockSize  是每个块的视觉宽度
            this.visualBlockSize = Math.floor((w - (this.config.columns - 1) * this.config.gutter) / this.config.columns);

        } else //在PC上， 图块大小保持不变，自动设置列数
        {
            // n = ( w+g)/(b+g);
            this.visualBlockSize = this.config.blockSize;
            this.config.columns = Math.floor((w + this.config.gutter) / (this.visualBlockSize + this.config.gutter));
        }

        let visualBlockSize = this.visualBlockSize;


        let t = `
            <style>
                #${this.gridlyID} { }  
                #${this.gridlyID} .gridly_brick{
                        width: ${visualBlockSize}px;
                        height: ${visualBlockSize}px;
                        background-color:transparent;
                      
                        }       
                
                #${this.gridlyID} .gridly_brick  .brickInner{
                        position:absolute;
                        left:10px; top:10px; right:10px;  bottom:10px;
                         width:${visualBlockSize - 22}px;
                        height:${visualBlockSize - 22}px;
                        border:1px  solid  lightgray; cursor:pointer;
                        font-size:2em;  color:#606060;  overflow:hidden;
                        /*flex 布局*/
                        display: flex;
                        /*实现垂直居中*/
                        align-items: center;
                        /*实现水平居中*/
                        justify-content: center;
           
                }
                
                #${this.gridlyID} .gridly_brick  .brickSelected{
                        
                        position:absolute;
                        left:0px;top:0px;width:32px;height:32px;
                        cursor:pointer;
           
                }
                
                #${this.gridlyID} .gridly_brick  .imageIndex{
                        
                        position:absolute;
                        left:0px;bottom:0px;width:100%;height:12px;
                        text-align:center;
                        color:darkgray;
                        font-family: Arial;
                        font-size:12px;
                }
                
                 #${this.gridlyID} .gridly_brick  .brickInner:hover{
                        
                        border:1px  solid  lightgray; cursor:pointer;
                        box-shadow:lightgray 0px 0px 10px;
           
                }
                    
                #${this.gridlyID} .gridly_brick  .brickInner img
                {
                   
                       max-width:${visualBlockSize - 22}px;
                       max-height:${visualBlockSize - 22}px;
                      
                       
                }
                    
                    
                #${this.gridlyID} .gridly_brick  .brickInner  .imageUploading
                {
                        
                        position:absolute; left:0; bottom:0; width:100%; height:100%;
                        color:white; font-size:0.9em;
                        /*flex 布局*/
                        display: flex;
                        /*实现垂直居中*/
                        align-items: center;
                        /*实现水平居中*/
                        justify-content: center;
                        
                }
                #${this.gridlyID} .gridly_brick  .brickInner  .uploadingProgress
                {
                        
                        position:absolute; left:0; bottom:0; width:100%; height:100%;
                        color:white; font-size:0.5em;
                        /*flex 布局*/
                        display: flex;
                        /*实现垂直居中*/
                        align-items: center;
                        /*实现水平居中*/
                        justify-content: center;
                }
                    
                
                #${this.gridlyID} .gridly_brick .deleteButton{
                        position:absolute;
                        top:5px; right:5px; width:20px; height:20px; 
                        cursor:pointer;
                        font-size:1.1em;
                        /*flex 布局*/
                        display: flex;
                        /*实现垂直居中*/
                        align-items: center;
                        /*实现水平居中*/
                        justify-content: center;
                   }
               
                        
             #${this.gridlyID} .gridly_brick .toolButton1{
                        position:absolute;
                        top:5px; right:5px; width:20px; height:20px; 
                        cursor:pointer;
                        font-size:1.1em;
                        /*flex 布局*/
                        display: flex;
                        /*实现垂直居中*/
                        align-items: center;
                        /*实现水平居中*/
                        justify-content: center;
                        }
             #${this.gridlyID} .gridly_brick .toolButton2{
                        position:absolute;
                        top:5px; right:26px; width:20px; height:20px; 
                        cursor:pointer;
                        font-size:1.1em;
                        /*flex 布局*/
                        display: flex;
                        /*实现垂直居中*/
                        align-items: center;
                        /*实现水平居中*/
                        justify-content: center;
                        }
             #${this.gridlyID} .gridly_brick .toolButton3{
                        position:absolute;
                        top:5px; right:47px; width:20px; height:20px; 
                        cursor:pointer;
                        font-size:1.1em;
                        /*flex 布局*/
                        display: flex;
                        /*实现垂直居中*/
                        align-items: center;
                        /*实现水平居中*/
                        justify-content: center;
                        }
               #${this.gridlyID} .gridly_brick .toolButton4{
                        position:absolute;
                        top:5px; right:68px; width:20px; height:20px; 
                        cursor:pointer;
                        font-size:1.1em;
                        /*flex 布局*/
                        display: flex;
                        /*实现垂直居中*/
                        align-items: center;
                        /*实现水平居中*/
                        justify-content: center;
                        }       
                #${this.gridlyID} .gridly_brick .toolButton5{
                        position:absolute;
                        top:5px; right:89px; width:20px; height:20px; 
                        cursor:pointer;
                        font-size:1.1em;
                        /*flex 布局*/
                        display: flex;
                        /*实现垂直居中*/
                        align-items: center;
                        /*实现水平居中*/
                        justify-content: center;
                        }       
             
            </style>
            
            <div id="${this.gridlyID}">
              
            </div>
            `;

        $('#' + this.domID).html(''); //清除旧的

        $(t).appendTo($('#' + this.domID));

        //按data构建brick

        for (let i = 0; i < data.length; i++)
        {
            let b = this.addImage(data[i]);
            if (this.config.showImageIndexNo) b.find('.imageIndex').html(i + 1); //设置序号
            if( data[i].title)  b.find('.imageIndex').html( data[i].title); //设置title
        }


        let gridly = $(`#${this.gridlyID}`);

        let action = this.getAction();

        if (action == 'new' || action == 'edit')
        {
            if (data.length < this.config.maxCount) this.insertNewButton();
        }
        //如果没有图片，且是只读，那么显示一个无的占位符
        if (action == 'view' && gridly.find('.gridly_brick .brickImage').length == 0)
        {
            this.insertNullButton();
        }

        gridly.gridly({
            base: visualBlockSize, // px
            gutter: this.config.gutter, // px
            columns: this.config.columns,
            callbacks: {
                reordering: function (objs) {
                    that.lastIDS = that.getIDS(objs);

                },
                reordered: function (objs, dragged) {

                    that.justDragged = false;
                    //如果有拖动，那么dragged 记录了当前被拖动的对象，否则就是undefined
                    if (dragged == undefined) return;

                    //如果点击 uploadNewImage，也会进入到这里来，那么它通常不会拖动，就视为点击
                    //必须这么做，因为下面把它删除，又加上，然后原来的点击事件在删除对象时也失效了， 这样它永远无法触发点击事件

                    if (dragged.find('.uploadNewImage').length > 0)
                    {

                        return;
                    }

                    that.justDragged = true;

                    let ids = that.getIDS(objs);

                    //可能新增上传的按钮被拖到中间去了， 把它删除一下，重新再加上
                    if (that.removeNewButton())
                    {
                        that.insertNewButton();
                        gridly.gridly('layout');
                    }


                    setTimeout(function () {
                        //console.info("reorder index ");
                        console.info("imageOrderChanged" );
                        console.dir( ids);
                        that.pSheet.Book.EM.fire("attachmentOrderChanged", [that.pSheet, that.config, ids]);
                        that.pSheet.Book.EM.fire("imageOrderChanged", [that.pSheet, that.Name, ids]);

                        if (that.config.showImageIndexNo)
                        {
                            var imgs = that.getCurrentSortedImageList();
                            for (let i = 0; i < imgs.length; i++)
                            {
                                $('#' + imgs[i].id).parent().find('.imageIndex').html(i + 1);
                            }
                        }
                    }, 200);

                }
            }
        });


        this.autoSetRowHeight();
        this.showOrHideWithWorkSheet();//不加可能会隐藏起来，原因不想找了
        this.justBuilding = false;
        this.pSheet.Book.EM.fire("imageGridlyInited", [this.pSheet, this.Name, this.data]);
        if (callback != null) callback();
    },

    rebuildGridly: function () {
        if (this.justBuilding) return;

        //console.info(" rebuild gridly");
        let imgs = this.getCurrentSortedImageList();


        this.buildGridly(imgs);
    },

    addToolButton: function (pos, icon, title, color, func) {
        let gridly = $(`#${this.gridlyID}`);
        let bricks = gridly.find('.gridly_brick').not(".uploadNewImageContainer");
        let html = `<div class="toolButton${pos}" >
                <span class="fa-stack fa-lg">
                 
                    <i class="fa fa-circle fa-stack-1x" style="color:${color};"></i>
                    <i class="fa ${icon}  fa-stack-1x" style="color:white;font-size:0.5em"  title="${title}"></i>
                   
                 </span>
                </div>
                `;
        bricks.append(html);

        bricks.find(`.toolButton${pos}`).on("click", function (e) {
            let id = $(e.currentTarget.parentElement).find(".brickImage").attr("id")
            if (func) func(id);
        });


    },

    /**
     * api  移除指定ID的图片，仅仅是界面上移除，并不做更多
     * @param id
     */
    removeItem: function (id) {
        let gridly = $(`#${this.gridlyID}`);
        let img = gridly.find('#' + id);
        let cb = img.parent();

        cb.remove();//把自已移除
        //2021.06.09 把 data 中的数据也删除
        for (let i = 0; i < this.data.length; i++)
        {
            if (this.data[i].id == id)
            {
                this.data.splice(i, 1); //把i位置的数据清除
                break;
            }
        }
    },


    /**
     * 弹出显示指定id的图片，并可以前后翻看
     * @param id
     */
    popupShow:function( id)
    {
        let that = this;
        let gridly = $(`#${this.gridlyID}`);

        let imgs = that.getCurrentSortedImageList();
        let pn = 0;  //当前点击的是第几张图片
        let items = [];
        for (let i = 0; i < imgs.length; i++)
        {
            if (imgs[i].id ==id)
            {
                pn = i;
            }
            items.push(imgs[i].url_img);
        }
        //整理成图片浏览所需要的数据格式
        let pb = $.photoBrowser({
            items: items,
            initIndex: pn,
            onOpen: function () {
                //绝对置顶，所有弹窗都显示都不能阻止
                $('.weui-photo-browser-modal').css("z-index", 99999990);
                $('.weui-photo-browser-modal').on("contextmenu", function(){return false; });
                //使用的swipe 是一个修改版 ，其中的  keyboardControl我改成 true, mousewheelControl改成 true,后，并没有生效
                // 原因不明，再者这个swipe.js 是一个被修改过的版本，原作者增加了 Photos 这个类，我也不想去改它了
                // 在此处 hack一下， 增加对weui-photo-browser-modal 的keydown处理也能实现按键翻页的功能，那就先这么用了
                // 有功夫或有需要时，再替换成 swipe.js的 3.4版 ，此版本，上述两个参数实测都是有效的


                $('.weui-photo-browser-modal').attr("tabIndex", 9);

                $('.weui-photo-browser-modal').on("keydown", function (e) {
                    // console.info(e.keyCode);
                    if ([VK_LEFT, VK_UP].contains(e.keyCode))
                    {
                        pb.slidePrev();
                    }
                    if ([VK_RIGHT, VK_DOWN, VK_SPACE].contains(e.keyCode))
                    {
                        pb.slideNext();
                    }
                    if ([VK_ESC].contains(e.keyCode))
                    {
                        pb.close();
                    }


                });

                $('.weui-photo-browser-modal').on("mousewheel", function (e) {

                    if( e.shiftKey)
                    {
                        var scale= pb.currentScale;
                        if (e.deltaX < 0 || e.deltaY<0 ) scale+=0.5;
                        if (e.deltaX > 0 || e.deltaY>0 ) scale-=0.5;
                        if( scale<0) scale=0.5;
                        pb.scale(scale);
                        return;
                    }



                    {
                        if (e.deltaY < 0) pb.slideNext();
                        if (e.deltaY > 0) pb.slidePrev();
                    }

                });

/*
                $('.weui-photo-browser-modal').append(`<input id="imageZoom" type="range" min="100" max="500" value="100" step="2" 
                    style=" position:absolute; top:100px; right:100px; width:200px;height:30px; z-index:9999999 ; pointer-events:auto; ">
                        `);

                $('.weui-photo-browser-modal #imageZoom').on('mouseup', function(evt)
                {
                    var v= $('.weui-photo-browser-modal #imageZoom').val();
                    console.info(v);
                    pb.scale(v/100);

                    evt.preventDefault();
                    evt.stopPropagation();// 其它浏览器下阻止冒泡


                });
*/

/*
                $('.weui-photo-browser-modal').append(`<div id="imageZoom" type="range" min="100" max="500" value="100" step="2" 
                    style=" position:absolute; top:100px; right:100px; width:200px;height:20px; z-index:9999999 ; pointer-events:auto; ">
                    <div id="draggable"><i class="fa fa-circle"></i></div></div>
                        `);

                $( "#draggable" ).draggable({ containment: "parent" ,axis: "x",
                    drag: function( event, ui ) {  console.info(JSON.stringify(ui)); }});
*/

       /*        $('.weui-photo-browser-modal').append(`<input id="imageZoom" type="button"
                    style=" position:absolute; top:100px; right:100px; width:20px;height:20px; z-index:99999991 ; "   ">
                        `);

                $('#imageZoom').on('mousedown',   function(evt)
                {
                    console.info("=====");
                    var scale= pb.currentScale;
                    scale+=0.5;

                    pb.scale(scale);

                    evt.preventDefault();
                    evt.stopPropagation();// 其它浏览器下阻止冒泡


                });
             */
                var watermark = that.pSheet.Book.EM.fire("imageOnPopupWithWatermark", [that.pSheet, that.Name],'' ) ||'';
                if( watermark!='')
                {
                    //2021.11.07 增加一个蒙层，显示水印
                    $('.weui-photo-browser-modal').find('.photo-container').append(
                        `
                            <div style="position:absolute;left:0, top:0;height:100%; width:100%; 
                               background-image:url(${watermark}) ;
                               background-repeat: repeat;
                               pointer-events:none ; z-index:999999 "
                               oncontextmenu=" return false;"
                               ></div>
                            
                            `
                    );
                }


                $('.weui-photo-browser-modal')[0].focus();

            },
            onClose: function () {
                //console.log("photoBrowser onClose");
                that.photoBrowser = null;
                setTimeout(function () {
                    $('.weui-photo-browser-modal').remove();
                }, 200);
            }
        });
        that.photoBrowser = pb; //记下来 ，便于脚本中控制
        pb.open();
    },

    /**
     * 把  data 数据 生成一个图片占位符，显示缩略图，并根据  action 控制删除
     * @param data
     * @param replaceDom
     */
    addImage: function (data, replaceDom) {
        let that = this;
        let gridly = $(`#${this.gridlyID}`);
        let html = `
                    <div class="brickInner  brickImage" id="${data.id}">
                        <img  class="image" src="${data.url_thumbnail}"  largeImage="${data.url_img}" " >
                    </div>
                    
                    <div class="imageIndex"></div>
                    `;

        if (this.getAction() != 'view' && !this.config.deleteButtonDisabled)
        {
            html += ` <div class="deleteButton" >
                        <span class="fa-stack fa-lg">
                            <i class="fa fa-circle fa-stack-1x" style="color:white;"></i>
                            <i class="fa fa-times-circle fa-stack-1x" style="color:red;"></i>
                        </span>
                      </div>
                    `;
        }


        let cb = replaceDom;
        if (replaceDom == undefined) //如果没有这个参数，那么就是新加一个图片缩略图，如果有提供此参数，就是在此位置替换原内容
        {
            cb = $(` <div class="gridly_brick">${html}</div>`);

            gridly.append(cb);

        } else
        {
            cb.html(html);  //替换内容
        }


        //注册事件
        cb.find('.brickImage').on('dblclick', function (e) {

            //console.info("dblclick");
            //2021.05 触发事件
            that.pSheet.Book.EM.fire("imageOnDBLClick", [that.pSheet, that.Name, data, e]);

        });

        //注意： click这种事件， 可以冒泡穿透， 但是 mouseenter,mouseout ，请注意，需要放在最上层对象里处理。
        // 不然，会出现 ，多次 enter, out . 比如“回“字形对象层次， 如果在外围对象上处理mouseEnter，那么当鼠标进入内口时， 就会触发 mouseout
        //再回到外口时，又触发 mouseenter，出去时，又触发mouseout，就混乱了 所以必须在内口上处理
        // 所以 把  mouseenter ,mouseout 放在 image 图片上处理，而不是 brickImage这个DIV上。
        cb.find('.image').on('mouseenter', function (e) {
            // console.info("mouseenter");
            //console.dir( e);
            that.currentImageData = data;
            //2021.05 触发事件
            that.pSheet.Book.EM.fire("imageOnMouseEnter", [that.pSheet, that.Name, data, e]);

        });

        cb.find('.image').on('mouseout', function (e) {
            //console.info("mouseout");
            that.currentImageData = null;
            //2021.05 触发事件
            that.pSheet.Book.EM.fire("imageOnMouseOut", [that.pSheet, that.Name, data, e]);

        });

        cb.find('.image').on('mousedown', function (e) {
            //console.info("mousedown");
            //2021.05 触发事件
            if (e.which == 3)
            {
                //console.info("右键按下");
                if (that.config.rightMouseDownToViewLargeImage)
                {
                    that.fullScreenViewImageOpen(data);
                }
            } else
            { //非右键点下， 都隐藏单独的大图浏览
                if (that.config.rightMouseDownToViewLargeImage)
                {
                    that.fullScreenViewImageClose();
                }
            }

            that.pSheet.Book.EM.fire("imageOnMouseDown", [that.pSheet, that.Name, data, e]);

        });

        cb.find('.image').on('mouseup', function (e) {
            //console.info("mouseup");
            //2021.05 触发事件


            if (that.config.rightMouseDownToViewLargeImage)
            {
                that.fullScreenViewImageClose();
            }

            that.pSheet.Book.EM.fire("imageOnMouseUp", [that.pSheet, that.Name, data, e]);

        });

        cb.find('.image').on('contextmenu', function (e) {
            e.preventDefault();


        });


        //2021.05.20增强单击动作：
        //如果是无选中标记状态，那么单击触发浏览大图操作
        // 如果是单选模式，那么取消所选，只选当前
        // 如果是多选模式，如果是按下了Shift则连续选中或连续取消选中。 如果没有按下Shift则是反选当前图片
        cb.find('.brickImage').on('click', function (e) {


            if (that.justDragged) return; //说明这是一个拖拽操作，不要触发click
            //

            //2021.05 触发事件
            that.pSheet.Book.EM.fire("imageOnClick", [that.pSheet, that.Name, data]);

            if (that.m_selectionMode == 'single')
            {
                gridly.find('.brickSelected').remove();
                that.$setSelected(cb, true);
            }

            if (that.m_selectionMode == 'multi')
            {
                if (cb.find('.brickSelected').length > 0) //如果已经选中了
                {
                    if (e.shiftKey) //如果按下了Shiftkey，那么把从自已开始的倒着向前循环，如果前一个也是选中状态，那么把前一个也取消选中，并继续循环，否则终止
                    {
                        //如果按下了ShiftKey，那么把此前选中的一直取消选中，直到碰到一个没有选中的为止
                        let imgs = that.getCurrentSortedImageList();
                        let pn = 0;  //当前点击的是第几张图片
                        let items = [];
                        for (let i = 0; i < imgs.length; i++)
                        {
                            if (imgs[i].id == data.id)
                            {
                                pn = i;
                                break;
                            }
                        }
                        //从自已开始 ，倒着向前找，把没有选中的都选中，直到碰到一个已经选中的为止
                        for (let i = pn; i >= 0; i--)
                        {
                            let tcb = $(gridly.find('#' + imgs[i].id).parent());
                            if (tcb.find('.brickSelected').length == 0) break;
                            that.$setSelected(tcb, false);
                        }

                    } else
                    {
                        that.$setSelected(cb, false);
                    }
                } else //如果当前没有选中，那么就选中
                {
                    if (e.shiftKey)
                    {
                        //如果按下了ShiftKey，那么把此前没有选中的一直选中，直到碰到一个选中的为止
                        let imgs = that.getCurrentSortedImageList();
                        let pn = 0;  //当前点击的是第几张图片
                        let items = [];
                        for (let i = 0; i < imgs.length; i++)
                        {
                            if (imgs[i].id == data.id)
                            {
                                pn = i;
                                break;
                            }
                        }
                        //从自已开始 ，倒着向前找，把没有选中的都选中，直到碰到一个已经选中的为止
                        for (let i = pn; i >= 0; i--)
                        {

                            let tcb = $(gridly.find('#' + imgs[i].id).parent());
                            if (tcb.find('.brickSelected').length > 0) break;
                            that.$setSelected(tcb, true);

                        }
                    } else
                    {
                        that.$setSelected(cb, true);
                    }
                }
            }

            if (that.m_selectionMode == 'none') //不能选中模式下，点击它就是大图显示
            {
                 that.popupShow( data.id);
            }


        });

        cb.find('.deleteButton').on('click', function () {

            //2021.05 允许定制删除提示
            $confirm("提示", that.config.promptInfoOnDelete || "确信要删除吗?", function () {

                //2021.05 触发事件 ,可以中断
                var continueDelete = that.pSheet.Book.EM.fire("imageOnDelete", [that.pSheet, that.Name, data], true);
                if (continueDelete == null) continueDelete = {success: true};

                if (continueDelete.success)
                {

                    cb.remove();//把自已移除
                    //2021.06.09 把 data 中的数据也删除
                    for (let i = 0; i < that.data.length; i++)
                    {
                        if (that.data[i].id == data.id)
                        {
                            that.data.splice(i, 1); //把i位置的数据清除
                            break;
                        }
                    }

                    //删除图片后，如果此时没有上传新图片按钮，那么重新把它另上来
                    //之所以没有上传新图片捥，是当前图片数量已经到达最大数量
                    let cb_new = $(`#${that.gridlyID} .gridly_brick .uploadNewImage`);
                    if (cb_new.length == 0 && !that.config.uploadButtonDisabled) that.insertNewButton();

                    gridly.gridly('layout');
                } else
                {
                    if (continueDelete.message != null) toastr.error(continueDelete.message);
                }

            });
        });

        return cb;
    }
    ,

    //全屏显示大图
    fullScreenViewImageOpen: function (data) {
        if (this.fullScreenDiv == null)
        {
            this.fullScreenDiv = $(`<div style="position:absolute;left:0;top:0;width:100%;height:100%;z-index:9999;background-color:#000000b0; 
            pointer-events: none; 
            display:table-cell;
            text-align:center;
            vertical-align:middle;"><img class="fullScreenImage" src="" style="display: inline-block; pointer-events: none; height:100%;" ></img></div>`);
            $(document.body).append(this.fullScreenDiv);
        }


        let imgContainer = this.fullScreenDiv.find('.fullScreenImage');

        //临时计算一下比例
        let h = this.fullScreenDiv.height();
        let w = this.fullScreenDiv.width();
        let delta = h * 1.0 / w;

        let img = $('#' + data.id).find('.image');
        let h2 = img.height();
        let w2 = img.width();
        let delta2 = h2 * 1.0 / w2;

        if (delta2 > delta)  // 图片的高宽比 比显示区域的高宽比要大，说明需要适应高度
        {
            imgContainer.css({"height": "100%", "width": "auto"});
        } else
        {
            imgContainer.css({"height": "auto", "width": "100%"});
        }

        imgContainer.attr("src", data.url_img);
        this.fullScreenDiv.css("display", "");
    },

    fullScreenViewImageClose: function () {
        if (this.fullScreenDiv != null)
        {
            this.fullScreenDiv.css("display", "none");
        }
    },
    removeNewButton: function () {
        //注意，是要把它的外围容器一并删除，不然会留一个空白
        let cb_new = $(`#${this.gridlyID} .gridly_brick .uploadNewImage`);
        if (cb_new.length == 0) return false;
        $(cb_new[0].parentNode).remove();
        return true;
    }

    ,

    /**
     * 插入一个 +  上传占位符
     */
    insertNewButton: function () {

        if (this.config.uploadButtonDisabled) return;

        let brickHTML = `
                        <div class="gridly_brick uploadNewImageContainer">
                            <div class="brickInner  uploadNewImage" >
                                <i class="fa fa-plus"></i>
                            </div>
                        </div>
            
            `;

        let gridly = $(`#${this.gridlyID}`);

        gridly.append($(brickHTML));

        let that = this;
        let cb_new = $(`#${this.gridlyID} .gridly_brick .uploadNewImage`);

        if (!that.pSheet.designMode)
        {
            cb_new.on('click', function () {

                //2021.05 触发事件 ,可以中断上传
                var continueUpload = that.pSheet.Book.EM.fire("imageOnNewUpload", [that.pSheet, that.Name], true);
                if (continueUpload == null) continueUpload = {success: true};
                if (!continueUpload.success)
                {
                    if (continueUpload.message) toastr.error(continueUpload.message);
                    return;
                }

                that.uploadNewImage();
            });
        }

    }
    ,

    //
    removeNullButton: function () {
        //注意，是要把它的外围容器一并删除，不然会留一个空白
        let cb = $($(`#${this.gridlyID} .gridly_brick .nullImage`)[0].parentNode);
        cb.remove();
    }
    ,

    /**
     * 插入一个无
     */
    insertNullButton: function () {
        let brickHTML = `
                        <div class="gridly_brick">
                            <div class="brickInner nullImage" >
                                无
                            </div>
                        </div>
            
            `;

        let gridly = $(`#${this.gridlyID}`);

        gridly.append($(brickHTML));

    }
    ,

    /**
     * 自动调整容器单元格的高度
     */
    autoSetRowHeight: function () {

        let that = this;
        let gridly = $(`#${this.gridlyID}`);
        gridly.gridly('layout');

        let thisH = gridly.height();
        if (thisH == 0) thisH = this.visualBlockSize + parseInt(this.config.gutter);

        let h = this.pCell.getBounds().height;

        if (h != thisH)
        {

            //2021.05.21 Bug修正 ： 下面pSheet.setRowHeight调整行高，会触发 相应的View重新计算卷滚条高度
            //如果pSheet是嵌在另一个Sheet中，也会重新计算 SheetContainer的卷滚条
            // 而之前是用  pSheet.RPC.setRowHeight 只能调整行高，不能让View重新计算卷滚条高度
            if (this.pCell.isMerged())
            {
                if (this.pCell.getMergedRange().endRow > this.pCell.rowIndex)
                {
                    if (h < thisH) //合并过，那么仅在区域太小时把它调大 ，只需要把
                    {
                        setTimeout(function () {
                            let h0 = that.pSheet.RPM.getRowHeight(that.pCell.rowIndex);
                            that.pSheet.setRowHeight(that.pCell.rowIndex, h0 + thisH - h);
                        }, 10);
                    } else
                    {
                        console.info("重要提示：如果UploadBrick有在行上合并，那么当它的高度撑大后，不会回调。要避免这种情况，请不要在行上有合并");
                    }
                } else
                {  //如果行上没有合并，那么没问题， 继续
                    setTimeout(function () {
                        that.pSheet.setRowHeight(that.pCell.rowIndex, thisH);
                    }, 10);
                }
            } else
            {

                setTimeout(function () {
                    that.pSheet.setRowHeight(that.pCell.rowIndex, thisH);
                }, 10);
            }
        }

        this.showOrHideWithWorkSheet();
        this.invalidate();

    }
    ,

    /**
     * 上传图片文件
     * @param button
     */
    uploadNewImage: function () {
        let that = this;
        let gridly = $(`#${this.gridlyID}`);

        //临时创建一个 file 选择对象
        let selectFile = $(`<input type="file"  
                            style="display:none;"  
                            multiple="multiple"  
                            accept="image/png,image/jpg,image/gif,image/JPEG" />
                        `);
        //必须加入文档中，不然不能触发事件
        selectFile.appendTo($(document.body));

        //模拟点击
        selectFile.click();
        //确认选择文件后，触发上传操作
        selectFile.unbind().change(function (e) {
            that.loadImg(e, selectFile);
        });

    }
    ,

    /**
     *
     * @param e
     * @param selectFile
     */
    loadImg: function (e, selectFile) {
        let that = this;
        let gridly = $(`#${this.gridlyID}`);
        let filePath = selectFile.val();
        let fileFormat = filePath.substring(filePath.lastIndexOf(".")).toLowerCase();
        //检查后缀名
        if (!fileFormat.match(/.png|.gif|.jpg|.jpeg/))
        {
            showError('文件格式必须为：png/jpg/jpeg');
            selectFile.remove(); //删除
            return;
        }
        //获取并记录图片的base64编码


        let files = e.target.files;
        for (let i = 0; i < files.length; i++)
        {

            ImageUtil.photoCompress(files[i], {size: that.config.blockSize}, function (base64) {

                //console.dir(  files[i]);

                // 图片的 base64 格式, 可以直接当成 img 的 src 属性值
                let dataURL = base64;

                that.removeNewButton();

                // 显示图片

                let uploadingCB = $(` 
                       <div class="gridly_brick">
                            <div class="brickInner  brickImage">
                               <img  src='${dataURL}'  onclick="imgDisplay(this)">
                               <div class="imageUploading"><i class="fa fa-circle-o-notch fa-spin fa-2x fa-fw"></i></div>
                               <div class="uploadingProgress">1</div>
                             </div>
         
                       </div>
                         `);

                gridly.append(uploadingCB);

                //如果已经存在的图片数量没有超过允许上传的数量
                let imageCount = gridly.find('.gridly_brick .brickImage').length;
                if (imageCount < that.config.maxCount)
                {

                    that.insertNewButton();
                }

                that.autoSetRowHeight();
                let progressInfo = uploadingCB.find(".uploadingProgress");

                //缩略图显示，已经ok了，下面开始上传

                let url = that.config.homeURL;

                that.config.fileIndex = imageCount; //图片序号
                let imgMaxSize = that.config.imgMaxSize;
                if (imgMaxSize == undefined) imgMaxSize = 1920;

                if (!url.endsWith("/") && url != '') url = url + "/";
                ImageUtil.uploadFile(url + "UploadFile", files[i],
                    {size: imgMaxSize},  // 压缩选项
                    that.parseConfig(), //直接用这个配置做上传选项，虽然可能属性多了一些，不影响上传
                    {
                        onloadstart: function (evt) {

                            //它是一个属性，不要用 ++;
                            that.pSheet.Book.uploadingCount = that.pSheet.Book.uploadingCount + 1;
                            console.info("=====开始上传");
                        },
                        onprogress: function (evt) {
                            console.dir(evt);
                            let percent = Math.round(evt.loaded / evt.total * 100);
                            //到100%后，还会等一会儿才触发上什完成，所以这里不能到100%
                            progressInfo.html('' + (percent - 1));
                            console.info("=====上传进度 " + Math.round(evt.loaded / evt.total * 100) + "%");
                        },
                        onerror: function (evt) {
                            console.dir(evt);
                            selectFile.remove(); //删除临时的选择文件的input
                            that._uploadSuccess(uploadDemoData, uploadingCB);
                        },

                        onComplete: function (evt) {
                            //  event.total是需要传输的总字节，
                            // event.loaded是已经传输的字节。
                            // 如果event.lengthComputable不为真，则event.total等于0

                            progressInfo.html('100');
                            console.info("=====上传完成");
                            //它是一个属性，不要用 --;
                            that.pSheet.Book.uploadingCount = that.pSheet.Book.uploadingCount - 1;

                            /*
                            let data = evt.target.responseText;
                            data = JSON.parse(data);
*/
                            let data = evt.data;
                            // console.dir(data);

                            selectFile.remove(); //删除临时的选择文件的input
                            //uploadingCB的内容会被真正的图片替换。
                            that._uploadSuccess(data, uploadingCB);
                        }

                    }
                )// end of uploadFile


            });// end of photoCompress


        }//end of for loop
    }
    ,

    _uploadSuccess: function (data, replaceDom) {
        //把显示上传进度的图片替换成上传结果

        //2021.12.10 上传的结果要加入到data中
        this.data.push( data);

        this.addImage(data, replaceDom);
        //2021.12.07 添加上传完成事件
        this.pSheet.Book.EM.fire("imageOnUploaded", [this.pSheet, this.Name, data]);

    }
    ,


    getIDS: function (bricks) {
        let ids = [];
        for (let i = 0; i < bricks.length; i++)
        {

            let brickImage = $(bricks[i]).find('.brickImage');
            if (brickImage.length > 0) ids.push(brickImage[0].id);
        }
        return ids;
    }
    ,

    /**
     * @api {getDataById} 函数   getDataById
     *
     * @apiDescription  getDataById(id)
     * <br><br> 根据图片ID得到图片的详细数据
     *
     * @apiName  getDataById
     * @apiGroup UploadBrick
     * @apiVersion 1.0.0
     *
     *
     *

     * @apiSuccess (返回值){Object} - 返回图片详细信息
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     *  let brick= book.sheet0.getBrick('Upload1');
     *  brick.getDataById('adaefadfaea');
     *
     */
    getDataById: function (id) {
        for (let i = 0; i < this.data.length; i++)
        {
            if (this.data[i].id == id) return this.data[i];
        }
        return null;
    }
    ,

    /**
     * gridly没有提供获取排序后的对象清单函数，一个办法是在初始化时记录初始状态，在排序后的回调中保存数据
     * 另一个办法就是自已解决：获取所有块，根据坐标计算它们的顺序，下面就是这个思路
     * @returns {Array}
     */
    /**
     * @api {getCurrentSortedImageList} 函数   getCurrentSortedImageList
     *
     * @apiDescription  getCurrentSortedImageList()
     * <br><br> 得到当前顺序的图片数据
     *
     * @apiName  getCurrentSortedImageList
     * @apiGroup UploadBrick
     * @apiVersion 1.0.0
     *
     *
     *

     * @apiSuccess (返回值){Array} - 返回图片信息数组
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     *  let brick= book.sheet0.getBrick('Upload1');
     *  brick.getCurrentSortedImageList();
     *
     */
    getCurrentSortedImageList: function () {


        let bricks = [];

        let brickImages = $(`#${this.gridlyID} .brickImage`);//获取所有划块


        //把它们的坐标获取，然后通过坐标重新对它们进行排序，再取排序后的结果
        for (let i = 0; i < brickImages.length; i++)
        {
            let img = $(brickImages[i]).find('.image')[0];
            let id = brickImages[i].id;


            let one = this.getDataById(id);
            one.x = $(brickImages[i]).offset().left,
                one.y = $(brickImages[i]).offset().top
            bricks.push(one);

        }
        bricks = bricks.sort(compare);

        return bricks;
    }
    ,

    /**
     * @api {getSelectedImageList} 函数   getSelectedImageList
     *
     * @apiDescription  getSelectedImageList()
     * <br><br> 得到当前顺序的具有选中标记的图片数据
     *
     * @apiName  getSelectedImageList
     * @apiGroup UploadBrick
     * @apiVersion 1.0.0
     *
     *
     *

     * @apiSuccess (返回值){Array} - 返回图片信息数组
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     *  let brick= book.sheet0.getBrick('Upload1');
     *  brick.getSelectedImageList();
     *
     */
    getSelectedImageList: function () {
        if (this.m_selectionMode == 'none') return [];

        let gridly = $(`#${this.gridlyID}`);
        let ret = [];
        let imgs = this.getCurrentSortedImageList();
        //从自已开始 ，倒着向前找，把没有选中的都选中，直到碰到一个已经选中的为止
        for (let i = 0; i < imgs.length; i++)
        {

            let tcb = $(gridly.find('#' + imgs[i].id).parent());
            if (tcb.find('.brickSelected').length == 0) continue;
            ret.push(imgs[i]);

        }
        return ret;
    }
    ,

    /**
     * @api {deleteSelectedImages} 函数   deleteSelectedImages
     *
     * @apiDescription  deleteSelectedImages()
     * <br><br> 删除具有选中标记的图片
     *
     * @apiName  deleteSelectedImages
     * @apiGroup UploadBrick
     * @apiVersion 1.0.0
     *
     *
     *

     * @apiSuccess (返回值){void} - 无返回数组
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     *  let brick= book.sheet0.getBrick('Upload1');
     *  brick.deleteSelectedImages();
     *
     */
    deleteSelectedImages: function () {
        let gridly = $(`#${this.gridlyID}`);
        let imgs = this.getCurrentSortedImageList();
        //从自已开始 ，倒着向前找，把没有选中的都选中，直到碰到一个已经选中的为止
        for (let i = 0; i < imgs.length; i++)
        {

            let tcb = $(gridly.find('#' + imgs[i].id).parent());
            if (tcb.find('.brickSelected').length == 0) continue;
            tcb.remove();

        }


        gridly.gridly('layout');

    }
    ,

    //为图片块增加一个远中标记或取消选中标记
    $setSelected: function (tcb, selected) {
        let that = this;
        if (selected)
        {
            tcb.append(`<img class="brickSelected" src="js/spreadsheet/img/gou5.png">`);
            tcb.find('.brickSelected').on("click", function (e) //点击这个勾选图标就取消选择
            {
                that.$setSelected(tcb, false);
            }.bind(this));
        } else
        {
            let t = tcb.find('.brickSelected');
            if (t) t.remove();
        }
    }
    ,

    /**
     * @api {setImageSelected} 函数   setImageSelected
     *
     * @apiDescription  setImageSelected(id, selected)
     * <br><br> 设置指定ID相关的图片选中或不选中
     *
     * @apiName  setImageSelected
     * @apiGroup UploadBrick
     * @apiVersion 1.0.0
     *
     *
     * @apiParam {String}  id 图片块的ID
     * @apiParam {boolean} selected  true表示需要选中， false表示取消选中

     * @apiSuccess (返回值){void} - 无返回数组
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     *  let brick= book.sheet0.getBrick('Upload1');
     *  brick.setImageSelected( 'xaerafag4eerqa' , true);
     *
     */
    setImageSelected: function (id, selected) {
        let gridly = $(`#${this.gridlyID}`);
        let tcb = $(gridly.find('#' + id).parent());
        this.$setSelected(tcb, selected);
    }
    ,

    /**
     * @api {isImageSelected} 函数   isImageSelected
     *
     * @apiDescription  isImageSelected(id)
     * <br><br> 指定ID相关的图片是否被选中
     *
     * @apiName  isImageSelected
     * @apiGroup UploadBrick
     * @apiVersion 1.0.0
     *
     *
     * @apiParam {String}  id 图片块的ID

     * @apiSuccess (返回值){boolean} -  true表示被选中， false表示没有选中
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     *  let brick= book.sheet0.getBrick('Upload1');
     *  let b = brick.isImageSelected( 'xaerafag4eerqa');
     *
     */

    isImageSelected:function(id)
    {
        let gridly = $(`#${this.gridlyID}`);
        let tcb = $(gridly.find('#' + id).parent());
        let t = tcb.find('.brickSelected');
        if( t.length==0) return false;
        return true;
    },

    /**
     * @api {selectAll} 函数   selectAll
     *
     * @apiDescription  selectAll()
     * <br><br> 将控件中的所有图片都打上选中标记
     *
     * @apiName  selectAll
     * @apiGroup UploadBrick
     * @apiVersion 1.0.0
     *
     *

     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     *  let brick= book.sheet0.getBrick('Upload1');
     *  brick.selectAll();
     *
     */
    selectAll: function () {
        let gridly = $(`#${this.gridlyID}`);
        let brickImages = $(`#${this.gridlyID} .brickImage`);//获取所有划块
        //把它们的坐标获取，然后通过坐标重新对它们进行排序，再取排序后的结果
        for (let i = 0; i < brickImages.length; i++)
        {
            let tcb = $(brickImages[i]).parent();
            this.$setSelected(tcb, true);
        }
    }
    ,

    /**
     * @api {unSelectAll} 函数   unSelectAll
     *
     * @apiDescription  unSelectAll()
     * <br><br> 将控件中的所有图片都取消选中标记
     *
     * @apiName  unSelectAll
     * @apiGroup UploadBrick
     * @apiVersion 1.0.0
     *
     *

     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     *  let brick= book.sheet0.getBrick('Upload1');
     *  brick.unSelectAll();
     *
     */
    unSelectAll: function () {
        let gridly = $(`#${this.gridlyID}`);
        let brickImages = $(`#${this.gridlyID} .brickImage`);//获取所有划块
        //把它们的坐标获取，然后通过坐标重新对它们进行排序，再取排序后的结果
        for (let i = 0; i < brickImages.length; i++)
        {
            let tcb = $(brickImages[i]).parent();
            this.$setSelected(tcb, false);
        }
    }
    ,


    //以下是Brick的功能函数
    setVisible: function (v) {
        try
        {
            if (v)
            {
                this.dom.css("display", "");
            } else
            {
                this.dom.css("display", "none");
            }
        } catch (err)
        {
            let that = this;
            setTimeout(function () {
                that.setVisible(v);
            }, 200);
        }
    }
    ,

    showOrHideWithWorkSheet: function () {

        console.info("调整可见性");
        if (this.pSheet.visible && this.visible)
        {
            this.showOrHide(true);
        } else
        {
            this.showOrHide(false);
        }

    }


});


export default UploadBrick;
