/**
 * Created by zengjun on 2017/4
 */


import Class from '../base/Class.js';
import DataBuffer from "./DataBuffer.js";
import EventManager from '../core/EventManage.js';

import UniformDataType from '../core/UniformDataType.js';
import ColumnProperty from "./ColumnProperty.js";
import ObjectType from "./ObjectType.js";
import AggregateDefine from "./AggregateDefine.js";
import AggregateType from "./AggregateType.js";
import DsRow from "./DsRow.js";
import ItemStatus from "./ItemStatus.js";
import AutoCheckType from "./AutoCheckType.js";
import AutoCheckConfig from "./AutoCheckConfig.js";
import DataProviderDefault from './DataProviderDefault.js';
import Util from '../util/Util.js';
import SqlParse from './SqlParse.js';
import DsItem from "./DsItem.js";
import Esprima from '../base/esprima.js';
import ObjectTool from "./ObjectTool.js";
import Tools from '../util/Tools.js';

import Dialect from './Dialect.js';
import DialectMySQL from './DialectMySQL.js';
import DialectSQLServer from './DialectSQLServer.js';
import DialectH2  from'./DialectH2.js';
import DialectOracle from './DialectOracle.js';
import DialectPostgres from './DialectPostgres.js';


const WITHLASTSELECT = "@WITH-LAST-SELECT@"; // 用来配合实现分页检索

//日志输出
var Logger = {
    info: function (s) {
        console.log(s);
    }
};

var  DataStore = Class.extend({

    static: {
        WITHLASTSELECT: "@WITH-LAST-SELECT@",  //用来配合实现分页检索

    },

    properties: {

        /**
         * @api {rowCount} 只读属性 [属性]rowCount
         * @apiName  rowCount
         * @apiDescription 结果集的行数
         * @apiGroup DataStore
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){int} - 结果集的行数
         * @apiExample {js}(示例：)
         * var ds=newDataStore('','select * from app_config');
         * ds.retrieve();
         * alert( ds.rowCount);
         */

        "rowCount": {
            get: function () { return this.getRowCount();}
        },


        /**
         * @api {DBRowCount} 只读属性 [属性]DBRowCount
         * @apiName  DBRowCount
         * @apiDescription 当分页检索时，rowCount属性是当前结果集检索出的条数，而本属性则是表示结果集最终能检索出的条数
         * @apiGroup DataStore
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){int} - 结果集可以检索出的数据条数，
         * @apiExample {js}(示例：)
         *
         * var ds=newDataStore('','select * from app_config');
         * ds.onceRetrieveCount=10;
         * ds.retrieve();
         * alert( ds.rowCount);
         * alert( ds.DBRowCount);
         */
        "DBRowCount":
            {
                get: function () {return this.getDBRowCount();}
            },

        /**
         * @api {columnCount} 只读属性 [属性]columnCount
         * @apiName  columnCount
         * @apiDescription 结果集的列数，包括字段，计算列
         * @apiGroup DataStore
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){int} - 结果集的列数
         * @apiExample {js}(示例：)
         * var ds=newDataStore('','select * from app_config');
         * ds.retrieve();
         * alert( ds.columnCount);
         */
        "columnCount": {
            get: function () { return this.getColumnCount();}
        },


        /**
         * @api {onceRetrieveCount} 可读可写属性 [属性]onceRetrieveCount
         * @apiName  onceRetrieveCount
         * @apiDescription 分页检索时，每页检索的条数
         * @apiGroup DataStore
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){int} - 分页检索时，每页检索的条数
         * @apiExample {js}(示例：)
         * alert( ds.onceRetrieveCount);
         */
        "onceRetrieveCount":
            {
                get: function () {return this.getOnceRetrieveCount();},
                set: function (val) { this.setOnceRetrieveCount(val);}

            },


        /**
         * @api {dataSourceName} 可读可写属性 [属性]dataSourceName
         * @apiName  dataSourceName
         * @apiDescription 结果集名称
         * @apiGroup DataStore
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){String} - 结果集名称
         * @apiExample {js}(示例：)
         * alert( ds.dataSourceName);
         */
        "dataSourceName":
            {
                get: function () {return this.m_DataSourceName;},
                set: function (val) { this.m_DataSourceName=val;}

            },
        /**
         * @api {retrieveContext} 可读可写属性 [属性]retrieveContext
         * @apiName  retrieveContext
         * @apiDescription 检索前，可以设置此属性，该属性会传递到DataProvider实现中，此属性做什么用，由DataProvider决定。
         * 在 reset()方法中，本属性并不会被清空，如果需要清除它，需要显示地调用 ds.retrieveContext=null;
         * @apiGroup DataStore
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){Variant} - 任意类型的数据
         * @apiExample {js}(示例：)
         * //
         */
        "retrieveContext":
            {
                get: function () {return this.m_retrieveContext==null?{}:this.m_retrieveContext  ;},
                set: function (val) { this.m_retrieveContext =val;}

            },


        /**
         * @api {absolute} 可读可写属性 [属性]absolute
         * @apiName  absolute
         * @apiDescription 检索前将光标绝对定位到指定位置
         * @apiGroup DataStore
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){int} - 检索前将光标绝对定位到指定位置
         * @apiExample {js}(示例：)
         //从第101条开始，检索10条数据
         * var ds=newDataStore('','select * from app_config');
         * ds.absolute=31;
         * ds.onceRetrieveCount=10;
         * ds.retrieve();
         */
        'absolute':
            {
                get: function () {return this.getAbsolute();},
                set: function (val) { this.setAbsolute(val);}
            },

        /**
         * @api {timeOut} 可读可写属性 [属性]timeOut
         * @apiName  timeOut
         * @apiDescription 检索数据超时时间.当一个查询在指定的超时间内，数据库服务器没有返回检索结果，则强制终止检索操作
         * @apiGroup DataStore
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){int} - 超时时间，单位为毫秒
         * @apiExample {js}(示例：)
         *
         * var ds=newDataStore('','select * from app_config');
         * ds.timeOut=3000;
         * ds.retrieve();
         * alert( ds.rowCount);
         */
        "timeOut": {
            get: function () { return this.m_timeOut;},
            set: function (val) { this.m_timeOut = val;}
        },

        /**
         * @api {error} 只读属性 [属性]error
         * @apiName  error
         * @apiDescription 异常信息对象
         * @apiGroup DataStore
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){int} code 异常信息代码
         * @apiSuccess (属性类型){String} message 异常信息
         * @apiExample {js}(示例：)
         *
         *  ds.update(true);
         *  alert( ds.error.message);
         */
        "error":
            {
                get: function () {return this.m_Error;}
            },
        /**
         * 事件管理对象
         */

        "EM": {
            get: function () { return this.getEvent();}
        },

        /**
         * 是否可以更新
         */

        /**
         * @api {updatable} 可读可写属性 [属性]updatable
         * @apiName  updatable
         * @apiDescription 结果集是否允许更新
         * @apiGroup DataStore
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){boolean} - 是否可以更新
         * @apiExample {js}(示例：)
         * ds.updateable=true;
         */
        "updatable": {
            get: function () { return this.m_Updatable;},
            set: function (val) { this.m_Updatable = val;}
        },


        /**
         * @api {updatableTable} 可读可写属性 [属性]updatableTable
         * @apiName  updatableTable
         * @apiDescription 可更新的表名称
         * @apiGroup DataStore
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){String} - 可更新的表名称
         * @apiExample {js}(示例：)
         * alert( ds.updatableTable);
         * ds.updatableTable='tableA';
         *
         */
        "updatableTable": {
            get: function () { return this.m_UpdateTable;},
            set: function (val) { this.setUpdatableTable(val);}
        },

        /**
         * @api {primaryKey} 可读可写属性 [属性]primaryKey
         * @apiName  primaryKey
         * @apiDescription 结果集的主键。当结果集同可更新时，它必须要有主键，如果是组合主键，多个列之间用英文逗号分隔
         * @apiGroup DataStore
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){String} - 结果集的主键
         * @apiExample {js}(示例：)
         * alert( ds.primaryKey);
         */
        "primaryKey": {
            get: function () { return this.getPrimaryKey();},
            set: function (val) { this.setPrimaryKey(val);}
        },

        /**
         * @api {updatableColumns} 可读可写属性 [属性]updatableColumns
         * @apiName  updatableColumns
         * @apiDescription 结果集中哪些字段允许更新 ， 字段名用|分隔。通配符*表示所有字段可更新 ，如果字段名前带有~符号，则表示它不
         * 允许更新。 详见示例
         * @apiGroup DataStore
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){String} -  结果集中哪些字段允许更新
         * @apiExample {js}示例1：
         *
         *
         * var d=newDataStore("","select * from t1");
         * ds.updatable=true;
         * ds.updatableTable="t1";
         * ds.primaryKey="id";
         * ds.updatableColumns="id|name|~code|c1|~c2|c3";
         * //上面表示 id, name , c1,c3可以更新 ，code, c2不可以更新
         *
         * @apiExample {js} 示例2：
         *
         *
         * var d=newDataStore("","select * from t1");
         * ds.updatable=true;
         * ds.updatableTable="t1";
         * ds.primaryKey="id";
         * ds.updatableColumns="id";
         * ds.updatableColumns="name";
         * ds.updatableColumns="c1|c3";
         * ds.updatableColumns="~code";
         * ds.updatableColumns="~c2";
         *
         * //上面的设置与示例1效果是一样的。 即该属性是一个追加性属性，而不是覆盖性属性。
         */
        "updatableColumns": {
            get: function () { return this.getUpdatableColumns();},
            set: function (val) { this.setUpdatableColumns(val);}
        },

        /**
         * @api {setValueWithTrigger} 可读可写属性 [属性]setValueWithTrigger
         * @apiName  setValueWithTrigger
         * @apiDescription 通常在修改结果集的值时，会同步触发itemChanged事件，可以使用本属性定义来禁止触发itemChanged事件。
         * 为什么会需要这样的一个开关呢？因为值的修改会同步触发itemChanged事件，如果是在
         * 做大批量的数据修改，每修改一行一列的值都触发事件，效率会很低，可以通过本属性来控制不触发事件。当本属性为
         * false时，相应的计算列也不会被修改，绑定到字段的界面，也不会被刷新重绘。请谨慎修改本属性，并小心它的不良后果。
         * @apiGroup DataStore
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){boolean} - 修改结果集的值时，是否触发itemChanged事件
         * @apiExample {js}(示例：)
         *
         * ds.setValueWithTrigger=false;
         * ds.setValue(0,"id",1);
         * ds.setValue(0,"name","abc");
         * ds.setValueWithTrigger=true;
         * ds.setValue(0,"code","123");
         *
         * 上述脚本执行时，不会触发 id, name的 itemChanged事件，而会触发 code的被改变事件
         */
        "setValueWithTrigger": {
            get: function () { return this.isSetValueWithTrigger();},
            set: function (val) { this.setSetValueWithTrigger(val);}
        },

        /**
         * @api {blobUnRetrieve} 可读可写属性 [属性]blobUnRetrieve
         * @apiName  blobUnRetrieve
         * @apiDescription 大文本字段是否检索。当结果集中有大文本字段（text, clob类型的字段)时，控制它们是否检索。
         * 如果本属性为false,则 retrieve 执行时，它们不会被检索，以减少网络传输，当然这些字段也就没有值。这是为懒人设计
         * 的开关，如果你喜欢用select * from  tableA ，并且不希望其中的大文本字段检索时，本属性就有用了。当然最好是使用
         * select col1, col2... from tableA 指明字段的方式来避免检索大文本。
         * @apiGroup DataStore
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){boolean} - 是否检索结果集中的大文本字段
         * @apiExample {js}(示例：)
         *
         * var ds=newDataStore("","select * from t1");
         * ds.blobUnRetrieve=true;
         * ds.retrieve(); //如果t1中有大文本，则它们不会检索出来。
         */
        "blobUnRetrieve": {
            get: function () { return this.isBlobUnRetrieve();},
            set: function (val) { this.setBlobUnRetrieve(val);}
        },

        /**
         * @api {currentFocusRow} 可读可写属性 [属性]currentFocusRow
         * @apiName  currentFocusRow
         * @apiDescription 当结果集是多行结果集时，并且它绑定到了spreadsheet上，可以用鼠标点击其中的具体行，此时被点中的
         * 行就是当前焦点行，当前焦点行的背景会亮显，以示突出。脱离了界面绑定，本属性就没有意义。
         * @apiGroup DataStore
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){int} - 当前焦点行
         * @apiExample {js}(示例：)
         * alert( ds.currentFocusRow);
         * //它等同于下面的语句
         * book.getDataSource('someDataSource').currentBindRow;
         */
        "currentFocusRow": {
            get: function () { return this.getCurrentFocusRow();},
            set: function (val) { this.setCurrentFocusRow(val);}
        },

        /**
         * @api {primaryBuffer} 只读属性 [属性]primaryBuffer
         * @apiName  primaryBuffer
         * @apiDescription DataStore内部维护着三个缓存区，主缓冲区，过滤缓冲区， 删除缓冲区，分放存放主要数据， 被过滤掉的数据，
         * 被删除的数据。本属性就是主缓冲区。
         * @apiGroup DataStore
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){DataBuffer} - 主缓冲区
         * @apiExample {js}(示例：)
         * alert( ds.primaryBuffer.getItemStatus(0, 2));
         */
        "primaryBuffer": {
            get: function () { return this.getPrimaryBuffer();}
        },

        /**
         * @api {deleteBuffer} 只读属性 [属性]deleteBuffer
         * @apiName  deleteBuffer
         * @apiDescription DataStore内部维护着三个缓存区，主缓冲区，过滤缓冲区， 删除缓冲区，分放存放主要数据， 被过滤掉的数据，
         * 被删除的数据。本属性就是删除缓冲区。
         * @apiGroup DataStore
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){DataBuffer} - 主缓冲区
         * @apiExample {js}(示例：)
         * alert( ds.primaryBuffer.getItemStatus(0, 2));
         */
        "deleteBuffer": {
            get: function () { return this.getDeleteBuffer();}
        },

        /**
         * @api {filterBuffer} 只读属性 [属性]filterBuffer
         * @apiName  filterBuffer
         * @apiDescription DataStore内部维护着三个缓存区，主缓冲区，过滤缓冲区， 删除缓冲区，分放存放主要数据， 被过滤掉的数据，
         * 被删除的数据。本属性就是过滤缓冲区。
         * @apiGroup DataStore
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){DataBuffer} - 主缓冲区
         * @apiExample {js}(示例：)
         * alert( ds.primaryBuffer.getItemStatus(0, 2));
         */
        "filterBuffer": {
            get: function () { return this.getFilterBuffer();}
        },


        /**
         * @api {orderBy} 可读可写属性 [属性]orderBy
         * @apiName  orderBy
         * @apiDescription 在检索时按指定的排序规则排序
         * @apiGroup Cell
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){String} - 排序规则
         * @apiExample {js}(示例：)
         * //示例
         *
         * ds.orderBy=" name asc , code desc ";
         * ds.retrieve();
         *
         */
        "orderBy": {
            get: function () { return this.getOrderBy();},
            set: function (val) { this.setOrderBy(val);}
        }
    },


    /**
     * @api {constructor} 函数   constructor
     *
     * @apiDescription constructor(connection, sql, dataProvider, dialect, columnInfo)
     * <br><br>构造函数
     *
     * @apiName  constructor
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {String} connection 数据库连接名称
     * @apiParam {String} sql 构建结果集的SQL语句
     * @apiParam {DataProvider} dataProvider 数据实际提供者对象
     * @apiParam {String} dialect 数据库类型
     * @apiParam {[Object]} columnInfo 字段信息
     *
     *
     * @apiSuccess (返回值){DataStore} - 返回自已
     *
     * @apiExample   {js}示例：
     *
     * //通常不使用本函数来构建DataStore，而是使用 newDataStore函数
     *
     * var ds=newDataStore('','select * from t1');
     *
     */
    constructor: function (connection, sql, dataProvider, dialect, columnInfo) {
        this.m_Select = sql;
        this.m_Event = new EventManager();
        this.m_Connection = connection;
        this.m_dbType = dialect;

        if (dataProvider)
        {
            if (Util.isString(dataProvider))
            {
                this.m_DataProvider = new DataProviderDefault(dataProvider);
            } else if (dataProvider.instanceOf('DataProvider'))
            {
                this.m_DataProvider = dataProvider;
            }
        }


        this.m_autoDeleteNoDataRow = true;
        this.guardLine = 200000; //数据检索的警戒线，超过了会在日志里输出
        this.needNotifyAggregateAndUnsureComputerInvalid = true;
        this.m_Dialect = null;
        this.m_Error = {code: 0, message: ''};
        this.m_PrimaryBuffer = new DataBuffer(0, this);//主数据缓冲区
        this.m_DeleteBuffer = new DataBuffer(0, this);//删除缓冲区
        this.m_FilterBuffer = new DataBuffer(0, this);//过滤缓冲区
        this.m_ColumnProperty = []; //列属性数组
        this.m_ColumnName2Index = {};//列名到列号的映射
        this.m_Express2DependList = {};// 表达式的依赖列表的缓存
        this.m_DepentList2DependCol = {};
        this.m_FieldCount = 0;//数据库字段数，注意，不包括后来创建的计算列
        this.m_OrderByConfig = [];


        this.m_Updatable = false;
        this.m_UpdateTable = "";

        this.m_PrimaryKey = "";
        this.m_UpdatableColumns = "";
        this.m_WhereClauseColumns = "";

        //保存下最后一次检索的SQL，备查
        this.m_SelectLastRetrieved = "";

        this.m_AdditionalWhere = ""; //检索时临时提供的参数
        this.m_AdditionalParam = []; //检索时临时提供的存储过程的参数

        this.m_NameInScript = ""; //在脚本中的名字
        this.m_DataSourceName=""; // 结果集的名称
        this.m_OrderBy = "";
        this.m_FilterBy = "";

        this.m_SetValueWithRiskMode = false; //设置值使用冒险模式
        this.m_SetValueWithTrigger = true; //设置值并触发事件


        this.m_AutoResetBeforeRetrieve = true; //检索前，自动清除当前数据

        //自动校验配置	vector<CAutoCheckConfig*>  m_VectorAutoCheck;
        this.m_VectorAutoCheck = [];

        //HashMap < /*String*/, HashMap < /*String*/, Integer >>
        this.m_QuickValue2Row = null;
        // 分类汇总
        this.m_GroupRules = "";
        //vector < vector < long > * > m_VectorGroups;
        this.m_VectorGroupCols = [];
        this.m_VectorGroups = [];
        /*  比如 假设共15行， {  {0} ,   { 0 ,4,7 ,10 } }  表示第零级汇总后所有的记录为一个组， 第一级汇总后，0-3行是一个组，
             4-6 行是一个组， 7-9   行是一个组， 10-14 行是一个组
             */
        //map < long, vector < long > * > m_MapGroups;
        this.m_MapGroups = [];
        /*  将上面的数据转换到本结构中就是：
             {    3->{1} , 6->{1} , 9->{1} , 14->{ 0, 1}} 表示 第3,6,9行后应该放置一个第一级的小计区，
             第14行后应该放置一个第一级的小计区 , 一个第0级的汇总区。

             */
        this.m_ListSeparator = ","; //聚合列为List时， 值之间的间隔符号

        //  加入以支持跳过前面的若干行
        this.m_absolute = 1;
        this.m_OnceRetrieveCount = 0;
        this.m_OnceRetrieveTimeout = 0; // retrieveOnce 检索数据超过多少时间后，停止　，它一timeout 是不一样的，后者通常是异常强制中止，　前者是人为控制停止　
        this.m_retrieveTimes = 0; //记录是第几次调用RetrieveOnceMore
        this.m_DBRowCount = null; // Select语句能返回多少行数据

        this.RemoteFactoryURL = "";

        this.m_AutoTrim = true;
        this.m_IsProcedure = false;
        this.m_ProcedureName = "";
        //ArrayList       /*<Integer>*/
        this.m_ProcedureParamType = []; // 存储过程参数类型

        //  缺省设置仅如下字段有值时，数据行被认为是空行
        this.m_AsNoDataIfOnlySuchColumnsHoldData = "id|gguid|inner_xh|lastmodifieddate|parent_id";

        //ArrayList</*String*/>
        this.sqlBatch = [];// new ArrayList/*</*String*/>*/();
        //ArrayList</*String*/>
        this.sqlBatch2 = [];

        this.m_blobUnRetrieve = false; //检索时对于大字段是不是不检索，有助于提高检索性能
        this.m_timeOut = 0;

        this.m_currentFocusRow = -1;
        this.m_retrieveContext =null;

        if (columnInfo != undefined) //如果直接提供了字段信息，那么直接初始化，不需要再用DataProvider.newDataStroe来创建了
        {
            this.init({
                success: true,
                dialect: dialect, //数据库类型
                value: columnInfo  //字段信息
            });

        } else //没有提供字段信息，但如果提供了DataProvider,那么，继续下面的处理
        {

            if (this.m_DataProvider) //如果指定了数据提供者，那么用它初始化自已
            {

                var colInfo = this.m_DataProvider.newDataStore(this.m_Connection, this.m_Select);

                if (colInfo.success)
                {
                    this.init(colInfo);
                } else
                {
                    this.onError(JSON.stringify(colInfo));
                }

            }
        }
        return this;
    },

    init: function (dsConfig) {
        if (!dsConfig)
        {
            this.onError("dsConfig is null ");
            return;
        }

        if (!dsConfig.success)
        {
            this.onError(dsConfig);
            return;
        }

        this.m_dbType = dsConfig['dialect']; //数据库类型
        this.m_Dialect = this.createDataAdapter(this.m_dbType, this);

        //	先复位
        this.reset('init');
        this.resetColumnInfo();


        // 清除额外的条件
        this.m_AdditionalWhere = "";
        this.m_AdditionalParam = [];

        //清除排序配置
        this.m_OrderBy = "";
        this.m_OrderByConfig.clear();


        this.parseColumnProperty(dsConfig.value);


    },

    parseColumnProperty: function (rsmd) {
        var colCount = rsmd.length;

        var /*ColumnProperty*/ cp;

        //创建一个数据访问对象，可以直接用字段名访问数据
        this.valueVisitor = function (ds, row) {
            this.m_ds = ds;
            this.m_row = row;
        }

        // SQL语句返回的列数

        this.m_FieldCount = colCount;
        var checkNullList = [];

        for (var i = 0; i < colCount; i++)
        {
            var one = rsmd[i];
            //与columnName是不同的 ，比如select  a.col as  c1 其中c1就是label 即显示名， col是列名
            //JDBC驱动能提供非常详细的信息，但是 node.js 中通常不能提供很详细的信息


            //把缺失的属性补上
            Util.merge(one, {
                columnLabel: null, //jdbc有 ,nodejs 无
                tableName: null,//jdbc有 ,nodejs 无
                columnName: null, //jdbc有 ,nodejs 无
                name: null, //nodejs有 jdbc无
                schemaName: null, //jdbc有 ,nodejs 无
                isWritable: true,//jdbc有 ,nodejs 无  这个属性需要进一步处理
                objType: ObjectType.isColumn,
                isAutoIncremen: false,// jdbc 有 ,nodejs 无
                dataType: 0,// jdbc有 ,nodejs 无
                dataTypeName: null, //jdbc有 ,nodejs 无
                type: null,  // nodejs有， jdbc无
                readOnly: false,//nodejs有，jdbc无
                precision: 0, //node.js  and  jdbc都有
                scale: 0,//nodejs and jdbc都有
                columnDisplaySize: null,// jdbc有 ,nodejs 无  表明字节数
                defaultValue: null,
                nullable: true,
                modifylog: false,
                length: null  //nodejs 有 jdbc无  表明字节数

            }, false);

            // var colName = one.columnName;
            //if (colName == null) colName = one.name; //
            //2019.09.11 修改，对于类似select * , id as aaa  from xxx 其中的 aaa 它的 columnName=id
            //  name=aaa 如果用 one.columnName 做 colName，就出现重名了，这是不对的，而且无法用别名来读取数据了
            var colName = one.name;
            colName = colName.trim().toLowerCase();

            // 必须在检索出表名属性后,在对重复的colName 作处理 , 见  Code_1
            cp = new ColumnProperty(this);
            cp.m_Name = colName;

            //JDBC可以提供下面三个属性，但是 nodejs中的驱动无法提供
            cp.m_DBTable = (one.tableName || '').trim();
            cp.m_DBName = (one.columnName || colName).trim();
            //模式名称， 但可能某些数据的驱动并未实现此功能　，比如 mysql的JDBC,比如　select * from aaa.bb 无法得到　aaa
            cp.m_DatabaseName = (one.schemaName || '').trim();

            //Logger.error(cp.m_DBName + "   " + cp.m_DBTable);
            //如果可写，或不可写但是自动增量
            cp.m_ObjType = one.objType;


            if (this.m_ColumnName2Index[colName] != undefined)
            {
                colName = cp.m_DBTable + "_" + cp.m_DBName;
                cp.m_Name = colName;
            }

            if (this.m_ColumnName2Index[colName] != undefined)
            {
                var t = colName;
                for (var ci = 1; ; ci++)
                {
                    colName = t + ci;
                    if (this.m_ColumnName2Index[colName] == undefined) break;
                }
            }

            // 列名称到所引号的映射
            this.m_ColumnName2Index[colName] = i;

            cp.m_UpdateWhereClause = false;
            cp.m_Updatable = false;
            cp.m_DataType = one.dataType;//数字型，不同的数据库，可能含义不同
            cp.m_DataTypeName = one.dataTypeName || one.type; //字段类型名称
            cp.m_UniformDataType = UniformDataType.normalize(cp.m_DataTypeName); //规整成几种类型
            cp.m_ColumnLabel = one.columnLabel || colName;

            cp.m_DefaultValue = one.defaultValue;

            cp.ColumnDisplaySize = one.columnDisplaySize || one.length || 0;
            cp.Precision = one.precision;
            cp.Scale = Math.max(0, one.scale); //  在连接Oracle时，一个numeric(22,0)字段得到的scale为-127 ，这个不知是不是驱动的Bug
            cp.Nullable = one.nullable;


            //cp.m_ColumnClassName = rsmd.getColumnClassName(i); //得到数据类型对应的Java类类型名称
            //  对于varchar类型　jdbc 驱动返回的Precision是０，为了统一使用getPrecision得到字段长度，
            // 做了如下的强制设置
            if (cp.Precision == 0) cp.Precision = cp.ColumnDisplaySize;


            //以字段名为属性名，用来直接读写数据
            /*注意两点:
                   1  是给 valueVisitor.prototype增加读写属性，而不是valueVisitor本身。因为它自已是一个函数（构造函数），
                   后面使用时，是需要创建它的实例，所以是需要给它的原型增加属性，之后创建的实例才能也具有这些属性
                   2 需要用 let tempCol 保存一下字段名，如果直接用colName,那么，无论用什么字段名去访问，得到的都是最后一个字段的数据
                */
            let tempCol = colName;
            Object.defineProperty(this.valueVisitor.prototype, colName.toLowerCase(), {
                set: function (newValue) {

                    this.m_ds.setValue(this.m_row, tempCol, newValue);
                },
                get: function () {

                    var v = this.m_ds.getValue(this.m_row, tempCol);
                    return v;
                }
            });


            this.m_ColumnProperty.push(cp);

            if (!cp.Nullable) checkNullList.push(i);


        }


        this.m_PrimaryBuffer.initColumnCount(colCount);
        this.m_FilterBuffer.initColumnCount(colCount);
        this.m_DeleteBuffer.initColumnCount(colCount);


        //字段属性初始化后，再做这些控制
        //空的检测
        for (let i = 0; i < checkNullList.length; i++)
        {
            this.addAutoCheck(checkNullList[i], AutoCheckType.CHECK_NULL, "", "");
        }


        return colCount > 0;


    },


    getOrderBy: function () {
        return this.m_OrderBy;
    },

    setOrderBy: function (v) {
        if (v == null) v = "";
        v = v.trim();
        this.m_OrderBy = v;
        //  触发事件

        this.EM.fire("orderByChanged");
    },


    /**
     * @api {row} 函数   row
     * @apiDescription  row(row)
     * <br>封装一个数据访问器，可以直接用字段名读写数据.
     * <font color=red>本函数仅能在客户端脚本中使用，服务端脚本不适用</font>
     * @apiName  row
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {int} rowNo 行号
     * @apiSuccess (返回值){Object} - 一个行数据包装对象，对象中以字段名做属性名

     * @apiExample   示例：
     *
     * var ds=newDataStore('',' select name , code from t1');
     * ds.retrieve();
     * alert( ds.row(0).name )    ;
     * ds.row(0).name='abc'  ;
     * ds.row(0).code=123;
     */
    row: function (row) {
        if (row < 0) return null;
        if (row > this.rowCount - 1) return null;
        var visiter = new this.valueVisitor(this, row);
        return visiter;
    },

    column: function (col, convertFunc) {
        var ret = [];
        var RC = this.rowCount;
        for (var row = 0; row < RC; row++)
        {
            let v = this.getValue(row, col);
            if (convertFunc) v = convertFunc(v);
            ret.push(v);
        }
        return ret;
    },

    clearBuffer: function () {
        this.m_PrimaryBuffer.clear();
        this.m_DeleteBuffer.clear();
        this.m_FilterBuffer.clear();
        this.sqlBatch.clear();
        this.sqlBatch2.clear();

    },

    /**
     * @api {reset} 函数   reset
     *
     * @apiDescription reset()
     * <br><br>将三个缓冲区的数据清空，所有标记复位。本函数执行后触发 afterReset事件
     *
     * @apiName  reset
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *

     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：

     ds.reset();

     */
    reset: function (action) {

        //action用来表明是因为什么原因而reset 在 DBPageBrick中有用到
        if (action == undefined) action = 'reset';

        this.m_retrieveTimes = 0;

        this.clearBuffer();
        this.m_DBRowCount = null;
        this.m_QuickValue2Row = null;
        this.clearError();


        this.EM.fire("afterReset", [action]);
    },

    /**
     *  用一个函数设置更新属性
     * @param table  允许更新的表
     * @param primaryKey   主键 ，组合主键的各个列用逗号分隔
     * @param updatableCols 允许更新的列
     * @param whereClauseCols 参与构造Where子句的列
     */


    /**
     * @api {setUpdateProperty} 函数   setUpdateProperty
     *
     * @apiDescription setUpdateProperty(table, primaryKey, updatableCols, whereClauseCols)
     * <br><br> 设置结果集的更新属性。
     *
     * @apiName  setUpdateProperty
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {String} table 可更新的表的名称
     * @apiParam {String} primaryKey 主键
     * @apiParam {String} updatableCols 哪些字段可更新 ,多个字段间用|分隔，*表示所有字段
     * @apiParam {String} whereClauseCols 参与构建where语句的字段,多个字段间用|分隔。用它来实现多种更新策略,详细请参阅DataStore的说明
     *
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     * var ds=newDataStore('','select * from t1');
     * ds.updatableTable='t1';
     * ds.primaryKey="id";
     * ds.updatableColumns="*";
     * ds.updateWhereClauseColumns ="id";
     *
     *
     * //上述脚本等同于如下脚本
     * var ds=newDataStore('','select * from t1');
     * ds.setUpdateProperty( "t1","id","*","id");
     *
     *
     */
    setUpdateProperty: function (table, primaryKey, updatableCols, whereClauseCols) {
        this.setUpdatable(true);
        this.setUpdatableTable(table);
        this.setPrimaryKey(primaryKey);
        this.setUpdatableColumns(updatableCols);
        this.setUpdateWhereClauseColumns(whereClauseCols);

    },


    getError: function () {

        return this.m_Error;
    },

    clearError: function () {
        this.m_Error = {code: 0, message: ''};
    },

    resetColumnInfo: function () {
        //清除列名到索引号的映射
        this.m_ColumnName2Index = {};
        //清除烈属性缓冲区
        this.m_ColumnProperty.clear();
        //清除自动校验缓冲区
        this.m_VectorAutoCheck.clear();
        this.m_Express2DependList = {};
        this.m_DepentList2DependCol = {};
        // 增加视同无数据设置
        this.m_AsNoDataIfOnlySuchColumnsHoldData = "";

    },


    /**
     * 设置数据库连接，并自动创建合适的数据库适配器
     * @param con
     */
    setConnection: function (con) {
        this.m_Connection = con;


    },


    createDataAdapter: function (con, ds) {

        // 使用MS SQL Server 2K驱动程序
        if (con.indexOf('sqlserver') >= 0) return new DialectSQLServer(ds);
        if (con.indexOf('h2') >= 0)     return new DialectH2(ds);
        if (con.indexOf('oracle') >= 0) return new DialectOracle(ds);
        if (con.startsWith("postgresql"))  return new DialectPostgres(ds);
        if (con.startsWith("mysql"))   return new DialectMySQL(ds);

        console.log("系统并未提供对" + con + "的特别支持，将使用缺省的DataAdapter");

        return new Dialect(ds);
    },


    getConnection: function () {
        return this.m_Connection;
    },


    /**
     * @param col
     * @return  返回列名对应的列号 ,如果不存在列，那么返回-1
     */

    /**
     * @api {col2Index} 函数   col2Index
     *
     * @apiDescription col2Index(col)
     * <br><br>将列名转换成列号。比如 select id , name ,code from t1
     * id,name,col的列号分别是 0，1，2
     *
     * @apiName  col2Index
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {String} col 列名 ,如果col是一个整数，那么直接返回该整数

     *
     * @apiSuccess (返回值){int} - 返回列名对应的列号
     *
     * @apiExample   {js}示例：

     var ds=newDataStore("","select id , name , value from app_config");
     var a= ds.col2Index('id')  //  a == 0
     var b= ds.col2Index(5) ; // b==5


     */
    col2Index: function (col) {
        var colIndex = -1;
        if (!isNaN(col)) return col;
        col = ('' + col).trim().toLowerCase();
        colIndex = this.m_ColumnName2Index[col];
        if (colIndex == undefined) colIndex = -1;
        return colIndex;

    },


    /**
     * @api {getColumnProperty} 函数   getColumnProperty
     *
     * @apiDescription getColumnProperty (col)
     * <br><br> 返回列的属性.
     *
     * @apiName  getColumnProperty
     * @apiGroup DataStore
     *
     * @apiParam {int/String} col 列号或列名
     *
     * @apiSuccess (返回值){ColumnProperty} - 返回列的属性,参看 ColumnProperty对象
     *
     * @apiExample   {js}示例：
     *
     * ds.getColumnProperty('code').getLabel();
     */
    getColumnProperty: function (col) {
        col = this.col2Index(col);
        if (col < 0 || col >= this.m_ColumnProperty.length) return null;
        return this.m_ColumnProperty[col];
    },


    /**
     * @api {getSelect} 函数   getSelect
     *
     * @apiDescription  getSelect()
     * <br><br> 得到创建本结果集的Select语句
     *
     * @apiName  getSelect
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     *
     * @apiSuccess (返回值){String} - 创建本结果集的Select语句
     *
     * @apiExample   {js}示例：
     *
     * alert( ds.getSelect());
     */
    getSelect: function () {
        return this.m_Select;
    },


    /**
     * @api {getSelectLastRetrieved} 函数   getSelectLastRetrieved
     *
     * @apiDescription  getSelectLastRetrieved()
     * <br><br> 得到最后一次执行检索时的完整的Select语句
     *
     * @apiName  getSelectLastRetrieved
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiSuccess (返回值){String} - 得到最后一次执行检索时的完整的Select语句
     *
     * @apiExample   {js}示例：
     *
     * var ds=newDataStore("","select * from t1");
     * ds.retrieve();
     * println( ds.getSelectLastRetrieved()); //  select * from t1
     * ds.retrieve("id>1");
     * println( ds.getSelectLastRetrieved()); //  select * from t1 where id>1
     * ds.setSelect(" select * from t1 where id>10");
     * ds.retrieve(" name='abc' " );
     * println( ds.getSelectLastRetrieved()); //  select * from t1 where id>10 and name='abc'
     */
    getSelectLastRetrieved: function () {
        return this.m_SelectLastRetrieved;
    },


    setSelectLastRetrieved: function (sql) {
        this.m_SelectLastRetrieved = sql;
    },


    /**
     * @api {getAdditionalWhere} 函数   getAdditionalWhere
     *
     * @apiDescription getAdditionalWhere()
     * <br><br>得到最后一次检索时的附加条件
     *
     * @apiName  getAdditionalWhere
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     *
     * @apiSuccess (返回值){String} - 得到最后一次检索时的附加条件
     *
     * @apiExample   {js}示例：
     * var ds=newDataStore("","select * from t1");
     * ds.retrieve();
     * println( ds.getAdditionalWhere()); //  空字符串
     * ds.retrieve("id>1");
     * println( ds.getAdditionalWhere()); //  id>1
     * ds.setSelect(" select * from t1 where id>10");
     */
    getAdditionalWhere: function () {
        return this.m_AdditionalWhere;
    },

    /**
     * 仅仅是把 select 替换一下，而不重新解析字段。这就要求新的select 与旧的select 在字段上应该是完全兼容的
     */

    /**
     * @api {replaceSelect} 函数   replaceSelect
     *
     * @apiDescription replaceSelect(select)
     * <br><br> 用指定的select语句替换本结果集的Select语句。前提条件是替换的select语句与当前结果集的select语句具有
     * 相同的字段及顺序。结果集虽然能重新创建，但是当结果集被绑定到界面上后，字段名通常不允许再改变，不然影响绑定。此时
     * 又想调整结果集的Select语句，就可以使用本函数。本函数仅仅是替换Select语句，并不重建整个结果集，即列信息不会重新
     * 构建，也不影响界面对结果集的绑定。
     *
     * @apiName  replaceSelect
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {String} select 用于替换的SQL语句
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     * var ds=newDataStore('', 'select id , name from t1');
     * //do something
     *
     * ds.replaceSelect("select id , name from t2"); // ok
     * ds.replaceSelect("select id , name from t2 where id>1"); // ok
     * ds.replaceSelect("select id , code as name from t3  "); // ok
     *
     * ds.replaceSelect("select id , code from t2"); // error 字段名清单不一至
     * ds.replaceSelect("select id , name, code from t2"); // error 字段个数不一至
     *
     *
     *
     */
    replaceSelect: function (select) {
        this.m_Select = select;
    },


    /**
     * sql 语句中的内联参数格式是 :参数名{ 缺省值}
     * 比如 select * from news_user where id>:工作本1!id{10} or birthday>=':birthday{2007.01.01}'
     *     or moblie like ':mobile{123}'
     *    其中有三个参数： 工作本1!id   , birthday , mobile
     * @param sql
     * @param useDetault
     * @returns {*}
     *
     *
     */

    parseInnerParameter: function (sql, useDetault) {

        // ([^\\{]+) 表示所有非{字符的组合
        //([^\\}]+) 表示所有非}字符的组合

        //注意，不要加参数 g ,加了，match的结果就是所有匹配的字符串内容。不加，就是第一个匹配的字符串，及各级()中的内容
        //这里需要的是后者
        // 正确匹配后， result[0]是整个匹配串，[1]是前后去掉空格后的内容 [2]是:与{之前的表达式内容 [3]是测试值内容
        var res = [/(::\s*([^\{]+)\s*\{\s*([^\}]*)\s*\})/,
            /(:\s*([^\{]+)\s*\{\s*([^\}]*)\s*\})/];


        for (let i = 0; i < res.length; i++)
        {
            let re = res[i];
            while (true)
            {
                var result = sql.match(re);
                if (result == null) break;
                if (result.length == 4)
                {
                    var total = result[0];
                    var express = result[2];
                    var demoValue = result[3];
                    var value = this.EM.fire("parseInnerParameter", [express]);
                    if (value == null) value = demoValue;
                    sql = sql.substring(0, result.index) + value + sql.substring(result.index + total.length);
                }
            }
        }


        return sql;


    },


    /**
     * 从 数据中 中检索数据，该方法用来解决远程取得数据后，把数据导入本DataStore中。
     *
     *
     * 注意，本函数用来从dbprovider返回的数据中把数据放到主缓冲区。它是一个内部使用的函数，data的数据与 ds.getData()的数据结构是不一样的
     * 如果是要在两个ds之间复制数据， 请使用 ds2.importData (ds1.getData());
     *
     *   本函数并不清空缓存 ！！！，如果需要，请调用 reset()来清除原有数据
     *
     * @param data   是一个ArrayList，其中的每个元素又是一ArrayList
     * @param where
     * @param triggerEvent 是否触发事件
     * @param dataType  'no'表示数据中是用列号， 'name'表示用列名
     * @param isResetBeforeRetrieve  在检索数据前，是不是已经清除了数据。
     * @param name2column  data中如果用的是列号，那么需要记录列号对应的列名，因为数据的列号并不一定与此时的列名能对应上，需要转换
     * 注意，本参数是一个事后标记参数，它并不用来控制清除数据，只是用来告知在本函数执行前是不是已经清除了数据，
     * 本参数的目的是在触发事件 retrieveEnd时，用来判断是不是需要重设置当前绑定定。
     * 当 isResetBeforeRetrieve==false时，是不需要重设置当前绑定行的
     *
     * @returns {number}
     */
    retrieveFromData: function (data,  where, triggerEvent,name2column, dataType, DBRowCount ,isResetBeforeRetrieve) {
        //处理数据

        if (dataType == undefined) dataType = 'no'; //默认，表data中是用列号记录数据

        if (triggerEvent == undefined) triggerEvent = true;

        if (!data) return 0;

        var rowCount = data.length;
        //2020.04.23  下面直接返回是不对的，没有数据，仍需要触发相关事件
        //if (rowCount == 0) return 0;

        var colCount = this.columnCount;

        this.m_DBRowCount=null;

        //先缓存字段统一类型
        var vt = [];
        var vi = {}; //字段名对应的序号
        var vi2=[]; //本地字段的列号对应到返回数据包中的列号,下标就是本地结果集的列号，对应的数据就是 c? 这样的以c开头的数据包中的列号
        for (let i = 0; i < colCount; i++)
        {
            let /*ColumnProperty*/ cp = this.getColumnProperty(i);
            vt.push(cp.getUniformDataType());
            let colName=cp.m_DBName.toLowerCase();//一定要转小写，因为oracle 可能会全部大写字段名，另外的又可能是小写字段名
            vi[colName] = i;

            //2021.01.19 增加，数据包中列号与本地列号可能不一至，形成的原因可能有：结果集是从一个数据库创建的，但是检索时可能
            // 从另一个数据库中检索，且这两个数据库中的字段顺序可能不一样，因此需要用列名做中介转换一下
            if(name2column)
            {
                if( name2column[colName]!=undefined)
                {
                    let ni = name2column[colName]; //字段在数据包中对应的C+索引号
                    vi2.push(ni);
                }else {
                    vi2.push('error'); //表示本地结果集中的字段在数据包中没有找到对应的字段
                }

            }
        }

        for (var r = 0; r < rowCount; r++)
        {

            var rowData = data[r];
            var tempRow = new DsRow(this, colCount);
            this.m_PrimaryBuffer.push(tempRow);

            //检索出来的数据行的初始状态设置成 isNotModified
            tempRow.m_RowStatus = ItemStatus.isNotModified;
            if (rowData.$rowStatus != undefined) tempRow.m_RowStatus = rowData.$rowStatus;


            //当在Retrieve前,可能设置了自定义的计算列,那么这些自定义的
            //计算列也应该需要初始化.当  索引号>=FieldCount的单元即用户自定义的计算列或者自定义列

            if (dataType == 'name') //2019.06.17 支持后台脚本生成的数据，直接用列名保存数据
            {
                for (let col in rowData)
                {
                    let v = rowData[col];
                    if (v == null) continue;
                    let colIndex = vi[col.toLowerCase()];
                    if (colIndex == undefined) continue; //不存在的字段忽略它


                    //类型一定要检查一下， 因为在上面的数据检索后，再远程传过来，此时v 的类型不一定与字段类型是一样的，
                    //要做个强制转换，对速度的影响，有待评估
                    v = ObjectTool.changeType(v, vt[colIndex]);

                    var /*DsItem*/ item = new DsItem();
                    item.m_CurrentValue = v;
                    item.m_OriginalValue = v; //如果是检索出来的数据 ，那么就应该是相待的，状态是未改变
                    item.m_ItemStatus = ItemStatus.isNotModified;

                    tempRow.setAt(colIndex, item);
                }

                //如果返回了全数数据行数，那么设置它
                if (DBRowCount != undefined) this.m_DBRowCount = DBRowCount;

            } else
            {
                for (let i = 0; i < colCount; i++)
                {
                    //如果是初始化用户自定义的计算列那么
                    if (i > this.m_FieldCount) continue;
                    //下面将数据库中的值取出来
                    //注意：要把列号转成字符型,这个是与DataProvider 相对应的
                    let ci= vi2[i];//i 是本地列索引号，要从vi2中找到它对应于数据包中的列号
                    if( ci=='error') continue; //在数据包中没有对应的字段，有可能存在这种情况：检索时对应的数据库中可能不存在某个字段
                    let v = rowData[ci];
                    let v0 = rowData['$colOriginal_' + ci];
                    if (v0 == undefined) v0 == null;

                    if (v == null && v0 == null) continue;

                    //   if (this.m_AutoTrim) v = v.trim();

                    //类型一定要检查一下， 因为在上面的数据检索后，再远程传过来，此时v 的类型不一定与字段类型是一样的，
                    //要做个强制转换，对速度的影响，有待评估
                    if (v != null) v = ObjectTool.changeType(v, vt[i]);

                    var /*DsItem*/ item = new DsItem();
                    item.m_CurrentValue = v;
                    item.m_OriginalValue = v; //如果是检索出来的数据 ，那么就应该是相待的，状态是未改变
                    item.m_ItemStatus = ItemStatus.isNotModified;


                    //不能用 v0==null判断，因为新插入的行,v0就是null
                    if (rowData['$colStatus_' + ci] != undefined)//如果有状态指示，那么
                    {

                        item.m_OriginalValue = v0 == null ? null : ObjectTool.changeType(v0, vt[i]);
                        item.m_ItemStatus = rowData['$colStatus_' + ci];
                    }
                    tempRow.setAt(i, item);

                }
            }


        }

        //当是完全检索时，直接把m_DBRowCount设置成rowcount
        if (this.m_absolute <= 1 && this.m_OnceRetrieveCount == 0) this.m_DBRowCount = rowCount;


        //分组信息仅仅保留总计
        this.BuildAllAsOneGroup(this.m_VectorGroups);


        //重大修改，2019.02.05 把通知非稳定函数失效，及触发 retrieveEnd事件改成异步，避免显示上的延迟

        var that = this;

        //[标记20201230]
        //2020.12.30 null 表示延时任务的key自动生成，即此任务一定会执行，不存在后续的key任务将之覆盖
        Tools.delayRun(null, function () {
            that.m_PrimaryBuffer.notifyAggregateAndUnsureComputerInvalid();
        }, 10);


        //触发检索结束事件
        if (triggerEvent)
        {
            //2020.12.30 null 表示延时任务的key自动生成，即此任务一定会执行，不存在后续的key任务将之覆盖
            Tools.delayRun(null , function () {
                that.EM.fire("retrieveEnd", [rowCount, where == WITHLASTSELECT ,isResetBeforeRetrieve]);
            }, 10);
        }

        return rowCount;

    },


    /**
     * @api {importData} 函数   importData
     *
     * @apiDescription importData(data)
     * <br><br>  从 data 中导入数据，导入的数据的状态等同于检索出来的数据状态
     * 本操作会触发 retrieveEnd事件
     *
     * @apiName  importData
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {Object} data 数据数组，它必须是某个DataStore使用.getData()得到的数据
     *
     * @apiSuccess (返回值){int} - 返回本次导入的数据条数
     *
     * @apiExample   {js}示例：
     *
     * var ds1=newDataStore("","select * from t1");
     * var ds2=newDataStore("","select * from t2");
     *
     * ds1.importData( ds2.getData());
     *
     */
    importData: function (data) {


        if (!data) return 0;

        var rc = data.length;
        var cc = this.columnCount;

        if (rc == 0) return 0;


        //先缓存字段统一类型
        var vt = [];
        var colName = [];

        for (let i = 0; i < cc; i++)
        {
            let /*ColumnProperty*/ cp = this.getColumnProperty(i);
            vt.push(cp.getUniformDataType());

            colName.push(cp.m_Name.toLocaleLowerCase());
        }


        for (var r = 0; r < rc; r++)
        {

            var rowData = data[r];
            var tempRow = new DsRow(this, cc);
            this.m_PrimaryBuffer.push(tempRow);

            //检索出来的数据行的初始状态设置成 isNotModified
            tempRow.m_RowStatus = ItemStatus.isNotModified;

            //当在Retrieve前,可能设置了自定义的计算列,那么这些自定义的
            //计算列也应该需要初始化.当  索引号>=FieldCount的单元即用户自定义的计算列或者自定义列

            for (let i = 0; i < cc; i++)
            {
                //如果是初始化用户自定义的计算列那么
                if (i > this.m_FieldCount) continue;
                //下面将数据库中的值取出来
                //注意：要把列号转成字符型,这个是与DataProvider 相对应的
                let col = colName[i];

                let v = rowData[col];


                if (v == undefined) continue;
                if (v == null) continue;


                //类型一定要检查一下，
                //要做个强制转换，对速度的影响，有待评估
                if (v != null) v = ObjectTool.changeType(v, vt[i]);

                var /*DsItem*/ item = new DsItem();
                item.m_CurrentValue = v;
                item.m_OriginalValue = v; //如果是检索出来的数据 ，那么就应该是相待的，状态是未改变
                item.m_ItemStatus = ItemStatus.isNotModified;


                tempRow.setAt(i, item);


            }


        }

        //当是完全检索时，直接把m_DBRowCount设置成rowcount
        //下面的处理有待细思
        // if (this.m_absolute <= 1 && this.m_OnceRetrieveCount == 0) this.m_DBRowCount = rowCount;


        //分组信息仅仅保留总计
        this.BuildAllAsOneGroup(this.m_VectorGroups);
        this.m_PrimaryBuffer.notifyAggregateAndUnsureComputerInvalid();

        //触发检索结束事件
        this.EM.fire("retrieveEnd", [rc, false]);
        return rc;
    },


    /**
     * @api {retrieveWithLastSelect} 函数   retrieveWithLastSelect
     *
     * @apiDescription  retrieveWithLastSelect()
     * <br><br> 最近做过的检索操作，重新做一次
     *
     * @apiName  retrieveWithLastSelect
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     *
     * @apiSuccess (返回值){int} - 返回本次检索出的数据条数
     *
     * @apiExample   {js}示例：
     *
     *
     * ds.retrieve(" id>100");
     *
     * ds.retrieveWithLastSelect(); // 使用 id>100这个条件，重新做一下检索，它等同于 ds.retrieve(" id>100");
     *
     */
    retrieveWithLastSelect: function () {
        return this.retrieve(DataStore.WITHLASTSELECT);
    },


    /**
     * @api {retrieve} 函数   retrieve
     *
     * @apiDescription   retrieve(where, resetBeforeRetrieve)
     * <br><br>按指定条件检索数据，当条件为 ""时，表示没有附加的条件。当数据源定义中本来就有where
     * 条件时，指定的条件与原有的条件是 and 的关系。
     *
     * @apiName  retrieve
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {String} where 检索条件
     * @apiParam {boolean} resetBeforeRetrieve 检索前，当前的缓冲区是不是要先清空,默认是true
     * @apiParam {int} onceRetrieveCount 检索数据条数限制，0 表示不限制
     * @apiParam {int} absolute 检索前，定位到哪一行。从1开始计数，例如： absolute=10 ，表示忽略前面9条数据，从第10条开始检索
     * @apiParam {Function} callback  检索完成后的回调事件，它在retrieveEnd事件触发之后再执行，可以理解为它是最后执行的
     * @apiParam {boolean} asyncCall  是否异步检索，默认是false即同步方式检索。 注意，并非是提供了 callback参数，检索就是异步的。
     * callback仅表示检索完成后执行，当异步检索时，在异步检索完成后，callback再被执行， 当同步检索时，在同步检索完成后，callback被执行。
     *
     * @apiSuccess (返回值){int} - 返回本次检索出的数据条数
     *
     * @apiExample   {js}示例：

     ds.retrieve( " name like '%abc%' ");
     ds.retrieve();
     ds.retrieve(" id>10", false);

     ds.setSelect("select * from t1 where id>10")
     ds.retrieve("id<100")
     //表示按照　select * from t1 where ( id>10 )  and (id<100)来检索数据　

     ds.setSelect("select * from t1 where id>10 or id<5")
     ds.retrieve("id<100")
     //表示按照　select * from t1 where ( id>10 or id<5)  and (id<100)来检索数据

     */
    retrieve: function (where, resetBeforeRetrieve, onceRetrieveCount , absolute, callback ,asyncCall,useCache) {

        if( asyncCall==undefined) asyncCall=false; //默认是同步检索
        if( useCache==undefined)  useCache=true; //默认是允许优先使用缓存，究竟是否使用缓存，还要看结果集的定义中是否允许缓存
        this.clearError();
        //清除数据
        if (resetBeforeRetrieve == undefined) resetBeforeRetrieve = true;
        if (resetBeforeRetrieve) this.reset('retrieve');

        //增加一个标记，因为 thisrow() 函数会受这个影响
        this.lastRetrieveIsResetBeforeRetrieve = resetBeforeRetrieve;

        if (this.m_Select == '')
        {
            this.onError("请先设置 Select ");
            return 0;
        }

        if (!this.m_DataProvider)
        {
            this.onError("请先设置DataProvider");
            return 0;
        }

        var sql = this.m_Select;

        //对内联参数的支持
        // 2020.04.23 内联参数改到后台解析，前台的SQL是加过密的，出现兼容性，仍放在这里，后期不排除取消此参数
        // 2020.04.24 某些功能（取数定义）仍需要前台内嵌参数的处理，暂时放开
        // 2020.04.25 如果没有提供参数解析上下文数据，那么就用前台解析，如果提供了，就用后台解析
        //2020.07.03 增加事件，方便retrieveContext 的提供，它不应该显示的通过设置属性来处理，因为 可能使用者会直接用retreive
        //来检索参数，而忽略了retrieveContext的提供 ， 把它改造成通过事件来提供，不需要手动提供，就样可以避免忘记提供此数据


        this.m_retrieveContext = this.EM.fire("buildRetrieveContext", [], null);


        if( Object.getOwnPropertyNames(this.retrieveContext).length == 0)
        {
            sql = this.parseInnerParameter(sql, false);
        }

        var originalSQL = sql; //不带where的select语句，主要用来支持在后端脚本中提供数据时，需要这个原始的SQL

        //  增加对按上一次语句进行重新检索的支持，用来实现分页检索
        if (where != WITHLASTSELECT)
        {

            this.m_AdditionalWhere = where;//where可能是数组

            // order by 做为参数带入后台进行处理，而不是前台处理

            /* 下面的是旧代码
            var sp = new SqlParse(sql);
            //2019.03 增加对修改orderBy的支持
            if (this.m_OrderBy != '') sp.orderBy = this.m_OrderBy;
            sp.linkUpWhere(sp.sql, where); //拼上 where，注意linkUpWhere中会对whre是数组时做串联处理
            sql = sp.sql;

            //转成数据库对应的方言
            sql = this.m_Dialect.toLocalSyntax(sql);

            // 保存下最后一次检索的SQL，备查
            this.m_SelectLastRetrieved = sql;
            */
        } else
        {
            sql = this.m_SelectLastRetrieved;
        }

        //2019.09.28 增加检索前的事件，可以在此终止检索
        if (!this.EM.fire("beforeRetrieve", [], true)) return -1;

        //2020.04.16 补丁
        //当做追加检索时，当前的 absolute可能会导致无法检索出数据，
        //比如 当前已经翻了几页，然后追加检索一条数据，此时如果不指定 absolute=0 ，那么在按条件追加检索数据时
        // 仍会滚到 this.m_absolute之后，再取数，此时导至数据已被滚过去，取不到了


        //2020.06.03 增加对异步检索的支持
        let  holdData=function(result)
        {
            if (!result.success)
            {
                this.onError(result.message,'retrieve');
                return 0;
            }

            // 保存下最后一次检索的SQL，备查及翻页时要用到
            this.m_SelectLastRetrieved = result.selectLastRetrieved;

            //2019.07.16 增加参数 result.dataType 用它来表明数据的格式
            let ret= this.retrieveFromData(result.value, where, true, result.name2column,
                result.dataType, result.DBRowCount , resetBeforeRetrieve);
            //callback 在 retrieveEnd事件之后再执行
            if(callback)  callback();
            return ret;
        }.bind(this);


        let result = this.m_DataProvider.retrieve(
            this.name,
            this.id,
            this.templateid,
            this.m_Connection,
            sql,
            '',  //标记20190716  如果是重新检索，where 已经拼在sql里了， 如果是重复上次检索，where也已经拼在SQL里了
            //如果提供了onceRetrieveCount，则用它，否则用当前的
            onceRetrieveCount==undefined? this.m_OnceRetrieveCount : onceRetrieveCount,
            //如果临时提供了absolute,则用它，否则用默认的
            absolute==undefined? this.m_absolute : absolute,
            //强制转成数组，因为在smartRetrieve时，传入的where是一个数组，此处统一强制转换成数组
            Util.isArray(this.m_AdditionalWhere) ? this.m_AdditionalWhere : [this.m_AdditionalWhere],
            originalSQL,
            this.m_OrderBy,
            this.retrieveContext ,
            asyncCall? holdData :null , //如果异步检索，那么把holdData做为回调函数
            useCache // 是否优先使用缓存
        );

        //当需要同步检索时， 上面的操作是同步的， result放的是数据包，那么现在拿住数据
        if( ! asyncCall) return holdData(result);

    },


    /**
     * 刷新检索第row行。当使用where检索出0行数据时，则把row行删除 ，但是并不放到删除缓冲区,如果检索出多行，那么仅使用第一行
     * @param row  需要刷新检索的条
     * @param where  刷新检索的where
     */



    /**
     * @api {refreshRetrieve} 函数   refreshRetrieve
     *
     * @apiDescription  refreshRetrieve(row, where)
     * <br><br> 使用where做检索条件检索数据，检索出来的数据覆盖到第row行。通常使用where只会检索出一条数据
     *
     * @apiName  refreshRetrieve
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {int} row 要刷新检索的行
     * @apiParam {String} where 检索条件
     *
     * @apiSuccess (返回值){int} - 返回本次检索出的数据条数
     *
     * @apiExample   {js}示例：
     *
     * ds.refreshRetrieve( 3 , "id=3"); // 用 id=3做条件检索数据覆盖到第4行（行中从0开始）
     *
     */
    refreshRetrieve: function (row, where) {

        this.clearError();

        if (this.m_Select == '')
        {
            this.onError("请先设置 Select ");
            return 0;
        }

        if (!this.m_DataProvider)
        {
            this.onError("请先设置DataProvider");
            return 0;
        }

        var sql = this.m_Select;
        //对内联参数的支持
        sql = this.parseInnerParameter(sql, false);
        var originalSQL = sql;

        //  增加对按上一次语句进行重新检索的支持，用来实现分页检索
        if (where != WITHLASTSELECT)
        {
            // this.m_AdditionalWhere = where; // 为什么要注释掉，参看下面的备注
            //   var sp = new SqlParse(sql);
            //   sp.linkUpWhere(sp.sql, where);//拼上 where，注意linkUpWhere中会对whre是数组时做串联处理
            //  sql = sp.sql;

            //转成数据库对应的方言
            //  sql = this.m_Dialect.toLocalSyntax(sql);

            // 保存下最后一次检索的SQL，备查
            //this.m_SelectLastRetrieved = sql;
            //注意本函数是刷新检索，它不应该改变 m_AdditionalWhere ，即不应该影响翻页操作
        } else
        {
            sql = this.m_SelectLastRetrieved;
        }


        var result = this.m_DataProvider.retrieve(
            this.name,
            this.id,
            this.templateid,
            this.m_Connection,
            sql,
            '',  //标记20190716  如果是重新检索，where 已经拼在sql里了， 如果是重复上次检索，where也已经拼在SQL里了
            1, //只检索一行
            0,
            //强制转成数组，因为在smartRetrieve时，传入的where是一个数组，此处统一强制转换成数组
            [where],
            originalSQL,
            '',
            this.retrieveContext
        );



        if (!result.success)
        {
            this.onError(result);
            return 0;
        }

        //不要触发事件，因为本函数是刷新某行数据，应避免触发事件，避免当前行改变事件等等
        this.retrieveFromData(result.value, where, false,result.name2column);
        //此时最后一行，它就是追加检索出来的

        //把第row行移除，并把最后一行，移到第row行处
        var buffer = this.getPrimaryBuffer();
        buffer.eraseRow(row);

        var RC = this.getRowCount();
        var lastRow = RC - 1;
        var list = buffer.GetBuffer();
        var /*DsRow*/ rowData = list[lastRow];
        list.removeAt(lastRow);
        list.splice(row, 0, rowData);
        buffer.notifyAggregateAndUnsureComputerInvalid();

        this.EM.fire("refreshRetrieveEnd", [row]);

    },


    BuildAllAsOneGroup: function (VectorGroups) {
        //TODO

    },


    reCreateGroups: function () {
        //TODO
    },


    getGroupRole: function () {
        return this.m_GroupRules;
    },

//      g1Col1|g1Col2 , g2Col1|g2Col2 ,  组之间用 , 分隔， 同组的各个列之间用 | 分隔。
    createGroups: function (rule) {
        //TODO
    },

//	清除分组信息
    ResetGroupsInfo: function () {
        this.m_VectorGroupCols.clear();
        this.m_VectorGroups.clear();
        this.m_MapGroups.clear();
    },

    /**
     * 功能描述： 将 m_VectorGroups 转换到 m_MapGroups ,这样方便 CDrawObject绘制
     */
    ConvertGroupsInfo: function () {
        //TODO
    },

//在 row 行后面需要增加一个 第groupNo 个分组组底信息
    AddGroupsMap: function (/*int*/row, /*int*/groupNo) {
        //TODO
    },

    BuildGroupInfo: function (/*String*/groupByCols, /*ArrayList*/ VectorGroups) {
        //TODO
    },

    /**
     * 注：20101102
     *  在vectorGroups中找界于 startRow, endRow之间的行号
     *  比如,按 “a,b”进行分组，当按a分组后，vectorGroups= [ [0,2] ],再按b进行数据分且时，得到的是[0,3]
     *  但是由于b是相对a的更明细的分组，所以，应该插值为 [0,2,3]
     *  a b
     *  1 1
     *  1 1
     *  2 1
     *  2 2
     *  显示会如下：
     *    ___________
     * 0 |   1   |  1   |
     * 1 |_____|      |
     * 2 |   2   |____|
     * 3 |_____|__2_|
     *  应该修正为
     *    ___________
     * 0 |   1   | 1    |
     * 1 |_____|____|
     * 2 |   2   |_1__|
     * 3 |_____|_2__|
     *
     *
     * @param row
     * @param VectorGroups
     * @return
     */
    /*int*/
    FindLowerRow: function (/*int*/startRow, /*int*/endRow, /*ArrayList*/VectorGroups) {
        //TODO
    },

    /*int*/
    FindNextDifferentRow: function (/*ArrayList*/ColIndexs, /*int*/row, /*int*/rc) {
        //TODO
    },


    /*boolean*/
    TwoRowIsEqualOnSomeColumns: function (ColIndexs, /*int*/row1, /*int*/row2) {

        //TODO
        return true;

    },

    ReCalculateComputer: function (/*int*/col) {
        //TODO
    },


    /**
     * @api {getValue} 函数   getValue
     *
     * @apiDescription  getValue(row,col[,original])
     * <br><br> 读取主缓冲区中row行col列的数据。每个缓冲区本质上就是一个二维数组。其中的每个元素包含3个属性
     * <ul><li>oldValue 检索出来的原始值</li>
     * <li>newValue 当前可能被编辑过的值</li>
     * <li>status 编辑状态</li></ul>
     * 其中 status 可以是：
     * <ul><li>新插入的数据:oldValue == newValue == null </li>
     * <li>新插入且被修改的数据: oldValue == null , newValue != null </li>
     * <li>检索出来的数据: oldValue == newValue </li>
     * <li>检索出来且被修改的数据： oldValue != newValue;</li>
     * </ul>
     * @apiName  getValue
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {int} row 行号
     * @apiParam {int/String} col 列号或列名
     * @apiParam {boolean} original 读取从数据库中检索出来的原始数据吗。可选参数，默认值为false表示取当前值
     *
     *
     * @apiSuccess (返回值){Variant} - 返回row行col列的数据
     *
     * @apiExample   {js}示例：
     *
     * var ds=newDataStore("","select id from app_config where id=1");
     * ds.retrieve();
     *
     * println( ds.getValue(0,"id")) ; // 1 当前还没有被修改，所以是1
     * println( ds.getValue(0,"id",true)) ; // 1 从数据库中取出来的值是1
     * println( ds.getValue(0,"id",false)) ; // 1 等同于  getValue(0,"id")
     *
     * ds.setValue(0,"id", 2);
     * println( ds.getValue(0,"id")) ; // 2 当前被修改成了2
     * println( ds.getValue(0,"id",true)) ; // 1  它从数据库中取出来时是1
     * println( ds.getValue(0,"id",false)) ; // 2  等同于  getValue(0,"id")
     *
     *
     *
     *
     */
    getValue: function (/*int*/row, col, /*boolean*/original) {
        return this.m_PrimaryBuffer.getValue(row, col, original);
    },


    /**
     * @api {getInt} 函数   getInt
     *
     * @apiDescription getInt(row, col [,  defaultValue])
     * <br><br> 对getValue的二次包装，先用getValue(row,col)获取数据，再转换成整数，如果为null或转换失败，则返回deaultValue
     *
     * @apiName  getInt
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {int} row 行号
     * @apiParam {int/String} col 列号或列名
     * @apiParam {int} defaultValue 如果结果集中的数据为null，则返回本默认值。可选参数，默认为0
     *
     * @apiSuccess (返回值){int} -
     *
     * @apiExample   {js}示例：
     *
     *  ds.setValue(0,"name","abc");
     *  println( ds.getInt(0,"name")); // 0
     *  ds.setValue(0,"name","123");
     *  println( ds.getInt(0,"name")); // 123
     *  ds.setValue(0,"name",null);
     *  println( ds.getInt(0,"name")); // 0
     *
     *
     */
    getInt: function (/*int*/row, col, /*int*/defaultValue) {
        if (defaultValue == undefined) defaultValue = 0;
        var obj = this.getValue(row, col);
        if (obj == null) return defaultValue;
        obj = ObjectTool.changeType(obj, UniformDataType.$Integer);
        return obj;
    },

    /**
     * @api {getString} 函数   getString
     *
     * @apiDescription getString(row, col [,  defaultValue])
     * <br><br> 对getValue的二次包装，先用getValue(row,col)获取数据，再转换成字符串，如果为null或转换失败，则返回deaultValue
     *
     * @apiName  getString
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {int} row 行号
     * @apiParam {int/String} col 列号或列名
     * @apiParam {String} defaultValue 如果结果集中的数据为null，则返回本默认值。可选参数，默认为“”
     *
     * @apiSuccess (返回值){String} -
     *
     * @apiExample   {js}示例：
     *
     *  ds.setValue(0,"name","abc");
     *  println( ds.getString(0,"name")); // 'abc'
     *  ds.setValue(0,"rq",newDate());
     *  println( ds.getString(0,"rq")); // '2018.01.01'
     *  ds.setValue(0,"name",null);
     *  println( ds.getString(0,"name")); // ''
     *
     *
     */
    getString: function (/*int*/row, col, /*int*/defaultValue) {
        if (defaultValue == undefined) defaultValue = '';
        var obj = this.getValue(row, col);
        if (obj == null) return defaultValue;
        obj = ObjectTool.changeType(obj, UniformDataType.$String);
        return obj;
    },

    /**
     * @api {getDate} 函数   getDate
     *
     * @apiDescription getDate(row, col [,  defaultValue])
     * <br><br> 对getValue的二次包装，先用getValue(row,col)获取数据，再转换成日期，如果为null或转换失败，则返回deaultValue
     *
     * @apiName  getDate
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {int} row 行号
     * @apiParam {int/String} col 列号或列名
     * @apiParam {getDate} defaultValue 如果结果集中的数据为null，则返回本默认值。可选参数，默认为null
     *
     * @apiSuccess (返回值){Date} -
     *
     * @apiExample   {js}示例：
     *
     *  ds.setValue(0,"name","abc");
     *  println( ds.getDate(0,"name")); // null
     *  ds.setValue(0,"name",'2018.01.1');
     *  println( ds.getDate(0,"rq")); // 2018/01/01
     *  ds.setValue(0,"rq",null);
     *  println( ds.getDate(0,"rq")); // null
     *
     *
     */
    getDate: function (/*int*/row, col, /*int*/defaultValue) {
        if (defaultValue == undefined) defaultValue = null;
        var obj = this.getValue(row, col);
        if (obj == null) return defaultValue;
        obj = ObjectTool.changeType(obj, UniformDataType.$Datetime);
        return obj;
    }
    ,

    /**
     * @api {getDouble} 函数   getDouble
     *
     * @apiDescription getDouble(row, col [,  defaultValue])
     * <br><br> 对getValue的二次包装，先用getValue(row,col)获取数据，再转换成小数，如果为null或转换失败，则返回deaultValue
     *
     * @apiName  getDouble
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {int} row 行号
     * @apiParam {int/String} col 列号或列名
     * @apiParam {double} defaultValue 如果结果集中的数据为null，则返回本默认值。可选参数，默认为0
     *
     * @apiSuccess (返回值){number} -
     *
     * @apiExample   {js}示例：
     *
     *  ds.setValue(0,"name","abc");
     *  println( ds.getDouble(0,"name")); // 0
     *  ds.setValue(0,"name","123");
     *  println( ds.getDouble(0,"name")); // 123.0
     *  ds.setValue(0,"name",null);
     *  println( ds.getDouble(0,"name")); // 0
     *
     *
     */
    getDouble: function (/*int*/row, col, /*int*/defaultValue) {
        if (defaultValue == undefined) defaultValue = 0;
        var obj = this.getValue(row, col);
        if (obj == null) return defaultValue;
        obj = ObjectTool.changeType(obj, UniformDataType.$Numeric);
        return obj;
    }
    ,

    /**
     * @api {evaluate} 函数   evaluate
     *
     * @apiDescription  evaluate(expression, row)
     * <br><br> 在row行中评估一个表达式的值。表达式中可以使用字段名参与运算。在评估表达式的值时，这此字段名用row行中相应字段的值
     * 来代入。
     *
     * @apiName  evaluate
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {String} expression  表达式
     * @apiParam {int} row 行号
     *
     * @apiSuccess (返回值){variant} - 返回表达式计算的结果
     *
     * @apiExample   {js}示例：
     *
     * var row=ds.insertRow(0);
     * ds.setValue(0,"sl",100);
     * ds.setValue(0,"dj", 12.34);
     * var je= ds.evaluate("sl*dj",0); // je=1234
     *
     */
    evaluate: function (expression, row) {
        if (expression.startsWith("=")) expression = expression.substring(1);
        return this.m_PrimaryBuffer.evaluate(expression, row);
    },


    /**
     * @api {setValue} 函数   setValue
     *
     * @apiDescription setValue(row,col,value)
     * <br><br>为指定的行列设置数据
     * <br>
     *   <ul>
     *       <li>行号+列号/列名 可以唯一定位一个数据单元。一个数据单元，它有三个属性：原始值（从数据库中检索出来的值），当前值，状态</li>
     *       <li>在将value设置到字段的当前值之前，先触发事件itemChangeAccept ，该事件如果返回false,则本函数终止，返回false </li>
     *       <li> 设置数据时，首先会将value转换成与col字段相同的数据类型，如果转换失败，则value被设置成null，row行col列单元格的当前值也被设置成空</li>
     *       <li>当value与字段当前的值一样时，本函数返回true，但并不做任何改变。</li>
     *       <li>在把字段的当前值设置为value后，根据当前值与原始值，设置该单元格的状态。然后触发itemChanged事件。</li>
     *       <li>如果有计算列引用col字段，首先将计算列设置成需要重算状态，但并不立即重新计算。并触发
     *         computerChanged事件。如果计算列需要回填，则立即重新评估该计算列的值，并回填到需回填的字段中。如果有UI绑定该字段，则UI刷新显示。</li>
     *   </ul>
     *
     * @apiName  setValue
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {int} row 行号
     * @apiParam {int/String} col 列号或列名
     * @apiParam {variant} value  任意类型的数据
     *
     * @apiSuccess (返回值){boolean} - true表示设置成功
     *
     * @apiExample   {js}示例：
     *
     *  ds.setValue(0,"rq", "aaa");//rq是Date型时，本函数将rq设置成null,因为'aaa'无法转换成日期。
     */
    setValue: function (row, col, v) {
        return this.m_PrimaryBuffer.setValue(row, col, v);
    },


    /**
     *       * 得到表达式中可能包含的列名称
     * @param expression_
     * @param dependColList
     *
     * */

    assembleDependColumns: function (expression_) {
        var dependColList = [];
        try
        {
            var expression = expression_.toLowerCase();
            //增加的缓存，以提高Filter的性能
            dependColList = this.m_Express2DependList[expression];
            if (dependColList != undefined) return dependColList;
            dependColList = [];


            var options = {attachComment: true, range: true, loc: true, sourceType: 'script', tokens: true};
            var tokens = Esprima.parse(expression, options).tokens;
            for (var i = 0; i < tokens.length; i++)
            {
                var token = tokens[i];
                if (token.type == 'Identifier')
                {
                    var col = token.value;
                    if (this.col2Index(col) >= 0) dependColList.push(col.toLowerCase());
                }
            }
        } catch (err)
        {
            alert(expression_ + "计算列定义错误：" + JSON.stringify(err));
        }

        this.m_Express2DependList[expression] = dependColList;
        return dependColList;
    },

    assembleToken: function (expression_) {
        var ret = [];
        try
        {
            var expression = expression_.toLowerCase();


            var options = {attachComment: true, range: true, loc: true, sourceType: 'script', tokens: true};
            var tokens = Esprima.parse(expression, options).tokens;
            for (var i = 0; i < tokens.length; i++)
            {
                var token = tokens[i];
                if (token.type == 'Identifier')
                {
                    var col = token.value;
                    ret.push(col);
                }
            }
        } catch (err)
        {
            alert(expression_ + "计算列定义错误：" + JSON.stringify(err));
        }

        return ret;
    },


    /**
     * @api {createComputer} 函数   createComputer
     *
     * @apiDescription createComputer(name, expression[, fillback])
     * <br><br>创建计算列。计算列是指一个列，它不是简单的字段，而是一个表达式。表达式中可以引用字段名进行四则运行，引用函数。
     *<br><b>注意：</b>
     * <ul>
     * <li>如果已经存在名为name的计算列，那么将不再创建新的计算列，而是修改该计算列的表达式为expression</li>
     * <li>如果指定了回填到fillback字段，那么计算列的值自动回填到fillback表明的字段中，可以实现计算列的值固化到字段中</li>
     * </ul>
     * @apiName  createComputer
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {String} name 计算列的名称
     * @apiParam {String} expression 表达式
     * @apiParam {String} fillback 将计算列的值回填到哪个字段中
     *
     * @apiSuccess (返回值){boolean} - true表示创建成功， false 表示创建失败
     *
     * @apiExample   {js}示例：
     *
     *   createComputer("c_je", "sj*dj" , "je");
     */
    createComputer: function (name, expression, fillback) {

        this.clearError();
        var colName = name.trim().toLowerCase();
        var /*ColumnProperty*/ pCP;
        var colIndex = this.col2Index(colName);
        if (colIndex >= 0)
        {
            pCP = this.getColumnProperty(colIndex);
            console.log(name + "已经存在，它将被重新定义。");
        } else
        {

            //第一步:增加列属性配置
            pCP = new ColumnProperty(this);

            //增加到列属性数组中
            this.m_ColumnProperty.push(pCP);
            //第二步:增加名称和索引的映射
            this.m_ColumnName2Index[colName] = this.m_ColumnProperty.length - 1;

            //第三步:增加缓冲区来盛放计算列的值

            //每一行都需要增加一个单元来放计算列的值
            this.m_PrimaryBuffer.AppendColumn();
            this.m_DeleteBuffer.AppendColumn();
            this.m_FilterBuffer.AppendColumn();

        }

        pCP.m_Name = colName;
        pCP.m_DBName = "";
        pCP.m_DBTable = "";
        pCP.m_ObjType = ObjectType.isUserComputer; //是自定义的计算列
        pCP.m_UpdateWhereClause = false;
        pCP.m_Updatable = false;
        pCP.m_ColumnLabel = colName;
        //计算列类型为未知道
        pCP.m_DataType = -9999;
        pCP.m_DataTypeName = "unknown";
        pCP.m_IncludeUnsureFunction = this.IncludeUnsureFunction(expression);

        pCP.m_FillBack = fillback || '';

        //设置公式
        pCP.m_Formula = expression;
        //处理依赖和影响关系
        var dc = this.assembleDependColumns(expression);
        var n = dc.length;
        for (var i = 0; i < n; i++)
        {
            var t2 = dc[i];
            colIndex = this.col2Index(t2);
            if (colIndex < 0)
            {
                this.onError("不存在列" + t2);
                return false;
            }

            var /*ColumnProperty*/  pDependCP = this.m_ColumnProperty[colIndex];
            if (pDependCP.m_AffectMap == null) pDependCP.m_AffectMap = [];
            if (pCP.m_DependMap == null) pCP.m_DependMap = [];
            pDependCP.m_AffectMap.push(pCP.m_Name); //某列影响自己
            pCP.m_DependMap.push(pDependCP.m_Name); //自己依靠某列
        }

        try
        {
            //增加以属性方式读取数据
            let tempCol = colName;
            Object.defineProperty(this.valueVisitor.prototype, colName.toLowerCase(), {
                set: function (newValue) {

                    console.log("计算列是只读的");
                },
                get: function () {

                    var v = this.m_ds.getValue(this.m_row, tempCol);
                    return v;
                }
            });
        } catch (err)
        {

        }

        return true;
    },

//

    /**
     * 是不是包含非确定性函数
     * @param Expression  公式
     * @returns {boolean}
     * @constructor
     */
    IncludeUnsureFunction: function (expression) {

        var UnsureFunction = ["thisrow", "rowcount"];


        var dc = this.assembleToken(expression);

        for (var i = 0; i < dc.length; i++)
        {
            var col = dc[i].toLowerCase();
            if (UnsureFunction.contains(col)) return true;
        }

        return false;
    },


    onError: function (msg,code) {
        if( code==undefined) code=-1;
        this.m_Error = {code: code, message: msg};
        this.EM.fire("onError", [code, msg]);
    },

    /**
     * 创建一个汇总列
     * @param aType
     * @param groupNo
     * @param name
     * @param expression
     * @returns {*}
     */
    createAggregate: function (/*int*/    aType, /*int*/ groupNo, /*String*/ name, /*String*/  expression) {
        this.clearError();
        var b = this.createComputer(name, expression);
        if (!b) return b;

        var /*ColumnProperty*/   pCP;
        var colIndex = this.col2Index(name);
        if (colIndex < 0)
        {
            this.onError("非法的聚合列");
            return false;
        }
        pCP = this.m_ColumnProperty[colIndex];
        pCP.m_ObjType = ObjectType.isAggregateComputer;
        pCP.m_AggregateDefine = new AggregateDefine(groupNo, aType);
        return true;
    },


    /**
     * @api {insertRow} 函数   insertRow
     *
     * @apiDescription  insertRow(beforeRow）
     * <br><br> 在指定行之前插入一行
     *
     * <br>细节</b><br>
     * <ul>
     *  <li>插入前会触发 insertRowPermit事件，如果此事件返回false ,则插入操作被终止</li>
     *  <li>插入行后，如果列定义了默认值，则该值被填充到字段中。如果默认值是以=开头，表示是一个表达式，则先计算该表达式的值，再填入字段中</li>
     *  <li>最后触发 afterInsertRow 事件</li>
     *
     * </ul>
     * @apiName  insertRow
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {int} beforeRow 在哪行之前插入。行号从0开始计
     *
     *
     * @apiSuccess (返回值){int} - 返回新插入行的行号
     *
     * @apiExample   {js}示例：
     *
     * var row= ds.insertRow(0); //在最前面插入一行
     * row=ds.insertRow( ds.rowCount);//在最后面插入行
     *
     */
    insertRow: function (beforeRow) {

        var rc = this.rowCount;
        var cc = this.columnCount;

        if (beforeRow < 0) beforeRow = 0;
        if (beforeRow >= rc) beforeRow = rc;

        if (!this.EM.fire("insertRowPermit", [beforeRow], true)) return -1;

        var /*DsRow*/   tempRow = new DsRow(this, cc);
        //新插入的数据行的初始状态设置成 isNew
        tempRow.setRowStatus(ItemStatus.isNew);
        var thisRow = this.m_PrimaryBuffer.InsertRow(beforeRow, tempRow);

        // 增加, 保证在插入一行后，分组信息仅仅保留总计
        this.BuildAllAsOneGroup(this.m_VectorGroups);

        this.m_PrimaryBuffer.notifyAggregateAndUnsureComputerInvalid();

        // 填充缺省值
        for (var i = 0; i < cc; i++)
        {
            var /*ColumnProperty*/  pCP = this.getColumnProperty(i);
            var defaultValue = pCP.getDefaultValue();
            if (defaultValue == null) continue;
            var s = ('' + defaultValue).trim();

            //如果是公式，那么先计算它
            if (s.startsWith("="))
            {
                s = this.evaluate(s.substring(1, s.length), thisRow);
            }
            //2019.08.16
            if (s == '') continue; //避免出现 null的字段被''填入导致出现需要保存，
            this.setValue(thisRow, i, s);
        }

        //v把触发事件放到填充缺省值之后，避免在afterinsert事件里取一些数据时还没有数据
        this.EM.fire("afterInsertRow", [thisRow]);

        return thisRow;

    },


    /**
     * @api {deleteRow} 函数   deleteRow
     *
     * @apiDescription deleteRow (row [ ,onlyFromBuffer] )
     * <br><br> 删除指定的行
     *  <br>细节</b><br>
     * <ul>
     *  <li>删除前会触发 beforeDeleteRow， 再触发 deleteRowPermit 事件，如果deleteRowPermit事件返回false ,则删除操作被终止</li>
     *  <li>数据从主缓冲区删除后，如果onlyFromBuffer=false ，删除的数据会放到删除缓冲区，否则不放入删除缓冲区
     *  </li>
     *  <li>最后触发 afterDeleteRow 事件</li>
     *
     * </ul>
     * @apiName  deleteRow
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {int} row 行号
     * @apiParam {boolean} onlyFromBuffer  true:表示从主缓冲区删除后，并不放入删除缓冲区缓冲区中 .false  表示从主缓冲区删除后，
     * 移到删除缓冲区。本参数是可选参数，默认是false
     *
     * @apiSuccess (返回值){boolean} - true表示删除成功
     *
     *
     *
     * @apiExample   {js}示例：
     *
     *  var rc= ds.getRowCount(); //客户端脚本中可以简写成 rc= ds.rowCount;
     *  for ( var row= rc-1; row>=0;row--) //注意：如果是删除全部数据 ，需要从后往前删除
     *  {
         *     ds.deleteRow(row);
         *  }
     *
     */
    deleteRow: function (row, onlyFromBuffer) {
        if (onlyFromBuffer == undefined) onlyFromBuffer = false;
        this.clearError();
        var rc = this.rowCount;

        if (row < 0 || row >= rc)
        {
            this.onError("DeleteRow 行号超界.");
            return false;
        }

        //  增加删除前事件，此时确信要删除了，但是尚未删除
        //一定要放在 fireDeleteRowPermit前面，　在这里可以做一些处理，并可能影响下面的是否允许继续删除
        //比如连动删除中，在beforeDelete中把子数据删除，之后，在　deleteRowPermit事件中可以检测子数据是否还在（比如不允许删除）
        // 然后要以此控制自己的删除是否还能继续
        this.EM.fire("beforeDeleteRow", [row]);

        var permit = this.EM.fire("deleteRowPermit", [row], true);
        if (!permit) return false; //如果不允许删除,那么直接返回

        //  新行,或者新修改的行直接删除,不需要放入到删除缓冲区中.
        var /*DsRow*/ dr = this.m_PrimaryBuffer.getRow(row);

        //如果仅仅是从缓冲区中删除，表示并不在数据库中删除，那么就不要移到删除缓冲区中，因为删除缓冲中的数据表示数据库中需要删除
        if (!onlyFromBuffer)
        {
            if (dr.m_RowStatus != ItemStatus.isNew && dr.m_RowStatus != ItemStatus.isNewModified) this.m_DeleteBuffer.push(dr);
        }

        this.m_PrimaryBuffer.eraseRow(row);

        //  增加事件触发，好让客户程序
        //　做同步响应，比如在AfterDeleteRow事件中删除界面行（列邦定模式下）
        //  或者重新设置邦定行(稀疏绑定模式下)

        // 增加, 保证在插入一行后，分组信息仅仅保留总计
        this.BuildAllAsOneGroup(this.m_VectorGroups);

        this.m_PrimaryBuffer.notifyAggregateAndUnsureComputerInvalid();
        //  增加结束

        this.EM.fire("afterDeleteRow", [row]);

        return true;

    },


    /**
     出于兼容性，请使用属性updatable
     * @param b
     */


    /**
     * @api {setUpdatable} 函数   setUpdatable
     *
     * @apiDescription  setUpdatable(b)
     * <br><br> 设置结果集是否可以更新 .本函数是属性  updatable的 setter函数。服务端脚本无法使用属性，请使用本函数
     *
     * @apiName  setUpdatable
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     *
     * @apiParam {boolean} b  true 表示允许更新
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *   ds.setUpdatable(true);
     *
     */
    setUpdatable: function (b) {
        this.m_Updatable = b;
    },


    /**
     * @api {isUpdatable} 函数   isUpdatable
     *
     * @apiDescription  isUpdatable()
     * <br><br> 本结果集是否可以更新 。 它是属性updatable的 getter函数。服务端脚本无法使用属性，请使用本函数
     *
     * @apiName  isUpdatable
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     *
     * @apiSuccess (返回值){boolean} - 返回本结果集是否可以更新
     *
     * @apiExample   {js}示例：
     *
     * if( !ds.isUpdatable()) alert("禁止更新 ");
     *
     */
    isUpdatable: function () {
        return this.m_Updatable;
    },


    /**
     * @api {setUpdatableTable} 函数   setUpdatableTable
     *
     * @apiDescription setUpdatableTable(table)
     * <br><br>设置可更新的表名称
     *
     * @apiName  setUpdatableTable
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {String} table 可更新的表名称
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *  var ds=newDataStore("","selec * from t1");
     *  ds.seUpdatableTable("t1");
     *
     */
    setUpdatableTable: function (table) {

        this.m_UpdateTable = table.trim();
        var cc = this.columnCount;
        for (var i = 0; i < cc; i++)
        {
            var /*ColumnProperty*/ cp = this.getColumnProperty(i);

            if (cp.m_ObjType == ObjectType.isColumn && cp.m_DBTable == '')
            {
                cp.m_DBTable = table;
            }
        }
    },


    /**
     * @api {getUpdatableTable} 函数   getUpdatableTable
     *
     * @apiDescription getUpdatableTable()
     * <br><br>可更新的表名称 ，本函数是属性updatableTable的 getter函数
     *
     * @apiName  getUpdatableTable
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *

     * @apiSuccess (返回值){int} - 返回可更新的表名称
     *
     * @apiExample   {js}示例：
     *
     *  var table=ds.getUpdatableTable();
     */
    getUpdatableTable: function () {
        return this.m_UpdateTable;
    },

    /**
     * 设置表的主键，组合主键用，分隔各个列，比如　  id, name
     * @param pk
     */

    /**
     * @api {setPrimaryKey} 函数   setPrimaryKey
     *
     * @apiDescription setPrimaryKey(pk)
     * <br><br>设置主键,本函数是属性primaryKey的setter函数
     *
     * @apiName  setPrimaryKey
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {String}  pk 主键
     *
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     * ds.setPrimaryKey("id");
     *
     */
    setPrimaryKey: function (pk) {
        var s = pk;
        s = s.replace(/\s/g, '');
        this.m_PrimaryKey = s;
    },

    /**
     * @api {getPrimaryKey} 函数   getPrimaryKey
     *
     * @apiDescription getPrimaryKey
     * <br><br>得到主键,本函数是属性primaryKey的getter函数
     *
     * @apiName  getPrimaryKey
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     *
     * @apiSuccess (返回值){String} - 返回主键
     *
     * @apiExample   {js}示例：
     *
     *  ds.getPrimaryKey();
     */
    getPrimaryKey: function () {
        return this.m_PrimaryKey;
    },


    /**
     * @api {setUpdatableColumns} 函数   setUpdatableColumns
     *
     * @apiDescription setUpdatableColumns(cols)
     * <br><br> 设置可更新的列，本函数是追加性设置，不是覆盖性，即多闪设置的效果是叠加的
     *
     * @apiName  setUpdatableColumns
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {String} cols 可更新的字段，多个字段间用英文逗号分隔，如果字段名前加上～则表示字段不可更新
     *
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *   ds.setUpdatableColumns("id");
     *   ds.setUpdatableColumns("name");
     *   //上面两句等效于下一句
     *   ds.setUpdatableColumns("id|name");
     *
     *   ds.setUpdatableColumns("~name");//将name设置成不可更新
     *
     */
    setUpdatableColumns: function (cols) {
        this.clearError();

        if (this.m_UpdateTable == (""))
        {
            this.onError("设置 UpdatableColumns前,先设置UpdateTable");
            return;
        }


        var s = cols;
        s = s.replace(/\s/g, '').toLowerCase();


        this.m_UpdatableColumns = s;

        var cols = s.split('|');

        var updatableCols = [];
        var unUpdatableCols = [];


        for (var i = 0; i < cols.length; i++)
        {
            var col = cols[i];
            var b = col.startsWith("~");
            if (b) col = col.substring(1);
            col = col.toLowerCase().trim();
            if (!b)
            {
                updatableCols.push(col);
            } else
            {
                unUpdatableCols.push(col);
            }
        }

        var cc = this.columnCount;
        for (var i = 0; i < cc; i++)
        {

            var /*ColumnProperty*/pcp = this.getColumnProperty(i);

            //如果不是列，那么忽略它
            if (pcp.m_ObjType != ObjectType.isColumn) continue;

            // 如果表不是可以更新的表,那么该列是不可以保存的

            // 增加对　update  database.table  set ... 语法的支持。即更新同一个服务器中另一个数据库中的表
            if (pcp.m_DBTable.toLowerCase() != this.m_UpdateTable.toLowerCase()
                && ( pcp.m_DatabaseName + "." + pcp.m_DBTable).toLowerCase() != this.m_UpdateTable.toLowerCase())
            {
                pcp.m_Updatable = false;
                continue;
            }

            if (unUpdatableCols.contains(pcp.m_DBName.toLowerCase()))
            {
                pcp.m_Updatable = false;
                continue;
            }

            if (updatableCols.contains(('*') || updatableCols.contains(pcp.m_DBName.toLowerCase())))
            {
                pcp.m_Updatable = true;
            }

        }
    },


    /**
     * @api {getUpdatableColumns} 函数   getUpdatableColumns
     *
     * @apiDescription getUpdatableColumns()
     * <br><br> 得到可以更新的列
     *
     * @apiName  getUpdatableColumns
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     *
     * @apiSuccess (返回值){String} - 得到可以更新的列
     *
     * @apiExample   {js}示例：
     *
     *
     *   var ds=newDataStore("","select id, name ,code from t1");
     *   ds.setUpdatableTable("t1");
     *   ds.setUpdatableColumns("id|name|~code");
     *   var t=ds.getUpdatableColumns( );// t=="id|name"
     *
     */
    getUpdatableColumns: function () {
        var /*String*/            ret = [];
        var /*ColumnProperty*/         pcp = null;
        var  n = this.columnCount;

        for (var i = 0; i < n; i++)
        {
            pcp = this.getColumnProperty(i);
            if (pcp.m_Updatable) ret.push(pcp.m_Name);

        }
        return ret.join("|");
    },


    /**
     * @api {setUpdateWhereClauseColumns} 函数   setUpdateWhereClauseColumns
     *
     * @apiDescription setUpdateWhereClauseColumns(colsString)
     * <br><br> <h3>设置参与构造Where子句的列，控制多用户的并发更新</h3>
     * <br>
     *
     *
     * 当设置某个列不参与构造Where子句时，只需要在列名字前加上&tilde;符号，
     * 比如 id|name|&tilde;code 表示 id, name参与构造Where子句，code不参与构造Where子句。
     * <br><i><font color=red>注意：本函数是追加性设置，不是覆盖性设置,详见示例</font></i>
     *
     * <br><b>详细说明：</b>
     * <br>
     * 当有多个用户同时修改相同的数据时， 如何进行控制呢?
     * 比如：000001号单据上的总金额为 500。这时，A，B两个操作员同时将该单据打开修改，A将总金额修改成 1000 ，
     * 并保存。B将总金额修改成2000 ，此时，在数据库中，该单据的总金额已经变成了 1000，而不是最初检索出来的 500 。那么，
     * 是否允许 B 操作员保存修改呢？
     * <ul>
     * <li>
     *   策略一：允许B保存修改，这样带来的结果是：多人同时打开数据进行修改时，后修改的将覆盖先修改的，以最后一个保存的结果为最终结果。
     * </li>
     *     <li>
     *   策略二：不允许 B 保存修改，这样带来的结果是：多人同时打开数据进行修改时，最先保存的数据是有效数据，其后的更新操作都失败。
     * </li>
     * </ul>
     *
     * DataStore 是通过 UpdateWhereClauseColumns 属性来控制多用户的并发更新。该属性定义了数据修改翻译成 Update
     * 语句中的where 子句的结构。比如 UpdateWhereClauseColumns =&quot; id | name | je &quot;  ,那么当用户修改了
     * 数据并调用Update 函数作保存时，被修改的数据被解释成  Update  ???   Set   col1= ??? , ...
     * Where  id=? and  name=? and je= ? 。 比如表 t1 中有列 id, name ,code, je  ,   检索出来某行数据
     * 为 id=1, name='wise', code='01', je=500 。A 、B 操作员同时将该数据检索出来修改。 A操作员将 je 修改成 1000 ,
     * 当A作保存时 ， DataStore 将A所做的操作解释成  Update  t1 set je=500 Where id=1 and name='wise' and je=500 。
     * 当A 保存成功后，数据库中该记录的je变成了1000 . 这时，B操作员将je  修改成 2000 , 当B作保存时 ， DataStore
     * 将B所做的操作解释成  Update  t1 set je=2000 Where id=1 and name='wise' and je=500 。显然 B操作员的操作不
     * 会生效。因为数据库中已经不存在 id=1 并且 name='wise' 并且 je=500  的记录了。这样实现了策略二的要求。如果令
     * UpdateWhereClauseColumns= &quot;id &quot; 那么 B 操作员的操作被解释成  Update  t1 set je=2000 Where id=1 。
     * 这时B的操作生效，实现策略一的要求。
     * <br>
     * 所以，出现在 UpdateWhereClauseColumns中的列越多，并发控制越严格。，出现在 UpdateWhereClauseColumns中的列越少，
     * 并发控制越宽松 。但表的主键必须出现在 UpdateWhereClauseColumns中。 比如表 t2 的主键是 name 和 code 组合而成，
     * 那么 name | code 必须出现在UpdateWhereClauseColumns中，否则无法正确更新数据。
     *
     * </br>
     *
     * 值得注意的是：DataStore 能够通过UpdateWhereClauseColumns来控制多用户同时多同一数据的更新，但对多用户对同一数据
     * 的删除不作控制，也就是说：如果Ａ操作员删除了单据 000001 , 然后　B 操作员再删除单据 000001 ,　虽然 B 操作员的删除
     * 操作没有删除任何数据（因为要删除的数据已经被删除），但系统仍然认为　B 操作员执行成功（因为他的目的的确已经达到）。
     *
     *
     *
     * @apiName  setUpdateWhereClauseColumns
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {String} colsString 字段清单，多个列之间用竖线分隔，比如 id|name|code
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     * ds.setUpdateWhereClauseColumns("id");
     * ds.setUpdateWhereClauseColumns("name");
     * ds.setUpdateWhereClauseColumns("code");
     * //此时 id, name ,code 3个字段参与 where子句的构建
     * ds.setUpdateWhereClauseColumns("~code");
     * //此时 id, name   2个字段参与 where子句的构建
     *
     */
    setUpdateWhereClauseColumns: function (colsString) {
        this.clearError();
        if (this.m_UpdateTable == "")
        {
            this.onError("先设置UpdateTable");
            return;
        }

        var s = colsString;
        s = s.replace(/\s/g, '').toLowerCase();


        //	拼装上 | 后保存到 m_UpdatableColumns;

        this.m_UpdatableColumns = s;

        var cols = s.split('|');

        var updatableCols = [];
        var unUpdatableCols = [];


        for (var i = 0; i < cols.length; i++)
        {
            var col = cols[i];
            var b = col.startsWith("~");
            if (b) col = col.substring(1);
            col = col.toLowerCase().trim();
            if (!b)
            {
                updatableCols.push(col);
            } else
            {
                unUpdatableCols.push(col);
            }
        }

        var m = this.columnCount;

        for (var i = 0; i < m; i++)
        {
            var /*ColumnProperty*/ pcp = this.getColumnProperty(i);

            if (pcp.m_DBTable.toLowerCase() != this.m_UpdateTable.toLowerCase())
            {
                pcp.m_UpdateWhereClause = false;
                continue;
            }


            if (updatableCols.contains(pcp.m_DBName.toLowerCase()) || updatableCols.contains('*')) pcp.m_UpdateWhereClause = true;
            if (unUpdatableCols.contains(pcp.m_DBName.toLowerCase())) pcp.m_UpdateWhereClause = false;


        }
    },

    /**
     * @api {getUpdateWhereClauseColumns} 函数   getUpdateWhereClauseColumns
     *
     * @apiDescription getUpdateWhereClauseColumns ()
     * <br><br>得到构建where子句的列名
     *
     * @apiName  getUpdateWhereClauseColumns
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     *
     * @apiSuccess (返回值){String} - 得到构建where子句的列名
     *
     * @apiExample   {js}示例：
     *
     *  alert( ds.getUpdateWhereClauseColumns());
     */
    getUpdateWhereClauseColumns: function () {

        var /*String*/            ret = [];
        var /*ColumnProperty*/         pcp = null;
        var   n = this.columnCount;

        for (var i = 0; i < n; i++)
        {
            pcp = this.getColumnProperty(i);
            if (pcp.m_UpdateWhereClause) ret.push(pcp.m_Name);

        }
        return ret.join("|");

    },


    /**
     * @api {update} 函数   update
     *
     * @apiDescription  update(resetFlag)
     * <br><br> 提交所有修改到数据库
     * <br>细节</br>
     * <ul>
     *     <li>对每一个修改过的行，触发 validate事件，在该事件中可以做一些校验。如果返回 false,则更新失败，本函数返回false</li>
     *   <li>然后将修改过的主缓冲区中数据被翻译成insert 或 update 语句， 删除缓冲区中的数据被翻译成delete语句，没有修改过
     *   的语句不做处理 ，这些 insert , update , delete语句被放到一个SQL数组</li>
     *
     *   <li>所有  executeBeforeUpdate 中的SQL语句被插入到 SQL数组之前</li>
     *   <li>所有  executeAfterUpdate 中的SQL语句被插入到 SQL数组之后</li>
     *   <li>所有这些语句在一个数据库事务中一并执行，要么全部成功，要么全部回滚.所能这要求所有SQL语句是在一个数据库服务器中执行的。</li>
     *   <li>多个DataStore 可以用全局函数   updateAll( ds1,ds2....) 来保证它们在一个事件中更新。详情查看全局函数updateAll </li>
     *
     * </ul>
     *
     * @apiName  update
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {boolean} resetFlag 是否复位标记 .可选参数，默认是true
     *
     * @apiSuccess (返回值){boolean} - 是否提交成功
     *
     * @apiExample {js}示例
     *
     * if(!ds.update(true)) alert( ds.error.message);
     *
     *
     * @apiExample   {js}示例2：
     *
     *
     * if( ds.update(false) &&  ds2.upate(false))
     * {
     *    ds.resetUpdate();
     *    ds2.resetUpdate();
     * }else
     * {
     *   alert( ds.error.message + ds2.error.message);
     * }
     *
     * @apiExample   {js}示例3：
     *
     *  var err= updateAll( ds1, ds2, ds3 ... ); // 最多可以20个一起更新
     *  if( err=='')
     *  {
     *    alert(' success ');
     *  }else
     *  {
     *    alert('error: '+  err);
     *  }
     *
     *
     */
    update: function (resetFlag) {

        if (resetFlag == undefined) resetFlag = true;
        var list = [];
        if (!this.getUpdateSQL(list)) return false;

        if (list.length == 0) return true; //不需要保存

        var result = this.m_DataProvider.update(this.name, this.id, this.templateid, this.m_Connection, list);

        if (!result.success)
        {
            this.onError(result.message);
            return false;
        }

        // 重新设置单元状态标志
        if (resetFlag) this.resetUpdate();
        return true;
    },


    /**
     * @api {getUpdateSQL} 函数   getUpdateSQL
     *
     * @apiDescription getUpdateSQL (vectorSql)
     * <br><br> 针对每个需要保存的行触发 Validate事件，在校验通过的情况下，把构造好的语句放入到 vectorSql中备用,
     * SQL语句是已经做过 toLocalSyntax的
     *
     * @apiName  getUpdateSQL
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam { Array} vectorSql 用来存放SQL语句的数组对象
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *
     * var sqls=[];
     *
     * ds1.getUpdateSQL(sqls);
     * ds2.getUpdateSQL(sqls);
     *
     * //把两个结果集提交的语句放到一起。
     *
     */
    getUpdateSQL: function (vectorSql) {


        if (vectorSql == null) return false;

        //update前的批量执行语句先放入
        for (var i = 0; i < this.sqlBatch.length; i++)
        {
            var s = this.sqlBatch[i];
            s = this.m_Dialect.toLocalSyntax(s);
            vectorSql.push(s);

            //为什么不用 vectorSql.addAll(sqlBatch);，因为要对其中的语句做toLocalSyntax
            //为什么不在sqlBatch中保存toLocalSyntax后的语句呢，1请查看execute函数的注释
            // 2是因为本函数可能多次调用，不能让sqlBatch中的语句被多次toLocalSyntax
        }

        //为什么不在结果集不允许更新时，直接return呢
        //因为可能使用 .executeBeforeUpdate 或executeAfterUpdate 添加了额外执行的SQL，这些SQL是允许 执行的
        //所以不能直接return 了事
        if (this.m_Updatable)
        {
            //---------------------------------------------------------------------------------------------
            //  在Update前复位 m_MaxIDMap ,这样强制 FillInMaxIDSafely 重新取得最大的ID
            //
            //
            //---------------------------------------------------------------------------------------------
            // m_MaxIDMap.clear();

            var n = this.m_DeleteBuffer.getRowCount();

            for (var row = 0; row < n; row++)
            {
                var sql = this.m_DeleteBuffer.AsserbleSQLDelete(row);
                if (sql == "") continue;
                //		 转换成数据库的字符集字符
                // if (m_CharsetAutoConvert) sql = DataStoreFactory./*String*/CharsetConvert(sql, m_Charset, m_DBCharset);
                //  因为加了多表联动，可能存在需要进行把数据多次过滤并多次调用　本函数
                //　就样，删除缓冲的解析就会多次，因此加入了不重复处理。
                if (!vectorSql.contains(sql)) vectorSql.push(sql);
            }


            n = this.m_PrimaryBuffer.getRowCount();
            var /*boolean*/ validateOk = true;
            for (var row = 0; row < n; row++)
            {
                var sqlList = [];
                var sql = '';

                var /*DsRow*/pRow = this.m_PrimaryBuffer.getRow(row);
                switch (pRow.m_RowStatus)
                {
                    case ItemStatus.isNew:
                    case ItemStatus.isNotModified:

                        break;
                    case ItemStatus.isNewModified:
                        if (!this.isTheRowHoldNoDataExceptSuchColumns(row))
                        {
                            validateOk = this.EM.fire("validate", [row], true);
                            if (!validateOk)
                            {
                                this.onError("validate 校验失败");
                                return false;
                            }
                            // 自动校验
                            if (!this.AutoCheck(row))
                            {
                                this.onError("AutoCheck 校验失败");
                                return false;
                            }
                            sqlList = this.m_PrimaryBuffer.AsserbleSQLInsert(row);
                        }
                        break;
                    case ItemStatus.isDataModified:
                        //  如果除了指定的字段，其它字段没有数据，那么表示要删除数据
                        //删除操作前，要做删除允许测试
                        if (this.isTheRowHoldNoDataExceptSuchColumns(row) && this.isAutoDeleteNoDataRow())
                        {
                            validateOk = this.EM.fire("deleteRowPermit", [row]);
                            if (!validateOk) return false;

                            sql = this.m_PrimaryBuffer.AsserbleSQLDelete(row);
                            Logger.info("第" + row + "行是检索出来的数据，在被修改后可被视为无数据的行，因此本行被删除。");
                            sqlList.push(sql);
                        } else
                        {
                            validateOk = this.EM.fire("validate", [row], true);
                            if (!validateOk)
                            {

                                this.onError("validate 校验失败");
                                return false;
                            }
                            // 自动校验
                            if (!this.AutoCheck(row))
                            {
                                this.onError("AutoCheck 校验失败");
                                return false;
                            }
                            sqlList = this.m_PrimaryBuffer.AsserbleSQLUpdate(row);
                        }
                        break;
                }

                for (var k = 0; k < sqlList.length; k++)
                {
                    sql = sqlList[k];
                    //	转换成数据库的字符集字符
                    //if (m_CharsetAutoConvert) sql = DataStoreFactory./*String*/CharsetConvert(sql, m_Charset, m_DBCharset);
                    if (!vectorSql.contains(sql)) vectorSql.push(sql);
                }

            }
        }

        //update后的批量执行语句最后放入
        for (var i = 0; i < this.sqlBatch2.length; i++)
        {
            var s = this.sqlBatch2[i];
            s = this.m_Dialect.toLocalSyntax(s);
            //注意，不能加上 ！vectorSql.contains(sql)的判断，这里的SQL是手工添加的，可能存在需要反复执行的语句
            vectorSql.push(s);

        }

        //sqlPreview 并不影响执行，所以可以在新线程中打印SQL语句,不要阻塞主线程的执行
        setTimeout(function () {
            for (var i = 0; i < vectorSql.length; i++)
            {
                this.EM.fire("sqlPreview", [vectorSql[i]]);
            }
        }.bind(this), 100);

        return true;
    },


    getUpdateCommand: function (vectorSql) {


        if (vectorSql == null) return false;


        //为什么不在结果集不允许更新时，直接return呢
        //因为可能使用 .executeBeforeUpdate 或executeAfterUpdate 添加了额外执行的SQL，这些SQL是允许 执行的
        //所以不能直接return 了事
        if (this.m_Updatable)
        {
            //---------------------------------------------------------------------------------------------
            //  在Update前复位 m_MaxIDMap ,这样强制 FillInMaxIDSafely 重新取得最大的ID
            //
            //
            //---------------------------------------------------------------------------------------------
            // m_MaxIDMap.clear();

            var n = this.m_DeleteBuffer.getRowCount();

            for (var row = 0; row < n; row++)
            {
                var sql = this.m_DeleteBuffer.AsserbleCommandDelete(row);
                if (sql == null) continue;
                vectorSql.push(sql);
            }


            n = this.m_PrimaryBuffer.getRowCount();
            var /*boolean*/ validateOk = true;
            for (var row = 0; row < n; row++)
            {
                var sqlList = [];
                var sql;

                var /*DsRow*/pRow = this.m_PrimaryBuffer.getRow(row);
                switch (pRow.m_RowStatus)
                {
                    case ItemStatus.isNew:
                    case ItemStatus.isNotModified:

                        break;
                    case ItemStatus.isNewModified:
                        if (!this.isTheRowHoldNoDataExceptSuchColumns(row))
                        {
                            validateOk = this.EM.fire("validate", [row], true);
                            if (!validateOk)
                            {
                                this.onError("validate 校验失败");
                                return false;
                            }
                            // 自动校验
                            if (!this.AutoCheck(row))
                            {
                                this.onError("AutoCheck 校验失败");
                                return false;
                            }
                            sqlList = this.m_PrimaryBuffer.AsserbleCommandInsert(row);
                        }
                        break;
                    case ItemStatus.isDataModified:
                        //  如果除了指定的字段，其它字段没有数据，那么表示要删除数据
                        //删除操作前，要做删除允许测试
                        if (this.isTheRowHoldNoDataExceptSuchColumns(row) && this.isAutoDeleteNoDataRow())
                        {
                            validateOk = this.EM.fire("deleteRowPermit", [row]);
                            if (!validateOk) return false;

                            sql = this.m_PrimaryBuffer.AsserbleCommandDelete(row);
                            Logger.info("第" + row + "行是检索出来的数据，在被修改后可被视为无数据的行，因此本行被删除。");
                            sqlList.push(sql);
                        } else
                        {
                            validateOk = this.EM.fire("validate", [row], true);
                            if (!validateOk)
                            {

                                this.onError("validate 校验失败");
                                return false;
                            }
                            // 自动校验
                            if (!this.AutoCheck(row))
                            {
                                this.onError("AutoCheck 校验失败");
                                return false;
                            }
                            sqlList = this.m_PrimaryBuffer.AsserbleCommandUpdate(row);
                        }
                        break;
                }

                for (var k = 0; k < sqlList.length; k++)
                {
                    sql = sqlList[k];
                    //	转换成数据库的字符集字符
                    //if (m_CharsetAutoConvert) sql = DataStoreFactory./*String*/CharsetConvert(sql, m_Charset, m_DBCharset);
                    if (!vectorSql.contains(sql)) vectorSql.push(sql);
                }

            }
        }


        //sqlPreview 并不影响执行，所以可以在新线程中打印SQL语句,不要阻塞主线程的执行
        setTimeout(function () {
            for (var i = 0; i < vectorSql.length; i++)
            {
                this.EM.fire("sqlPreview", [vectorSql[i]]);
            }
        }.bind(this), 100);

        return true;
    },

    /**
     * @api {isSaveNeeded} 函数   isSaveNeeded
     *
     * @apiDescription isSaveNeeded()
     * <br><br> 返回本DataStore 是否需要保存
     *
     * @apiName  isSaveNeeded
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *

     * @apiSuccess (返回值){boolean} - 返回本DataStore 是否需要保存
     *
     * @apiExample   {js}示例：
     *
     *
     * if( ds.isSaveNeeded()) alert("数据发生了修改，请注意保存");
     *
     */
    isSaveNeeded: function () {
        if (this.m_DeleteBuffer.getRowCount() > 0) return true;

        var n = this.m_PrimaryBuffer.getRowCount();
        //主缓冲区
        for (var i = 0; i < n; i++)
        {

            var rowStatus = this.m_PrimaryBuffer.getRow(i).getRowStatus();
            //如果该行是新修改的行
            if (rowStatus == ItemStatus.isNewModified)
            { // 如果除了指定字段外还有其字段有数据，那么才是需要保存的
                if (!this.isTheRowHoldNoDataExceptSuchColumns(i)) return true;
            }

            //如果是被修改的行 ,那么需要保存
            if (rowStatus == ItemStatus.isDataModified) return true;

        }

        //过滤缓冲区
        n = this.m_FilterBuffer.getRowCount();
        for (var i = 0; i < n; i++)
        {

            rowStatus = this.m_FilterBuffer.getRow(i).getRowStatus();
            if (rowStatus == ItemStatus.isNewModified)
            { // 如果除了指定字段外还有其字段有数据，那么才是需要保存的
                if (!this.isTheRowHoldNoDataExceptSuchColumns(i)) return true;
            }

            //如果是被修改的行 ,那么需要保存
            if (rowStatus == ItemStatus.isDataModified) return true;

        }

        return false;
    },


    AutoCheck: function (row) {
        var RC = this.getRowCount();
        for (var i = 0; i < this.m_VectorAutoCheck.length; i++)
        {
            var /*AutoCheckConfig*/    acc = this.m_VectorAutoCheck[i];
            switch (acc.type)
            {
                case AutoCheckType.CHECK_NULL:

                    var ret = false;
                    if (this.getValue(row, acc.col) != null)
                    {
                        if (this.getString(row, acc.col).trim() != "") ret = true;
                    }

                    if (ret) break;

                    var CP = this.getColumnProperty(acc.col);

                    var info = "必须录入" + (RC == 1 ? "" : "第" + (row + 1) + "行中的") + CP.getColumnLabel();
                    Logger.info(info);
                    this.EM.fire("autoCheckError", [this.getUpdatableTable(), row, this.getColumnName(acc.col), acc.type, info]);
                    return false;

                default:
                    this.EM.fire("autoCheckError", [this.getUpdatableTable(), row, this.getColumnName(acc.col),
                        acc.type, "尚未实现的自动校验功能："]);
                    return false;
            }
        }
        return true;
    },


    isColNeedCheckNull: function (col) {
        var colIndex = this.col2Index(col);
        if (colIndex < 0) return false;

        for (var i = 0; i < m_VectorAutoCheck.length; i++)
        {
            var /*AutoCheckConfig*/            acc = this.m_VectorAutoCheck[i];
            if (acc.col != colIndex) continue;
            if (acc.type == AutoCheckType.CHECK_NULL) return true;
        }

        return false;
    },


    /**
     * 注册自动检测
     * @param col  被检测的列
     * @param checkType   检测类型
     * <p>可以是AutoCheckConfig中定义的类型
     * <table border="1" width="100%" id="table1" cellspacing="0" cellpadding="0" bordercolorlight="#808080" bordercolordark="#FFFFFF">
     * <tr><td>AutoCheckConfig.CHECK_NULL    = 1</td><td> 数据不能为空的校验</tr>
     * <tr><td>AutoCheckConfig.CHECK_UNIQUE    = 2</td><td>数据唯一性的校验</tr>
     * <tr><td>AutoCheckConfig.CHECK_UNIQUE_WARN    = 4</td><td>数据可选择的唯一性校验</tr>
     * <tr><td>AutoCheckConfig.CHECK_UNIQUE_UNLESS_NULL    = 8</td><td>数据必须唯一，除非为空</tr>
     * <tr><td>AutoCheckConfig.CHECK_UNIQUE_WARN_UNLESS_NULL    = 16</td><td>数据最好唯一，除非为空</tr>
     *
     * </table>
     * </p>
     * @param where  用于拼装 select 语句的where子句的条件表达式，表示当满足where定义的条件时才检测，格式遵循sql语法
     * @param find  查找表达式，格式遵循计算列表达式的语法，表示该行必须满足find 定义的表达式时才检测。
     *<pre>
     *  示例：
     *      AddAutoCheck( "name", 2 , "pID=1", "pID==1")
     *
     *     表示对本DataStore中pID等于1的数据以及数据库中pID等于1的数据做数据唯一性的校验，
     * 即 pid等于1的所有数据中，name 不能重复。
     * 其中where参数为"pID=1" 它符合sql语句where条件的写法，
     * 其中find参数为"pID==1" 它符合计算列的语法
     *
     * 细节：
     *    当使用本函数注册自动检测后，在使用 Update 函数执行更新时，
     * 根据这些注册的检测类型对需要保存的数据（每一行都）做检测，如
     * 果版检测没有通过，那么Update 直接返回false。当自动检测通过后，
     * 才触发事件 Validate ，之后的过程请参看 Update函数。
     * <pre>
     */

    addAutoCheck: function (col, /*int*/
                            checkType, /*String*/
                            where, /*String*/
                            find) {


        col = this.col2Index(col);

        var ct = checkType;

        var /*AutoCheckConfig*/    pACC = null;

        //检查行列是否越界
        if (col < 0) return;
        if (col >= this.getColumnCount()) return;

        //如果是唯一性检测,那么自动加上空检测
        if (ct == AutoCheckType.CHECK_UNIQUE || ct == AutoCheckType.CHECK_UNIQUE_WARN)
        {

            this.addAutoCheck(col, AutoCheckType.CHECK_NULL, "", "");
        }

        //覆盖已经有的登记

        var haveRegister = false;
        var n = this.m_VectorAutoCheck.length;
        for (var i = 0; i < n; i++)
        {
            pACC = this.m_VectorAutoCheck[i];

            if (pACC.col == col && pACC.type == ct)
            {
                haveRegister = true;
                break;
            }
        }

        if (!haveRegister)
        {
            pACC = new AutoCheckConfig(col, ct, where, find);
            this.m_VectorAutoCheck.push(pACC);
        }

    },


    resetBufferUpdate: function (/*DataBuffer*/       Buffer) {
        var row, rc = Buffer.getRowCount();
        var col, cc = Buffer.getColumnCount();
        var /*DsRow*/   pRow;
        var /*DsItem*/ pItem;
        //处理组缓冲区,注意,当Update成功后,缓冲区的数据应该被视为从Database中
        //检索出来的一样,如果行状态是IsNew, 那么该行状态保持为IsNew
        // 因此有数据的行的状态都应该是 IsNotModified,不应该存在其他的状态
        // 没有数据的行仍然保持 IsNew
        for (row = 0; row < rc; row++)
        {
            pRow = Buffer.getRow(row);

            if (pRow.m_RowStatus == ItemStatus.isNew) continue;

            // 如果行是新修改状态，并且除指定列外没有更多数据，那么仍保持它的状态，
            //这样当接下来，如果其它字段被修改，那么它将被当作insert处理，否则在update时仍被忽略
            if (pRow.m_RowStatus == ItemStatus.isNewModified)
            {
                // 重大bug修正，isTheRowHoldNoDataExceptSuchColumns要带上buffer参数
                if (this.isTheRowHoldNoDataExceptSuchColumns(row, undefined, Buffer)) continue;
            }

            // 如果行状态是数据发生修改，并且除指定列外没有更多数据，那么在update时，此行被执行为删除指令
            //因此在复位标志时，此行应该被设置成 isNewModified状态，即被当作新插入的行
            // 各个字段当它们有数据时，被当作新修改，没有数据的字段当作新
            if (pRow.m_RowStatus == ItemStatus.isDataModified)
            {
                // 重大bug修正，isTheRowHoldNoDataExceptSuchColumns要带上buffer参数
                if (this.isTheRowHoldNoDataExceptSuchColumns(row, undefined, Buffer))
                {
                    pRow.m_RowStatus = ItemStatus.isNewModified;
                    for (col = 0; col < cc; col++)
                    {
                        pItem = pRow.getAt(col);
                        //如果单元为空,那么忽略
                        if (pItem == null) continue;
                        // 把初始数据清空，如果当前数据不为空，那么状态改成新修改，否则改成新
                        pItem.m_OriginalValue = null;
                        if (pItem.m_CurrentValue == null)
                        {
                            pItem.m_ItemStatus = ItemStatus.isNew;
                        } else
                        {
                            pItem.m_ItemStatus = ItemStatus.isNewModified;
                        }

                    }
                    continue;
                }
            }

            // 能到这里来，显示行的状态应该是isNewModified 或者 isDataModified ，并且除指定列外还有更多数据
            //当Update成功后,缓冲区的数据应该被视为从Database中，即设置成　isNotModified
            pRow.m_RowStatus = ItemStatus.isNotModified;
            for (col = 0; col < cc; col++)
            {
                pItem = pRow.getAt(col);
                //如果单元为空,那么忽略
                if (pItem == null) continue;
                //如果单元是被修改或者新修改,那么将原来值与当前值同步
                if (pItem.m_ItemStatus == ItemStatus.isDataModified || pItem.m_ItemStatus == ItemStatus.isNewModified)
                { //复制值
                    // 此时  OriginalValue 与 CurrentValue指向的是同一个对象，不用担心当修改CurrentValue时，
                    //  OriginalValue会被一并修改，因为修改CurrentValue 是将CurrentValue指向一个新的值。
                    pItem.m_OriginalValue = pItem.m_CurrentValue;
                }
                //将单元的状态设置成没有被修改
                pItem.m_ItemStatus = ItemStatus.isNotModified;
            }
        }

    },

    /**
     *复位更新标记，执行本操作后，DataStore中的数据就等效于刚从数据库中检索出来一样
     *更新单个DataStore 时，本函数一般不需要调用，使用  Update(true)时，当更新成功后，
     *本函数会自动调用。本函数通常用在多个DataStore需要协同更新作为一个整体事务时，例如：
     *<pre>
     *
     *if (  ds1.Update(false) && ds2.Update(false)
     *{
	 *     ds1.resetUpdate();
	 *     ds2.resetUpdate();
	 *     connection.commit();
	 *}else
     *{
	 *     connection.rollBack();
	 *}
     *
     *</pre>
     */


    /**
     * @api {resetUpdate} 函数   resetUpdate
     *
     * @apiDescription  resetUpdate()
     * <br><br> 复位更新标记，执行本操作后，DataStore中的数据就等效于刚从数据库中检索出来一样
     *  本函数一般不需要手工调用，使用  ds.update(true)时，当更新成功后，
     *  本函数会自动调用。本函数通常用在多个DataStore需要协同更新作为一个整体事务时，详见示例
     *
     * @apiName  resetUpdate
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *

     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     * //下面仅做演示，在 Java代码中可以如下使用，在脚本中，通常不再需要自已处理事务
     *
     * if ( ds1.update(false) && ds2.update(false))
     * {
	     *     ds1.resetUpdate();
	     *     ds2.resetUpdate();
	     *     connection.commit();
	     * }else
     *{
	     *     connection.rollBack();
	     *}
     *
     *
     */
    resetUpdate: function () {
        this.sqlBatch.clear();//把用 executeBeforeUpdate 加进来的SQL清除
        this.sqlBatch2.clear(); //把用 executeAfterUpdate 加进来的SQL清除

        this.resetBufferUpdate(this.m_PrimaryBuffer);
        this.resetBufferUpdate(this.m_FilterBuffer);

        //清除删除缓冲区
        this.m_DeleteBuffer.clear();

    },


    /**
     * @api {getFormatedValue} 函数   getFormatedValue
     *
     * @apiDescription  getFormatedValue(row,col, original)
     * <br><br>  得到格式化后的数据 ， 字符及日期型数据，外围会用单引号括起来
     *
     * @apiName  getFormatedValue
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {int} row  行号
     * @apiParam {int/String} col 列号可列名
     * @apiParam {boolean} original  true表示取从数据库中取出的原始值，false表示取当前值
     *
     * @apiSuccess (返回值){ String} - 返回格式化后的字符串数据
     *
     * @apiExample   {js}示例：
     *
     *
     *   ds.setValue(0,"date1", newDate());
     *   ds.getFormatedValue(0,"date1") ; //  '2018.01.01'
     *
     */
    getFormatedValue: function (/*int*/row, /*String*/col, /*boolean*/ original) {


        var colIndex = col2Index(col); // colIndex 已经是内部坐标
        if (colIndex < 0) return null;

        var /*DsItem*/        item = this.m_PrimaryBuffer.getItem(row, colIndex);
        var /*ColumnProperty*/       cp = this.getColumnProperty(colIndex);
        return this.m_PrimaryBuffer.GetFormatedTextValue(item, cp, original);
    },


    /**
     * @api {getRowCount} 函数   getRowCount
     *
     * @apiDescription  getRowCount()
     * <br><br> 返回行数 ,本函数是属性 rowCount 对应的getter函数.在服务端脚本中不能使用属性时，使用本函数来代替
     *
     * @apiName  getRowCount
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {String} where 检索条件
     * @apiParam {boolean} resetBeforeRetrieve 先清空,默认是true
     *
     * @apiSuccess (返回值){int} - 返回本次检索出的数据条数
     *
     * @apiExample   {js}示例：
     *
     *   var rc=  ds.getRowCount();
     *
     */
    getRowCount: function () {
        return this.m_PrimaryBuffer.getRowCount();
    },


    /**
     * @api {getColumnName} 函数   getColumnName
     *
     * @apiDescription getColumnName( index)
     * <br><br> 得到列号对应的列名
     *
     * @apiName  getColumnName
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {int} index 列号
     *
     * @apiSuccess (返回值){String} - 返回列号对应的列名
     *
     * @apiExample   {js}示例：
     *
     * for( var i=0;i<ds.getColumnCount();i++)
     * {
         *     println( ds.getColumnName(i));
         * }
     *
     */
    getColumnName: function (index) {
        if (index < 0 || index >= this.columnCount) return "";
        var /*ColumnProperty*/pCP = this.getColumnProperty(index);
        return pCP.m_Name;
    },


    /**
     * @api {getColumnCount} 函数   getColumnCount
     *
     * @apiDescription getColumnCount()
     * <br><br> 属性 columnCount 的 getter函数 ，服务端脚本请使用本方法而不是使用属性
     *
     * @apiName  retrieve
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {String} where 检索条件
     * @apiParam {boolean} resetBeforeRetrieve 先清空,默认是true
     *
     * @apiSuccess (返回值){int} - 返回本次检索出的数据条数
     *
     * @apiExample   {js}示例：
     *
     *  var cc= ds.getColumnCount();
     */
    getColumnCount: function () {
        return this.m_PrimaryBuffer.getColumnCount();
    }
    ,


    getDialect: function () {
        if (this.m_Dialect == null) this.m_Dialect = new Dialect(this);
        return this.m_Dialect;
    }
    ,

    /**<p>&nbsp;&nbsp;&nbsp;&nbsp;
     *     设置数据库适配器。因为数据库间的差异，必须为不同的数据库设置不同的适配器。幸运的是，当使用
     * SetConnection 时，系统会自动判断并设置对应的适配器。如果无法找到合适的适配器，就使用缺省的
     * 适配器。缺省的适配器适用于 MS SQL Server
     * </p>
     *<p>&nbsp;&nbsp;&nbsp;&nbsp;
     * 可以根据需要定制自己的适配器，可以从DataAdapterDefault派生一个子类，
     * 或新建一个类实现DataAdapter接口
     *
     * </p>
     *
     * @param da
     */

    setDialect: function (da) {
        this.m_Dialect = da;
    }
    ,


    getEvent: function () {
        if (this.m_Event == null) this.m_Event = new EventManage();
        return this.m_Event;
    }
    ,


    /**
     * @api {getPrimaryBuffer} 函数   getPrimaryBuffer
     *
     * @apiDescription getPrimaryBuffer()
     * <br><br> 得到主记缓冲区对象 ，是属性 primaryBuffer的getter函数
     *
     * @apiName  getPrimaryBuffer
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     *
     * @apiSuccess (返回值){DataBuffer} - 返回缓冲区对象
     *
     * @apiExample   {js}示例：
     *
     * var buffer= ds.getPrimaryBuffer(); //服务端脚本中不能使用属性时，换成这样使用
     *
     * var buffer= ds.primaryBuffer;  //客户端脚本中可以就样使用
     *
     */
    getPrimaryBuffer: function () {
        return this.m_PrimaryBuffer;
    }
    ,


    /**
     * @api {getDeleteBuffer} 函数   getDeleteBuffer
     *
     * @apiDescription getDeleteBuffer()
     * <br><br> 得到删除记缓冲区对象，是属性 deleteBuffer的getter函数
     *
     * @apiName  getDeleteBuffer
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     *
     * @apiSuccess (返回值){DataBuffer} - 返回缓冲区对象
     *
     * @apiExample   {js}示例：
     *
     * var buffer= ds.getDeleteBuffer(); //服务端脚本中不能使用属性时，换成这样使用
     *
     * var buffer= ds.deleteBuffer;  //客户端脚本中可以就样使用
     *
     */
    getDeleteBuffer: function () {
        return this.m_DeleteBuffer;
    }
    ,

    /**
     * @api {getFilterBuffer} 函数   getFilterBuffer
     *
     * @apiDescription getFilterBuffer()
     * <br><br> 得到过滤记缓冲区对象，是属性 filterBuffer的getter函数
     *
     * @apiName  getFilterBuffer
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     *
     * @apiSuccess (返回值){DataBuffer} - 返回缓冲区对象
     *
     * @apiExample   {js}示例：
     *
     * var buffer= ds.getFilterBuffer(); //服务端脚本中不能使用属性时，换成这样使用
     *
     * var buffer= ds.filterBuffer;  //客户端脚本中可以就样使用
     *
     */
    getFilterBuffer: function () {
        return this.m_FilterBuffer;
    }
    ,


    /**
     * @api {importTxtSource} 函数   importTxtSource
     *
     * @apiDescription importTxtSource(str)
     * <br><br> 从字符串中导入数据 。 字符串中以回车做行分隔符，以tab做列分隔符
     *
     * @apiName  importTxtSource
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {String} str 数据字符串
     *
     * @apiSuccess (返回值){int} - 返回本次读入的数据条数
     *
     * @apiExample   {js}示例：
     *
     *
     * var  data="a\t123\taaa\nb\t443\tbbb\n";
     * ds.importTxtSource(data); //导入2行3列数据，列从0开始自动对应
     *
     */
    importTxtSource: function (str) {
        var rc = this.m_PrimaryBuffer.ImportTxt(str, 0);
        var rowcount = this.getRowCount();
        //触发检索结束事件
        this.EM.fire("retrieveEnd", [rowcount]);
        return rc;
    },


    /**
     * @api {importCSVSource} 函数   importCSVSource
     *
     * @apiDescription  importCSVSource (str, delimiter)
     * <br><br> 导入  csv格式的数据从字符串中
     *
     * @apiName  importCSVSource
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {String} str  csv格式的数据字符串
     * @apiParam {String} delimiter 列分隔符，默认是逗号
     *
     * @apiSuccess (返回值){int} - 返回本次导入的数据条数
     *
     * @apiExample   {js}示例：
     *
     * ds.importCSVSource( data);
     *
     */
    importCSVSource: function (str, delimiter) {
        /*int*/
        rc = m_PrimaryBuffer.ImportCSV(new /*String*/Reader(/*String*/), delimiter, false);
        /*int*/
        rowcount = getRowCount();
        //触发检索结束事件
        this.EM.fireRetrieveEnd(rowcount, false);
        return rc;
    },


    /**
     * @api {importXMLSource} 函数   importXMLSource
     *
     * @apiDescription  importXMLSource (xmlSource)
     * <br><br>  从xml格式的字符串中导入数据 。格式如下
     *
     <pre>

     <?xml version="1.0" encoding="UTF-8" ?>
     <datastore><table rowcount="2"/>
     <column_define name="id" no="0" type="varchar"/>
     <column_define name="gguid" no="1" type="varchar"/>
     <column_define name="inner_xh" no="2" type="int"/>
     <column_define name="lastmodifieddate" no="3" type="datetime"/>

     <row no="1"><c no="0">5b73eba3d856627120cfc96c</c>
     <c no="1">5b73eba0d85638ba40a69595</c>
     <c no="3">2018.08.15%2017%3A01%3A11.000</c>
     </row>
     <row no="2"><c no="0">5b73eba3d856627120cfc96d</c>
     <c no="1">5b73eba0d85638ba40a69595</c>
     <c no="3">2018.08.15%2017%3A01%3A11.000</c>
     </row>
     </datastore>
     </pre>
     * 其中的数据内容做了escape编码处理
     *
     *
     * @apiName  importXMLSource
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {String} xmlSource 字符串数据
     *
     * @apiSuccess (返回值){int} - 返回本次导入的数据条数
     *
     * @apiExample   {js}示例：
     *
     *  ds.importXMLSource( xmlData);
     *
     */
    importXMLSource: function (xmlSource) {
        var rc = this.m_PrimaryBuffer.ImportXMLSource(xmlSource);
        var rowcount = getRowCount();
        //触发检索结束事件
        this.EM.fireRetrieveEnd(rowcount, false);
        return rc;
    },


    setFilterBy: function (filterBy) {
        this.m_FilterBy = filterBy;
    },


    /*String*/
    getFilterBy: function () {
        return this.m_FilterBy;
    },


    fastFilter: function (col, value) {

        //TODO
    },


    filter: function (role) {

    },


    /**
     * 把col , Operator, value 组装成合适的用于Retrieve 的语句
     * @param col
     * @param Operator
     * @param value
     */
    assembleRetrieve: function (/*String*/      col, /*String*/       Operator, /*Object*/    value) {
    },


    /**
     * @api {getCreateTableSyntax} 函数   getCreateTableSyntax
     *
     * @apiDescription getCreateTableSyntax(tableName,primaryKey,AllColumnIgnoreType)
     * <br><br>  得到建表语句 ,本函数仅能在服务端脚本中使用
     *
     * @apiName  getCreateTableSyntax
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {String} tableName 表名称
     * @apiParam {String} primaryKey 主键
     * @apiParam {boolean} AllColumnIgnoreType 是否忽略字段类型
     *
     * @apiSuccess (返回值){String} - 返回建表语句
     *
     * @apiExample   {js}示例：
     *
     *   var sql= ds.getCreateTableSyntax( "t1", "id", false);
     */
    getCreateTableSyntax: function (/*String*/       tableName, /*String*/    primaryKey, /*boolean*/     AllColumnIgnoreType) {

    },

    /**
     * 将自己克隆一份
     * @return
     */

    clone: function () {

    },


    /**
     * @api {find} 函数   find
     *
     * @apiDescription find (expression,   startRow,  endRow)
     * <br><br> 在结果集中查询数据 。 本函数在客户端脚本与服务端脚本中 expression定义的规则是不一样的。 在客户端，表达式是一个合法的javascript
     * 表达式即可。 在服务端功能只能使用四则运行及比较运算( == , >,<,>=,<=)及少量的函数
     *
     * @apiName  find
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {String} expression 查询条件，这是一个表达式， 本函数返回从startRow到endRow之间，使用expression的值为true
     * 的第一条数据。
     * @apiParam {int} startRow 从哪行开始查询（包含该行)
     * @apiParam {int} endRow 查找到哪行结束（包含该行)
     *
     *
     * @apiSuccess (返回值){int} - 返回满足条件的每一行行号
     *
     * @apiExample   {js}示例：
     *
     *
     *   var row= ds.find('  name=="aaa" ' , 0, 9999);
     *
     *
     */
    find: function (/*String*/     expression, /*int*/        startRow, /*int*/        endRow) {

    },


    fastFind: function (col, v) {

    },

    /**
     * @api {setAbsolute} 函数   setAbsolute
     *
     * @apiDescription setAbsolute(row)
     * <br><br> 直接将光标定位到指定的行.本函数是属性 absolute的setter函数，在服务端脚本中不能使用此属性，使用本函数代替
     *
     * @apiName  setAbsolute
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {int} row 在检索前，将光标定位到哪行
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *  ds.setAbsolute(100);
     */
    setAbsolute: function (/*int*/     row) {
        this.m_absolute = Math.max(1, row);
    },

    /**
     * 跳过前面的 rc 行　，　通常本函数在Retrieve函数调用之前调用以跳过若干行。本函数等效于　SetAbsolute( ic+1);
     */
    ignoreTopRow: function (/*int*/   ir) {
        this.setAbsolute(ir + 1);
    },


    /**
     * @api {getAbsolute} 函数   getAbsolute
     *
     * @apiDescription getAbsolute（）
     * <br><br> 本函数是 abaolute属性的getter函数，服务端脚本请使用本函数而不是使用abaolute属性
     *
     * @apiName  getAbsolute
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     *
     * @apiSuccess (返回值){int} - 在检索数据前，将光标定位到哪一行
     *
     * @apiExample   {js}示例：
     *
     *  var a= ds.getAbsolute();
     *
     */
    getAbsolute: function () {
        return Math.max(1, this.m_absolute); //==1时是没有忽略的行，这个要注意
    },


    /**
     * @api {setOnceRetrieveCount} 函数   setOnceRetrieveCount
     *
     * @apiDescription setOnceRetrieveCount(rc)
     * <br><br> 本函数是 onceRetrieveCount属性的setter函数，服务端脚本请使用本函数而不是属性
     *
     * @apiName  setOnceRetrieveCount
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {int} rc 分页检索一次检索几条数据
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *   ds.setOnceRetrieveCount(100);
     */
    setOnceRetrieveCount: function (/*int*/   rc) {
        this.m_OnceRetrieveCount = Math.max(0, rc);
    },


    /*int*/


    /**
     * @api {getOnceRetrieveCount} 函数   getOnceRetrieveCount
     *
     * @apiDescription getOnceRetrieveCount()
     * <br><br> 是属性 onceRetrieveCount 的getter函数
     *
     * @apiName  getOnceRetrieveCount
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     *
     * @apiSuccess (返回值){int} - 分页检索时一页检索多少行
     *
     * @apiExample   {js}示例：
     *
     *   var t=ds.getOnceRetrieveCount();
     *
     */
    getOnceRetrieveCount: function () {
        return this.m_OnceRetrieveCount;
    },

    /**
     *  对短时间使用光标的支持
     */

    /**
     * @api {setOnceRetrieveTimeout} 函数   setOnceRetrieveTimeout
     *
     * @apiDescription setOnceRetrieveTimeout（time)
     * <br><br> 设置分页检索时的超时时间，单位：毫秒
     *
     * @apiName  setOnceRetrieveTimeout
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {int} time 超时时间，毫秒
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *
     * ds.setOnceRetrieveTimeout(5000);
     *
     */
    setOnceRetrieveTimeout: function (/*int*/     rc) {
        this.m_OnceRetrieveTimeout = Math.max(0, rc);
        //如果一次检索数据有时间限制，那么通常它就是为了控制不要长时间占用数据库，所以设置成检索完成一次后就自动关闭光标
        this.m_CloseResultsetAfterRetrieveOnce = this.m_OnceRetrieveTimeout > 0;

    },


    /**
     * @api {getOnceRetrieveTimeout} 函数   getOnceRetrieveTimeout
     *
     * @apiDescription getOnceRetrieveTimeout()
     * <br><br> 得到分页检索的超时时间
     *
     * @apiName  getOnceRetrieveTimeout
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *

     *
     * @apiSuccess (返回值){int} - 返回分页检索时，检索一页数据的超时时间
     *
     * @apiExample   {js}示例：
     *
     *   alert( ds.getOnceRetrieveTimeout());
     */
    getOnceRetrieveTimeout: function () {
        return this.m_OnceRetrieveTimeout;
    },


    /**
     * @api {getDBRowCount} 函数   getDBRowCount
     *
     * @apiDescription getDBRowCount()
     * <br><br> 得到当前Select及条件下能检索出的数据条数
     *
     * @apiName  getDBRowCount
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *

     *
     * @apiSuccess (返回值){int} - 得到当前Select及条件下能检索出的数据条数
     *
     * @apiExample   {js}示例：
     *
     *
     * var ds=newDataStor('','select * from app_config');
     *
     * ds.setOnceRetrieveCount(10);
     * ds.retrieve();
     *
     * alert( ds.getRowCount());  // 当前检索出的数据行数
     * alert( ds.getDBRowCount()); //完全检索时，能检索出的数据行数
     *
     *
     */
    getDBRowCount: function (callback) {

        console.info("getDBRowCount");
        if (this.m_DBRowCount != null)
        {
            if (callback != undefined) callback(this.m_DBRowCount);
            return this.m_DBRowCount;
        }


        //传到后台的是当前检索语句，语句中已经合并上了条件
        if (callback != undefined)
        {
            this.m_DataProvider.getDBRowCount(this.name, this.id, this.templateid, this.m_Connection,
                this.m_SelectLastRetrieved,
                Util.joinWhere(this.m_AdditionalWhere),
                Util.isArray(this.m_AdditionalWhere) ? this.m_AdditionalWhere : [this.m_AdditionalWhere],
                this.m_OrderBy,
                this.retrieveContext,
                callback);
            return;

        } else
        {

            var result = this.m_DataProvider.getDBRowCount(this.name, this.id, this.templateid, this.m_Connection,
                this.m_SelectLastRetrieved,
                Util.joinWhere(this.m_AdditionalWhere),
                Util.isArray(this.m_AdditionalWhere) ? this.m_AdditionalWhere : [this.m_AdditionalWhere],
                this.m_OrderBy,
                this.retrieveContext );

            if (!result.success)
            {
                this.onError(result);
                return 0;
            }
            return result.DBRowCount;
        }
    },


    /**
     * @api {getData} 函数   getData
     *
     * @apiDescription  getData(beginRow ,rowCount)
     * <br><br> 把当前结果集中的数据，从beginRow开始， 总共rowCount行数据导出到数组里。当本函数运行在客户端时，它返回的
     * 是一个JS数组，当本函数运行在服务端时，它返回的是 ArrayList<HashMap<String,Object>>
     *
     * @apiName  getData
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {int} beginRow 从哪行开始取数据
     * @apiParam {int} rowCount 取多少行的数据
     *
     * @apiSuccess (返回值){ Array} - 返回一个数组对象，里面存放结果集的数据
     *
     * @apiExample   {js}客户端脚本示例：
     *
     * //客户端脚本
     * var  data= ds1.getData(0 , 10);
     * ds2.retrieveFromData(data ,"", false);
     *
     * * @apiExample   {js}服务端脚本示例：
     *
     * //服务端脚本
     * var  data= ds1.getData(0 , 10);
     * ds2.retrieve (data  );
     */
    getData: function (/*int*/      beginRow, /*int*/     rowCount) {
        var data = [];

        if (beginRow == undefined) beginRow = 0;
        if (rowCount == undefined) rowCount = this.getRowCount();

        var start = beginRow;
        var end = Math.min(beginRow + rowCount, this.getRowCount());
        var cc = this.getColumnCount();
        var colName = [];
        for (let i = 0; i < cc; i++)
        {
            colName.push(this.getColumnName(i).toLocaleLowerCase());
        }

        for (var i = start; i < end; i++)
        {
            var rowData = {};
            data.push(rowData);
            for (var j = 0; j < cc; j++)
            {
                rowData[colName[j]] = this.getValue(i, j);
            }
        }
        return data;
    },


    getSafeData: function () {
        var data = [];

        var cc = this.getColumnCount();
        for (var i = 0; i < this.getRowCount(); i++)
        {
            var rowData = {};
            data.push(rowData);
            for (var j = 0; j < cc; j++)
            {
                var v = this.getValue(i, j);
                if (v == null) v = this.getColumnProperty(j).getValueInsteadOfNull();
                rowData[this.getColumnName(j)] = v;
            }
        }
        return data;
    },


    /*boolean*/
    isAutoTrim: function () {
        return this.m_AutoTrim;
    },


    setAutoTrim: function (/*boolean*/ autoTrim) {
        this.m_AutoTrim = autoTrim;
    },


    /**
     * @api {executeBeforeUpdate} 函数   executeBeforeUpdate
     *
     * @apiDescription  executeBeforeUpdate(sql)
     * <br><br> 在DataStore执行update() 时，先执行指定的sql语句 ，再执行DataStore自身的SQL。
     * 本函数可以多次调用
     *
     * @apiName  executeBeforeUpdate
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {String} sql sql语句
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *
     *  var ds=newDataStore("","select * from t1");
     *  ds.setUpdateProperty("t1","id","*","id");
     *  .....
     *  ds.executeBeforeUpdate(" delete from t2 where ...");
     *  ds.executeBeforeUpdate(" delete from t3 where ...");
     *
     *  ds.executeAfterUpdate(" delete from t4 where ...");
     *  ds.executeBeforeUpdate(" delete from t5 where ...");
     *
     *  ds.update(true);
     *
     *
     *  上面ds.update(true)在执行时，它将按如下顺序执行对数据库的处理：
     *  delete from t2 where ...
     *  delete from t3 where ...
     *  ds提交数据执行的SQL
     *  delete from t4 where ...
     *  delete from t5 where ...
     *
     *
     *
     */
    executeBeforeUpdate: function (/*String*/  sql) {

        //此时sql不需要本地化，在getUpdateSQL中会做本地化
        this.sqlBatch.push(sql);

    },

    /**
     * @api {executeAfterUpdate} 函数   executeAfterUpdate
     *
     * @apiDescription  executeAfterUpdate(sql)
     * <br><br> 在DataStore执行update() 时，先执DataStore本身提交的SQL语句，然后再执行行指定的sql语句。
     * 本函数可以多次调用
     *
     * @apiName  executeAfterUpdate
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {String} sql sql语句
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *
     *  var ds=newDataStore("","select * from t1");
     *  ds.setUpdateProperty("t1","id","*","id");
     *  .....
     *  ds.executeBeforeUpdate(" delete from t2 where ...");
     *  ds.executeBeforeUpdate(" delete from t3 where ...");
     *
     *  ds.executeAfterUpdate(" delete from t4 where ...");
     *  ds.executeBeforeUpdate(" delete from t5 where ...");
     *
     *  ds.update(true);
     *
     *
     *  上面ds.update(true)在执行时，它将按如下顺序执行对数据库的处理：
     *  delete from t2 where ...
     *  delete from t3 where ...
     *  ds提交数据执行的SQL
     *  delete from t4 where ...
     *  delete from t5 where ...
     *
     *
     *
     */
    executeAfterUpdate: function (/*String*/   sql) {

        //此时sql不需要本地化，在getUpdateSQL中会做本地化
        this.sqlBatch2.push(sql);

    },

    /**
     * 注意不能对sql 做 DataAdapter da=this.getDataAdapter();
     sql= da.toLocalSyntax(sql);操作。因为 execute  executeRemote互相调用，可能导致被多次toLocalSyntax 而发生错误
     应该在被执行前再 进行方言化
     */

    /*String*/
    execute: function (/*String*/           sql) {
        //TODO

    },


    clearValue: function (/*int*/   row, col) {
        return this.setValue(row, col, null);
    },


    /**
     * @api {getAsNoDataIfOnlySuchColumnsHoldData} 函数   getAsNoDataIfOnlySuchColumnsHoldData
     *
     * @apiDescription   getAsNoDataIfOnlySuchColumnsHoldData()
     * <br><br> 仅仅哪些字段有数据时，仍被视为空行
     *
     * @apiName  getAsNoDataIfOnlySuchColumnsHoldData
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *

     * @apiSuccess (返回值){String} - 返回仅仅哪些字段有数据时，仍被视为空行
     *
     * @apiExample   {js}示例：
     *
     *  ds.getAsNoDataIfOnlySuchColumnsHoldData();
     */
    getAsNoDataIfOnlySuchColumnsHoldData: function () {
        return m_AsNoDataIfOnlySuchColumnsHoldData;
    },


    /**
     * @api {setAsNoDataIfOnlySuchColumnsHoldData} 函数   setAsNoDataIfOnlySuchColumnsHoldData
     *
     * @apiDescription setAsNoDataIfOnlySuchColumnsHoldData(cols)
     * <br>一行数据有许多字段，有时需要控制如果仅仅是指定的字段有数据，其它字段没有数据时，则把该行视为没有有效数据的空行。
     * 比如当某行仅仅是id，gguid字段有数据时，设为此行没有有效的数据，视做空行，在保存时该行被忽略。<br>
     * 该功能通常用来实现录入界面默认N行的需求。当给结果集默认插入一定行数，并且一些字段也设置了数据，但是除了这些被初始设置了数据的字段外，
     * 当其它字段没有录入数据时，它们在视觉上仍是空行，不需要保存到数据库中。
     *     <br><br>
     *
     * @apiName  setAsNoDataIfOnlySuchColumnsHoldData
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {String} col字段名，多个字段名之间用竖线分隔。如果要取消某个字段，则在字段名前加上～
     * @apiParam {boolean} resetBeforeRetrieve 先清空,默认是true
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *  var ds=newDataStore("","select * from t1");
     *  ds.setUpdateProperty("t1","id","*","id");
     *  ds.setAsNoDataIfOnlySuchColumnsHoldData("id|gguid|inner_xh|lastmodifydate");
     *  var row=ds.insertRow(0);
     *  ds.setValue(row,"id", newGUID());
     *  ds.setValue( row, "gguid", ds.getValue(row,"id"));
     *  ds.setValue(row,"inner_xh",1);
     *  ds.update();
     *
     *  //上述脚本执行时，并不会的数据保存到数据库中，因为除了指定的字段外，没有数据，它被视为空行而忽略
     */
    setAsNoDataIfOnlySuchColumnsHoldData: function (cols) {
        var list = this.m_AsNoDataIfOnlySuchColumnsHoldData.split("|");

        var ss = cols.split("|");

        for (var i = 0; i < ss.length; i++)
        {
            var col = ss[i].trim();
            if (col === "") continue;
            if (col.startsWith("~"))
            {
                col = col.substring(1);
                if (list.contains(col)) list.remove(col);
            } else
            {
                if (!list.contains(col)) list.push(col);
            }
        }


        this.m_AsNoDataIfOnlySuchColumnsHoldData = list.join('|');

    },


    /**
     * @api {isTheRowHoldNoDataExceptSuchColumns} 函数   isTheRowHoldNoDataExceptSuchColumns
     *
     * @apiDescription  isTheRowHoldNoDataExceptSuchColumns(row,cols [,buffer])
     * <br><br>除去指定的列外，其它字段是不是没有数据
     * <font color=red>注意　本函数返回　false并不表示本行一定有数据，仅表示除某些字段外没有数据为假，因为当cols==""时本函数返回false</font>
     *
     * @apiName  isTheRowHoldNoDataExceptSuchColumns
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {int} row 行号
     * @apiParam {String} cols 字段列表，多个字段名之间用竖线分隔
     *  @apiParam {DataBuffer} buffer 哪个缓冲区，可选参数，默认为primryuffer
     *
     * @apiSuccess (返回值){boolean} - 除去指定的列外，其它字段是不是没有数据
     *
     * @apiExample   {js}示例：
     *
     *
     *  var b= ds.isTheRowHoldNoDataExceptSuchColumns(row,"id|gguid|inner_xh|lastmodifydate");
     *
     */
    isTheRowHoldNoDataExceptSuchColumns: function (row, /*optional*/cols, /*optional DataBuffer*/ buffer) {
        var ret = false;
        if (row < 0) return false;

        //console.info("isTheRowHoldNoDataExceptSuchColumns");

        if (cols == undefined) cols = this.m_AsNoDataIfOnlySuchColumnsHoldData;
        if (buffer == undefined) buffer = this.m_PrimaryBuffer;

        if (row >= buffer.getRowCount()) return false;
        if (cols == "") return false;

        var colList = cols.split('|');

        for (var j = 0; j < buffer.getColumnCount(); j++)
        {
            var /*ColumnProperty*/     CP = this.getColumnProperty(j);
            if (CP.getObjType() != ObjectType.isColumn) continue;
            var colName = CP.getName();
            if (colList.contains(colName)) continue;
            var v = buffer.getValue(row, j);
            if (v == null) continue;
            var vs = ('' + v).trim();
            if (vs != "")
            {
                Logger.info("isTheRowHoldNoDataExceptSuchColumns 函数，" + colName + "=" + vs + " 返回返回 false");
                return false;
            }

        }
        return true;
    },


    /*boolean*/
    isCloseResultsetAfterRetrieveOnce: function () {
        return this.m_CloseResultsetAfterRetrieveOnce;
    },


    setCloseResultsetAfterRetrieveOnce: function (/*boolean*/    closeResultsetAfterRetrieveOnce) {
        this.m_CloseResultsetAfterRetrieveOnce = closeResultsetAfterRetrieveOnce;
    },


    getColumnInfo: function () {
        var ret = {};

        ret.columnproperty = this.m_ColumnProperty;
        ret.name2index = this.m_ColumnName2Index;
        ret.fieldcount = this.m_FieldCount;
        return ret;

    },

    setColumnInfo: function (colInfo) {

        this.m_ColumnProperty = colInfo.columnproperty;
        this.m_ColumnName2Index = colInfo.name2index;
        this.m_FieldCount = conInfo.fieldcount;


        this.m_PrimaryBuffer.initColumnCount(m_FieldCount);
        this.m_FilterBuffer.initColumnCount(m_FieldCount);
        this.m_DeleteBuffer.initColumnCount(m_FieldCount);

    },


    getTag: function (name) {

        return this.m_Tag[name];
    },

    setTag: function (name, value) {
        this.m_Tag[name] = value;

    },


    /*boolean*/
    isNeedNotifyAggregateAndUnsureComputerInvalid: function () {
        return this.needNotifyAggregateAndUnsureComputerInvalid;
    },


    setNeedNotifyAggregateAndUnsureComputerInvalid: function (/*boolean*/    needNotifyAggregateAndUnsureComputerInvalid) {
        this.needNotifyAggregateAndUnsureComputerInvalid = needNotifyAggregateAndUnsureComputerInvalid;
        if (needNotifyAggregateAndUnsureComputerInvalid) this.m_PrimaryBuffer.notifyAggregateAndUnsureComputerInvalid();
    },


    /*boolean*/
    isBlobUnRetrieve: function () {
        return this.m_blobUnRetrieve;
    },


    setBlobUnRetrieve: function (/*boolean*/      unRetrieve) {
        this.m_blobUnRetrieve = unRetrieve;
    },


    /*boolean*/
    /**
     * @api {isSetValueWithTrigger} 函数   isSetValueWithTrigger
     *
     * @apiDescription  isSetValueWithTrigger()
     * <br><br> 修改数据触发itemChanged事件吗。本函数是属性 setValueWithTrigger 的setter函数
     *
     * @apiName  isSetValueWithTrigger
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     *
     * @apiSuccess (返回值){int} - 修改数据触发itemChanged事件吗
     *
     * @apiExample   {js}示例：
     *
     * var b= ds.isSetValueWithTrigger()
     */
    isSetValueWithTrigger: function () {
        return this.m_SetValueWithTrigger;
    },

    /**
     * @api {setSetValueWithTrigger} 函数   setSetValueWithTrigger
     *
     * @apiDescription  setSetValueWithTrigger(b)
     * <br><br> 设置修改数据是否触发itemChanged事件，本函数是属性setValueWithTrigger的setter函数
     *
     * @apiName  setSetValueWithTrigger
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     *
     * @apiParam {boolean} b  true表示修改数据需要触发itemChanged事件
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     * ds.setSetValueWithTrigger(true);
     */
    setSetValueWithTrigger: function (/*boolean*/      mSetValueWithTrigger) {
        this.m_SetValueWithTrigger = mSetValueWithTrigger;
    },

    /**
     * @api {getJSON} 函数   getJSON
     *
     * @apiDescription  getJSON(row,forceConvertToString)
     * <br><br> 把指定的行的数据包装成一个JSON数据对象返回.本函数与row(row)函数是有区别的。row(row)返回的是一个行数据访问对象，
     * 它的属性是可读可写的。可写表示设置的值以回写到DataStore中
     *
     * @apiName  getJSON
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {int} row 行号
     * @apiParam {boolean} forceConvertToString true表示所有字段数据都转换成字符串
     *
     * @apiSuccess (返回值){Object} - 把指定的行的数据包装成一个JSON对象返回
     *
     * @apiExample   {js}示例：
     *
     * var ds=newDataStore("","select id,name  from t1");
     * ds.retrieve();
     * var rd=ds.getJSON(0);
     * alert( rd.id);
     * alert( rd.name);
     * rd.name='abc'; //仅仅是改变rd.name,并不影响ds的数据
     *
     * ds.row(0).name='ab'; //等同于 ds.setValue(0,"name",'ab');
     *
     */
    getJSON: function (/*int*/     row, /*boolean*/     forceConvertToString) {
        var ret = {};
        if (row < 0 || row > (this.getRowCount() - 1)) return ret;
        var n = this.getColumnCount();
        for (var col = 0; col < n; col++)
        {
            var v = forceConvertToString ? this.getString(row, col) : this.getValue(row, col);
            ret[this.getColumnName(col).toLowerCase()] = v;
        }

        return ret;
    },


    /**
     * @api {isDataModified} 函数   isDataModified
     *
     * @apiDescription  isDataModified(row,col)
     * <br><br> 判断指定的行列是不是检索出来的数据，并且被修改了
     *
     * @apiName  isDataModified
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {int} row 行号
     * @apiParam {int/String} col 列号或列名
     *
     * @apiSuccess (返回值){boolean} - 判断指定的行列是不是检索出来的数据，并且被修改了
     *
     * @apiExample   {js}示例：
     *
     * ds.retrieve();
     * ds.setValue(0,"rq", newDate());
     * var b= ds.isDataModified(0,"rq"); // true
     *
     */
    isDataModified: function (/*int*/     row, col) {
        if (row < 0) return false;
        if (col < 0) return false;

        return this.getPrimaryBuffer().getItemStatus(row, col) == ItemStatus.isDataModified;
    },


    /**
     * @api {isModified} 函数   isModified
     *
     * @apiDescription  isModified(row,col)
     * <br><br> 判断指定的行列是不是被修改了.本函数等价于 isDataModified(row,col) || isNewModified(row,col)
     *
     * @apiName  isModified
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {int} row 行号
     * @apiParam {int/String} col 列号或列名
     *
     * @apiSuccess (返回值){boolean} - 判断指定的行列是不是被修改了
     *
     * @apiExample   {js}示例：
     *
     * ds.retrieve();
     * ds.setValue(0,"rq", newDate());
     * var b= ds.isModified(0,"rq"); // true
     *
     * var row=ds.insertRow(0);
     * ds.setValue(row,"rq",newDate());
     * var b1= ds.isModified(row,"rq"); //  true
     * var b2= ds.isDataModified(0,"rq"); // false 不是检索出来的数据，无论是否实修改，本函数都返回false
     * var b3= ds.isNewModified(0,"rq"); // true
     */
    isModified: function (/*int*/    row, col) {
        if (this.isDataModified(row, col)) return true;
        if (this.isNewModified(row, col)) return true;

        return false;
    },


    /**
     * @api {isNew} 函数   isNew
     *
     * @apiDescription  isNew(row,col)
     * <br><br> 判断指定的行列是不是新插入的行，并且没有被修改了.
     *
     * @apiName  isNew
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {int} row 行号
     * @apiParam {int/String} col 列号或列名
     *
     * @apiSuccess (返回值){boolean} - 判断指定的行列是不是新插入的行，并且没有被修改了.
     *
     * @apiExample   {js}示例：
     *

     * var row=ds.insertRow(0);
     * var b1= ds.isNew(row,"rq"); //  true
     * ds.setValue(row,"rq",newDate());
     * var b1= ds.isModified(row,"rq"); //  true
     * var b2= ds.isDataModified(0,"rq"); // false 不是检索出来的数据，无论是否实修改，本函数都返回false
     * var b3= ds.isNewModified(0,"rq"); // true
     * var b4= ds.isNew(row,"rq"); //  false
     */
    isNew: function (/*int*/   row, col) {
        if (row < 0) return false;
        if (col < 0) return false;
        return this.getPrimaryBuffer().getItemStatus(row, col) == ItemStatus.isNew;
    },


    /**
     * @api {isNew} 函数   isNew
     *
     * @apiDescription  isNew(row,col)
     * <br><br> 判断指定的行列是不是新插入的行，并且没有被修改了.
     *
     * @apiName  isNew
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {int} row 行号
     * @apiParam {int/String} col 列号或列名
     *
     * @apiSuccess (返回值){boolean} - 判断指定的行列是不是新插入的行，并且没有被修改了.
     *
     * @apiExample   {js}示例：
     *
     * var row=ds.insertRow(0);
     * var b1= ds.isNew(row,"rq"); //  true
     * ds.setValue(row,"rq",newDate());
     * var b1= ds.isModified(row,"rq"); //  true
     * var b2= ds.isDataModified(0,"rq"); // false 不是检索出来的数据，无论是否实修改，本函数都返回false
     * var b3= ds.isNewModified(0,"rq"); // true
     * var b4= ds.isNew(row,"rq"); //  false
     */
    isNewModified: function (/*int*/      row, col) {
        if (row < 0) return false;
        if (col < 0) return false;
        return this.getPrimaryBuffer().getItemStatus(row, col) == ItemStatus.isNewModified;
    },


    /**
     * @api {isNotModified} 函数   isNotModified
     *
     * @apiDescription  isNotModified(row,col)
     * <br><br> 判断指定的行列是不是检索出来的行，并且没有被修改了.
     *
     * @apiName  isNotModified
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {int} row 行号
     * @apiParam {int/String} col 列号或列名
     *
     * @apiSuccess (返回值){boolean} - 判断指定的行列是不是检索出来的行，并且没有被修改了.
     *
     * @apiExample   {js}示例：
     *
     * ds.retrieve();
     * var b=ds.isNotModifyed(0,"rq"); // true
     * ds.setValue(0,"rq",newDate());
     * var b1= ds.isNotModified(0,"rq"); //  false
     * var b2= ds.isDataModified(0,"rq"); // true
     * var b3= ds.isNewModified(0,"rq"); // false  检索出来的数据，无论是否修改，都是false
     * var b4= ds.isNew(0,"rq"); //  false
     */
    isNotModified: function (/*int*/     row, col) {
        if (row < 0) return false;
        if (col < 0) return false;
        return this.getPrimaryBuffer().getItemStatus(row, col) == ItemStatus.isNotModified;
    },


    /*int*/
    getGuardLineOnRetrieve: function () {

        return this.guardLine;
    },


    setGuardLineOnRetrieve: function (/*int*/      gl) {
        this.guardLine = gl;

    },


    /*boolean*/
    isEdited: function (/*int*/       row, col) {
        var /*DsItem*/  item = this.getPrimaryBuffer().getItem(row, col);
        if (item == null) return false;
        return item.m_isEdited == 1;
    },


    /*boolean*/
    columnExists: function (/*String*/     colName) {

        var col = colName.trim().toLowerCase();
        return this.m_ColumnName2Index[col] != undefined;
    },


    /**
     * @api {moveDown} 函数   moveDown
     *
     * @apiDescription moveDown(row)
     * <br><br>把row行往下移一行，即把row行与row+1
     * 行位置互捣一下
     *
     * @apiName  moveDown
     * @apiGroup DataStore
     * @apiVersion 1.0.0
     *
     * @apiParam {int} row 行号
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     * ds.moveDown(3);
     */
    moveDown: function (/*int*/    row) {
        var buffer = this.getPrimaryBuffer();
        if (row >= this.getRowCount() - 1) return;
        if (row < 0) return;
        var list = buffer.GetBuffer();
        var /*DsRow*/ rowData = list[row];

        list.removeAt(row);
        list.splice(row + 1, 0, rowData);
        buffer.notifyAggregateAndUnsureComputerInvalid();

    },


    setBalloonTip: function (/*int*/    row, /*String*/  col, /*String*/    type, /*String*/    info) {
        this.m_PrimaryBuffer.setBalloonTip(row, col, type, info);
    },


    /*String*/
    getBalloonTip: function (/*int*/      row, /*String*/     col, /*String*/     type) {
        return this.m_PrimaryBuffer.getBalloonTip(row, col, type);
    },


    /*
         得到分组后，增加了多少个分组行

         比如  {    3->{1} , 6->{1} , 9->{1} , 14->{ 0, 1}} 表示 第3,6,9行后应该放置一个第一级的小计区，
         第14行后应该放置一个第一级的小计区 , 一个第0级的汇总区。
         总共增加了5个汇总行
         */


    /*int*/
    getGroupsRowCount: function (/*int*/       startRow, /*int*/       endRow, /*int*/       underGroupLevel) {
//TODO;

    },


    /**
     * 自动删除空行自动删除
     * @returns {boolean|*}
     */
    isAutoDeleteNoDataRow: function () {
        return this.m_autoDeleteNoDataRow;
    },


    setAutoDeleteNoDataRow: function (/*boolean*/   autoDelete) {
        this.m_autoDeleteNoDataRow = autoDelete;

    },


    setAutoResetBeforeRetrieve: function (/*boolean*/      autoReset) {
        this.m_AutoResetBeforeRetrieve = autoReset;

    },


    /*boolean*/
    isAutoResetBeforeRetrieve: function () {
        return this.m_AutoResetBeforeRetrieve;
    },


    setCurrentFocusRow: function (/*int*/      row) {
        this.m_currentFocusRow = row;

    },


    /*int*/
    getCurrentFocusRow: function () {
        return this.m_currentFocusRow;
    },


    setColumnFormat: function (/*String*/    col, /*String*/    format) {

        var /*ColumnProperty*/    cp = this.getColumnProperty(col);
        if (cp == null) return;
        cp.setFormat(format);
    }
});

export default DataStore;
