/**
 * Created by zengjun
 *
 */


import Class from '../base/Class.js';
import Cell from './Cell.js';
import Row from './Row.js';
import Util from '../util/Util.js';
import Tools from '../util/Tools.js';
import Color from '../gdi/Color.js';
import Range from './Range.js';
import Map from '../util/Map.js';
import Esprima from '../base/esprima.js';
import Evaluate from '../eval/Evaluate.js';
import EvaluatePool from '../eval/EvaluatePool.js';
import ClipBoard from './ClipBoard.js';
import MergedRange from '../core/MergedRange.js';
import CommandManage from '../cmd/CommandManage.js';
import WorkSheetView from './WorkSheetView.js';
import RedefineBecause from './RedefineBecause.js';
import DataSourceConfig from './DataSourceConfig.js';
import DrawTool from '../mouse/DrawTool.js';
import RowPropertyManage from './RowPropertyManage.js';
import ColumnPropertyManage from './ColumnPropertyManage.js';

import PopupSheetContainer from '../brick/PopupSheetContainer.js';
import ALIGNMENT from './ALIGNMENT.js';

import CMD_SetRangeProperty from '../cmd/CMD_SetRangeProperty.js';
import CMD_SetRangeBorder from '../cmd/CMD_SetRangeBorder.js';
import CMD_SetColumnWidth from '../cmd/CMD_SetColumnWidth.js';
import CMD_SetWidthBetweenTwoColumn from '../cmd/CMD_SetWidthBetweenTwoColumn.js';
import CMD_SetHeightBetweenTwoRow from '../cmd/CMD_SetHeightBetweenTwoRow.js';
import CMD_SetSheetName from '../cmd/CMD_SetSheetName.js';

import CMD_SetRowHeight from '../cmd/CMD_SetRowHeight.js';
import CMD_InsertMultiRow from '../cmd/CMD_InsertMultiRow.js';
import CMD_InsertMultiColumn from '../cmd/CMD_InsertMultiColumn.js';
import CMD_DeleteMultiRow from '../cmd/CMD_DeleteMultiRow.js';
import CMD_DeleteMultiColumn from '../cmd/CMD_DeleteMultiColumn.js';
import CMD_Paste from '../cmd/CMD_Paste.js';
import CMD_Merge from '../cmd/CMD_Merge.js';
import CMD_UnMerge from '../cmd/CMD_UnMerge.js';


//sheet的相关操作，把它们分开的目的一是为了方便维护，二是为了减小本类的大小，把代码拆分到小文件中便于维护
import ISheetBorderDesign from './ISheet/ISheetBorderDesign.js'; //边框设置
import ISheetMergeOrNot from './ISheet/ISheetMergeOrNot.js';
import ISheetDimension from './ISheet/ISheetDimension.js';
import ISheetInsertDelete from './ISheet/ISheetInsertDelete.js';
import ISheetFocusRange from './ISheet/ISheetFocusRange.js';
import ISheetSaveAs from './ISheet/ISheetSaveAs.js';
import ISheetPrint from './ISheet/ISheetPrint.js';
import ISheetPaste from './ISheet/ISheetPaste.js';


import Property from "./Property.js";
import Rectangle from "../gdi/Rectangle.js";


var WorkSheet = Class.extend({

        components: [
            ISheetMergeOrNot,  //合并/取消合并
            ISheetDimension, //行列，尺寸设置
            ISheetInsertDelete,//插入删除行列的操作
            ISheetFocusRange,
            ISheetBorderDesign,//边框设置
            ISheetPrint, // 打印相关
            ISheetPaste, //粘贴excel
            ISheetSaveAs //数据导出，另存

        ],


        properties: {

            /**
             * @api {name} 可读可写属性 [属性]name
             * @apiName  name
             * @apiDescription 工作表的名称
             * @apiGroup WorkSheet
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){String} - 工作表的名称
             * @apiExample {js}(示例：)
             * //示例
             */
            "name": {
                get: function () {return this.Name;},
                set: function (val) {this.setName(val);}
            },


            /**
             * @api {guid} 可读可写属性 [属性]guid
             * @apiName  guid
             * @apiDescription 工作表的GUID
             * @apiGroup WorkSheet
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){String} - 工作表的GUID
             * @apiExample {js}(示例：)
             * //示例
             */
            "guid": {
                get: function () {return this.GUID;},
                set: function (val) {this.GUID = val;}
            },

            /**
             * @api {printPermit} 可读可写属性 [属性]printPermit
             * @apiName  printPermit
             * @apiDescription 工作表是否可打印
             * @apiGroup WorkSheet
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){boolean} - 工作表是否可打印
             * @apiExample {js}(示例：)
             * //示例
             */
            "printPermit":
                {
                    get: function () {return this.m_printPermit;},
                    set: function (val) {this.m_printPermit = val;}
                },

            /**
             * @api {printerDeviceName} 可读可写属性 [属性]printerDeviceName
             * @apiName  printerDeviceName
             * @apiDescription 使用哪个打印机进行打印
             * @apiGroup WorkSheet
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){String} - 使用哪个打印机进行打印
             * @apiExample {js}(示例：)
             * //示例
             */
            "printerDeviceName": {
                get: function () {return this.m_PrinterDeviceName || '';},
                set: function (val) {this.m_PrinterDeviceName = val;}
            },


            /**
             * @api {code} 可读可写属性 [属性]code
             * @apiName  code
             * @apiDescription 工作表的code
             * @apiGroup WorkSheet
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){String} - 工作表的code
             * @apiExample {js}(示例：)
             * //示例
             */
            "code": {
                get: function () {return this.Code;},
                set: function (val) {this.setCode(val);}
            },


            /**
             * @api {alternantBackground} 可读可写属性 [属性]alternantBackground
             * @apiName  alternantBackground
             * @apiDescription 工作表中绑定的多行结果集， 显示时是否对厅偶行使用不同的背景色，默认是true
             * @apiGroup WorkSheet
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){boolean} - 多条结果集显示时是否使用交替背景色
             * @apiExample {js}(示例：)
             * //示例
             */
            alternantBackground: {
                get: function () {return this.m_alternantBackground;},
                set: function (val) {
                    this.m_alternantBackground = val;
                    this.View.repaint();
                }
            },
            /**
             * @api {index} 只读属性 [属性]index
             * @apiName  index
             * @apiDescription 工作表的序号
             * @apiGroup WorkSheet
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){int} - 工作表的序号
             * @apiExample {js}(示例：)
             * //示例
             */
            "index": {
                get: function () { return this.getIndex();}
            },


            /**
             * @api {editable} 可读可写属性 [属性]editable
             * @apiName  editable
             * @apiDescription 工作表是否可以编辑，如果是false ，则无论单元格的editable属性为何值，单元格都不可编辑
             * 如果工作表的editable=true,则单元格的editable=true 时允许编辑
             * @apiGroup WorkSheet
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){editable} - 工作表是否允许编辑
             * @apiExample {js}(示例：)
             * //示例
             *
             * sheet.editable=false;
             */

            'editable': {
                get: function () {return this.Editable;},
                set: function (val) {this.Editable = val; }
            },
            /**
             * @api {paperColor} 可读可写属性 [属性]paperColor
             * @apiName  paperColor
             * @apiDescription 工作表的背景色
             * @apiGroup WorkSheet
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){Color} - 工作表的背景色
             * @apiExample {js}(示例：)
             * //示例
             *
             * sheet.paperColor= newColor(255,200,200);
             */
            "paperColor": {
                get: function () {return this.getPaperColor();},
                set: function (val) {this.setPaperColor(val);}
            },

            /**
             * @api {placeholderColor} 可读可写属性 [属性]placeholderColor
             * @apiName  placeholderColor
             * @apiDescription 工作表中单元格占位提示显示的文字颜色 ，默认为灰色
             * @apiGroup WorkSheet
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){Color} - 作表中单元格占位提示显示的文字颜色
             * @apiExample {js}(示例：)
             * //示例
             *
             * sheet.placeholderColor= “#a0a0a0”;
             *
             */
            "placeholderColor": {
                get: function () {return this.getPlaceholderColor();},
                set: function (val) {this.setPlaceholderColor(val);}
            },

            /**
             * @api {rowCount} 只读属性 [属性]rowCount
             * @apiName  rowCount
             * @apiDescription 工作表的数据行数
             * @apiGroup WorkSheet
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){int} - 工作表的数据行数
             * @apiExample {js}(示例：)
             * //示例
             */
            "rowCount": {
                get: function () {return this.getRowCount();}
            },

            /**
             * @api {columnCount} 只读属性 [属性]columnCount
             * @apiName  columnCount
             * @apiDescription 工作表的数据列数
             * @apiGroup WorkSheet
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){int} - 工作表的数据列数
             * @apiExample {js}(示例：)
             * //示例
             */
            "columnCount": {
                get: function () {return this.getColumnCount();}
            },

            /**
             * @api {workBook} 只读属性 [属性]workBook
             * @apiName  workBook
             * @apiDescription 工作表所在的工作本对象
             * @apiGroup WorkSheet
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){WorkBook} - 工作表所在的工作本对象
             * @apiExample {js}(示例：)
             * //示例
             */
            "workBook": {
                get: function () {return this.Book;}
            },

            /**
             * @api {selection} 只读属性 [属性]selection
             * @apiName  selection
             * @apiDescription 工作表当前选中区域
             * @apiGroup WorkSheet
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){Range} - 工作表当前选中区域
             * @apiExample {js}(示例：)
             * //示例
             */
            'selection':
                {
                    get: function () { return this.getSelection();}
                },

            /**
             * @api {designMode} 可读可写属性 [属性]designMode
             * @apiName  designMode
             * @apiDescription 工作表处于设计状态吗
             * @apiGroup WorkSheet
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){boolean} - 工作表处于设计状态吗
             * @apiExample {js}(示例：)
             * //示例
             */
            "designMode":
                {
                    get: function () {return this.DesignMode},
                    set: function (val) {this.DesignMode = val}
                },

            /**
             * @api {loadMoreWhenScrollToTop} 可读可写属性 [属性]loadMoreWhenScrollToTop
             * @apiName  loadMoreWhenScrollToTop
             * @apiDescription 当在移动设备上划屏到顶端时，是否触发 loadMoreOnTop 事件 ,默认是false
             * @apiGroup WorkSheet
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){boolean} - 当在移动设备上划屏到顶端时，是否触发 loadMoreOnTop 事件
             * @apiExample {js}(示例：)
             * //示例
             *
             * sheet.loadMoreWhenScrollToTop=true;
             *
             */
            "loadMoreWhenScrollToTop":
                {
                    get: function () {return this.m_loadMoreWhenScrollToTop},
                    set: function (val) {this.m_loadMoreWhenScrollToTop = val}
                },

            /**
             * @api {loadMoreWhenScrollToBottom} 可读可写属性 [属性]loadMoreWhenScrollToBottom
             * @apiName  loadMoreWhenScrollToBottom
             * @apiDescription 当在移动设备上划屏到底部时，是否触发 loadMoreWhenScrollToBottom 事件 ,默认是false
             * @apiGroup WorkSheet
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){boolean} - 当在移动设备上划屏到底部时，是否触发 loadMoreWhenScrollToBottom 事件
             * @apiExample {js}(示例：)
             * //示例
             *
             * sheet.loadMoreWhenScrollToTop=true;
             *
             */
            "loadMoreWhenScrollToBottom":
                {
                    get: function () {return this.m_loadMoreWhenScrollToBottom},
                    set: function (val) {this.m_loadMoreWhenScrollToBottom = val}
                },

            "EM": //事件管理器
                {
                    get: function () { return this.Book.EM;}
                },


            /**
             * @api {currentPrintingPage} 可读可写属性 [属性]currentPrintingPage
             * @apiName  currentPrintingPage
             * @apiDescription 当前打印页号
             * @apiGroup WorkSheet
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){int} - 当前打印页号
             * @apiExample {js}(示例：)
             * //示例
             */
            "currentPrintingPage": {
                get: function () { return this.getCurrentPrintingPage();},
                set: function (val) { this.setCurrentPrintingPage(val);}
            },
            /**
             * 打印比例
             */

            /**
             * @api {printScale} 可读可写属性 [属性]printScale
             * @apiName  printScale
             * @apiDescription 打印比例。取值0-400之间
             * @apiGroup WorkSheet
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){int} - 打印比例
             * @apiExample {js}(示例：)
             * //示例
             */
            "printScale": {
                get: function () { return this.getPrintScale();},
                set: function (val) { this.setPrintScale(val);}
            },


            "viewScale": {
                get: function () { return this.getViewScale();},
                set: function (val) { this.setViewScale(val);}
            },


            /**
             * @api {gridLineVisible} 可读可写属性 [属性]gridLineVisible
             * @apiName  gridLineVisible
             * @apiDescription 是否显示网格线
             * @apiGroup WorkSheet
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){boolean} - 是否显示网格线
             * @apiExample {js}(示例：)
             * //示例
             */
            "gridLineVisible":
                {
                    get: function () { return this.getGridLineVisible();},
                    set: function (val) { this.setGridLineVisible(val);}
                },

            /**
             * @api {gridLineColor} 可读可写属性 [属性]gridLineColor
             * @apiName  gridLineColor
             * @apiDescription 网络线颜色
             * @apiGroup WorkSheet
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){Color} - 网络线颜色
             * @apiExample {js}(示例：)
             * //示例
             */
            "gridLineColor":
                {
                    get: function () { return this.getGridLineColor();},
                    set: function (val) { this.setGridLineColor(val);}
                },

            /**
             * @api {paintPermit} 可读可写属性 [属性]paintPermit
             * @apiName  paintPermit
             * @apiDescription 是否允许绘制
             * @apiGroup WorkSheet
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){boolean} - 是否允许绘制
             * @apiExample {js}(示例：)
             * //示例
             */
            "paintPermit":
                {
                    get: function () {return this.isPaintPermit();},
                    set: function (v) { this.setPaintPermit(v);}
                },

            /**
             * @api {visible} 可读可写属性 [属性]visible
             * @apiName  visible
             * @apiDescription 工作本显示或隐藏
             * @apiGroup WorkSheet
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){boolean} - 工作本显示或隐藏
             * @apiExample {js}(示例：)
             * //示例
             */
            "visible": {
                get: function () {return this.isVisible();},
                set: function (val) { this.setVisible(val);}
            },


            /**
             * @api {visibleInTabContainer} 只读属性 [属性]visibleInTabContainer
             * @apiName  visibleInTabContainer
             * @apiDescription 工作本如果是Tab容器中，那么它当前是否可见
             * @apiGroup WorkSheet
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){boolean} - 工作本显示或隐藏
             * @apiExample {js}(示例：)
             * //示例
             */
            "visibleInTabContainer": {
                get: function () {return this.isVisibleInTabContainer();},

            },


            /**
             * @api {xOffset} 可读可写属性 [属性]xOffset
             * @apiName  xOffset
             * @apiDescription 当前显示的水平方向上卷滚的位置
             * @apiGroup WorkSheet
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){int} - 当前显示的水平方向上卷滚的位置
             * @apiExample {js}(示例：)
             * //示例
             */
            "xOffset":
                {
                    get: function () { return this.getXOffset();},
                    set: function (val) { this.setXOffset(val);}
                },

            /**
             * @api {yOffset} 可读可写属性 [属性]yOffset
             * @apiName  yOffset
             * @apiDescription 当前显示的垂直方向上卷滚的位置
             * @apiGroup WorkSheet
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){int} - 当前显示的垂直方向上卷滚的位置
             * @apiExample {js}(示例：)
             * //示例
             */
            "yOffset":
                {
                    get: function () { return this.getYOffset();},
                    set: function (val) { this.setYOffset(val);}
                },


            /**
             * @api {ignoreBeforeRowWhenPrinting} 可读可写属性 [属性]ignoreBeforeRowWhenPrinting
             * @apiName  ignoreBeforeRowWhenPrinting
             * @apiDescription 本工作表打印时，第几行之前的内容不打印
             * @apiGroup WorkSheet
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){int} - 本工作表打印时，第几行之前的内容不打印
             * @apiExample {js}(示例：)
             * //示例
             *
             *   book.sheet0.ignoreBeforeRowWhenPrinting=3; //0，1，2，这几行不打印
             */

            ignoreBeforeRowWhenPrinting:
                {
                    get: function () { return this.m_ignoreBeforeRowWhenPrinting;},
                    set: function (val) {
                        this.m_ignoreBeforeRowWhenPrinting = val;

                    }
                },

            /**
             * @api {highlightDataSourceCurrentRow} 可读可写属性 [属性]highlightDataSourceCurrentRow
             * @apiName  highlightDataSourceCurrentRow
             * @apiDescription 本工作表是否需要亮显多行结果集的当前行
             * @apiGroup WorkSheet
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){boolean} - 本工作表是否需要亮显多行结果集的当前行
             * @apiExample {js}(示例：)
             * //示例
             *
             *   book.sheet0.highlightDataSourceCurrentRow=false;
             */



            "highlightDataSourceCurrentRow":
                {
                    get: function () { return this.HighlightDataSourceCurrentRow;},
                    set: function (val) {
                        this.highlightDataSourceCurrentRow = val;
                        this.View.repaint();
                    }
                },

            /**
             * @api {workSheetView} 可读可写属性 [属性]workSheetView
             * @apiName  workSheetView
             * @apiDescription 工作表关联的UI对象
             * @apiGroup WorkSheet
             * @apiVersion 1.0.0
             * @apiSuccess (属性类型){WorkSheetView} - 工作表关联的UI对象
             * @apiExample {js}(示例：)
             * //示例
             */
            "workSheetView": {
                get: function () { return this.View;},
                set: function (view) { this.$setWorkSheetView(view);}
            },

            "pageHeader":
                {
                    get: function () {
                        if (this.m_pageHeader == null)
                        {
                            this.m_pageHeader = this.addExpression("$pageHeaderForMe", '');
                            this.m_pageHeader.setPropertyValue("text-align", ALIGNMENT.Center);

                        }
                        return this.m_pageHeader;
                    },

                },

            "pageFooter":
                {
                    get: function () {
                        if (this.m_pageFooter == null)
                        {
                            this.m_pageFooter = this.addExpression("$pageFooterForMe", '="第 "+pageIndex()+" 页，共 "+pageCount()+" 页"');
                            this.m_pageFooter.setPropertyValue("text-align", ALIGNMENT.Center);
                        }
                        return this.m_pageFooter;
                    }

                }


        },


        constructor: function (/*WorkBook*/book,
                               /*String*/name,
                               /*String*/guid,
                               /*String*/ code,
                               /*boolean*/visible,
                               /*int*/defaultRowCount,
                               /*int*/defaultColumnCount,
                               /*boolean*/paintPermit) {

            this.Book = book;//一定要在最先设置好

            this.GUID = guid || Tools.newGUID();
            this.Code = code || this.GUID;
            this.Visible = visible || true;
            this.Name = name;
            //如果本sheet在Tab容器中，那么它是否处于激活可见状态，当在tab容器中时，visible=false,可以用本函数判断是不是激活状态
            this.VisibleInTabContainer = false;

            this.CM = new CommandManage(this);//// Redo Undo管理

            defaultRowCount = Math.min(Math.max(defaultRowCount, 1), 100000);// 10万
            defaultColumnCount = Math.min(Math.max(defaultColumnCount, 1), 10000);// 1万


            this.RPM = new RowPropertyManage(defaultRowCount, this);//行属性管理对象
            this.CPM = new ColumnPropertyManage(defaultColumnCount, this); //列属性管理对象
            //数据
            this.SheetRows = [];
            for (var i = 0; i < defaultRowCount; i++)
            {
                this.SheetRows.push(null);  //插入空，占位
            }


            this.View = null; //WorkSheetView

            if (book.withGUI)
            {
                var view = new WorkSheetView(this);
                this.setWorkSheetView(view);
            }


            this.m_ignoreBeforeRowWhenPrinting = 0; //打印时忽略第几行之前的内容

            this.m_printPermit = true; //是否允许打印

            this.floatToTopStartRow = -1;
            this.floatToTopEndRow = -1;

            this.PaintPermit = paintPermit; // 是否允许绘制
            this.BalloonVisible = false;

            this.pSheetContainer = null; // CellPanel_如果自已被嵌入到一个cellpanel中，那么设置它


            this.PlaceholderColor = Color.LIGHTGRAY;

            this.PaperColor = 'transparent'; // 页面背景
            this.GridLineColor = Color.LIGHTGRAY; // 网格线颜色
            this.GridLineVisible = true; //网线是否显示

            this.XOffset = 0;
            this.YOffset = 0;

            this.xPrintOffset = 0;
            this.yPrintOffset = 0;

            this.ColumnMovePermited = true; // 是否允许移动列的位置
            this.Selection = new Range(0, 0, 0, 0); // 当前选中的范围
            this.MovingSelection = null; // 当前选中的范围被拖到的位置

            this.popupContainer = null;  //TODO

            this.oneGroupPrintOnOnePageMap = null; //HashMap < Integer, Boolean >

            this.MergeMap = new Map(); // HashMap<Cell, MergedRange> 单元格合并映射，因为key是对象，所以不能简单地用 {}模拟映射
            this.AliasMap = {};//  HashMap<String, Cell> 别名

            this.cellPanelList = []; //ArrayList < CellPanel>

            //背景配置
            this.backgroundImageMap = {}; // TreeMap < Integer, HashMap < String, BackGroundImageConfig >>

            this.DesignMode = true;

            this.freezedRow = 0;
            this.freezedCol = 0;

            this.tabHightLight = false;

            this.focusRange = []; // new ArrayList < Range > ();

            this.currentFocusRangeIndex = -1;

            // 为了打印

            this.pageFormat = {};//new PageFormat()
            this.pageRange = [];

            this.m_currentPrintingPage; // 当前打印页号，以1开始
            this.m_printScale = 0; //0表示自动适应页宽
            this.m_viewScale = 100; // 100%显示

            this.printServiceName = ""; // 打印设备名称

            this.formulaBackFillBindEnabled = true;

            this.rebuildCellDefineAfterInsertOrDeleteRow = true; //插入行后，需要自动重建公式定义

            // 表达式数据对象，不需要显示的表达式对象，只有数据，不需要显示
            this.ExpressionMap = null; //HashMap < String, Cell >
            this.ExpressionList = null;//ArrayList < Cell >


            this.Editable = true;
            this.isPrintable = true;

            this.contextMenuMap = null;  //LinkedHashMap < String, ContextMenuConfig >

            this.HighlightDataSourceCurrentRow = true;

            this.m_loadMoreWhenScrollToTop = false;
            this.m_loadMoreWhenScrollToBottom = false;


            this.m_alternantBackground = true; //多行绑定自动显示交替背景

            this.multiRowDataSourceSomeConfigCache = {}; //多行绑定的一些运行时缓存，详见 Cell.drawCell

        },
        _$WorkSheet: function () {

            //
            this.clear();

        },


        clear: function () {

            this.Book = null;
            this.CM = null; // Redo Undo管理

            this.View = null;
            this.SheetRows = null; // Protect方式访问

            this.CPM = null; // 列属性管理对象
            this.RPM = null; // 行属性管理对象

            this.Selection = new Range(0, 0, 0, 0); // 当前选中的范围
            this.MovingSelection = null; // 当前选中的范围被拖到的位置

            this.MergeMap = null;
            this.oneGroupPrintOnOnePageMap = null;
            this.AliasMap = null;

            this.focusRange = null;

            this.ExpressionMap = null;
            this.ExpressionList = null;

            this.tag = null;


        },


        getWorkSheetView: function () {
            return this.View;
        },

        /**
         * @api {unFreezeCell} 函数   unFreezeCell
         *
         * @apiDescription unFreezeCell()
         * <br><br> 取消卷滚的锁定
         *
         * @apiName  unFreezeCell
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         *
         *
         * @apiSuccess (返回值){void} - 无返回值
         *
         * @apiExample   {js}示例：
         *
         *  sheet.unFreezeCell();
         *
         */
        unFreezeCell: function () {

            this.freezedRow = 0;
            this.freezedCol = 0;

            if (this.View != null) this.View.repaint();

        },


        getName:function()
        {
            return this.Name;
        },

        getCode:function()
        {
            return this.Code;
        },

        /**
         * @api {freezeCell} 函数   freezeCell
         *
         * @apiDescription freezeCell (row, col)
         * <br><br>在指定行列处锁定工作表的卷滚
         *
         * @apiName  freezeCell
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         * @apiParam {int} row 行号
         * @apiParam {int} col 列号
         *
         * @apiSuccess (返回值){void} - 无返回值
         *
         * @apiExample   {js}示例：
         *
         *  //示例
         */
        freezeCell: function (row, col) {

            var /*Cell*/ cell = this.cells(row, col);

            if (cell == null)
            {

                return;
            } // 可能是row, col 超界引起的
            this.freezedRow = row;
            this.freezedCol = col;
            if (this.View != null) this.View.repaint();
        },


        /**
         * @api {getFreezedCell} 函数   getFreezedCell
         *
         * @apiDescription getFreezedCell()
         * <br><br>得到锁定坐标处的单元格对象
         *
         * @apiName  getFreezedCell
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         *
         *
         * @apiSuccess (返回值){Cell} - 得到锁定坐标处的单元格对象
         *
         * @apiExample   {js}示例：
         *
         *  //示例
         */
        getFreezedCell: function () {
            if (this.freezedRow == 0 && this.freezedCol == 0) return null;// 都为0时表示不锁定
            return this.cells(this.freezedRow, this.freezedCol);
        },

        /**
         * 增加显示对象
         * @param view
         */
        setWorkSheetView: function (/*WorkSheetView*/  view) {

            var /*WorkBookView*/    bookview = this.Book.getWorkBookView();
            // 先去除旧的
            if (this.View != null && bookview != null)
            {
                bookview.remove(this.View);
            }

            this.View = view;

            if (bookview != null)
            {
                bookview.addWorkSheetView(view);
            }

            var active = this.Book.getActiveSheet() == this;
            this.setActive(active);

        },


        setCode: function (newCode) {
            var oldCode = this.Code;
            if (newCode == null) return;
            if (newCode.trim() == '') return;

            if (oldCode == newCode) return;

            var t = newCode;

            if (Book.getWorkSheet(t) != null) return;
            this.Code = newCode;
            //TODO 检测公式中有没有用到CODE，如果有，那么自动修改公式
        },


        setName: function (newname) {
            // 把感叹号和冒号去掉
            newname = newname.replace(/[!]/g, '');//名字中不允许有！
            var oldName = this.Name;
            if (newname == null) return;
            if (newname.trim() == "") return;
            // 不是设计模式的工作表不允许改名
            // if (!this.designMode()) return;

            if (oldName == newname) return;

            var t = newname;

            var n = 0;
            while (this.Book.getWorkSheet(t) != null) // 找到不重复的名称
            {
                n++;
                t = newname + n;
            }

            var scene = {};
            var scene = new Property();
            scene.put('newname', t);


            var cmd = new CMD_SetSheetName(this, scene);
            cmd.execute();

        },


        $setName: function (newname) {

            if (this.Name == newname) return;

            var oldName = this.Name;
            this.Name = newname;
            var BookView = this.Book.View;
            if (BookView != null) BookView.setTabTitle(this.guid, this.Name);

            if (!oldName.equals(""))
            { //TODO
                this.Book.RebuildAllDefine(this, new RedefineBecause(RedefineBecause.RebuildDefineBecauseSheetRename, 0, 0, 0, 0));
                this.Book.EM.fire("nameChanged", [this, oldName, this.Name]);
            }
        },


        /**
         * 不要用本函数激活sheet，而是用book.setActieSheet来激活，本函数在那个函数中被调用
         * @param active
         */
        setActive: function (active) {
            if (this.View == null) return;


            this.View.setActive(active);

            // 把Tab也同步到合适的Tab页
            if (active)
            {
                var /*WorkBookView*/ bookView = this.Book.View;

                bookView.setActiveTab(this.guid);
                bookView.resetHScrollBarScrollRange();
                bookView.resetVScrollBarScrollRange();
            }


            //有一个小控件，它是独立的DOM对象，当切换sheet时，要把它们显示或隐藏


        },

        /**
         * 本函数不会直接使用，它是在setActiveSheet 中,以及 setYoffset , setXOffset 中 调用，用来正确显示或隐藏一些小控件
         */
        bricksShowOrHideWithWorkSheet: function () {
            var RC = this.rowCount;
            var CC = this.columnCount;
            for (var row = 0; row < RC; row++)
            {
                for (var col = 0; col < CC; col++)
                {
                    var cell = this.$cells(row, col);
                    if (cell == null) continue;
                    var bc = cell.getBrickCount();
                    if (bc == 0) continue;
                    for (var bi = 0; bi < bc; bi++)
                    {
                        var brick = cell.getBrick(bi);
                        brick.showOrHideWithWorkSheet();
                    }
                }
            }
        },


        getIndex: function () {
            return this.Book.getWorkSheetIndex(this);
        }
        ,


        getPaperColor: function () {
            return this.PaperColor;
        }
        ,


        setPaperColor: function (color) {
            if (color == null) return;

            this.PaperColor = color;
            if (this.View != null) this.View.repaint();

        }
        ,

        getPlaceholderColor: function () {
            return this.PlaceholderColor;
        }
        ,


        setPlaceholderColor: function (color) {
            if (color == null) return;

            this.PlaceholderColor = color;
            if (this.View != null) this.View.repaint();

        }
        ,

        isPaintPermit: function () {
            return this.PaintPermit;
        }
        ,


        setPaintPermit: function (paintPermit) {
            this.PaintPermit = paintPermit;
            if (!paintPermit) return;
            if (this.View != null && paintPermit && this.Book.activeSheet == this) this.View.repaint();

        }
        ,


        getGridLineColor: function () {
            return this.GridLineColor;
        }
        ,


        setGridLineColor: function (color) {
            if (color == null) return;
            if (color.equals(this.GridLineColor)) return;
            this.GridLineColor = color;
            if (this.View != null) this.View.repaint();

        }
        ,


        /**
         * @api {getColumnPropertyManage} 函数   getColumnPropertyManage
         *
         * @apiDescription  getColumnPropertyManage()
         * <br><br> 得到列属性管理对象
         *
         * @apiName  getColumnPropertyManage
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         *
         *
         *
         * @apiSuccess (返回值){getColumnPropertyManage} - 返回列属性管理对象
         *
         *
         * @apiExample   {js}示例：
         *
         *  var cpm= book.sheet0.getColumnPropertyManage();
         *  cpm.setColumnWidth(0,0); //隐藏第一列
         *
         */

        getColumnPropertyManage: function () {
            return this.CPM;
        }
        ,


        /**
         * @api {getRowPropertyManage} 函数   getRowPropertyManage
         *
         * @apiDescription  getRowPropertyManage()
         * <br><br> 得到行属性管理对象
         *
         * @apiName  getRowPropertyManage
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         *
         *
         *
         * @apiSuccess (返回值){ RowPropertyManage } - 返回行属性管理对象
         *
         *
         * @apiExample   {js}示例：
         *
         *  var rpm= book.sheet0.getRowPropertyManage();
         *  rpm.setRowHeight(0,0); //隐藏第一行
         *
         */
        getRowPropertyManage: function () {
            return this.RPM;
        }
        ,


        /**
         * @api {cells} 函数   cells
         *
         * @apiDescription cells (行，列，或别名)
         * <br><br>
         *
         * @apiName  cells
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         * @apiParam (1个参数){variant}  parm  单元格名称，或别名
         *
         * @apiParam (2个参数){int}  row 行号
         * @apiParam (2个参数){int}  col 列号
         *
         * @apiSuccess (返回值){Cell} - 返回Cell对象
         *
         * @apiExample   {js}示例：
         *
         *  sheet.cells("a1");
         *  sheet.cells("abcde");
         *  sheet.cells( 0,0);
         *
         */
        cells: function () {
            if (arguments.length == 1)
            {
                var rc = this.cellNameToRowColumn(arguments[0]);
                if (rc == null) return null;
                return this.cells(rc.row, rc.col);
            }
            if (arguments.length == 2)
            {
                var row = arguments[0];
                var col = arguments[1];
                return this.createCell(row, col);
            }
            return null;
        }
        ,


        getXPrintOffset: function () {
            return this.xPrintOffset;
        }
        ,

        setXPrintOffset: function (printOffset) {
            this.xPrintOffset = printOffset;
        }
        ,

        getYPrintOffset: function () {
            return this.yPrintOffset;
        }
        ,

        setYPrintOffset: function (printOffset) {
            this.yPrintOffset = printOffset;
        }
        ,

        getXOffset: function () {
            return this.XOffset;
        }
        ,

        getYOffset: function () {

            return this.YOffset;
        }
        ,

        setXOffset: function (xoffset) {
            if (this.XOffset == xoffset) return;


            //最大值，不能超过

            /*
                        //下面的处理，是针对手机上，水平方向没有卷滚条时，水平划屏的处理
                        var hScrollBar = $('#' + this.Book.View.containerID + ' .viewContainser .hScrollBar');

                        let screenWidth=0;
                        if( hScrollBar.find('.content').width() ==0)
                        {
                            screenWidth = hScrollBar.width();


                            if (this.pSheetContainer)
                            {
                                hScrollBar = $('#' + this.pSheetContainer.containerID + '_hScrollBar');
                                // = $('#' + this.pSheetContainer.containerID + '_hScrollBar_content').width() - hScrollBar.width();
                                screenWidth = hScrollBar.width();
                            }


                                //划动的距离
                                var left = (this.View.startTouchX - this.View.lastTouchX ) / this.View.devicePixelRatio;
                                if (left > 200) left = 200
                                left = screenWidth - left;
                                //console.info("top=" + top);
                                this.Book.View.dragToolDOM.css("left", left + "px");

                                this.releaseDragTool = true;


                            xoffset = 0;

                        }

            */


            if (xoffset < 0) xoffset = 0;


            this.XOffset = xoffset;


            if (this.View != null)
            {
                this.View.forceCurrentEditControlGiveUpFocus();
                this.View.repaint();
            }


            this.resetHScrollBarScrollRange();

            //2019.07.18增加，显示或隐藏小控件
            this.bricksShowOrHideWithWorkSheet();
        }
        ,

        resetHScrollBarScrollRange: function () {
            if (this.pSheetContainer != null)
            {

                this.pSheetContainer.resetHScrollBarScrollRange();
            } else
            {
                var /*WorkBookView*/ BookView = this.Book.View;

                if (!this.designMode)
                {
                    if (this.View.includeStretchColumn()) this.View.reCalcuColumnsWidth();
                }

                BookView.resetHScrollBarScrollRange();
            }

        }
        ,

        resetVScrollBarScrollRange: function () {
            if (this.pSheetContainer != null)
            {
                this.pSheetContainer.resetVScrollBarScrollRange();
            } else
            {
                if (!this.designMode)
                {
                    if (this.View.includeStretchRow()) this.View.reCalcuRowsHeight();
                }

                var /*WorkBookView*/ BookView = this.Book.View;
                BookView.resetVScrollBarScrollRange();
            }

        }
        ,


        setYOffset: function (yoffset) {

            yoffset= Math.floor(yoffset);//取整
            //console.info(" yoffset=" + yoffset);

            let oldYOffset= this.YOffset;

            if (this.YOffset == yoffset)
            {
                //console.info(" yoffset not change , return;");
                return false;
            }

            if (yoffset < 0)
            {


                var top = -40 - (this.View.startTouchY - this.View.lastTouchY ) / this.View.devicePixelRatio;
                if (top >= 0) top = 0;
                //console.info("top=" + top);
                this.Book.View.loadMoreDOM.css("top", top + "px");

                // let t= this.workBook.EM.fire("loadMoreOnTopTipInfo", [this]);
                // if( t!=undefined)  this.Book.View.loadMoreDOM.html(t);
                if (this.loadMoreWhenScrollToTop)
                {
                    this.Book.View.loadMoreDOM.html("按住一会儿再松开则刷新");
                } else
                {
                    this.Book.View.loadMoreDOM.html("松手吧，已经拉到顶了");
                }
                this.releaseDragToLoadMoreTop = true;


                yoffset = 0;
            } else
            {
                if (this.loadMoreWhenScrollToTop)
                {
                    this.Book.View.loadMoreDOM.css("top", "-100px");
                    this.releaseDragToLoadMoreTop = false;
                }
            }


            //最大值，不能超过


            var vScrollBar = $('#' + this.Book.View.containerID + ' .viewContainser .vScrollBar');
            var h = vScrollBar.find('.content').height() - vScrollBar.height();

            if (this.pSheetContainer)
            {
                vScrollBar = $('#' + this.pSheetContainer.containerID + '_vScrollBar');
                h = $('#' + this.pSheetContainer.containerID + '_vScrollBar_content').height() - vScrollBar.height();
            }


            //卷到了最下面
            //2020.05.24增加 ，用在卷滚条拉到最下面时触发事件
            this.justScrolledToBottom = yoffset == h;


            if (yoffset > h)
            {

                var top = (this.View.startTouchY - this.View.lastTouchY ) / this.View.devicePixelRatio;
                if (top > 40) top = 40;
                top = window.innerHeight - top;
                //console.info("top=" + top);
                this.Book.View.loadMoreDOM.css("top", top + "px");


                let t = "松手吧，已经拉到底了";
                if (this.loadMoreWhenScrollToBottom)
                {
                    t = this.workBook.EM.fire("loadMoreTipInfoWhenScrollToBottom", [this]);
                    if (t == undefined) t = "松开刷新"
                }
                this.Book.View.loadMoreDOM.html(t);


                this.releaseDragToLoadMoreBottom = true;


                yoffset = h;

            } else
            {
                if (this.loadMoreWhenScrollToBottom)
                {
                    this.releaseDragToLoadMoreBottom = false;
                }
            }

            if (yoffset < 0) yoffset = 0;

            //2020.03.29 增加再次判断 ，可能是不需要真正滚动的
            if (this.YOffset == yoffset)
            {
                //console.info(" yoffset not change , return;");
                return false;
            }

            //console.info(" yOffset=" + yoffset);
            this.YOffset = yoffset;
            var view = this.View;

            //重绘
            if (view != null)
            {
                view.forceCurrentEditControlGiveUpFocus();
                view.repaint();


                //2020.06.05  采用图像平移的办法，性能有一定的提高，但是在手机上，有跨域问题
                // 虽然尝试了不少方法，没能完美触发。通过服务器将图片转一下，这个让服务器增加了不少流量，不是好的办法
                // 用服务器转一下的办法，没问题，也尝试了， 在快的手机上，更快，在慢的手机上，快得不多
                // 所以暂时放弃，还原成整体重绘
                /*
                let delta= yoffset - oldYOffset ; //偏移量
                //整个画布尺寸
                let client = view.getBounds();
                //固定行高
                let fixedRowHeight= view.CPM.getColumnHeadHeight()+view.RPM.getFixedRowHeight();

                let h1=client.height- fixedRowHeight;//滚动区高度

                 if( Math.abs( delta)>= h1  )
                {
                   view.repaint();
                }else
                {
                    let g = view.graphics2d;


                    if(delta>0)//向上滚
                    {
                       // console.info("向上滚"+delta);
                        var imgData=g.ctx.getImageData(0,fixedRowHeight+1+delta ,client.width ,h1-delta );
                        g.clearRect(0, fixedRowHeight+1, client.width, h1);
                        g.ctx.putImageData(imgData,0, fixedRowHeight+1);
                        view.repaint( new Rectangle( 0, client.height - delta, client.width , delta));

                    }else //向下
                    {
                       // console.info("向下滚"+delta);
                        var imgData=g.ctx.getImageData(0,fixedRowHeight+1 ,client.width ,h1-delta );
                        g.clearRect(0, fixedRowHeight+1, client.width, h1);
                        g.ctx.putImageData(imgData,0, fixedRowHeight+1-delta);
                        view.repaint(new Rectangle(  0, fixedRowHeight+1, client.width , Math.abs(delta)));

                    }
                }
                */

            }

            //标记20180903-01
            if (this.pSheetContainer)
            {
                this.pSheetContainer.forceCurrentEditControlGiveUpFocus();
                this.pSheetContainer.vScrollTo(yoffset);
            } else
            {

                var /*WorkBookView*/ BookView = this.Book.View;
                if (BookView) BookView.vScrollTo(yoffset);
            }

            //显隐藏控件
            this.bricksShowOrHideWithWorkSheet();

            return true;
        },


        vScroll: function (delta) {
            let oldYOffset= this.yOffset;
            let y = this.yOffset;
            y += delta;
            return this.setYOffset(y);
        }
        ,

        hScroll: function (delta) {
            var x = this.xOffset;
            x += delta;
            this.setXOffset(x);
        }
        ,

        getColumnMovePermited: function () {

            return this.ColumnMovePermited;
        }
        ,

        setColumnMovePermited: function (permited) {
            this.ColumnMovePermited = permited;

        }
        ,

// 防止被改掉，所以返回一个Clone
        getSelection: function () {
            return this.Selection.clone();
        }
        ,


        setSelection: function () {
            if (arguments.length == 1)
            {
                var p1 = arguments[0];
                if (Util.isClas$(p1))
                {
                    if (p1.instanceOf(Cell)) //如果只有一个参数，且是一个Cell对象，那么
                    {
                        var row = p1.rowIndex;
                        var col = p1.columnIndex;
                        return this.$setSelection(row, col, row, col);
                    }

                    if (p1.instanceOf(Range))
                    {
                        return this.$setSelection(p1.startRow, p1.startCol, p1.endRow, p1.endCol);
                    }


                }

            }

            if (arguments.length == 2)
            {
                var p1 = arguments[0];
                var p2 = arguments[1];

                if (Util.isClas$(p1) && Util.isClas$(p2))
                {
                    if (p1.instanceOf(Cell) && p2.instanceOf(Cell))
                    {
                        var startRow = p1.rowIndex;
                        var startCol = p1.columnIndex;
                        var endRow = p2.rowIndex;
                        var endCol = p2.columnIndex;

                        return this.$setSelection(startRow, startCol, endRow, endCol);
                    }
                }
            }

            if (arguments.length >= 4)
            {
                var startRow = arguments[0];
                var startCol = arguments[1];
                var endRow = arguments[2];
                var endCol = arguments[3];
                var forceTriggerEvent = arguments[4];

                return this.$setSelection(startRow, startCol, endRow, endCol, forceTriggerEvent);
            }


        }
        ,

        $setSelection: function (startRow, startCol, endRow, endCol, forceTriggerEvent) {

            startRow = Math.min(this.getRowCount() - 1, Math.max(0, startRow));
            startCol = Math.min(this.getColumnCount() - 1, Math.max(0, startCol));
            endRow = Math.min(this.getRowCount() - 1, Math.max(0, endRow));
            endCol = Math.min(this.getColumnCount() - 1, Math.max(0, endCol));

            if (startRow == this.Selection.startRow && startCol == this.Selection.startCol &&
                endRow == this.Selection.endRow && endCol == this.Selection.endCol && !forceTriggerEvent)
            {
                return;
            }

            var range = new Range(startRow, startCol, endRow, endCol);

            var oldRow = this.Selection.startRow;
            var oldCol = this.Selection.startCol;

            var needRedrawRects = [];

            this.Selection = this.$copyRangeAndRepaint(this.Selection, range, needRedrawRects);

            var newRow = this.Selection.startRow;
            var newCol = this.Selection.startCol;

            if (newRow != oldRow || newCol != oldCol || forceTriggerEvent)
            {
                this.workBook.EM.fire("cellFocusLost", [this, oldRow, oldCol]);
                this.workBook.EM.fire("cellFocusGained", [this, newRow, newCol]);

            }


            //重绘
            var d = 2 * 3; //高清比例 *3个外扩点
            for (var i = 0; i < needRedrawRects.length; i++)
            {
                var rc = needRedrawRects[i];
                if (this.designMode) rc.append(d, d, d, d);  //设计模式下，有个外框，占用了相临单元格几个像素，所以要
                this.View.repaint(rc);
            }
            //选择变了， 题头也重绘一下
            this.View.repaintHeader();


        }
        ,


        setMovingSelection: function (range) {
            var needRedrawRects = [];
            this.MovingSelection = this.$copyRangeAndRepaint(this.MovingSelection, range, needRedrawRects);
            this.View.repaint();


        }
        ,

        /**
         * <pre>
         *   把 range 赋给 target ,但是在赋之前，要对range 做一些处理， 如果它与某些合并区有交叉，
         *   那么要把range扩展到包含整个合并区
         *
         *           __ ___________________________________
         *          |__| ____A____|_____B______|____C______|
         *          | 1 |                     _____________ |___________|
         *          | 2 |_________|_________________________|
         *
         *                如上图，当鼠标在C1点下，并拖拽到C2时，最终是A1:C2被选中：理由是：
         *                  当鼠标在C1点下，选中区的起始位置是c1, 拖拽到C2时，终止位置是C2，
         *        但是由于B2和C2合并了，所以选中区域应该是C1，C2，B2所占的最大区域：B1:C2
         *        但是B1又和A1合并，所以所以选中区域应该是A1，C１，C2所占的最大区域：A1:C2
         *
         * </pre>
         *
         * @param target
         * @param range
         */

        $copyRangeAndRepaint: function (target, range, needRedrawRects) {

            var /*Range*/ r = range;

            if (range != null) // 如果要设置的区域不为NULL，那么检查有没有合并区与它相交
            {
                // 判断合并情况

                var sr = range.startRow;
                var sc = range.startCol;
                var er = range.endRow;
                var ec = range.endCol;

                // 如果在range表示的范围中的单元格是参与了合并的，且合并的区域超出了range表示的区域，
                // 那么应该使用 range和这些被合并区域的并集来做为当前选中区域
                for (var row = range.startRow; row <= range.endRow; row++)
                {
                    for (var col = range.startCol; col <= range.endCol; col++)
                    {
                        var cell = this.$cells(row, col);
                        if (cell == null) continue; // 单元格是空的，那么跳过它
                        if (this.MergeMap.containsKey(cell))
                        {
                            var /*MergedRange*/ tr = this.MergeMap.get(cell);

                            sr = Math.min(tr.mergeStart.rowIndex, sr);
                            sc = Math.min(tr.mergeStart.columnIndex, sc);

                            er = Math.max(tr.mergeEnd.rowIndex, er);
                            ec = Math.max(tr.mergeEnd.columnIndex, ec);
                        }
                    }
                }

                r = new Range(sr, sc, er, ec);

                if (target != null)
                {
                    if (target.equals(r)) return target;

                }

            } // 检查结束并对r 做了相应处理

            // 下面计算重绘区域

            var /*Rectangle*/  oldrc = null, newrc = null;

            if (this.View != null && target != null)
            {
                oldrc = this.View.getRectangleOfRange(false, target);
            }

            target = r;

            if (this.View != null && target != null
            )
            {
                newrc = this.View.getRectangleOfRange(false, target);

            }

            needRedrawRects.push(oldrc);
            needRedrawRects.push(newrc);


            // 递归，直到范围不再增大为止
            /**
             * <pre>
             *           ______________________________________
             *          |___| ____A____|_____B_____|____C______|
             *          | 1 |______________________|___________|
             *          | 2 |_________|________________________|
             *
             *                如上图，当鼠标在C1点下，并拖拽到C2时，最终是A1:C2被选中：理由是：
             *                  当鼠标在C1点下，选中区的起始位置是c1, 拖拽到C2时，终止位置是C2，
             *        但是由于B2和C2合并了，所以选中区域应该是C1，C2，B2所占的最大区域：B1:C2
             *        但是B1又和A1合并，所以所以选中区域应该是A1，C１，C2所占的最大区域：A1:C2
             *
             * </pre>
             */
            if (target != null) target = this.$copyRangeAndRepaint(target, target, needRedrawRects);

            return target;
        }
        ,

        getMovingSelection: function () {
            return this.MovingSelection;
        }
        ,


        getCellValue: function (row, col, cellInnerRow) {
            var cell = this.cells(row, col);
            if (cell == null)
            {
                //Tools.log("非法的行列(" + row + "," + col + ")");
                return "";
            }
            return cell.getValue(cellInnerRow);
        }
        ,


        getCellText: function (row, col, cellInnerRow) {
            var cell = this.cells(row, col);
            if (cell == null)
            {
                //Tools.log("非法的行列(" + row + "," + col + ")");
                return "";
            }
            return cell.getShowText(cellInnerRow);

        }
        ,


        setCellValue: function (row, col, value) {
            var /*Cell*/    cell = this.createCell(row, col);
            cell.setValue(value);

        }
        ,

// 不创建Cell的方式访问
        $cells: function (row, col) {
            if (row < 0) return null;
            if (col < 0) return null;

            if (this.RPM == null) return null;
            if (this.CPM == null) return null;

            if (row >= this.rowCount) return null;
            if (col >= this.columnCount) return null;
            var /*Row*/  r = this.SheetRows.get(row);
            if (r == null) return null;
            return r.getCell(col);

        }
        ,

        /**
         * 如果单元格不存在，那么创建它，否则直接返回它
         *
         * @param row
         * @param col
         * @return
         */
        createCell: function (row, col) {
            var /*Cell*/cell = null;

            if (row == -9999) //虚拟单元格
            {
                if (col >= this.ExpressionList.length) return null;
                return ExpressionList.get(col);
            }

            if (row < 0) return null;
            if (col < 0) return null;
            if (row >= this.RPM.getRowCount()) return null;
            if (col >= this.CPM.getColumnCount()) return null;

            var r = this.SheetRows.get(row);
            if (r != null) cell = r.$getCell(col);

            if (cell != null) return cell;

            var /*Row*/R = this.getRow(row);

            cell = new Cell(this, R);
            R.$setCell(col, cell);

            return cell;
        }
        ,


        /**
         * 用于表单设计中，取单元格的属性。因为单元格可能是空的，此时返回缺省属性
         * @param row
         * @param col
         * @param propertyName
         * @returns {*}
         */
        getCellProperty: function (row, col, propertyName) {
            var /*Property*/ p = this.Book.PM[0];
            var cell = this.$cells(row, col);
            if (cell != null) p = cell.getPropertyObject();
            var v = p.get(propertyName);
            return v;
        }
        ,

        setCellProperty: function (row, col, propertyName, propertyValue) {
            var /*Cell*/    cell = this.createCell(row, col);
            cell.setProperty(propertyName, propertyValue);

        }
        ,


        getRangeProperty: function (range, propertyName) {
            return null;
        }
        ,

// 设置选中区域的属性
        setRangeProperty: function (/*Range*/range, /*String*/  propertyName, /*Object*/  propertyValue) {

            // 为空，什么都不发生
            if (range == null) return;
            if (propertyName == null || propertyValue == null) return;

            var scene = new Property();
            scene.put("range", range);
            scene.put("propertyname", propertyName);
            scene.put("propertyvalue", propertyValue);

            var cmd = new CMD_SetRangeProperty(this, scene);
            cmd.execute();

        }
        ,


        /**
         * @api {setRangeBorder} 函数   setRangeBorder
         *
         * @apiDescription setRangeBorder (startRow, startCol, endRow, endCol, whichSide,  style, width, color)
         * <br><br> 给指定的区域设置边框
         *
         * @apiName  setRangeBorder
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         * @apiParam {int} startRow 起始行
         * @apiParam {int} startCol 起始列
         * @apiParam {int} endRow 终止行
         * @apiParam {int} endCol  终止列
         * @apiParam {int} whichSide 哪个边框
         * <ul>
         * <li><b>0</b>:左边框</li>
         * <li><b>1</b>:右边框</li>
         * <li><b>2</b>:上边框</li>
         * <li><b>3</b>:下边框</li>
         * <li><b>4</b>:内部水平边框</li>
         * <li><b>5</b>:内部垂直边框</li>
         * <li><b>6</b>:外围边框</li>
         * <li><b>7</b>:内部边框</li>
         * <li><b>8</b>:所有边框</li>
         *
         * </ul>
         *
         * @apiParam {int} style 边框风格
         * <ul>
         * <li>1：实线</li>
         * <li>2,3,4：虚线</li>
         * <li>5：双线</li>
         *</ul>
         *
         * @apiParam {int} width 边框线粗细  0 - 3
         * @apiParam {String} color 颜色。css颜色，比如 "red" ,"green" , "#ff0000"都可以
         *
         * @apiSuccess (返回值){void} - 无返回值
         *
         * @apiExample   {js}示例：
         *
         *  sheet.setRangeBorder(0,0,10,10,8, 1,1, "#ff0000");
         *
         */
        setRangeBorder: function (startRow, startCol, endRow, endCol, whichSide, /*String*/style, width, color) {

            var /*Range*/range = new Range(startRow, startCol, endRow, endCol);


            var scene = new Property();
            scene.put("range", range);
            scene.put("side", whichSide);
            scene.put("style", style);
            scene.put("width", width);

            scene.put("color", color);

            var cmd = new CMD_SetRangeBorder(this, scene);
            cmd.execute();
        }
        ,


        /**
         * @api {merge} 函数   merge
         *
         * @apiDescription merge (startRow, startCol ,endRow, endCol)
         * <br><br>合并单元格
         *
         * @apiName  merge
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         * @apiParam {int} startRow 开始的行号
         * @apiParam {int} startCol 开始的列号
         * @apiParam {int} endRow 结束的行号
         * @apiParam {int} endCol 结束的列号
         *
         * @apiSuccess (返回值){void} - 无返回值
         *
         * @apiExample   {js}示例：
         *
         *  sheet.merge(0,0,3,5);
         *
         */
        merge: function () {
            var args = Array.prototype.slice.call(arguments);
            args.push(CMD_Merge);
            this.mergeOrNot.apply(this, args);

        }
        ,

        /**
         * @api {unMerge} 函数   unMerge
         *
         * @apiDescription unMerge (startRow, startCol ,endRow, endCol)
         * <br><br>取消合并单元格
         *
         * @apiName  unMerge
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         * @apiParam {int} startRow 开始的行号
         * @apiParam {int} startCol 开始的列号
         * @apiParam {int} endRow 结束的行号
         * @apiParam {int} endCol 结束的列号
         *
         * @apiSuccess (返回值){void} - 无返回值
         *
         * @apiExample   {js}示例：
         *
         *  sheet.merge(0,0,3,5);
         *
         */
        unMerge: function () {
            var args = Array.prototype.slice.call(arguments);
            args.push(CMD_UnMerge);
            this.mergeOrNot.apply(this, args);

        }
        ,

        mergeOrNot: function () {
            var startRow = undefined;
            var startCol, endRow, endCol;
            var CMD;

            if (arguments.length == 2)
            {
                var range = arguments[0];
                CMD = arguments[1];
                if (Util.isClas$(range))
                {
                    if (range.instanceOf(Range))
                    {
                        startRow = range.startRow;
                        startCol = range.startCol;
                        endRow = range.endRow;
                        endCol = range.endCol;
                    }
                }
            }

            if (arguments.length == 5)
            {
                startRow = arguments[0];
                startCol = arguments[1];
                endRow = arguments[2];
                endCol = arguments[3];
                CMD = arguments[4];
            }
            if (startRow == undefined) return;


            // 如果是一个单元格，那么就不需要做合并
            if (startRow == endRow && startCol == endCol) return;

            var scene = new Property();
            scene.put("startrow", startRow);
            scene.put("startcol", startCol);
            scene.put("endrow", endRow);
            scene.put("endcol", endCol);

            var cmd = new CMD(this, scene);
            cmd.execute();

        }
        ,


        /**
         * @api {setColumnWidth} 函数   setColumnWidth
         *
         * @apiDescription setColumnWidth  (colIndex, width)
         * <br><br>设置列宽
         *
         * @apiName  setColumnWidth
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         * @apiParam {int} colIndex 列号
         * @apiParam {int} width 宽度， 单位：像素
         *
         * @apiSuccess (返回值){void} - 无返回值
         *
         * @apiExample   {js}示例：
         *
         *  sheet.setColumnWidth(1,100);
         *
         */
        setColumnWidth: function (colIndex, width) {

            var scene = new Property();
            scene.put("col", colIndex);
            scene.put("width", width);
            CMD_SetColumnWidth
            var cmd = new CMD_SetColumnWidth(this, scene);
            cmd.execute();

        }
        ,


        setWidthBetweenTwoColumn: function (col1, col2, delta) {
            var scene = new Property();
            scene.put("col1", col1);
            scene.put("col2", col2);

            scene.put("delta", delta);
            var /*CMD_SetWidthBetweenTwoColumn*/ cmd = new CMD_SetWidthBetweenTwoColumn(this, scene);

            cmd.execute();
        }
        ,


        setHeightBetweenTwoRow: function (row1, row2, delta) {
            var scene = new Property();
            scene.put("row1", row1);
            scene.put("row2", row2);

            scene.put("delta", delta);

            var /*CMD_SetHeightBetweenTwoRow*/
                cmd = new CMD_SetHeightBetweenTwoRow(this, scene);
            cmd.execute();

        }
        ,

        /**
         * @api {setRowHeight} 函数   setRowHeight
         *
         * @apiDescription setRowHeight (rowIndex, height)
         * <br><br>设置行高
         *
         * @apiName  setRowHeight
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         * @apiParam {int} rowIndex 行号
         * @apiParam {int} height 行高
         *
         * @apiSuccess (返回值){void} - 无返回值
         *
         * @apiExample   {js}示例：
         *
         *  sheet.setRowHeight(2,30);
         *
         */
        setRowHeight: function (rowIndex, height) {
            var scene = new Property();
            scene.put("row", rowIndex);
            scene.put("height", height);

            var /*CMD_SetRowHeight*/cmd = new CMD_SetRowHeight(this, scene);
            cmd.execute();

        }
        ,

        getMergeMap: function () {
            return this.MergeMap;
        }
        ,


        getAllMergeRange()
        {
            var ranges = this.MergeMap.values();
            var ret = [];
            for (var i = 0; i < ranges.length; i++)
            {
                var r = ranges[i].getMergedRange();
                if (!ret.contains(r)) ret.push(r);
            }
            return ret;
        }
        ,


        /**
         * 得到var.getName 指定单元格的值 本函数用在公式解析中，取单元格的值 。其中有检测循环引用
         * @param jep   : Evaluate对象，来用评估
         * @param token    变量名称
         * @param cellInnerRow
         * @returns {*}
         */
        getValueForJEP: function (/*Cell*/ clientCell, token, cellInnerRow) {

            var errorRet = 0;

            var varName = token;

            var isCell = varName.indexOf('!') < 0 && varName.indexOf(':') < 0;//没有!也没有：表示是单元格名称或别名
            var isRange = varName.indexOf('!') < 0 && varName.indexOf(':') > 0; //没有！，但是有:表示是本sheet中的单元格区域
            var isOtherSheet = varName.indexOf('!') > 0;


            if (isCell)
            {
                var needShowText = varName.startsWith("$");
                var ce = this.cells(varName)

                if (ce === clientCell && clientCell != null)
                {
                    console.log("getValueForJEP :" + this.getName() + " (" + varName + ")循环引用");
                    return errorRet;
                }
                if (ce == null) return null;
                var v = needShowText ? ce.getShowText(cellInnerRow) : ce.getValue(cellInnerRow);
                return v;
            }

            if (isRange)
            {

                var needShowText = varName.startsWith("$");
                var range = this.rangeNameToRange(varName);
                var vector = []; // new Vector/* <Object> */();
                for (var row = range.startRow; row <= range.endRow; row++)
                {
                    for (var col = range.startCol; col <= range.endCol; col++)
                    {
                        var ce = this.cells(row, col);
                        if (ce === clientCell && clientCell != null)
                        {
                            console.log(" getValueFromWorkSheet 单元格:" + this.getName() + " (" + varName + ")循环引用" +
                                this.cells(row, col).getName());
                            return errorRet;
                        }
                        var v = null;
                        if (ce != null) v = needShowText ? ce.getShowText(cellInnerRow) : ce.getValue(cellInnerRow);
                        vector.push(v);
                    }
                }
                return vector;
            }

            if (isOtherSheet)
            {
                var t = varName.split('!');
                var sheetName = t[0];
                var cellName = t[1];

                var sheet = this.Book.getWorkSheet(sheetName);
                if (sheet == null)
                {
                    console.log(sheetName + "不存在");
                    return errorRet;
                }

                return sheet.getValueForJEP(clientCell, cellName, cellInnerRow);

            }


            return null;

        }
        ,


        /**
         * @api {rowColToName} 函数   rowColToName
         *
         * @apiDescription  rowColToName  (row, col)
         * <br><br>
         *
         * @apiName  rowColToName
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         * @apiParam {int} row 行号
         * @apiParam {int} col  列号
         *
         * @apiSuccess (返回值){String} - 返回行列对应的单元格的名称
         *
         * @apiExample   {js}示例：
         *
         *  sheet.rowColToName( 0,0) ; // A1
         *
         */
        rowColToName: function (row, col) {
            var s = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
            var a = Math.floor(col / 26)
            var ret = "";
            var b = col - a * 26;

            if (col >= 26)
            {
                ret = s.substring(a - 1, a) + s.substring(b, b + 1);
            } else
            {
                ret = s.substring(col, col + 1);
            }
            return ret + (row + 1);
        }
        ,


        /**
         * @api {cellNameToRowColumn} 函数   cellNameToRowColumn
         *
         * @apiDescription cellNameToRowColumn(cellName)
         * <br><br> 单元格的名字转换成行列
         *
         * @apiName  cellNameToRowColumn
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         * @apiParam {String}  cellName 单元格名称

         *
         * @apiSuccess (返回值){int} row 行号
         * @apiSuccess (返回值){int} col 列号
         *
         *
         * @apiExample   {js}示例：
         *
         *  var p= sheet.cellNameToRowColumn("B3");
         *  println( p.row); //2
         *  println( p.col); //1
         *
         */
        cellNameToRowColumn: function (cellName) {

            if (cellName.startsWith("$")) cellName = cellName.substring(1);

            var ret = {row: 0, col: 0} //一定不能改，会影响下面的计算

            if (cellName.equals(""))
            {
                ret.row = this.getRowCount() - 1;
                ret.col = this.getColumnCount() - 1;
                return ret;
            }

            // 别名优先。如果别名取成类似单元格名，比如c1,那么别名优先
            var /*Cell*/   cell = this.getCellByAlias(cellName);
            if (cell != null)
            {
                ret.row = cell.rowIndex;
                ret.col = cell.columnIndex;
                return ret;
            }

            //其次看看 结果集.列名能不能匹配


            //通常的单元格名称是列字母加行号，所以只需要从后往前取，直到不是数字为止
            var t = cellName.toUpperCase();

            var col = t.match(/[A-Z]+/) + ''; //把列字母取出来
            var row = t.match(/[\d]+/) + '';

            for (var i = 0; i < col.length; i++)
            {

                ret.col = ret.col * 26 + col.charCodeAt(i) - 64;
            }

            ret.col--;
            ret.row = parseInt(row) - 1;

            return ret;


        }
        ,


        /**
         * 把 A1:B1 这样的名字转换成Range
         *
         * @param rangeName
         * @return
         */
        rangeNameToRange: function (rangeName) {

            var t = rangeName.split(':');
            var s1 = t[0];
            var s2 = t[1];

            var rc1 = this.cellNameToRowColumn(s1);
            var rc2 = this.cellNameToRowColumn(s2);
            return new Range(rc1.row, rc1.col, rc2.row, rc2.col);
        }
        ,


        getGridLineVisible: function () {

            return this.GridLineVisible;
        }
        ,

        setGridLineVisible: function (v) {
            if (this.GridLineVisible == v) return;
            this.GridLineVisible = v;
            if (this.View != null) this.View.repaint();

        }
        ,


        /**
         * 插入多行，在startRow到endRow的位置插入多行。
         */

        /**
         * @api {insertMultiRow} 函数   insertMultiRow
         *
         * @apiDescription insertMultiRow (startRow, endRow)
         * <br><br>插入多行。插入的行将做为startRow到endRow .
         *
         * @apiName  insertMultiRow
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         * @apiParam {int} startRow 开始行号
         * @apiParam {int} endRow  结束行号
         *
         * @apiSuccess (返回值){void} - 无返回值
         *
         * @apiExample   {js}示例：
         *
         *   sheet.insertMultiRow(3,5);//插入 3行， 这3行将成为序号为3-5行
         *
         */
        insertMultiRow: function (startRow, endRow) {

            var scene = new Property();
            scene.put("startrow", startRow);
            scene.put("endrow", endRow);

            var cmd = new CMD_InsertMultiRow(this, scene);
            cmd.execute();

        }
        ,

        /**
         * @api {insertMultiColumn} 函数   insertMultiColumn
         *
         * @apiDescription insertMultiColumn(startCol, endCol)
         * <br><br>插入多列。插入的列将做为startCol到endCol .
         *
         * @apiName  insertMultiColumn
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         * @apiParam {int} startCol 开始列号
         * @apiParam {int} endCol  结束列号
         *
         * @apiSuccess (返回值){void} - 无返回值
         *
         * @apiExample   {js}示例：
         *
         *  sheet.insertMultiColumn(4,6);
         *
         */
        insertMultiColumn: function (startCol, endCol) {

            var scene = new Property();
            scene.put("startcol", startCol);
            scene.put("endcol", endCol);

            var cmd = new CMD_InsertMultiColumn(this, scene);
            cmd.execute();

        }
        ,

        /**
         * @api {deleteRow} 函数   deleteRow
         *
         * @apiDescription deleteRow
         * <br><br>删除第row行，行号从0开始
         *
         * @apiName  deleteRow(row)
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         * @apiParam {int} row 行号
         *
         * @apiSuccess (返回值){void}  - 无返回值
         *
         * @apiExample   {js}示例：
         *
         *  sheet.deleteRow(4);
         *
         */
        deleteRow: function (row) {
            this.deleteMultiRow(row, row);
        }
        ,


        /**
         * @api {deleteMultiRow} 函数   deleteMultiRow
         *
         * @apiDescription deleteMultiRow (startRow, endRow)
         * <br><br>将startRow, endRow之间的所有行全部删除
         *
         * @apiName  deleteMultiRow
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         * @apiParam {int} startRow 开始行号
         * @apiParam {int} endRow 结束行号
         *
         * @apiSuccess (返回值){void}  - 无返回值
         *
         * @apiExample   {js}示例：
         *
         *  sheet.deleteMultiRow(3,6);
         *
         */
        deleteMultiRow: function (startRow, endRow) {

            var scene = new Property();
            scene.put("startrow", startRow);
            scene.put("endrow", endRow);

            var cmd = new CMD_DeleteMultiRow(this, scene);
            cmd.execute();

        }
        ,


        /**
         * @api {deleteColumn} 函数   deleteColumn
         *
         * @apiDescription deleteColumn (col)
         * <br><br>删除col列
         *
         * @apiName  deleteColumn
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         * @apiParam {int} col 列号
         *
         * @apiSuccess (返回值){void}  - 无返回值
         *
         * @apiExample   {js}示例：
         *
         *  sheet.deleteColumn(3);
         *
         */
        deleteColumn: function (col) {
            this.deleteMultiColumn(col, col);
        }
        ,

        /**
         * @api {deleteMultiColumn} 函数   deleteMultiColumn
         *
         * @apiDescription deleteMultiColumn (startCol, endCol)
         * <br><br>将startCol, endCol之间的所有行全部删除
         *
         * @apiName  deleteMultiColumn
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         * @apiParam {int} startCol 开始列号
         * @apiParam {int} endCol 结束列号
         *
         * @apiSuccess (返回值){void}  - 无返回值
         *
         * @apiExample   {js}示例：
         *
         *  sheet.deleteMultiColumn(3,6);
         *
         */
        deleteMultiColumn: function (startCol, endCol) {

            var scene = new Property();
            scene.put("startcol", startCol);
            scene.put("endcol", endCol);

            var cmd = new CMD_DeleteMultiColumn(this, scene);
            cmd.execute();

        }
        ,


// 如果Row为Null，那么创建它

        getRow: function (row) {
            if (row < 0) return null;
            var rc = this.RPM.getRowCount();
            if (row >= rc) return null;
            var /*Row*/     R = this.SheetRows[row];
            if (R == null)
            {
                R = new Row(this, this.CPM.getColumnCount());
                this.SheetRows[row] = R;
            }
            return R;
        }
        ,

        /**
         *
         * @param row
         * @param R
         */
        $setRow: function (/*int*/  row, /*Row*/R) {
            if (row < 0) return;
            var rc = this.RPM.getRowCount();
            if (row >= rc) return;
            this.SheetRows [row] = R;

        }
        ,


        getColumn: function (col) {
            if (col < 0) return null;
            var cc = this.CPM.getColumnCount();
            if (col >= cc) return null;
            var ret = [];
            var rc = this.RPM.getRowCount();
            for (var row = 0; row < rc; row++)
            {
                ret.push(this.$cells(row, col));
            }
            return ret;
        }
        ,

        $setColumn: function (col, cellCol) {
            if (col < 0) return;
            var cc = this.CPM.getColumnCount();
            if (col >= cc) return;

            var rc = this.RPM.getRowCount();
            for (var row = 0; row < rc; row++)
            {
                var /*Row*/   R = this.SheetRows[row];
                if (R == null) continue;
                R.$setCell(col, cellCol[row]);
            }

        }
        ,

        rebuildCellsDefine: function (/*WorkSheet*/     trigger, /*RedefineBecause*/        because) {
            var rc = this.RPM.getRowCount();
            var cc = this.CPM.getColumnCount();
            for (var row = 0; row < rc; row++)
                for (var col = 0; col < cc; col++)
                {
                    var cell = this.$cells(row, col);
                    if (cell == null) continue;
                    cell.RebuildDefine(trigger, because);
                }
        }
        ,

        onContextMenuClicked: function (menuName) {

            var p = DrawTool.c_last; //当前的鼠标点，相对View的
            if (p != null)
            {
                var isPrint = false;
                var row = this.View.GetRowAtPoint(isPrint, p.y);
                if (Util.isObject(row)) row = row.row;//atRow可能是一个对象，参看GetRowAtPoint
                var col = this.View.GetColAtPoint(isPrint, p.x, row);
                var cell = this.cells(row, col);
                var innerRow = -1;
                if (cell != null) innerRow = cell.getInnerRowAtPoint(p);
                this.Book.EM.fire("onContextMenuClicked", [this, cell, innerRow, menuName]);
            }
        }
        ,


        refreshEditStyleOfCellAtPoint: function () {
            var p = DrawTool.c_last; //当前的鼠标点，相对View的
            if (p != null)
            {
                var isPrint = false;
                var row = this.View.GetRowAtPoint(isPrint, p.y);
                if (Util.isObject(row)) row = row.row;//atRow可能是一个对象，参看GetRowAtPoint
                var col = this.View.GetColAtPoint(isPrint, p.x, row);
                var cell = this.cells(row, col);
                var innerRow = -1;
                if (cell != null) innerRow = cell.getInnerRowAtPoint(p);
                this.refreshEditStyleOfCell(this, cell, innerRow);

            }
        }
        ,

        refreshEditStyleOfCell: function (sheet, cell, innerRow) {

            if (cell.isMerged()) cell = cell.getLeftTopCorner();
            this.Book.EM.fire("refreshEditStyleOfCell", [sheet, cell, innerRow]);
        }
        ,


        /**
         * @api {getDataSourceNameAtCurrentMousePoint} 函数   getDataSourceNameAtCurrentMousePoint
         *
         * @apiDescription getDataSourceNameAtCurrentMousePoint()
         * <br><br> 返回当前鼠标所在单元格绑定的数据源的名称
         *
         * @apiName  getDataSourceNameAtCurrentMousePoint
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *

         *
         * @apiSuccess (返回值){String} - 返回当前鼠标所在单元格绑定的数据源的名称
         *
         * @apiExample   {js}示例：
         *
         *  var name= sheet.getDataSourceNameAtCurrentMousePoint();
         *
         */
        getDataSourceNameAtCurrentMousePoint: function () {
            var p = DrawTool.c_last; //当前的鼠标点，相对View的
            if (p != null)
            {
                var isPrint = false;
                var row = this.View.GetRowAtPoint(isPrint, p.y);
                if (Util.isObject(row)) row = row.row;//atRow可能是一个对象，参看GetRowAtPoint
                var col = this.View.GetColAtPoint(isPrint, p.x, row);
                var cell = this.cells(row, col);
                var innerRow = -1;

                if (cell != null)
                {
                    // 合并的单元格应该复制左上角的单元格的数据
                    if (cell.isMerged()) cell = cell.getLeftTopCorner();

                    var /*DBBindConfig*/              dbc = cell.getBind();
                    if (dbc != null)
                    {
                        return dbc.getDataSource();
                    }
                }
            }
            return "";
        }
        ,


        /**
         *
         * 把当前选中区域的设置导出到本在存储
         * @returns {string}
         */
        exportSelectionDesign: function () {
            var /*ClipBoard*/     cp = new ClipBoard(this, this.Selection, ClipBoard.COPY, true);
            var s = cp.Scene.toString();
            window.localStorage['workSheetExportSelection'] = s;
            return s;

        }
        ,

        /**
         * 从本地存储中复制单元格设置
         */
        importDesign: function () {
            var s = window.localStorage['workSheetExportSelection'];
            s = JSON.parse(s);
            if (s['startRow'] == undefined) return;
            var source = new Property();
            source.property = s;
            var cpr = new Range(source.get("startRow"), source.get("startCol"), source.get("endRow"), source.get("endCol"));

            var scene = new Property();

            scene.put("action", ClipBoard.COPY);
            scene.put("startrow", this.Selection.startRow);
            scene.put("startcol", this.Selection.startCol);
            scene.put("range", cpr);
            scene.put("source", source);
            scene.put("withsize", true);


            var cmd = new CMD_Paste(this, scene);
            cmd.execute();


        }
        ,

        /**
         * @api {copy} 函数   copy
         *
         * @apiDescription copy（）
         * <br><br> 执行复制操作
         *
         * @apiName  copy
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *

         *
         * @apiSuccess (返回值){void} - 无返回值
         *
         * @apiExample   {js}示例：
         *
         *   sheet.copy();
         *
         */
        copy: function () {
            this.copyOrCut(ClipBoard.COPY);
        }
        ,

        copyOrCut: function (action) {
            var /*ClipBoard*/     cp = new ClipBoard(this, this.Selection, action);
            this.Book.setClipBoard(cp);

            //增加对单元格数据的复制

            var p = DrawTool.c_last; //当前的鼠标点，相对View的
            if (p != null)
            {
                var isPrint = false;
                var row = this.View.GetRowAtPoint(isPrint, p.y);
                if (Util.isObject(row)) row = row.row;//atRow可能是一个对象，参看GetRowAtPoint
                var col = this.View.GetColAtPoint(isPrint, p.x, row);
                var cell = this.cells(row, col);
                var innerRow = -1;
                if (cell != null)
                {
                    // 合并的单元格应该复制左上角的单元格的数据
                    if (cell.isMerged()) cell = cell.getLeftTopCorner();
                    innerRow = cell.getInnerRowAtPoint(p);

                    var text = cell.getShowText(innerRow);

                    console.info(text);
                    Tools.copyToClipboard(text);
                }

            }

        }
        ,
        /**
         * @api {cut} 函数   cut
         *
         * @apiDescription cut（）
         * <br><br> 执行剪切操作
         *
         * @apiName  cut
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *

         *
         * @apiSuccess (返回值){void} - 无返回值
         *
         * @apiExample   {js}示例：
         *
         *   sheet.cut();
         *
         */
        cut: function () {
            this.copyOrCut(ClipBoard.CUT);
        }
        ,

        /**
         * @api {paste} 函数   paste
         *
         * @apiDescription paste（）
         * <br><br> 执行粘贴操作
         *
         * @apiName  paste
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *

         *
         * @apiSuccess (返回值){void} - 无返回值
         *
         * @apiExample   {js}示例：
         *
         *   sheet.paste();
         *
         */
        paste: function () {
            var /*ClipBoard*/        cp = this.Book.getClipBoard();
            if (cp == null) return;
            // 可能已经删除了
            if (cp.Sheet == null) return;
            var /*Range*/    cpr = cp.range;

            if (cpr.startRow == this.Selection.startRow && cpr.startCol == this.Selection.startCol) return;

            var scene = new Property();
            scene.put("action", cp.Action);
            scene.put("startrow", this.Selection.startRow);
            scene.put("startcol", this.Selection.startCol);
            scene.put("range", cpr);
            scene.put("source", cp.Scene);

            var cmd = new CMD_Paste(this, scene);
            cmd.execute();

        }
        ,


        /**
         * @api {getColumnCount} 函数   getColumnCount
         *
         * @apiDescription getColumnCount()
         * <br><br>得到列数
         *
         * @apiName  getColumnCount
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *

         *
         * @apiSuccess (返回值){int} - 返回工作表的列数
         *
         * @apiExample   {js}示例：
         *
         *  var cc= sheet.getColumnCount();
         *
         */
        getColumnCount: function () {

            return this.CPM.getColumnCount();
        }
        ,

        /**
         * @api {getRowCount} 函数   getRowCount
         *
         * @apiDescription getRowCount()
         * <br><br>得到行数
         *
         * @apiName  getRowCount
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *

         *
         * @apiSuccess (返回值){int} - 返回工作表的行数
         *
         * @apiExample   {js}示例：
         *
         *  var rc= sheet.getRowCount();
         *
         */
        getRowCount: function () {

            return this.RPM.getRowCount();
        }
        ,


        /**
         * @api {isCellNull} 函数   isCellNull
         *
         * @apiDescription isCellNull
         * <br><br> 单元格是null吗， 并不是判断单元格内的数据是否是空而是判断指定行列中是不是空单元格，即单元格对象本身是否为 null
         *
         * @apiName  isCellNull
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         * @apiParam (1个参数){variant}  parm  单元格名称，或别名
         *
         * @apiParam (2个参数){int}  row 行号
         * @apiParam (2个参数){int}  col 列号
         *
         * @apiSuccess (返回值){boolean} - 指定的单元格是空的吗
         *
         * @apiExample   {js}示例：
         *
         *  sheet.isCellNull(3,4);
         *  sheet.isCellNull("a1");
         *
         */
        isCellNull: function () {
            if (arguments.length == 2)
            {
                var row = arguments[0];
                var col = arguments[1];
                return this.$cells(row, col) == null;
            }

            if (arguments.length == 1)
            {
                var cellName = arguments[0];
                var rc = this.cellNameToRowColumn(cellName);
                if (rc == null) return true;
                return this.$cells(rc.row, rc.col) == null;
            }
            return true;
        }
        ,


        /**
         * 本函数在Cell_.setAlias 中调用 , 设置别名
         * @param cell
         * @param alias
         * @param replace
         * @returns {*}
         */
        $setCellAlias: function (/*Cell*/ cell, /*String*/  alias, /*boolean*/ replace) {
            if (cell == null) return "";

            if (!alias) alias = ""; //取消别名
            if (alias == '') alias = null;


            // 如果是取消别名
            if (alias == null)
            {
                var oldAlias = cell.getAlias().toLowerCase();
                if (this.AliasMap[oldAlias]) delete this.AliasMap[oldAlias];

            } else
            {
                alias = alias.toLowerCase();
                if (this.AliasMap[alias])
                {
                    var /*Cell*/     c = this.AliasMap[alias];
                    if (c !== cell)
                    {
                        if (replace) //如果替换　，那么先清除旧的别名缓存，再设置上别名
                        {
                            c.setAlias("");
                            this.AliasMap[alias] = cell;
                            return "";
                        } else
                        {
                            var errorInfo = c.name + "的别名是" + alias + "，不能为" + cell.name + "定义相同的别名";
                            this.Book.EM.fire('error', [errorInfo]);
                            return errorInfo;
                        }
                    }
                }
                this.AliasMap[alias] = cell;
            }

            return "";
        }
        ,


        /**
         * 强制别名优先
         */
        getCellByAlias: function (alias) {
            var s = alias;
            if (s != null) s = s.toLowerCase();
            if (this.AliasMap[s]) return this.AliasMap[s];

            //支持  query_1.col1 这样的直接用结果集.字段名方式来直接访问结果集数据
            //创建虚拟单元格来处理这些没有实际坐标的假单元格，这样，可以把它们与实际的单元格用相同的方式进行访问
            var /*Cell*/ cell = null;
            var t = alias.split("[.]");
            if (t.length >= 2)
            {
                var datasource = t[0];
                var dbcol = alias.substring(datasource.length() + 1);
                this.addExpression(alias, datasource, dbcol); //此函数不会重复添加  注册虚拟单元格
                cell = this.getExpression(alias);

            }
            return cell;

        }
        ,


        setViewScale: function (scale) {
            if (this.m_viewScale == scale) return;
            if (scale < 0) return;
            if (scale > 1000) return;
            this.m_viewScale = scale;
            if (this.View != null) this.View.repaint();

        }
        ,

        getViewScale: function () {
            return this.m_viewScale;
        }
        ,


        getPrintServiceName: function () {
            return this.printServiceName;
        }
        ,

        setPrintServiceName: function (/*String*/  printServiceName) {
            this.printServiceName = printServiceName;
        }
        ,

        /**
         * 解析表达式引用了哪些单元格
         */

        parseDependList: function (expression) {
            var ret = [];
            if (expression == null) return ret;
            expression = expression.trim();
            if (expression.startsWith("=")) expression = expression.substring(1); // 去掉等号

            var options = {attachComment: true, range: true, loc: true, sourceType: 'script', tokens: true};
            var tokens = [];
            try
            {
                tokens = Esprima.parse(expression, options).tokens;
            } catch (err)
            {
                console.error(expression + " 表达式定义不合法：" + err);
            }

            for (var i = 0; i < tokens.length; i++)
            {
                var token = tokens[i];
                if (token.type != 'Identifier') continue;

                var varName = token.value;

                var isCell = varName.indexOf(':') < 0;// 没有：表示是单元格名称或别名
                var isRange = varName.indexOf(':') > 0; // 有:表示是单元格区域
                var isOtherSheet = varName.indexOf('!') > 0;


                var mr = null;
                var sheet = null;

                if (isOtherSheet)
                {
                    var t = varName.split('!');
                    var sheetName = t[0];
                    varName = t[1];

                    sheet = this.Book.getWorkSheet(sheetName);
                    if (sheet == null)
                    {
                        console.log(sheetName + "不存在");
                        continue;
                    }
                } else
                {
                    sheet = this;
                }


                var dependShowText = varName.startsWith("$");
                //是否是引用显示值

                if (isCell)
                {
                    var ce = sheet.cells(varName);
                    if (ce == null)
                    {
                        //可能是不正确的别名，单元格名，也可能是函数，忽略它
                        continue;
                    }
                    mr = new MergedRange(ce, ce);
                }

                if (isRange)
                {
                    //TODO  在range 中，dependShowText可能需要重新判断一下，但是使用较少，所以先不管了
                    var range = sheet.rangeNameToRange(varName);
                    if (range == null) continue;
                    mr = new MergedRange(sheet, range);
                }

                if (mr == null) continue;

                var da = {sheetGuid: sheet.guid, mergedRange: mr, token: token, dependShowText: dependShowText};
                ret.push(da);

            }

            return ret;
        }
        ,


        parseInnerParameter: function (str, innerRow, useDemoData, moreVars) {

            // ([^\\{]+) 表示所有非{字符的组合
            //([^\\}]+) 表示所有非}字符的组合

            //注意，不要加参数 g ,加了，match的结果就是所有匹配的字符串内容。不加，就是第一个匹配的字符串，及各级()中的内容
            //这里需要的是后者
            // 正确匹配后， result[0]是整个匹配串，[1]是前后去掉空格后的内容 [2]是:与{之前的表达式内容 [3]是测试值内容
            var res = [/(::\s*([^\{]+)\s*\{\s*([^\}]*)\s*\})/,
                /(:\s*([^\{]+)\s*\{\s*([^\}]*)\s*\})/];


            for (let i = 0; i < res.length; i++)
            {
                let re = res[i];
                while (true)
                {
                    var result = str.match(re);
                    if (result == null) break;
                    if (result.length == 4)
                    {
                        var total = result[0];
                        var express = result[2];
                        var demoValue = result[3];
                        var value = demoValue;
                        if (!useDemoData) value = this.evaluate(express, innerRow, false, demoValue, moreVars);
                        if (value == null) value = demoValue;
                        str = str.substring(0, result.index) + value + str.substring(result.index + total.length);
                    }
                }
            }

            return str;


        }
        ,


        /**
         * @api {evaluate} 函数   evaluate
         *
         * @apiDescription evaluate(formula,  innerRow,  returnException, defaultValue)
         * <br><br> 计算指定公式的值。
         * <br><br>
         *  <b>细节</b><br>
         *  与常规的表达式不同的是，单元格公式中的数据，可能是单个数据，也可能是一个数组
         *  比如  sum( a1:b10)  ,它是对数组进制运算，结果是一个数值，再比如  sl * dj  ,当sl 与 dj 中的一个是数组时，
         *  结果也是一个数组。总结起来，规则如下：
         * <ul>
         * <li>四则运算 ， 当参与运算的任意一个数据是数组时，结果就是数组</li>
         * <li> 非聚合类函数：当其中的任意一个参数是数组时，结果也是数组</li>
         * <li> 聚合类函数：无论参数是否数组， 结果都是单一的数值 ，比如 max( a1,b1,c1)  , sum( a1:b2)  , count( c3)</li>
         * </ul>
         *
         * 由于在js中并不能对直接对数组做四则运算（会转换成字符串进行运算），比如 [1,2] + [3,4 ] 需要= [1+3 , 2+4] 即对相同索引号的数据进行运算
         * 因此，可以考虑把四则运算转换成自定义的函数，通过函数来实现上述运算方法，这样， 就把表达式运算全部转换成函数调用。
         * 通过exprima 分析语法树时， 所有的二元运算，都会解析成BinaryExpression， 一元运算解析为 UnaryExpression
         * 而在这里的公式中只允许存在四则运算及函数调用，所以可以通过分析语法树，来实现四则运算转换成函数调用。比如把 + 转换成 add 函数，
         *  3+4 转换成 add(3,4)
         *
         *
         * @apiName  evaluate
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         * @apiParam {String} formula 公式，即表达式
         * @apiParam {int} innerRow 单元格内子行号。当公式中引用的单元格绑定到多行结果集时，使用innerRow行的数据参与表达式解析
         * @apiParam {String} returnException true 表示直接返回异常信息， false 表示发生异常时返回defaultValue
         *
         * @apiParam {variant} defaultValue 默认值
         * @apiParam {Object} moreVars 更多的变量值

         *
         * @apiSuccess (返回值){int} - 返回表达式的计算结果
         *
         * @apiExample   {js}示例：
         *
         *  sheet.evaluate("= a1+b1" );
         *  sheet.evaluate(" =a1+b1" , -99 , true, 4);
         *
         */
        evaluate: function (/*String*/     formula, /*int*/innerRow, /*boolean*/returnException, defaultValue, moreVars) {


            moreVars = moreVars || {};

            //2019.04.09 在没有初始化之前，计算列什么的，不要计算，
            //2020.07，03 去掉了这个控制， 因为 在单据打开后，检索数据时，initOK还没有被设置 ，但是它又需要计算动态检索数据包，
            //if (!this.Book.initOK) return null;

            if (innerRow == undefined) innerRow = -99; //为什么要用-99 参看　Cell_.getValue(row)中的说明
            returnException = returnException || true;


            var ret = null;

            if (formula == null) return null;
            if (formula == "") return "";

            var ev = EvaluatePool.getInstance();//从池中得到对象

            var funcFormula = ev.convertFormulaToFunction(formula, moreVars);


            moreVars.$sheet = this;
            moreVars.$clientCell = null;
            moreVars.$innerRow = innerRow;


            //  把与当前状态有关的数据加入进JEP
            window['worksheet_currentPrintingPage'] = this.getCurrentPrintingPage() + 1;
            window['worksheet_currentPrintingPageCount'] = this.getCurrentPrintingPageCount();


            var ret = ev.evaluate(moreVars, funcFormula);
            EvaluatePool.recycle(ev); //释放回池中
            return ret;


        }
        ,


        isVisibleInTabContainer: function () {
            return this.VisibleInTabContainer;
        },


        isVisible: function () {

            return this.Visible;
        }
        ,

        setVisible: function (visible) {
            if (this.Visible == visible) return;

            if (this == this.Book.activeSheet)
            {
                console.error("活动sheet不能设置visible=false ,请先激活另一个sheet");
                return;
            }

            this.Visible = visible;

            //相应的tab 也可同步可见性
            var view = this.Book.View;
            if (this.pSheetContainer != null)
            {
                view.setTabVisible(this.guid, false);
            } else
            {
                if (view != null) view.setTabVisible(this.guid, visible);
            }

        }
        ,


        checkRowIsMultiRowDataSource: function (r) {
            var needRedrawRects = [];
            var r1 = r;
            r1 = this.$copyRangeAndRepaint(r, r, needRedrawRects);


            var CN = this.columnCount;
            for (var row = r1.startRow; row <= r1.endRow; row++)
            {
                var dsn = [];
                var bindCount = 0;

                for (var col = 0; col < CN; col++)
                {
                    var cell = this.$cells(row, col);
                    if (cell == null) continue;
                    if (cell.Bind == null) continue;
                    if (dsn.contains(cell.Bind.dataSource)) continue;
                    if (cell.Bind.dsType != DataSourceConfig.MultiRowDataSource) continue;


                    dsn.push(cell.Bind.dataSource);
                    bindCount++;
                }

                if (dsn.length == 0)
                {
                    this.RPM.setBindCount(row, 0);
                    this.RPM.setMultiRowDataSourceName(row, '');
                }
                if (dsn.length == 1)
                {
                    this.RPM.setMultiRowDataSourceName(row, dsn[0]);
                    this.RPM.setBindCount(bindCount);
                }
                if (dsn.length > 1)
                {
                    this.RPM.setMultiRowDataSourceName(row, dsn[0]);
                    this.RPM.setBindCount(bindCount);
                    alert("不建议在同一行中绑定多个多行数据源");
                }


            }

        }
        ,

        /**
         * @api {addExpression} 函数   addExpression
         *
         * @apiDescription addExpression(name, p2,p3)
         * <br><br> 增加一个表达式，它被视为一个单元格，它类似于一个单元格，只是没有行列坐标
         *
         * @apiName  addExpression
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         *
         * @apiParam (2个参数){String} name 表达式名称
         * @apiParam (2个参数){String} p2  公式定义
         *
         * @apiParam (3个参数){String} name 表达式名称
         * @apiParam (3个参数){String} p2  数据源名称
         * @apiParam (3个参数){String} p3  字段名称
         *
         *
         *
         * @apiSuccess (返回值){Cell} - 返回一个Cell对象，但它并不是工作表中具有坐标的单元格，而是一个虚拟的单元格，它能参与公式
         * 计算，但是没有坐标
         *
         * @apiExample   {js}示例：
         *
         *  sheet.addExpression( "je" , "=sl*dj");
         *
         *  sheet.addExpression("je", "ds1", "je"); //一个虚拟单元格，绑定到数据源ds1的je字段上
         *
         */
        addExpression: function (/*String*/  name, /*String*/ p2, /*String*/p3) {

            var expression = "";
            var dataSource = "";
            var dbCol = "";
            if (p3 == undefined)
            {
                expression = p2;
            } else
            {
                dataSource = p2;
                dbCol = p3;
            }

            //2020.11.12 patch
            //因为有时候定义表达式时，可能省略了=号，那么检查一下
            expression= expression.trim();
            if( !expression.startsWith("=")) expression="="+expression;

            if (this.ExpressionMap == null) this.ExpressionMap = {}; //HashMap < String, Cell > ();
            if (this.ExpressionList == null) this.ExpressionList = []; //ArrayList < Cell > ();

            name = name.toLowerCase();

            var /*Cell*/  cell = null;
            if (this.ExpressionMap[name] != undefined)
            {
                cell = this.ExpressionMap[name];
            } else
            {
                cell = new Cell(this, null);
                this.ExpressionMap[name] = cell;
                this.ExpressionList.push(cell);
            }

            if (expression != '')
            {

                cell.setValue(expression);
            } else
            {
                cell.setBind(dataSource, dbCol);
            }
            return cell;
        }
        ,


        /**
         * @api {removeExpression} 函数   removeExpression
         *
         * @apiDescription removeExpression(name)
         * <br><br>删除一个虚拟单元格定义
         *
         * @apiName  removeExpression
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         * @apiParam {String} name 虚拟单元格名称
         *
         * @apiSuccess (返回值){boolean} - true表示删除成功
         *
         * @apiExample   {js}示例：
         *
         *  sheet.removeExpression("je");
         *
         */
        removeExpression: function (/*String*/  name) {
            if (this.ExpressionMap == null) return false;
            name = name.toLowerCase();

            var cell = null;
            if (this.ExpressionMap[name])
            {
                delete this.ExpressionMap[name];
                return true;
            } else
            {
                return false;
            }

        }
        ,


        /**
         * @api {getExpression} 函数   getExpression
         *
         * @apiDescription getExpression(name)
         * <br><br>获取虚拟单元格对象
         *
         * @apiName  getExpression
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         * @apiParam {String} name 虚拟单元格名称
         *
         * @apiSuccess (返回值){Cell} - 返回虚拟单元格对象 ,null表示不存在
         *
         * @apiExample   {js}示例：
         *
         *  var cell=sheet.getExpression("je");
         *  alert( cell.getValue());
         *
         */
        getExpression: function (/*String*/  name) {
            if (this.ExpressionMap == null) return null;
            name = name.toLowerCase();
            var cell = null;
            return this.ExpressionMap[name];
        }
        ,

        isFormulaBackFillBindEnabled: function () {

            return this.formulaBackFillBindEnabled;
        }
        ,

        setFormulaBackFillBindEnabled: function (auto) {
            this.formulaBackFillBindEnabled = auto;

        }
        ,


        /**
         * @api {reCalculateAllFormula} 函数   reCalculateAllFormula
         *
         * @apiDescription reCalculateAllFormula()
         * <br><br>  通知所有的公式重新计算
         * <br><br>在单元格的一些函数比如 setBind ，它们通常需要通知引用自己的单元格 重新计算，当需要多次调用
         * setBind时，这个通知操作就比较影响性能，所以通常可能关闭通知操作，并在合适的时候调用本函数来通知所有公式重新计算
         *
         *
         * @apiName  reCalculateAllFormula
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *

         * @apiSuccess (返回值){void} - 无返回值
         *
         * @apiExample   {js}示例：
         *
         *  sheet.reCalculateAllFormula();
         *
         */
        reCalculateAllFormula: function () {
            var RC = this.rowCount;
            var CC = this.columnCount;
            for (var row = 0; row < RC; row++)
            {
                for (var col = 0; col < CC; col++)
                {
                    var cell = this.$cells(row, col);
                    if (cell == null) continue;
                    cell.setInvalid();
                }
            }
        }
        ,


        /**
         * @api {makeSureAllFormulaValid} 函数   makeSureAllFormulaValid
         *
         * @apiDescription
         * <br><br> 确保每个公式都已经进行计算。 与reCalculateAllFormula不同的是，本函数会检查worksheet上的每个单元格，
         *如果单元格定义了公式，如果它已经做了计算，那么不再做任何处理，如果尚未计算，则马上进行计算。而 reCalculateAllFormula
         * 则是强制所有有公式的单元格，无论是否进行过计算，都强制重新再计算一次。
         *
         * @apiName  makeSureAllFormulaValid
         * @apiGroup Cell
         * @apiVersion 1.0.0
         *
         * @apiSuccess (返回值){void} - 无返回值
         *
         * @apiExample   {js}示例：
         *
         *  //示例
         *for( var i=0;i<book.getWorkSheetCount(); i++)
         *{
             *
             *  var sheet=book.getWorkSheet(i);
             *  sheet.reCalculateAllFormula();
             *}
         *
         *
         */
        makeSureAllFormulaValid: function () {
            var RC = this.rowCount;
            var CC = this.columnCount;
            for (var row = 0; row < RC; row++)
            {
                for (var col = 0; col < CC; col++)
                {
                    var cell = this.$cells(row, col);
                    if (cell == null) continue;
                    if (cell.isFormula())
                    {
                        if (!cell.isValid) cell.setInvalid();
                    }
                }
            }

        }
        ,

        removeBackgroundImage: function (id) {

            for (var layer in this.backgroundImageMap)
            {
                var layerMap = this.backgroundImageMap[layer];
                if (layerMap[id] != undefined)
                {
                    delete layerMap[id];
                    return;
                }
            }

            if (this.View != null) this.View.repaint();
        }
        ,

        /**
         * 增加一个背景设置
         * @param id
         * @param url
         * @param startCellName
         * @param endCellName
         * @param x_position
         * @param y_position
         * @param x_repeat
         * @param y_repeat
         * @param layer
         * @param needPrint
         * @returns {String}  ‘‘表示成功， 否则返回错误提示’
         */
        addBackGroundImage: function (/*String*/id,
                                      /*String*/ url,
                                      /*String*/startCellName,
                                      /*String*/ endCellName,
                                      /*String*/x_position,
                                      /*String*/y_position,
                                      /*String*/ x_repeat,
                                      /*String*/ y_repeat,
                                      /*int*/   layer,
                                      /*boolean*/   needPrint) {

            var lay = '' + layer;
            var /*HashMap<String , BackGroundImageConfig>*/  map;
            //背景是分层存放的，首先由层次号得到对应的列表
            if (this.backgroundImageMap[lay])
            {
                map = this.backgroundImageMap[lay];
            } else
            {
                map = {};
                this.backgroundImageMap[lay] = map;
            }

            if (map[id]) return id + "已经存在，请重新取个名称";

            var config = {
                id: id,
                url: url,
                x_position: x_position,
                y_position: y_position,
                x_repeat: x_repeat,
                y_repeat: y_repeat,
                startCell: startCell,
                endCell: endCell,
                needPrint: needPrint
            };

            map[id] = config;

            if (this.View != null) this.View.repaint();

            return "";

        }
        ,


        /**
         * @api {getBrick} 函数   getBrick
         *
         * @apiDescription getBrick(name)
         * <br><br>获取指定名称的控件对象。注意，由于控件是加在单元内的，只有单元格内控件名称保持唯一即可。所以不同的单元中插入相同
         * 名字的控件，是允许的。但是此时使用本函数获取控件时，只会返回第一个使用该名称的控件。
         *
         * @apiName  getBrick
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         * @apiParam {String} name 控件名称
         *
         * @apiSuccess (返回值){Brick} - 返回控件对象。<br>参看：<a href='../../Brick/doc/index.html'>Brick</a>
         *
         * @apiExample   {js}示例：
         *
         *  var cb= sheet.getBrick("cb1");
         *  cb.setTitle("确定");
         */
        getBrick: function (/*String*/    name) {
            var RC = this.rowCount;
            var CC = this.columnCount;
            var brick = null;
            for (var row = 0; row < RC; row++)
            {
                for (var col = 0; col < CC; col++)
                {
                    var /*Cell*/        cell = this.$cells(row, col);
                    if (cell == null) continue;
                    brick = cell.getBrick(name);
                    if (brick != null) return brick;
                }
            }
            return null;

        }
        ,

        /**
         * @api {getBricks} 函数   getBricks
         *
         * @apiDescription getBricks()
         * <br><br>获取本sheet上的所有控件
         *
         * @apiName  getBricks
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *

         *
         * @apiSuccess (返回值){Array} - 返回控件数组
         *
         * @apiExample   {js}示例：
         *
         *  var cbs= sheet.getBricks();
         *  alert( cbs.length);
         */
        getBricks: function () {
            let ret=[];
            let RC = this.rowCount;
            let CC = this.columnCount;

            for (var row = 0; row < RC; row++)
            {
                for (var col = 0; col < CC; col++)
                {
                    let /*Cell*/        cell = this.$cells(row, col);
                    if (cell == null) continue;
                    let bn= cell.getBrickCount();
                    for( var bi=0;bi<bn;bi++)
                    {
                        let brick = cell.getBrick(bi);
                        ret.push(brick);
                    }
                }
            }
            return ret;

        }
        ,


        findContextMenu: function (/*LinkedHashMap*/ map, /*String*/ name) {
            //TODO
        }
        ,


        addContextMenu: function (/*String*/    name, /*String*/caption, /*String*/  parentMenuName) {
            //TODO
        }
        ,

        removeContextMenu: function (/*String*/    name) {
            //TODO

        }
        ,

        setContextMenuEnabled: function (/*String*/ name, /*boolean*/ enabled) {
            //TODO
        }
        ,


        isContextMenuEnabled: function (/*String*/  name) {
            //TODO
        }
        ,


        setPrintable: function (b) {
            this.isPrintable = b;

        }
        ,

        isPrintable: function () {
            return this.isPrintable;
        }
        ,

        setRebuildCellDefineAfterInsertOrDeleteRow: function (/*boolean*/  b) {
            this.rebuildCellDefineAfterInsertOrDeleteRow = b;

        }
        ,


        isRebuildCellDefineAfterInsertOrDeleteRow: function () {

            return this.rebuildCellDefineAfterInsertOrDeleteRow;
        }
        ,


        setFloatToTop: function (startRow, endRow) {
            this.floatToTopStartRow = startRow;
            this.floatToTopEndRow = endRow;

        }
        ,


        getFloatToTopStartRow: function () {
            return floatToTopStartRow;
        }
        ,

        getFloatToTopEndRow: function () {
            return floatToTopEndRow;
        }
        ,


        /**
         * @api {popup} 函数   popup
         *
         * @apiDescription popup(title, width, height, modal , closeIcon,buttonAction)
         * <br><br>将Sheet以弹出框方式显示。
         * <ul>
         *     <li>当前活动Sheet不能弹出显示，不是当前Sheet才允许弹出显示</li>
         *     <li>当只有一个WorkSheet时，不允许弹出显示</li>
         *     <li>弹出显示的Sheet与整个WorkBook仍是处于同一个页面中，弹出Sheet仅仅是显示方式发生变化，它的数据结构保持不变，访问方式保持不变</li>
         *     <li>原来在脚本中怎么访问本sheet，现在仍是一样的访问方式</li>
         *     <li>sheet弹出显示的状态，就是上次关闭时的状态。这也再次证实弹出显示的Sheet仅仅是显示方式变化，其它不变</li>
         *     <li>Sheet一旦弹出显示， 它只能是弹出显示或不显示，回不到之前的Tab显示状态</li>
         *     <li>与 openDialogURL 不同， openDialogURL显示的是一个新页面，与当前页面不能互相访问，
         *     只能通过传递参数及返回参数的方式进行通信。但是弹出显示的Sheet与其它Sheet仍处于同一页面</li>
         *     <li> 注意， 本函数执行一次后，弹窗的标题，尺寸，按钮等设置项就生效了，并且当使用不同的参数再次执行本函数时，参数并不会生效。除非先使用
         *     popupDispose()函数销毁弱窗 。</li>
         *     </ul>
         *
         * @apiName  popup
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         * @apiParam {String} icon 弹框左上角的图标，示例  'fa fa-cog' ,可用的图标参看 http://www.fontawesome.com.cn/faicons/
         * @apiParam {String} title 弹框的标题，如果为''表示不显示标题
         * @apiParam {int} width 弹框的宽度
         * @apiParam {int} height 弹框的高度
         * @apiParam {boolean} modal 是否是模态显示，本参数只能是true
         * @apiParam {boolean} closeIcon 弹框右上角是否显示关闭按钮
         * @apiParam {JSON} buttonAction 一个JSON对象，用来描述弹框需要显示的按钮，及点击后执行的操作
         *
         * <ul>
         *     <li>本参数为null表示不显示任何按钮</li>
         *     <li>如果没有一个名为 cancel 的设置项，则系统自动添加它，即自动添加一个取消按钮.如果不相有一个“取消”按钮，那么
         *     必须定制 cancel 设置项。 详见示例
         *     </li>
         *
         * <ul>
         *  @apiParam {JSON} moreConfig 一个JSON对象，用来设置更多的属性
         *  <ul>
         *      <li>backgroundDismiss:  boolean型，true表示在弹窗外点击时，是否关闭弹窗.默认为 false </li>
         *      <li>valign: 窗口垂直对齐方式。默认是'center', 可选 'top','center','bottom' </li>
         *      <li>padding: int型，默认为10 ， 窗口边框尺寸</li>
         *  </ul>
         *
         * @apiSuccess (返回值){void} - 没有返回值
         *
         * @apiExample   {js}示例：
         *
         *
         *  book.sheet1.popup( 'fa fa-cog','选择' , 600,400, true, true ,
         *   {
             *         action1:
             *         {
             *           text: 'action1!',
             *           btnClass: 'btn-red',
             *           action: function ()
             *           {
             *             alert( '点我，不关闭弹窗');
             *             return false; //表示不关闭弹窗
             *           }
             *         } ,
             *
             *         cancel:
             *         {
             *           text: 'action2!',
             *           btnClass: 'btn-green',
             *           action: function ()
             *           {
             *             alert( '点我，关闭弹窗');
             *             return true; //表示不关闭弹窗
             *           }
             *         }  //如果，如果这里没有一个名为 cancel 的设置项，则系统自动添加它
             *
             *   });
         *
         *
         *
         */
        popup: function (/*String*/ icon, /*String*/ title,
                         /*int*/width,
                         /*int*/  height,
                         /*boolean*/ modal,
                         /*boolean*/ closeIcon,
                         buttonAction, moreConfig) {

            if (moreConfig == undefined) moreConfig = {};

            let config = Util.merge(moreConfig, {backgroundDismiss: false, valign: 'center', padding: 10});

            if (this.popupDialog != null)
            {
                if (this.popupDialog.isOpen()) return;
                this.popupDialog.open();
                return;
            }

            if (this == this.Book.activeSheet)
            {
                alert("当前activeSheet不能popup ,请将当前activeSheet设置成其它sheet后，再将本worksheet弹出显示");
                return;
            }

            this.setVisible(false);

            var that = this;

            if (buttonAction == null) buttonAction = {};


            if (Object.getOwnPropertyNames(buttonAction).length > 0)
            {
                if (buttonAction['cancel'] == undefined)
                {
                    buttonAction.cancel =
                        {
                            text: "&nbsp;取&nbsp;&nbsp;消&nbsp;",
                            btnClass: "btn-warning",
                            action: function () {
                                //不用做任何事， 下面的onClose 会做清除
                            }
                        };

                }
            }


            if (closeIcon && title == '') title = '&nbsp;'; //如果显示X按钮，但是又没有标题，那么X按钮也看不到，强加一个标题

            var valign = '';

            if ((height + '').indexOf("%") > 0)
            {
                height = height.replace(/%/g, '');
                height = Math.floor(window.innerHeight * height * 1.0 / 100);
                valign = 'bottom';
            }

            var p = {
                lazyOpen: true,
                theme: 'light',
                boxWidth: width + "px",
                useBootstrap: false,
                backgroundDismiss: config.backgroundDismiss, // this will just close the modal
                icon: icon,

                closeIcon: closeIcon, // || Object.getOwnPropertyNames(buttonAction).length == 0, //如果没有任何按钮，那么强制显示X按钮 , 03.28去掉这个
                closeIconClass: 'fa fa-close',
                animation: 'zoom',
                closeAnimation: 'scale',
                title: title,
                content: `<div  id="popup${this.guid}"   
                            style="width:100%; 
                                    height:${height}px; 
                                    border:1px solid lightgray;" >
                            <div>`,


                onContentReady: function () {

                    //每次打开Dialog时，都会重新初始化  content ,所以需要重新构建一个PopupSheetContainer
                    var psc = new PopupSheetContainer(that.guid + 'popup', {sheetGUID: that.guid}, 0, 0, 0, 0);
                    psc.createDOM(that.Book, $(`#popup${that.guid}`));
                    psc.resize();
                    that.setVisible(true); // 可见标记恢复可见状态，不然上面的brick可能无法显示

                    //2020.06.17 修正
                    //如果本sheet中又有嵌入sheet，那么需要对它的size做个修正
                    //原因是当float面板关闭时，本Sheet从面板中移除，此时SheetContainer中 resize被触发了
                    // 将width ,height 都调整成了100px (为什么是100px，在没搞清楚)
                    // 当本Sheet再次浮动显示时，嵌入的Sheet就成了100px的大小 ，
                    // 解决办法是在浮动显示时将SheetContainer.resize强制调用一下，同理，popup也类似处理
                    let bs= that.getBricks();

                    for( let i=0;i< bs.length;i++)
                    {
                        let brick= bs[i];
                        if( brick.type=='SheetContainer')
                        {
                            setTimeout(function(){ brick.resize();},100);

                        }
                    }



                },

                onOpenBefore: function () {
                    if (valign == 'bottom')
                    {
                        $('.jconfirm-cell').css('vertical-align', config.valign); //容器靠底边
                        $('.jconfirm-box').css({'padding': config.padding + 'px'}); // 边框不要有
                        $('.jconfirm-content-pane').css('margin', 0); // 默认底边有15px的留白， 去掉
                    }
                },
                onClose: function () {

                    //添加到弹框的content 中的内容 ，在close时，会被释放掉，这个会破坏workSheetView , 所以在它破坏前
                    //先把sheetView移回WorkBookView中
                    that.restoreViewToWorkBookView();
                    that.pSheetContainer = null; //取消与容器的关联，避免某些操作触发pSheetContainer卷滚条相关的操作

                    that.setVisible(false); // 可见标记恢复不可见状态，


                }
            };

            if ((width + '').indexOf("%") > 0)
            {
                p.boxWidth = width;
                p.useBootstrap = false;
                p.animation = 'none'; //不要动画
                p.offsetBottom = 0;
                p.offsetTop = 0;


            } else if (width <= 12)
            {
                p.columnClass = 'col-md-' + width;
                p.useBootstrap = true;
            } else
            {
                p.boxWidth = width + "px";
                p.useBootstrap = false;
            }


            if (Object.getOwnPropertyNames(buttonAction).length > 0)
            {
                p.buttons = buttonAction;
                this.popupDialog = $.confirm(p);
            } else  //没有按钮,
            {
                this.popupDialog = $.dialog(p);
            }
            this.popupDialog.open();

        }
        ,


        /**
         * 当sheetView在弹出窗口中显示，且弹窗关闭前，需要把sheetView移回到bookView中，
         *
         */
        restoreViewToWorkBookView: function () {
            this.setVisible(false);
            $(this.View.viewContainer).css("display", "none");
            var viewContainer = $('#' + this.Book.View.containerID + " .viewContainser");
            viewContainer.append(this.View.viewContainer);

            //如果有 tab控件，那么tab控件中的sheet 也要

        }
        ,


        /**
         * @api {popupDispose} 函数   popupDispose
         *
         * @apiDescription popupDispose( )
         * <br><br> 如果Sheet以弹出框方式显示。使用本函数可以关闭弹窗。并且销毁弹窗的标题，尺寸，按钮等设置项。

         *
         * @apiName  popupDispose
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         *
         * @apiSuccess (返回值){void} - 没有返回值
         *
         * @apiExample   {js}示例：
         *
         *
         *  book.sheet1.popupDispose();
         *
         *
         */
        popupDispose()
        {
            if (this.popupDialog != null)
            {
                this.popupDialog.close();
                this.popupDialog = null;
            }
        }
        ,

        /**
         * @api {popupClose} 函数   popupClose
         *
         * @apiDescription popupClose( )
         * <br><br> 如果Sheet以弹出框方式显示。使用本函数可以关闭弹窗。
         * 与 popupDispose不同的是，要函数并不销毁弹窗的标题，尺寸，按钮等设置项。

         *
         * @apiName  popupClose
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         *
         * @apiSuccess (返回值){void} - 没有返回值
         *
         * @apiExample   {js}示例：
         *
         *
         *  book.sheet1.popupClose();
         *
         *
         */
        popupClose()
        {
            if (this.popupDialog != null)
            {
                this.popupDialog.close();

            }
        }
        ,


        noticeSheetContainerResize:function()
        {
            //2020.06.17 修正
            //如果本sheet中又有嵌入sheet，那么需要对它的size做个修正
            //原因是当float面板关闭时，本Sheet从面板中移除，此时SheetContainer中 resize被触发了
            // 将width ,height 都调整成了100px (为什么是100px，在没搞清楚)
            // 当本Sheet再次浮动显示时，嵌入的Sheet就成了100px的大小 ，
            // 解决办法是在浮动显示时将SheetContainer.resize强制调用一下，同理，popup也类似处理
            let bs= this.getBricks();

            for( let i=0;i< bs.length;i++)
            {
                let brick= bs[i];
                if( brick.type=='SheetContainer')
                {
                    setTimeout(function(){ brick.resize();},100);

                }
            }
        },

        /**
         * @api {float} 函数   float
         *
         * @apiDescription
         * <br><br> 把本worksheet 在指定位置按指定大小浮动显示。它与popup不同的是，浮动显示是非模态的
         *
         * @apiName  float
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         * @apiParam {String} icon 浮动窗口左上角的图标，如果为'' 表示不需要显示图标
         * @apiParam {String} title 浮动窗口的标题
         * @apiParam {Object} position 浮动窗口坐标 参看 https://jspanel.de/api.html#options/position
         * @apiParam {Object} size 浮动窗口尺寸 参看 https://jspanel.de/api.html#options/panelSize
         * @apiParam {Object} icon 浮动窗口左上角的图标，如果为'' 表示不需要显示图标
         * @apiParam {String} theme [可选参数]浮动窗口的样式风格可以是  primary , default , success , warning , danger,light,dark
         * @apiParam {Object} moreConfig  [可选参数] 更多的配置，比如事件回调等，详细参看 https://jspanel.de/api.html#options/overview
         *
         *
         *
         * @apiSuccess (返回值){void} - 无返回值
         *
         * @apiExample   {js}示例：
         *
         *  //示例
         *
         *    book.sheet1.float('fa fa-cog', "浮动示例",
         *    {
         *       my: 'right-center',
         *       at: 'right-center',
         *       autoposition: 'down',
         *       offsetX: -5,
         *       offsetY: 5
         *   }, {width:300,height:400});
         *
         *
         *
         */
        float: function (icon, title, position, size, theme, moreConfig) {

            var that = this;
            if (this.floatPanel != null) return;

            if (moreConfig == undefined) moreConfig = {};
            moreConfig.position = position;
            moreConfig.headerTitle = title;
            if (icon != '') moreConfig.headerTitle = "<i class='  " + icon + "'></i>&nbsp;" + title;
            moreConfig.theme = theme || 'default';
            moreConfig.panelSize = size;

            moreConfig.content = `<div  id="popup${this.guid}"   
                            style="width:100%; 
                                    height:100%; 
                                    border:1px solid lightgray;" >
                            <div>`;


            moreConfig.callback = function () {

                //每次打开Dialog时，都会重新初始化  content ,所以需要重新构建一个PopupSheetContainer
                var psc = new PopupSheetContainer(that.guid + 'popup', {sheetGUID: that.guid}, 0, 0, 0, 0);
                psc.createDOM(that.Book, $(`#popup${that.guid}`));
                psc.resize();
                that.setVisible(true); // 可见标记恢复可见状态，不然上面的brick可能无法显示

                //2020.06.17 修正
                //如果本sheet中又有嵌入sheet，那么需要对它的size做个修正
                //原因是当float面板关闭时，本Sheet从面板中移除，此时SheetContainer中 resize被触发了
                // 将width ,height 都调整成了100px (为什么是100px，在没搞清楚)
                // 当本Sheet再次浮动显示时，嵌入的Sheet就成了100px的大小 ，
                // 解决办法是在浮动显示时将SheetContainer.resize强制调用一下，同理，popup也类似处理
                let bs= that.getBricks();

                for( let i=0;i< bs.length;i++)
                {
                    let brick= bs[i];
                    if( brick.type=='SheetContainer')
                    {
                        setTimeout(function(){ brick.resize();},100);

                    }
                }

            };

            moreConfig.onclosed = function () {

                //添加到弹框的content 中的内容 ，在close时，会被释放掉，这个会破坏workSheetView , 所以在它破坏前
                //先把sheetView移回WorkBookView中
                that.restoreViewToWorkBookView();
                that.pSheetContainer = null; //取消与容器的关联，避免某些操作触发pSheetContainer卷滚条相关的操作

                that.floatPanel = null;
                that.setVisible(false); // 可见标记恢复不可见状态，
            };

            this.floatPanel = jsPanel.create(moreConfig);
        }
        ,

        /**
         * @api {floatPanelClose} 函数   floatPanelClose
         *
         * @apiDescription floatPanelClose( )
         * <br><br> 如果Sheet以浮动方式显示。使用本函数可以关闭弹窗。如果不是浮动显示，本函数什么也不做
         *

         *
         * @apiName  floatPanelClose
         * @apiGroup WorkSheet
         * @apiVersion 1.0.0
         *
         *
         * @apiSuccess (返回值){void} - 没有返回值
         *
         * @apiExample   {js}示例：
         *
         *
         *  book.sheet1.floatPanelClose();
         *
         *
         */
        floatPanelClose: function () {
            if (this.floatPanel == null) return;
            this.floatPanel.close();
        }
        ,

        setBalloonVisible: function (/*boolean*/  v) {

            if (this.BalloonVisible == v) return;
            this.BalloonVisible = v;
            if (this.View != null) this.View.repaint();

        }
        ,


        isBalloonVisible: function () {

            return this.BalloonVisible;
        }
        ,

        balloonTipEdit: function (/*String*/ dsname, /*String*/  dbcol, /*int*/    innerRow) {
            //TODO
            this.setBalloonVisible(true);
        }
        ,


        menuConfig2HashMap: function (/*JSONArray*/   ja) {
            //TODO
            return null;
        }
        ,

        showPopupMenu: function (/*String*/menuConfig) {
            //TODO

        }
        ,


        isEditing: function () {
            if (this.View == null) return false;
            return this.View.$getCurrentEditingEdit() != null;
        }


    }
    )
;

export default WorkSheet;
