/**
 * Created by 三宝爹 on 2017/8/6.
 */


import Class from '../base/Class.js';
import RowProperty from './RowProperty.js';
import Tools from '../util/Tools.js';
import Util from '../util/Util.js';


var RowPropertyManage = Class.extend({

    static:
        {
            defaultRowHeight: 25
        },

    properties: {

        /**
         * @api {rowCount} 只读属性 [属性]rowCount
         * @apiName  rowCount
         * @apiDescription WorkSheet的行数
         * @apiGroup RowPropertyManage
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){int} - 行数
         * @apiExample {js}(示例：)
         * var rpm= book.sheet0.getRowPropertyManage();
         * var rc= rpm.rowCount;
         *
         */
        "rowCount": {
            get: function () { return this.getRowCount();}
        },

        /**
         * @api {rowHeadWidth} 可读可写属性 [属性]rowHeadWidth
         * @apiName  rowHeadWidth
         * @apiDescription WorkSheet的行头的宽度
         * @apiGroup RowPropertyManage
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){int} - 行头的宽度
         * @apiExample {js}(示例：)
         * var rpm= book.sheet0.getRowPropertyManage();
         * rpm.rowHeadWidth=50;
         *
         */
        "rowHeadWidth":
            {
                get: function () {return this.getRowHeadWidth();},
                set: function (val) { this.setRowHeadWidth(val);}
            }

    },

    constructor: function (rc, sheet) {
        this.Sheet = sheet;
        this.RowHeadWidth = 50;
        this.rowPropertyList = []; // ArrayList<RowProperty>

        for (var i = 0; i < rc; i++)
        {
            var /*RowProperty*/ RP = new RowProperty();
            this.rowPropertyList.push(RP);

        }
    },


    $$insertBefore: function (beforeRow) {
        if (beforeRow < 0) beforeRow = 0;
        var /*RowProperty*/ RP = new RowProperty();
        var rc = this.getRowCount();
        if (beforeRow >= rc)
        {
            this.rowPropertyList.push(RP); // 在最后追加
        } else
        {
            this.rowPropertyList.splice(beforeRow, 0, RP); // 在beforeRow 位置追加
        }
    },

    $insertMultiRow: function (startRow, endRow) {
        for (var i = startRow; i <= endRow; i++)
        {
            this.$$insertBefore(i);
        }
    },

    $deleteRow: function (row) {
        if (row < 0) return;

        var rc = this.getRowCount();
        if (row >= rc) return;
        this.rowPropertyList.splice(row, 1);

    },

    /*RowProperty*/ getRowProperty: function (row) {
        if (row < 0) return null;
        var rc = this.getRowCount();
        if (row >= rc) return null;
        return this.rowPropertyList[row];

    },

    $setRowProperty: function (row, /*RowProperty*/ rp) {
        if (row < 0) return;
        var rc = this.getRowCount();
        if (row >= rc) return;
        this.rowPropertyList.splice(row, 0, rp);

    },

    getRowCount: function () {

        return this.rowPropertyList.length;
    },

    getFixedRowCount: function () {
        var /*Cell*/ cell = this.Sheet.getFreezedCell();
        if (cell == null) return 0;
        return cell.rowIndex;
    },

    /*
     * 得到行标区的宽度
     */
    getRowHeadWidth: function () {
        return this.RowHeadWidth;
    },

    /*
     * 设置行标区的宽度
     */
    setRowHeadWidth: function (Width) {
        this.RowHeadWidth = Math.max(0, Width);
    },


    /**
     * @api {getRowHeight} 函数   getRowHeight
     *
     * @apiDescription getRowHeight(row )
     * <br><br> 得到行的高度
     *
     * @apiName  getRowHeight
     * @apiGroup RowPropertyManage
     * @apiVersion 1.0.0
     *
     * @apiParam {int} row  行号 ,从0开始计数
     *
     *
     * @apiSuccess (返回值){int} - 返回指定行的高度
     *
     *
     * @apiExample   {js}示例：
     *
     *   var h= book.sheet0.getRowPropertyManage().getRowHeight(0);
     *
     */
    getRowHeight: function (row, /*boolean*/ originalHeight) {

        if (row < 0) return 0;
        if (row >= this.getRowCount()) return 0;
        if (originalHeight == undefined) originalHeight = false;
        //如果是强制得到设置的高度
        if (originalHeight)
        {
            return Util.safeInt(this.rowPropertyList[row].height);
        } else
        {
            //如果行被设置成隐藏，那么返回0


            if (this.rowPropertyList[row].isHidden)
            {
                return 0;
            } else
            {
                return Util.safeInt(this.rowPropertyList[row].height);
            }
        }

    },

    getRowStretchHeight: function (row) {
        if (row < 0) return 0;
        if (row >= this.getRowCount()) return 0;
        //如果是强制得到设置的高度

        return Util.safeInt(this.rowPropertyList[row].stretchHeight);

    },

    setRowStretchHeight: function (row, newHeight) {
        if (newHeight == null || newHeight == undefined) return;

        if (row < 0) return;
        if (row >= this.getRowCount()) return;
        if (newHeight == this.rowPropertyList[row].stretchHeight) return;
        if (newHeight < 0) return;


        this.rowPropertyList[row].stretchHeight = Util.safeInt(newHeight);

        var /*WorkSheetView*/  view = this.Sheet.View;

        if (view != null)
        {
            // 要函数会让各行重绘的，所以不要再维生素绛
            view.reCalcuRowsHeight();


        }
    },


    /**
     * 得到Row的Y坐标偏移，注意不是实际的绘制时用的y坐标，实际显示坐标还要转换
     */
    getRowY: function (row) {
        var ret = 0;
        for (var i = 0; i < row; i++)
        {
            if (i >= this.rowPropertyList.length) return ret;

            let mrdsn = this.rowPropertyList[i].multiRowDataSourceName;
            if (mrdsn == '')
            {
                ret += this.getRowHeight(i); // 因为RowHeight可能出现通过计算得到，所以不要直接用this.rowPropertyList[row].height
            } else
            {
                let lastSameBindRow = this.findLastRowWhichBindToMultiRowDataSource(mrdsn);
                if (row > lastSameBindRow)
                {
                    var oneDataRowHeight = 0;//多行数据源区，一行数据的高度
                    for (let di = i; di <= lastSameBindRow; di++)
                    {
                        oneDataRowHeight += this.getRowHeight(di);
                    }
                    let dbRC = Math.max(1, this.Sheet.Book.getDataSource(mrdsn).dataStore.rowCount);

                    ret += oneDataRowHeight * dbRC;
                    i = lastSameBindRow;
                    continue;
                } else //如果行是在多行绑定区间内，那么只计算第一个数据行中，row行的Y
                {
                    for (let di = i; di < row; di++)
                    {
                        ret += this.getRowHeight(di);
                    }
                    return ret;
                }

            }
        }
        return ret;
    },


    /**
     * @api {setRowHeight} 函数   setRowHeight
     *
     * @apiDescription setRowHeight(row ,newHeight)
     * <br><br> 设置行的高度
     *
     * @apiName  setRowHeight
     * @apiGroup RowPropertyManage
     * @apiVersion 1.0.0
     *
     * @apiParam {int} row  行号 ,从0开始计数
     * @apiParam {int} newHeight  高度  ,单位px
     *
     *
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     *
     * @apiExample   {js}示例：
     *
     *   var h= book.sheet0.getRowPropertyManage().getRowHeight(0);
     *   book.sheet0.getRowPropertyManage().setRowHeight(0,100);
     *
     */
    setRowHeight: function (row, newHeight, needCalculateStretchHeight) {
        if (newHeight == null || newHeight == undefined) return;

        if (row < 0) return;
        if (row >= this.getRowCount()) return;
        if (isNaN(newHeight)) newHeight = RowProperty.defaultRowHeight;

        var oldHeight = this.rowPropertyList[row].height;
        if (newHeight == oldHeight) return;
        if (newHeight < 0) return;
        if (needCalculateStretchHeight == undefined) needCalculateStretchHeight = true; //设置行高后，是否需要触发弹性行重新计算行高

        this.rowPropertyList[row].height = newHeight;


        var /*WorkSheetView*/ view = this.Sheet.View;

        //自已内部，也可能需要在尺寸变化后，做一点什么
        this.Sheet.rowHeightChanged(row);


        if (view != null)
        {
            // 重绘
            view.repaint();
            // 触发事件
            view.EM.fire('rowHeightChanged', [this.Sheet, row, oldHeight, newHeight]);

            //重新计算弹性列的宽度
            if (needCalculateStretchHeight) view.reCalcuRowsHeight();

        }

    },


    //	 得到本行中单元格绑定到的数据源的中行数最多的那个数据源
    /*DataSourceConfig*/  getMaximalDataSourceConfig: function (row) {
        var dsNameList = [];
        var /*WorkBook*/ book = this.Sheet.workBook;
        var MaxRowCount = 1;
        var /*DataSourceConfig*/  MaxDSC = null;
        var CC = this.Sheet.columnCount;
        for (var col = 0; col < CC; col++)
        {
            var /*Cell*/ cell = this.Sheet.$Cells(row, col);
            if (cell == null) continue;
            var bind = cell.getBind();
            if (bind == null) continue;
            var dsName = bind.dataSource;
            if (dsNameList.contains(dsName)) continue; // 已经判断过的就不要再判断了
            dsNameList.push(dsName);
            var /*DataSourceConfig*/  dsc = book.getDataSource(dsName);

            var mrc = dsc.dataStore.rowCount;
            if (mrc > MaxRowCount)
            {
                MaxRowCount = mrc;
                MaxDSC = dsc;
            }
        }

        return MaxDSC;

    },

    getFixedRowHeight: function () {
        return this.getRowY(this.getFixedRowCount());
    },

    getAllRowsHeight: function () {
        var ret = 0;
        var n = this.getRowCount();
        //return  this.getRowY(n-1) +this.getRowHeight(n-1);
        //上面的做法是有问题的， 问题在于 this.getRowHeight(n-1)只是得到最后一行的单行高度，如果这是
        //绑定到一个多行数据源上，那么是需要*行数的
        return this.getRowY(n);
    },

    AddBindCount: function (row) {
        if (row < 0) return;
        if (row >= this.getRowCount()) return;

        this.rowPropertyList[row].dbColCount += 1;

    },

    setBindCount: function (row, c) {
        if (row < 0) return;
        if (row >= this.getRowCount()) return;

        this.rowPropertyList[row].dbColCount = c;

    },
    setMultiRowDataSourceName: function (row, dsn) {
        if (row < 0) return;
        if (row >= this.getRowCount()) return;
        this.rowPropertyList[row].multiRowDataSourceName = dsn;

        //把绑定到dsn的多行之间，都设置成绑定到dsn
        if (dsn != '')
        {
            var startRow = this.findFirstRowWhichBindToMultiRowDataSource(dsn);
            var endRow = this.findLastRowWhichBindToMultiRowDataSource(dsn);
            for (let i = startRow; i <= endRow; i++)
            {
                this.rowPropertyList[i].multiRowDataSourceName = dsn;
            }
        }
        //重新计算所有行的Y坐标
        this.rebuildAllRowsY();
    },

    getMultiRowDataSourceName: function (row) {
        if (row < 0) return '';
        if (row >= this.getRowCount()) return '';
        return this.rowPropertyList[row].multiRowDataSourceName;
    },

    /**
     * 得到绑定到指定的多行数据源的单元格，一行数据的高度
     * @param mrdsn
     * @returns {number}
     */
    getOneDataRowHeightOfDataSource: function (mrdsn) {
        var firstSameBindRow = this.findFirstRowWhichBindToMultiRowDataSource(mrdsn);
        var lastSameBindRow = this.findLastRowWhichBindToMultiRowDataSource(mrdsn);


        var oneDataRowHeight = 0;//多行数据源区，一行数据的高度
        for (let di = firstSameBindRow; di <= lastSameBindRow; di++)
        {
            oneDataRowHeight += this.getRowHeight(di);
        }
        return oneDataRowHeight
    },

    /**
     * 查找绑定到指定的多行结果集的第一行
     * @param mrdsn
     */
    findFirstRowWhichBindToMultiRowDataSource: function (mrdsn) {
        var rc = this.rowCount;
        for (var i = 0; i < rc; i++)
        {
            if (this.rowPropertyList[i].multiRowDataSourceName.equalsIgnoreCase(mrdsn)) return i;
        }
        return -1;

    },


    /**
     * 查找最后一行，它绑定到指定的多行结果集
     * @param mrdsn
     */
    findLastRowWhichBindToMultiRowDataSource: function (mrdsn) {
        var rc = this.rowCount;
        for (var i = rc - 1; i >= 0; i--)
        {
            if (this.rowPropertyList[i].multiRowDataSourceName.equalsIgnoreCase(mrdsn)) return i;
        }
        return -1;
    },


    ReleaseBindCount: function (row) {
        if (row < 0) return;
        if (row >= this.getRowCount()) return;

        this.rowPropertyList[row].dbColCount -= 1;
    },

    getBindedCount: function (row) {
        if (row < 0) return 0;
        if (row >= this.getRowCount()) return 0;

        return this.rowPropertyList[row].dbColCount;
    },

    /*boolean*/ isRowHidden: function (row) {
        if (row < 0) return false;
        var rc = this.getRowCount();
        if (row >= rc) return false;
        return this.rowPropertyList[row].isHidden;

    },

    setRowHidden: function (row, /*boolean*/ hidden) {
        if (row < 0) return;
        var rc = this.getRowCount();
        if (row >= rc) return;
        if (this.rowPropertyList[row].isHidden == hidden) return;
        this.rowPropertyList[row].isHidden = hidden;

        var /*WorkSheetView*/ view = this.Sheet.View;
        if (view != null) view.repaint();

    },

    /*boolean*/ isAutoAsyncHeightWithDBRowHeight: function (row) {
        if (row < 0) return true;
        var rc = this.getRowCount();
        if (row >= rc) return true;
        return this.rowPropertyList[row].autoAsyncHeightWithDBRowHeight;
    },

    setAutoAsyncHeightWithDBRowHeight: function (row, /*boolean*/ auto) {
        if (row < 0) return;
        var rc = this.getRowCount();
        if (row >= rc) return;
        if (this.rowPropertyList[row].autoAsyncHeightWithDBRowHeight == auto) return;
        this.rowPropertyList[row].autoAsyncHeightWithDBRowHeight = auto;

        var /*WorkSheetView*/ view = this.Sheet.View;
        if (view != null) view.repaint();

    },

    setColumnScrollEnable: function (row, enable) {
        if (row < 0) return;
        var rc = this.getRowCount();
        if (row >= rc) return;
        if (this.rowPropertyList[row].columnScrollEnabled == enable) return;
        this.rowPropertyList[row].columnScrollEnabled = enable;

    },

    isColumnScrollEnabled: function (row) {
        if (row < 0) return true;
        var rc = this.getRowCount();
        if (row >= rc) return true;
        return this.rowPropertyList[row].columnScrollEnabled;
    }
    ,

    getRowScrolledRowCount: function (row) {
        return 0;
    },

    setRowScrolledRowCount: function (row, h) {
        //TODO
    },

    /**
     * 重建所有行的Ywww坐标
     */
    rebuildAllRowsY: function () {
        //TODO  事先计算好Y，提高速度
    }


});


export default RowPropertyManage;
