/**
 * 文本Size计算， 取自 zrender ,略做调整
 *
 * 2020.06.08 增加指定宽度，计算文字高度
 */


import Util from '../util/Util.js';
import Rectangle from './Rectangle.js' ;


let textWidthCache = {};
let textWidthCacheCounter = 0;
let TEXT_CACHE_MAX = 5000;


let retrieve = Util.retrieve;

function getTextWidth(text, textFont)
{
    let key = text + ':' + textFont;
    if (textWidthCache[key])
    {
        return textWidthCache[key];
    }

    let textLines = (text + '').split('\n');
    let width = 0;

    for (let i = 0, l = textLines.length; i < l; i++)
    {
        // measureText 可以被覆盖以兼容不支持 Canvas 的环境
        width = Math.max(  measureText(textLines[i], textFont).width, width);
    }

    if (textWidthCacheCounter > TEXT_CACHE_MAX)
    {
        textWidthCacheCounter = 0;
        textWidthCache = {};
    }
    textWidthCacheCounter++;
    textWidthCache[key] = width;

    return width;
}

function getTextRect(text, textFont, textAlign, textBaseline)
{
    let textLineLen = ((text || '') + '').split('\n').length;

    let width = getTextWidth(text, textFont);

    let lineHeight = getTextWidth('国', textFont);
    let height = textLineLen * lineHeight;

    let rect = new Rectangle(0, 0, width, height);
    // Text has a special line height property
    rect.lineHeight = lineHeight;

    switch (textBaseline)
    {
        case 'bottom':
        case 'alphabetic':
            rect.y -= lineHeight;
            break;
        case 'middle':
            rect.y -= lineHeight / 2;
            break;
        // case 'hanging':
        // case 'top':
    }


    switch (textAlign)
    {
        case 'end':
        case 'right':
            rect.x -= rect.width;
            break;
        case 'center':
            rect.x -= rect.width / 2;
            break;
        // case 'start':
        // case 'left':
    }

    return rect;
}

function adjustTextPositionOnRect(textPosition, rect, textRect, distance)
{

    let x = rect.x;
    let y = rect.y;

    let height = rect.height;
    let width = rect.width;

    let textHeight = textRect.height;

    let halfHeight = height / 2 - textHeight / 2;

    let textAlign = 'left';

    switch (textPosition)
    {
        case 'left':
            x -= distance;
            y += halfHeight;
            textAlign = 'right';
            break;
        case 'right':
            x += distance + width;
            y += halfHeight;
            textAlign = 'left';
            break;
        case 'top':
            x += width / 2;
            y -= distance + textHeight;
            textAlign = 'center';
            break;
        case 'bottom':
            x += width / 2;
            y += height + distance;
            textAlign = 'center';
            break;
        case 'inside':
            x += width / 2;
            y += halfHeight;
            textAlign = 'center';
            break;
        case 'insideLeft':
            x += distance;
            y += halfHeight;
            textAlign = 'left';
            break;
        case 'insideRight':
            x += width - distance;
            y += halfHeight;
            textAlign = 'right';
            break;
        case 'insideTop':
            x += width / 2;
            y += distance;
            textAlign = 'center';
            break;
        case 'insideBottom':
            x += width / 2;
            y += height - textHeight - distance;
            textAlign = 'center';
            break;
        case 'insideTopLeft':
            x += distance;
            y += distance;
            textAlign = 'left';
            break;
        case 'insideTopRight':
            x += width - distance;
            y += distance;
            textAlign = 'right';
            break;
        case 'insideBottomLeft':
            x += distance;
            y += height - textHeight - distance;
            break;
        case 'insideBottomRight':
            x += width - distance;
            y += height - textHeight - distance;
            textAlign = 'right';
            break;
    }

    return {
        x: x,
        y: y,
        textAlign: textAlign,
        textBaseline: 'top'
    };
}

/**
 * Show ellipsis if overflow.
 *
 * @param  {string} text
 * @param  {number} containerWidth
 * @param  {string} textFont
 * @param  {string} [ellipsis='...']
 * @param  {Object} [options]
 * @param  {number} [options.maxIterations=3]
 * @param  {number} [options.minChar=0] If truncate result are less
 *                  then minChar, ellipsis will not show, which is
 *                  better for user hint in some cases.
 * @param  {number} [options.placeholder=''] When all truncated, use the placeholder.
 * @return {string}
 */
function truncateText(text, containerWidth, textFont, ellipsis, options)
{
    if (!containerWidth)
    {
        return '';
    }

    options = options || {};

    ellipsis = retrieve(ellipsis, '...');
    let maxIterations = retrieve(options.maxIterations, 2);
    let minChar = retrieve(options.minChar, 0);

    let cnCharWidth = getTextWidth('国', textFont);

    // Consider proportional font?
    let ascCharWidth = getTextWidth('a', textFont);
    let placeholder = retrieve(options.placeholder, '');

    // Example 1: minChar: 3, text: 'asdfzxcv', truncate result: 'asdf', but not: 'a...'.
    // Example 2: minChar: 3, text: '维度', truncate result: '维', but not: '...'.
    let contentWidth = containerWidth = Math.max(0, containerWidth - 1); // Reserve some gap.
    for (let i = 0; i < minChar && contentWidth >= ascCharWidth; i++)
    {
        contentWidth -= ascCharWidth;
    }

    let ellipsisWidth = getTextWidth(ellipsis);
    if (ellipsisWidth > contentWidth)
    {
        ellipsis = '';
        ellipsisWidth = 0;
    }

    contentWidth = containerWidth - ellipsisWidth;

    let textLines = (text + '').split('\n');

    for (let i = 0, len = textLines.length; i < len; i++)
    {
        let textLine = textLines[i];
        let lineWidth = getTextWidth(textLine, textFont);

        if (lineWidth <= containerWidth)
        {
            continue;
        }

        for (let j = 0; ; j++)
        {
            if (lineWidth <= contentWidth || j >= maxIterations)
            {
                textLine += ellipsis;
                break;
            }

            let subLength = j === 0
                ? estimateLength(textLine, contentWidth, ascCharWidth, cnCharWidth)
                : lineWidth > 0
                    ? Math.floor(textLine.length * contentWidth / lineWidth)
                    : 0;

            textLine = textLine.substr(0, subLength);
            lineWidth = getTextWidth(textLine, textFont);
        }

        if (textLine === '')
        {
            textLine = placeholder;
        }

        textLines[i] = textLine;
    }

    return textLines.join('\n');
}

function estimateLength(text, contentWidth, ascCharWidth, cnCharWidth)
{
    let width = 0;
    let i = 0;
    for (let len = text.length; i < len && width < contentWidth; i++)
    {
        let charCode = text.charCodeAt(i);
        width += (0 <= charCode && charCode <= 127) ? ascCharWidth : cnCharWidth;
    }
    return i;
}

/**
 * 在指定宽度下， 文字占用的高度
 * @param textWidth   宽度
 * @param str   文字内容
 * @param FontName
 * @param FontSize
 * @param FontBold
 * @param FontItalic
 * @param FontUnderline
 * @param lineSpacing  行间距
 * @returns {number}  返回整个文字占用的高度
 */
function getTextHeightInWidth(textWidth, str, FontName, FontSize, FontBold, FontItalic, FontUnderline, lineSpacing)
{
    /*首先是要把内容拆分成多行
    基于如下原则
    1  先计算一个汉字的宽度，一个字母的宽度，假设是等宽字体（主要是为了速度）
    2 然后逐个字符累加宽度，直到超出 textRect的宽度。这里不考虑英文单词不拆分的问题，
  */

    textWidth -=4;//会出现计算出来高度不够的情况，所以临时把宽度改小一点，具体以后再调试，不良后果是有时高度会过大。但是总比过小显示不全要好一点
    if (str == null) str = " ";


    if (lineSpacing === undefined) lineSpacing = 2;
    //把强制回车，换成回车符
    str = str.replace(/\\n/g, '\n');

    //参数初始化及检测


    if (FontName === '') FontName = 'STHeitiSC-Light'; // mac osX上

    if (['FontAwesome', 'physiology'].contains(FontName))
    {
        //  如果str中有单引号，可能需要特别处理一下
        //如果str 是上述字体，那么把字面值  \ue004 这样的字符串，转成对应的字符
        //这里要搞清楚， 字面字符串，与unicode字符串。  '\ue004' 与 '\\ue004'区别
        //在设计面板中直接输入的\ue004 实际上
        str = eval("'" + str + "'");
    }


    FontBold = FontBold || false;
    FontItalic = FontItalic || false;
    FontUnderline = FontUnderline || false;

    //字体设置
    let font = (FontItalic ? "italic" : "normal") + " normal " +
        (FontBold ? "bold" : "normal") + " " + FontSize + "px  " + FontName;

    let ctx = Util.getContext();
    ctx.font = font;



    let zh_w = ctx.measureText("国", font).width;
    let min_zs_one_row = Math.max(1, Math.floor(textWidth / zh_w)); //最少一个字

    let splitedStr = [];

    //先按回车拆分
    let strs = str.split('\n');
    for (let ti = 0; ti < strs.length; ti++)
    {
        str = strs[ti];
        while (str.length > 0)
        {

            let p = min_zs_one_row;
            //取一段字符，并计算它的实际宽度
            let one = str.substring(0, p);
            let oneWidth = ctx.measureText(one, font).width;

            while (true)
            {
                if (p >= str.length)
                {
                    splitedStr.push({str: str, width: oneWidth});
                    str = "";
                    break;
                }

                let ch = str.substring(p, p + 1);
                let ch_width = ctx.measureText(ch).width;
                //console.info(ch + "  width: " + ch_width);
                oneWidth += ch_width;

                if (oneWidth > textWidth)
                {
                    //到p这里，已经超过宽度了，所以要少一个字符进行换行
                    splitedStr.push({str: str.substring(0, p), width: oneWidth - ch_width});
                    str = str.substring(p);
                    break;
                } else
                {
                    p++;
                }
            }
        }

    }

    let lineHeight = FontSize; //通常字号就是文字的高度

    let underlineSpacing = 0;

    if (FontUnderline) underlineSpacing = 1;

    //lineSpacing是行间隔，最后一行不需要间隔
    return (lineHeight + lineSpacing + underlineSpacing) * splitedStr.length - lineSpacing;

}

function measureText  (text, textFont) {
    let ctx = Util.getContext();
    ctx.font = textFont || '12px sans-serif';
    return ctx.measureText(text);
}

let TextUtil = {

    getWidth: getTextWidth,

    getRectangle: getTextRect,

    adjustTextPositionOnRect: adjustTextPositionOnRect,

    truncateText: truncateText,

    getTextHeightInWidth:getTextHeightInWidth,

    measureText:measureText

};


export default TextUtil;
