/**
 * Created by zengjun on 2018/10/7.
 *
 * 逻辑绘图接口，它并不具体地实现绘图，而是把绘图动作抽象成命令形式，交给第三方绘图接口去实现
 * 主要用于显示输出格式转换，比如生成PDF,或输出到打印机
 */


import Class from '../base/Class.js';
import Color from './Color.js';
import Rectangle from './Rectangle.js';
import Util from '../util/Util.js';

import ALIGN from '../core/ALIGNMENT.js';
import Graphics from "./Graphics.js";


var cmd = {
    'contextSave': 1,
    'contextRestore': 2,

    'clearRect': 5,
    'setColor': 6,
    'setStrokeColor': 7,
    'setFillColor': 8,

    'drawRect': 10,
    'moveTo': 11,

    'setLineWidth': 3,
    'setLineCap': 12,
    'setLineDash': 13,
    'lineTo': 14,
    'drawLine': 15,
    'beginPath': 16,
    'closePath': 17,
    'scale': 18,
    'setClip': 19,
    'fillRect': 20,
    'stroke': 21,
    'drawImage': 22,
    'drawStringWordwrap': 23,
    'drawStringSingleRow': 24,
    'translate': 25

};

var GraphicsLogic = Graphics.extend({


    properties:
        {
            "invalidateRect":
                {
                    get: function () { return this.InvalidateRect;},
                    set: function (rc) { this.setInvalidateRect(rc);}
                }
        },

    constructor: function () {

        this.command = [];
    },


    reInit: function () {
        this.command = [];
    },

    /**
     * 保存绘图上下文
     */
    save: function () {
        this.command.push([cmd.contextSave]);

    },

    /**
     * 恢复绘图上下文
     */
    restore: function () {
        this.command.push([cmd.contextRestore]);
    },

    setInvalidateRect: function (rc) {
        if (rc == null)
        {
            this.InvalidateRect = null;
            return;
        }

        this.InvalidateRect = rc.clone();
    },

    color: function (c) {
        if (Util.isClas$(c))
        {
            if (c.instanceOf(Color)) return c;
        }
        return new Color(c);
    },

    clearRect: function (x, y, width, height) {
        this.command.push(
            [cmd.clearRect, x, y, width, height]
        );
    },


    setColor: function (color) {
        color = this.color(color);
        this.setStrokeColor(color);
        this.setFillColor(color);
    },

    setStrokeColor: function (color) {
        color = this.color(color);
        this.command.push(
            [
                cmd.setStrokeColor,
                color.color
            ]
        );

    },


    setFillColor: function (color) {
        color = this.color(color);
        this.command.push(
            [
                cmd.setFillColor,
                color.color
            ]
        );


    },

    setLineWidth: function (n) {
        this.command.push(
            [
                cmd.setLineWidth,
                n
            ]
        );

    },

    setLineCap: function (cap) {
        this.command.push(
            [
                cmd.setLineCap,
                cap
            ]
        );
    },


    setLineDash: function (dash) {
        this.command.push(
            [
                cmd.setLineDash,
                dash
            ]
        );
    },


    drawRect: function () {


        if (arguments.length == 1)
        {
            var rect = arguments[0];
            this.command.push(
                [
                    cmd.drawRect,
                    rect.x,
                    rect.y,
                    rect.width,
                    rect.height
                ]
            );

        }
        if (arguments.length == 4)
        {
            this.command.push(
                [
                    cmd.drawRect,
                    arguments[0],
                    arguments[1],
                    arguments[2],
                    arguments[3]
                ]
            );


        }
        return this;
    },


    moveTo: function (x, y) {
        this.command.push(
            [
                cmd.moveTo,
                x,
                y
            ]
        );
    },

    translate: function (x, y) {
        this.command.push(
            [
                cmd.translate,
                x,
                y
            ]
        );
    },

    lineTo: function (x, y) {
        this.command.push(
            [
                cmd.lineTo,
                x,
                y
            ]
        );
    },

    drawLine: function (x1, y1, x2, y2) {

        this.moveTo(x1, y1);
        this.lineTo(x2, y2);


    },

    beginPath: function () {
        this.command.push(
            [
                cmd.beginPath
            ]
        );
    }
    ,
    closePath: function () {
        this.command.push(
            [
                cmd.closePath
            ]
        );
    }
    ,

    scale: function (xScale, yScale) {

        this.command.push(
            [
                cmd.scale,
                xScale,
                yScale
            ]
        );
    },

    clip: function (rc) {

        //beginPath()的调用非常重要
        /* 在调用clip()前，多次的rect后，缺省情况下，裁剪区域是这些rect的并集，而不是交集，除非修改
        ctx.globalCompositeOperation属性。但是现在需要的不是并集，也不是交集，而是清除之前的裁剪区，使用新的裁剪区
        所以通过调用beginPath() 来重置当前的路径,现在的裁剪区就是rc，不会受旧裁剪区的影响
        *
        * */
        this.command.push(
            [
                cmd.setClip,
                rc.x,
                rc.y,
                rc.width,
                rc.height
            ]
        );
    }
    ,


    fillRect: function () {
        if (arguments.length == 1)
        {
            var rect = arguments[0];
            this.command.push(
                [
                    cmd.fillRect,
                    rect.x,
                    rect.y,
                    rect.width,
                    rect.height
                ]
            );

        }
        if (arguments.length == 4)
        {
            this.command.push(
                [
                    cmd.fillRect,
                    arguments[0],
                    arguments[1],
                    arguments[2],
                    arguments[3]
                ]
            );


        }
        return this;
    }
    ,


    stroke: function () {
        this.command.push(
            [
                cmd.stroke
            ]
        );
    }
    ,

    /**
     * 绘制图象
     * @param img
     * @param x
     * @param y
     */
    drawImage: function (img, x, y, w, h, x2, y2, w2, h2) {
        this.command.push(
            [
                cmd.drawImage,
                img.src,
                x, y, w, h, x2, y2, w2, h2,
                window.location.origin
            ]
        );
    }
    ,

    /**
     * 多行文字的绘制
     * @param textRect
     * @param str
     * @param bkMode
     * @param bkcolor
     * @param textColor
     * @param FontName
     * @param FontSize
     * @param FontBold
     * @param FontItalic
     * @param FontUnderline
     * @param hAlign
     * @param vAlign
     * @param lineSpacing
     */
    drawStringWordwrap(textRect, str, bkMode, bkColor, textColor, fontName, fontSize, fontBold,
                       fontItalic, fontUnderline, hAlign, vAlign, lineSpacing)
    {

        if (!str) return;

        if (['FontAwesome', 'physiology'].contains(fontName))
        {
            //TODO 如果str中有单引号，可能需要特别处理一下
            //如果str 是上述字体，那么把字面值  \ue004 这样的字符串，转成对应的字符
            //这里要搞清楚， 字面字符串，与unicode字符串。  '\ue004' 与 '\\ue004'区别
            //在设计面板中直接输入的\ue004 实际上保存的是 '\\ue004'

            str = eval("'" + str + "'");
        }

        this.command.push(
            [
                cmd.drawStringWordwrap,

                textRect.x,
                textRect.y,
                textRect.width,
                textRect.height,
                str,
                bkMode,
                bkColor,
                textColor,
                fontName,
                fontSize,
                fontBold,
                fontItalic,
                fontUnderline,
                hAlign,
                vAlign,
                lineSpacing
            ]
        );

    }
    ,


    /**
     * 单行显示 ，不进制坐标的计算，这样速度会快些
     * @param textRect
     * @param str
     * @param bkMode
     * @param bkcolor
     * @param textColor
     * @param FontName
     * @param FontSize
     * @param FontBold
     * @param FontItalic
     * @param FontUnderline
     * @param hAlign
     * @param vAlign
     */
    drawStringSingleRow(textRect, str, bkMode, bkColor, textColor, fontName, fontSize, fontBold,
                        fontItalic, fontUnderline, hAlign, vAlign)
    {
        if (!str) return;

        if (['FontAwesome', 'physiology'].contains(fontName))
        {
            //TODO 如果str中有单引号，可能需要特别处理一下
            //如果str 是上述字体，那么把字面值  \ue004 这样的字符串，转成对应的字符
            //这里要搞清楚， 字面字符串，与unicode字符串。  '\ue004' 与 '\\ue004'区别
            //在设计面板中直接输入的\ue004 实际上保存的是 '\\ue004'

            str = eval("'" + str + "'");
        }
        this.command.push(
            [
                cmd.drawStringSingleRow,

                textRect.x,
                textRect.y,
                textRect.width,
                textRect.height,
                str,
                bkMode,
                bkColor,
                textColor,
                fontName,
                fontSize,
                fontBold,
                fontItalic,
                fontUnderline,
                hAlign,
                vAlign
            ]
        );
    }

});

export default GraphicsLogic;
