/**
 * Created by zengjun on 2017/4/28.
 */


import Class from '../base/Class.js';
import ItemStatus from "./ItemStatus.js";
import Util from "../util/Util.js";
import ColumnProperty from "./ColumnProperty.js";
import ObjectType from "./ObjectType.js";
import ObjectTool from "./ObjectTool.js";
import DsItem from "./DsItem.js";
import Types from "./Types.js";

import Evaluate from '../eval/Evaluate.js';
import EvaluatePool from '../eval/EvaluatePool.js';

import UniformDataType from '../core/UniformDataType.js';

var DataBuffer = Class.extend({

    properties: {
        /**
         * 直接访问没物控部数据
         */
        /**
         * @api {buffer} 只读属性 [属性]buffer
         * @apiName  buffer
         * @apiDescription 直接访问内部数据
         * @apiGroup DataBuffer
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){Object} - 内部数据对象
         * @apiExample {js}(示例：)
         * //示例
         */
        "buffer": {
            get: function () {return this.m_Buffer;}
        },
        /**
         * 数据行数
         */
        /**
         * @api {rowCount} 只读属性 [属性]rowCount
         * @apiName  rowCount
         * @apiDescription 数据行数
         * @apiGroup DataBuffer
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){int} - 数据行数
         * @apiExample {js}(示例：)
         * //示例
         */
        "rowCount": {
            get: function () { return this.getRowCount();}
        },
        /**
         * @api {columnCount} 只读属性 [属性]columnCount
         * @apiName  columnCount
         * @apiDescription 结果集的列数
         * @apiGroup DataBuffer
         * @apiVersion 1.0.0
         * @apiSuccess (属性类型){int} - 结果集的列数
         * @apiExample {js}(示例：)
         * //示例
         */
        "columnCount": {
            get: function () { return this.getColumnCount();}
        }
    },

    constructor: function (colCount, ds) {


        this.m_ColumnCount = colCount;
        this.m_ds = ds;
        this.m_Buffer = [];


    },


    /**
     * @api {clear} 函数   clear
     *
     * @apiDescription clear()
     * <br><br>  清空数据
     *
     * @apiName  clear
     * @apiGroup DataBuffer
     * @apiVersion 1.0.0
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     */
    clear: function () {
        this.m_Buffer = [];
    },


    // 出于兼容性考虑
    GetBuffer: function () {
        return this.m_Buffer;
    },


    /**
     * @api {getItem} 函数   getItem
     *
     * @apiDescription getItem (row, col)
     * <br><br> 得到 指定行列的数据单元对象
     *
     * @apiName  getItem
     * @apiGroup DataBuffer
     * @apiVersion 1.0.0
     *
     * @apiParam {int} row 行号
     * @apiParam {int} col 列号
     *
     * @apiSuccess (返回值){DsItem} - 返回数据单元对象
     *
     * @apiExample   {js}示例：
     *
     *  var buffer= ds.getPrimaryBuffer();
     *  var item= buffer.getItem(0,0);
     *  alert( JSON.toString(item));
     *
     */
    getItem: function (row, col)  //DsItem
    {
        if (row < 0 || row >= this.getRowCount()) return null;
        if (col < 0 || col >= this.m_ds.getColumnCount()) return null;
        var dr = this.m_Buffer[row];
        var item = dr.getAt(col);
        return item;
    },

    /**
     * 将列名转换成列号
     * @param col
     * @returns {*}
     */
    col2Index: function (col) {
        var colIndex = -1;
        if (!isNaN(col)) return col;
        if (Util.isString(col)) colIndex = this.m_ds.col2Index(col);
        return colIndex;
    },


    /**
     * @api {getItemStatus} 函数   getItemStatus
     *
     * @apiDescription  getItemStatus (row, col)
     * <br><br> 得到指定单元格的数据状态
     *
     * @apiName  getItemStatus
     * @apiGroup DataBuffer
     * @apiVersion 1.0.0
     *
     * @apiParam {int} row 行号
     * @apiParam {int/String} col 列号或列名
     *
     * @apiSuccess (返回值){int} - 返回本数据的状态，可以是如下值
     * <ul>
     *    <li> 0: isNotModified 没有被修改</li>
     *    <li> 1: isDataModified 数据被修改</li>
     *    <li> 2: isNew 新插入的数据</li>
     *    <li> 3: isNewModified 新增加并被修改的数据</li>
     *    <li> 4: isAvailable 是有效的计算出来的有效数据</li>
     *    <li> 5: isInvalid  计算出来的数据,但当前数据失效，需要重算</li>
     * </ul>
     *
     * @apiExample   {js}示例：
     *
     *  var buf= ds.getPrimaryBuffer();
     *  var s= buf.getItemStatus(0,"name");
     *
     */
    getItemStatus: function (row, col) {
        var colIndex = this.col2Index(col);

        //列不存在，那么返回
        if (colIndex < 0) return ItemStatus.isNotModified;

        var item = this.getItem(row, colIndex);

        if (item != null) return item.m_ItemStatus;
        if (this.getRow(row) == null) return ItemStatus.isNotModified;

        var rowStatus = this.getRow(row).getRowStatus();
        //
        // 如果该Cell的对象还不存在，那么要看它所在的行的状态是什么
        switch (rowStatus)
        {
            case ItemStatus.isDataModified:
            case ItemStatus.isNotModified:
                return ItemStatus.isNotModified;
            case ItemStatus.isNew:
            case ItemStatus.isNewModified:
                return ItemStatus.isNew;
            default:
                return ItemStatus.isNew;
        }
    },

    /**
     * @api {setItemStatus} 函数   setItemStatus
     *
     * @apiDescription  setItemStatus (row, col, status)
     * <br><br> 设置数据单元的状态。结果集什自动维护数据单元的状态，不般情况下，不要使用本函数来修改状态
     *
     * @apiName  (row, col, status)
     * @apiGroup DataBuffer
     * @apiVersion 1.0.0
     *
     * @apiParam {int} row 行号
     * @apiParam {int/String} col 列号或列名
     * @apiParam {int} status  状态，可以是如下值
     *   <ul>
     *    <li> 0: isNotModified 没有被修改</li>
     *    <li> 1: isDataModified 数据被修改</li>
     *    <li> 2: isNew 新插入的数据</li>
     *    <li> 3: isNewModified 新增加并被修改的数据</li>
     *    <li> 4: isAvailable 是有效的计算出来的有效数据</li>
     *    <li> 5: isInvalid  计算出来的数据,但当前数据失效，需要重算</li>
     * </ul>
     *
     * @apiSuccess (返回值){void} - 无返回值
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     */
    setItemStatus: function (row, col, status) {
        var colIndex = this.col2Index(col);
        var item = this.getItem(row, colIndex);
        if (item == null) return;

        item.m_ItemStatus = status;
    },


    getColumnProperty: function (col) {
        return this.m_ds.getColumnProperty(col);
    },

    /**
     * @api {getRow} 函数   getRow
     *
     * @apiDescription getRow(row)
     * <br><br> 得到数据行对象
     *
     * @apiName  getRow
     * @apiGroup DataBuffer
     * @apiVersion 1.0.0
     *
     * @apiParam {int} row 行号
     *
     * @apiSuccess (返回值){DsRow} - 返回内部数据行对象
     *
     * @apiExample   {js}示例：
     *
     *  var one= ds.getPrimaryBuffer().getRow(0);
     *
     */
    getRow: function (row) {
        if (row < 0) return null;
        if (row >= this.getRowCount()) return null;
        return this.m_Buffer[row];
    },

    eraseRow: function (row) {
        // 不作参数校验，请调用者自己保证参数的合法
        this.m_Buffer.splice(row, 1);
    },


    /**
     * ------------------------------------------------------------------------
     * ------------------------------------------------------- 得到指定行列的值
     *
     * @param row
     * @param col
     * @param original
     *            true: 取最初值 , false: 取当前值
     * @return 得到指定行列的值 ★★★★★ 特别注意： 返回的是值的克隆，这样防止其它程序修改 GetValue返回的对象
     *         的值而影响将缓冲区中的值
     *         ------------------------------------------------------
     *         ------------
     *         -------------------------------------------------------------
     */

    getValue: function (row, colIndexOrName, original) {
        original = original || false;
        if (row >= this.m_Buffer.length || row < 0) return null;
        var col = this.col2Index(colIndexOrName);

        var dr = this.m_Buffer[row];

        if (col < 0 || col >= dr.m_Items.length) return null;

        var item = dr.getAt(col);

        // 如果是自定义的计算列
        var cp = this.getColumnProperty(col);
        //当datastore 与界面绑定时，在读取数据前，可能需要强制回填。因为读取的数据可能正在编辑中
        this.m_ds.EM.fire("beforeGetValue", [row, cp.m_Name]);

        if (cp.m_ObjType == ObjectType.isUserComputer)
        {
            return this.GetUserComputerValue(row, col);
        }
        // 如果是聚合列
        if (cp.m_ObjType == ObjectType.isAggregateComputer)
        {
            return this.GetAggregateComputerValue(row, col);
        }
        // 如果是列
        if (item == null) return null;
        var retValue = null;
        if (original)
        {
            retValue = ObjectTool.ObjectClone(item.m_OriginalValue);
        } else
        {
            retValue = ObjectTool.ObjectClone(item.m_CurrentValue);
        }

        return retValue;
    },

    // 该函数不应该用在open ,insertrow 函数中,通常该函数用在访问 pItem时,
    // 需要事先判断 pItem是否==NULL ,如果为空,那么需要用此函数来创建pItem


    NEW_ITEM: function (/*DsItem*/ item, /*DsRow*/ pRow, /*int*/  col) {
        if (item != null) return item;
        item = new DsItem();
        pRow.setAt(col, item);
        // 该单元的状态可以根据行的状态来确定
        if (pRow.rowStatus == ItemStatus.isNew ||
            pRow.rowStatus == ItemStatus.isNewModified)
        {
            item.m_ItemStatus = ItemStatus.isNew;
        }
        if (pRow.rowStatus == ItemStatus.isNotModified ||
            pRow.rowStatus == ItemStatus.isDataModified)
        {
            item.m_ItemStatus = ItemStatus.isNotModified;
        }

        return item;
    },

    // 注意同步修改　InsureColumnValueValid

    GetUserComputerValue: function (/*int*/ row, /*int*/ col) {

        var /*DsRow*/ dr = this.m_Buffer[row];

        var /*DsItem*/  item = dr.getAt(col);
        var /*ColumnProperty*/   pCP = this.getColumnProperty(col);
        if (item == null)
        {
            item = this.NEW_ITEM(item, dr, col);
            item.m_ItemStatus = ItemStatus.isInvalid;
        }

        if (pCP.m_ObjType != ObjectType.isUserComputer) return ObjectTool.ObjectClone(item.m_CurrentValue);
        // 如果该计算列的值是有效的,那么直接返回该值
        //  增加对不确定性函数的支持
        if (item.m_ItemStatus == ItemStatus.isAvailable &&
            !pCP.m_IncludeUnsureFunction)
        {
            return ObjectTool.ObjectClone(item.m_CurrentValue);
        }

        this.InternalRecalculateComputer(item, pCP.m_Formula, pCP.m_DependMap, row);
        return ObjectTool.ObjectClone(item.m_CurrentValue);

    },

    InternalRecalculateComputer: function (/*DsItem*/item, /*String*/ expression, /* Hashtable*/ dependMap, /*int*/ row) {
        // 评估该计算列表达式的值
        item.m_CurrentValue = this.Eval(expression, dependMap, row);
        // 修改值和状态
        item.m_ItemStatus = ItemStatus.isAvailable;
    },

    // 注意同步修改　InsureColumnValueValid

    GetAggregateComputerValue: function (row, /*int*/  col) {
        return null;
    },


    AggregateRange: function (row, groupNo) {
        return null;
    }
    ,

    /**
     * 评估计算列的值
     * @param expression
     * @param dependMap
     * @param row
     * @returns {*|Object}
     * @constructor
     */
    Eval: function (expression, dependMap, row) {

        var vars = {};
        var ev = EvaluatePool.getInstance();//从池中得到对象

        if (dependMap != null) // 比如计算列表达式为 "100" 这样的没有引用其它字段时，dependMap就==null
        {
            for (let i = 0; i < dependMap.length; i++)
            {

                let dependCol = dependMap[i];
                let v = this.getValue(row, dependCol);
                //  如果值为空，那么就用InsteadOfNull来替换
                if (v == null) v = this.m_ds.getColumnProperty(dependCol).getValueInsteadOfNull();

                vars[dependCol] = v;

            }
        }


        //  把与当前状态有关的数据加入进JEP
        window[Evaluate.datastore_internal_thisrow] = row ;
        //2020.05.24 如果是翻页模式，那么翻页后，页面要增加，如果不是翻页而是追加模式，那么行号不需要加上定位行数


        if(  this.m_ds.lastRetrieveIsResetBeforeRetrieve) window[Evaluate.datastore_internal_thisrow] =row + this.m_ds.getAbsolute() - 1;
        window[Evaluate.datastore_internal_rowcount] = this.m_Buffer.length;
        window[Evaluate.datastore_internal_absolute] = this.m_ds.getAbsolute();

        var ret = ev.evaluate(vars, expression);
        EvaluatePool.recycle(ev);//释放回池中
        return ret;


    }
    ,


    Eval_Aggregate: function (expression, pDependMap, row, pDefine, range) {

        return null;

    }
    ,


    Aggregate_Sum: function (expression, pDependMap, range, colIndex) {
        return null;
    }
    ,


    Aggregate_Cumulative: function (expression, pDependMap, range, row, colIndex) {

        return null;

    }
    ,


    Aggregate_FirstRow: function (expression, pDependMap, range, colIndex) {
        return null;
    }
    ,


    Aggregate_LastRow: function (expression, pDependMap, range, colIndex) {
        return null;
    }
    ,


    Aggregate_Avg: function (expression, pDependMap, range, colIndex) {
        return null;

    }
    ,


    Aggregate_Max: function (expression, pDependMap, range, colIndex) {

        return null;
    }
    ,


    Aggregate_Min: function (expression, pDependMap, range, colIndex) {

        return null;
    },


    Aggregate_Count: function (expression, pDependMap, range, colIndex) {
        return null;
    },


    Aggregate_List: function (expression, pDependMap, range, colIndex) {

        return null;
    },


    Aggregate_DistinctList: function (expression, pDependMap, range, colIndex) {
        return null;
    },

    Aggregate_DistinctSum: function (expression, pDependMap, range, colIndex) {

        return null;
    },


    Aggregate_DistinctCount: function (expression, pDependMap, range, colIndex) {
        return null;
    },

    evaluate: function (expression, row) {
        var dc = this.m_ds.assembleDependColumns(expression);
        return this.Eval(expression, dc, row);

    },


    /**
     * @api {getRowCount} 函数   getRowCount
     *
     * @apiDescription getRowCount()
     * <br><br> 得到行数， 本函数是属性 rowCount的getter函数
     *
     * @apiName  getRowCount
     * @apiGroup DataBuffer
     * @apiVersion 1.0.0
     *

     * @apiSuccess (返回值){int} - 行数
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     */
    getRowCount: function () {
        return this.m_Buffer.length;
    },

    /**
     * @api {getColumnCount} 函数   getColumnCount
     *
     * @apiDescription getColumnCount()
     * <br><br> 得到列数，本函数是属性columnCount的getter函数
     *
     * @apiName  getColumnCount
     * @apiGroup DataBuffer
     * @apiVersion 1.0.0
     *

     *
     * @apiSuccess (返回值){int} - 返回列数
     *
     * @apiExample   {js}示例：
     *
     *  //示例
     */
    getColumnCount: function () {
        return this.m_ColumnCount;

    },

    push: function (/*DsRow*/    r) {
        this.m_Buffer.push(r);
    },

    /**
     * 在每一行的末尾扩充一列
     *
     */
    AppendColumn: function () {
        this.m_ColumnCount++;
        var n = this.m_Buffer.length;
        for (var i = 0; i < n; i++)
        {
            var /*DsRow*/ r = this.m_Buffer[i];
            r.m_Items.push(null);
        }
    },

    /**
     *该函数在 beforeRow处插入一行，新插入的行变成 第 beforeRow行 ，这里的beforerow 是内部坐标
     * --------
     * @param beforeRow
     * @param dr
     * @return 返回插入行的行号，（ 是内部坐标）
     */

    InsertRow: function (/*int*/beforeRow, /*DsRow*/ dr) {


        var rc = this.getRowCount();
        if (beforeRow >= rc)
        {
            this.push(dr);
            return this.getRowCount() - 1;
        } else
        {
            this.m_Buffer.splice(beforeRow, 0, dr);
            return beforeRow;
        }
    },


    setValue: function (/*int*/row, colNameOrIndex, /*Object*/     v) {
        //console.info(" setValue row="+row+" col="+ colNameOrIndex+"  value="+v);

        if (row < 0) return false;
        if (row >= this.getRowCount()) return false;

        var col = this.col2Index(colNameOrIndex);
        if (col < 0) return false;

        var /*DsRow*/  dr = this.getRow(row);
        var /*DsItem*/ item = this.getItem(row, col);
        var /*ColumnProperty*/pCP = this.getColumnProperty(col);
        if (pCP == null) return; //正常不会出现

        if (pCP.m_ObjType == ObjectType.isUserComputer) return false;
        if (pCP.m_ObjType == ObjectType.isAggregateComputer) return false;

        if (item == null) item = this.NEW_ITEM(item, dr, col);

        if (null == item.m_CurrentValue && v == null) return true;

        // 将值转换成合适的类型
        // 确保 类型是对应的，newValue 值是克隆的一份，同时确保在 外部程序修改参数 v 后，对本缓冲区不产生影响
        var newValue = null;
        try
        {
            newValue = this.changeType(v, pCP.getUniformDataType());
            //2020.08补丁：当编辑格式设置成日期，但是字段又是varchar(10) , 数据会超出长度
            // 所以做了数据截取，避免出现字自超过长度而无法保存
            //2020.09.21 修正，增加了 m_Updatable的前置判断，因为当结果集不需要更新时，通常它是查询结果集，或是
            // 用类似  select  '' as col  from dual 这样的语句创建的结果集，此时不需要对数据进行截取
            if( pCP.m_Updatable)
            {
                if (Util.isString(newValue))
                {
                    //2020.08.13 补丁： mysql 并不能返回字符串的长度，所以，只有当precision>0 时才做处理
                    var precision = pCP.getPrecision();
                    if (precision > 0 && newValue.length > precision) newValue = newValue.substring(0, precision);
                }
            }
        } catch (e)
        {
            console.log(e);
        }

        var lastValue = item.m_CurrentValue;
        //  当旧值和新值都为空时，直接返回
        if (lastValue == null && newValue == null) return true;

        // 如果值与设置的值相等，直接返回
        if (item.m_CurrentValue != null && newValue != null)
        {
            if (this.ObjectCompare(item.m_CurrentValue, newValue) == 0) return true;
        }

        if (this.m_ds.m_SetValueWithTrigger) // 允许触发事件
        { // 触发是否接受值改变事件
            if (!this.m_ds.EM.fire("itemChangeAccept", [row, pCP.m_Name, newValue], true)) return false;
        }

        item.m_CurrentValue = newValue;
        //表示该数据已经做过改变，注意，它并不表示currentValue与oraginalValue是不是相等，
        // 仅仅表示currentValue被改动过，可能是改了又改回去 ，所以它并不表示数据被改成不一样了。
        item.m_isEdited = 1;

        // 修改状态
        //
        // 仅当 ( pCP->m_ObjType == isColumn) 时,才改变行的状态.
        // 因为如果仅仅是因为修改了 pCP->m_ObjType == isUserColumn 的列的值
        // 不应该导致 Insert 或者 Update 语句的解析,和 Validate事件的触发
        //
        switch (item.m_ItemStatus)
        {
            case ItemStatus.isNotModified:
                item.m_ItemStatus = ItemStatus.isDataModified;
                if (pCP.m_ObjType == ObjectType.isColumn) dr.m_RowStatus = ItemStatus.isDataModified;
                break;
            // 如果状态是数据被修改
            case ItemStatus.isDataModified:
            {

                if (item.m_CurrentValue == null && item.m_OriginalValue != null) break;
                if (item.m_CurrentValue != null && item.m_OriginalValue == null) break;

                // 是不是需要修改行的状态
                var /*boolean*/  needSetRowState = false;

                // 如果二者都 为null ,
                if (item.m_CurrentValue == null && item.m_OriginalValue == null)
                {
                    item.m_ItemStatus = ItemStatus.isNotModified;
                    needSetRowState = true;
                }

                // 如果又回设成最初的值,那么本单元就应该是没有改变
                if (item.m_CurrentValue != null && item.m_OriginalValue != null)
                {
                    if (this.ObjectCompare(item.m_CurrentValue, item.m_OriginalValue) == 0)
                    {
                        item.m_ItemStatus = ItemStatus.isNotModified;
                        needSetRowState = true;
                    }
                }

                if (!needSetRowState) break;

                //
                // 如果整行都是isNotModifyed,那么行的状态要回复成isNotModified
                var /*boolean*/       allIsNotModified = true;

                for (var ti = 0; ti < this.getColumnCount(); ti++)
                {

                    var /*ColumnProperty*/ tCP = this.getColumnProperty(ti);
                    // 不是数据列,忽略它
                    if (tCP.m_ObjType != ObjectType.isColumn) continue;
                    var st = this.getItemStatus(row, ti);
                    if (st == ItemStatus.isDataModified)
                    {
                        allIsNotModified = false;
                        break;
                    }
                }

                if (allIsNotModified)
                {
                    dr.m_RowStatus = ItemStatus.isNotModified;
                }
                break;
            }

            case ItemStatus.isNew:
                item.m_ItemStatus = ItemStatus.isNewModified;
                if (pCP.m_ObjType == ObjectType.isColumn) dr.m_RowStatus = ItemStatus.isNewModified;
                break;
            case ItemStatus.isNewModified:
            {
                if (item.m_CurrentValue == null && item.m_OriginalValue != null) break;
                if (item.m_CurrentValue != null && item.m_OriginalValue == null) break;

                // 是不是需要修改行的状态
                var /*boolean*/ needSetRowState = false;

                // 如果二者都 为null ,
                if (item.m_CurrentValue == null && item.m_OriginalValue == null)
                {
                    item.m_ItemStatus = ItemStatus.isNew;
                    needSetRowState = true;
                }

                // 如果又回设成最初的值,那么本单元就应该是没有改变
                if (item.m_CurrentValue != null && item.m_OriginalValue != null)
                {
                    if (ObjectCompare(item.m_CurrentValue, item.m_OriginalValue) == 0)
                    {
                        item.m_ItemStatus = ItemStatus.isNew;
                        needSetRowState = true;
                    }
                }

                if (!needSetRowState) break;

                //  修改严重漏洞
                // 如果整行都是isNotModifyed,那么行的状态要回复成isNotModified
                var allIsNotModified = true;

                for (var ti = 0; ti < this.getColumnCount(); ti++)
                {

                    var /*ColumnProperty*/        tCP = this.getColumnProperty(ti);
                    // 不是数据列,忽略它
                    if (tCP.m_ObjType != ObjectType.isColumn) continue;
                    var st = this.getItemStatus(row, ti);
                    if (st == ItemStatus.isNewModified)
                    {
                        allIsNotModified = false;
                        break;
                    }
                }

                if (allIsNotModified)
                {
                    dr.m_RowStatus = ItemStatus.isNew;
                }
                break;
            }// end of case

        } // end of switch

        // 如果修改值并触发事件，并且在事件 ItemChangeAccept事件中没有禁止触发
        // ItemChanged事件，那么触发ItemChanged事件
        if (this.m_ds.m_SetValueWithTrigger)
        {
            // 因为lastvalue 没有用了，为了性能，所以不用克隆
            // 把 newValue 克隆一下，再做参数 ，可以防止在事件中通过newValue 改变m_currentValue
            //出于性能考虑，没有做克隆，如果有负作用，改成用克隆的值
            this.m_ds.EM.fire("itemChanged", [row, pCP.m_Name, lastValue, newValue]);
        }

        // 通知它将影响的计算列:数据发生变化,需要重新计算

        var ChangedComputer = {}; // map < _bstr_t, long >
        // ChangedComputer; //
        // _bstr_t 记录列名称， long
        // 记录行号

        this.NotifyAffectCellInvalid(row, pCP, ChangedComputer);

        for (var computer in ChangedComputer)
        {

            // 事件触发
            if (this.m_ds.m_SetValueWithTrigger) this.m_ds.EM.fire("computerChanged", [ChangedComputer[computer], computer]);
            // 看有没有计算列回填
            var ci = this.col2Index(computer);
            var /*ColumnProperty*/          cpCP = this.getColumnProperty(ci);
            var /*String*/ fillback = cpCP.getFillBack();
            if (fillback != "")
            {
                var fillBackIndex = this.col2Index(fillback);
                // 如果回填字段合法，且不是本字段，那么回填
                if (fillBackIndex >= 0 && fillBackIndex != col) this.setValue(row, fillBackIndex, this.getValue(row, ci));
            }

        }

        ChangedComputer = {};
        return true;

    },

    // 通知影响到的单元将自己的状态修改成非法,并第归通知下去
    // 本函数并不把pCP对应的列放入到 ChangedComputerMap中，因为：
    // 在 setValue 中，被设置的col的itemchanged事件已经触发了，之后需要触发的是它影响到的计算列改变事件。
    // 而在notifyAggregateAndUnsureComputerInvalid中，
    // 手工已经把pCP对应的列加入到了ChangedComputerMap中
    NotifyAffectCellInvalid: function (/*int*/ row, /*ColumnProperty*/ pCP, /*Object*/ChangedComputerMap) {
        if (pCP.m_AffectMap == null) return; // 本函数 第一次调用时，它不为空，但是进入递归调用后，就难保，所以必须检测。

        var rc = this.m_ds.getRowCount();
        for (var i = 0; i < pCP.m_AffectMap.length; i++)
        {
            var affectCol = pCP.m_AffectMap[i];
            var colIndex = this.col2Index(affectCol);

            // 如果已经失效不要再次失效,
            if (ChangedComputerMap[affectCol] != undefined) continue;
            // 如果没有加入到事件触发映射中，那么加入他
            ChangedComputerMap[affectCol] = row;
            // 如果被影响的列是聚合型的计算列，那么整个列的值都失效
            /*
             * ( 粒度比较粗，更好的作法是查看该单元格的影响范围 if colindex
             * 不是分类列，那么她仅仅影响所在的组所所包含的行范围，否则影响全部行
             */
            var /*ColumnProperty*/     affectCP = this.getColumnProperty(colIndex);
            if (affectCP.m_ObjType == ObjectType.isAggregateComputer)
            {
                // 将行号设置成 -1 , 表明所有的的都改变
                ChangedComputerMap[affectCol] = -1;
                for (var i = 0; i < rc; i++)
                {
                    var pItem = this.getItem(i, colIndex);
                    if (pItem != null) pItem.m_ItemStatus = ItemStatus.isInvalid;
                }
                // 第归调用
                this.NotifyAffectCellInvalid(-1, affectCP, ChangedComputerMap);
            } else
            // 仅仅该行的值失效
            {
                if (row < 0)
                {
                    for (var i = 0; i < rc; i++)
                    {
                        var pItem = this.getItem(i, colIndex);
                        if (pItem != null) pItem.m_ItemStatus = ItemStatus.isInvalid;
                    }
                } else
                {
                    var pItem = this.getItem(row, colIndex);
                    if (pItem != null) pItem.m_ItemStatus = ItemStatus.isInvalid;
                }

                // 第归调用
                this.NotifyAffectCellInvalid(row, affectCP, ChangedComputerMap);

            }

        }
    },

    // 通知所有的聚合列和使用了不稳定函数的计算列它们的值发生改变
    // 本函数 通常在插入，删除，检索，过滤，排序后需要调用
    notifyAggregateAndUnsureComputerInvalid: function () {
        if (!this.m_ds.isNeedNotifyAggregateAndUnsureComputerInvalid()) return;

        var /*ColumnProperty*/  pCP;
        var /*DsItem*/ pItem;
        //
        var /*Hashtable*/ChangedComputer = {};//new Hashtable(); // map< _bstr_t , long>
        // ChangedComputer;  _bstr_t 记录列名称， long 记录行号

        var cc = this.getColumnCount();

        var rc = this.getRowCount();
        for (var i = 0; i < cc; i++)
        {
            pCP = this.getColumnProperty(i);
            if (pCP.m_ObjType == ObjectType.isAggregateComputer || pCP.isIncludeUnsureFunction())
            {
                for (var r = 0; r < rc; r++) // 将给列的数据设置成非法
                {
                    pItem = this.getItem(r, i);
                    if (pItem != null) pItem.m_ItemStatus = ItemStatus.isInvalid;
                }

                // 标记 需要把自己加入到列表中 ，下面的这一行原来没有，导致某些计算列无法回填
                ChangedComputer[pCP.getName()] = -1;
                // 然后通知所有依赖本列的列应该刷新数据.
                this.NotifyAffectCellInvalid(-1, pCP, ChangedComputer);
            }
        }

        // 再触发计算列发生变化事件，同时检测回填
        for (var computer in   ChangedComputer)
        {   // 事件触发
            if (this.m_ds.m_SetValueWithTrigger) this.m_ds.EM.fire("computerChanged", [ChangedComputer[computer], computer]);
            //看有没有计算列回填
            var ci = this.col2Index(computer);
            var /*ColumnProperty*/     cpCP = this.getColumnProperty(ci);
            var fillback = cpCP.getFillBack();

            if (fillback != "")
            {
                var fillBackIndex = this.col2Index(fillback);
                // 如果回填字段合法，且不是本字段，那么回填

                if (fillBackIndex >= 0)
                {
                    for (var r = 0; r < rc; r++)
                    {
                        this.setValue(r, fillBackIndex, this.getValue(r, ci));
                    }
                }
            }

        }

    },

    initColumnCount: function (c) {
        this.m_ColumnCount = c;
    },

    IsOneOfPrimaryKey: function (colName) {
        var col = colName + "";
        col = "," + col.trim().toLowerCase() + ",";
        var pk = this.m_ds.m_PrimaryKey + "";
        pk = "," + pk.trim().toLowerCase() + ",";
        if (pk.indexOf(col) >= 0) return true;
        return false;
    },

    /**
     * ------------------------------------------------------------------------
     * -- ----------------------------------------------------------------------
     * -- ---------------- 函数名称 : CDataStore::GetFormatedTextValue 功能描述 :
     * 得到格式化后的以字符方式表示的值,该函数被 update调用 "123" 表示数值型 "'abc'"表示字符型 ,
     * 如果有单引号.那么将被替换成两个单引号 如果是 postgresql 那么 \ 被替换成 \\ "'2001.01.01'" 表示日期型
     * "NULL" 表示空 返回类型 : _bstr_t ****** 关于字符串的处理需要同步修改函数 BuildSQLFormatString
     * 参数说明 : CdsItem *pItem 参数说明 : CColumnProperty *pCP 参数说明 : VARIANT_BOOL
     * original 修改日志 : 创建时间 2001.6.7 ★ 该函数仅用于 AssembleSQL????等函数中,
     * 如果需要取得某列文本化的值,请 使用函数 GetString 或者属性 Text *
     * --------------------------------
     * ------------------------------------------
     * --------------------------------
     * --------------------------------------------------------
     */

    GetFormatedTextValue: function (/*DsItem*/  pItem, /*ColumnProperty*/    pCP, /*boolean*/   original) {
        if (original == undefined) original = false;
        if (pItem == null) return this.m_ds.getDialect().getFormatedTextValue(null, pCP.m_DataType, pCP.Scale);
        return this.m_ds.getDialect().getFormatedTextValue((original ? pItem.m_OriginalValue : pItem.m_CurrentValue),
            pCP.m_UniformDataType, pCP.Scale);
    },

    getFormatedValue: function (/*int*/  row, /*String*/    colIndexOrName, /*boolean*/   original) {

        var col = this.col2Index(colIndexOrName);
        var /*DsItem*/    item = this.getItem(row, col);
        if (item == null) return "";
        var /*ColumnProperty*/    cp = this.getColumnProperty(col);
        return this.GetFormatedTextValue(item, cp, original);
    },

    /**
     * 构造删除语句
     * @param row
     * @return
     */

    AsserbleSQLDelete: function (row) {
        var /*DsItem*/   pItem;
        var /*ColumnProperty*/  pCP;
        var /*DsRow*/ pRow = this.m_Buffer[row];
        var i, n = this.getColumnCount();
        var sql = "";
        var WhereClause = "";
        var temp = "";
        // 如果是新插入的行或者是新插入并且被修改但又被删除的行,
        // 不需要构造删除语句
        if (pRow.m_RowStatus == ItemStatus.isNew || pRow.m_RowStatus == ItemStatus.isNewModified) return "";

        for (i = 0; i < n; i++)
        {
            pCP = this.getColumnProperty(i);
            if (!this.IsOneOfPrimaryKey(pCP.m_DBName)) continue;

            pItem = pRow.getAt(i);
            if (pItem == null) // 1
            {
                temp = pCP.m_DBName + " is NULL";
            } else
            {
                temp = this.GetFormatedTextValue(pItem, pCP, true);
                if (temp == "NULL")
                {
                    temp = pCP.m_DBName + " is NULL";
                } else
                {
                    temp = pCP.m_DBName + " = " + temp;
                }
            } // 1

            if (WhereClause == "")
            {
                WhereClause = temp;
            } else
            {
                WhereClause += " and " + temp;
            }

        }

        sql = "Delete From  " + this.m_ds.m_UpdateTable;

        if (WhereClause == (""))
        {
            throw new Exception("没有用于构建where子句的字段");
        } else
        {
            sql += "  Where (" + WhereClause + ")";
        }

        return sql;

    },

    /**
     * 拼一个删除命令数据包，示例: { command:'delete', table:'test1' , where:[ 'id=1' , "name='abc'" ] }
     * @param row
     * @returns {*}
     * @constructor
     */
    AsserbleCommandDelete: function (row) {
        var /*DsItem*/   pItem;
        var /*ColumnProperty*/  pCP;
        var /*DsRow*/ pRow = this.m_Buffer[row];
        var i, n = this.getColumnCount();
        var sql = {command: 'delete', dataSourceName: this.m_ds.dataSourceName,  table: this.m_ds.m_UpdateTable};

        var WhereClause = [];
        var primaryKeyValue = {};
        var temp = "";
        // 如果是新插入的行或者是新插入并且被修改但又被删除的行,
        // 不需要构造删除语句
        if (pRow.m_RowStatus == ItemStatus.isNew || pRow.m_RowStatus == ItemStatus.isNewModified) return null;

        for (i = 0; i < n; i++)
        {
            pCP = this.getColumnProperty(i);
            if (!this.IsOneOfPrimaryKey(pCP.m_DBName)) continue;

            pItem = pRow.getAt(i);
            if (pItem == null) // 1
            {
                temp = pCP.m_DBName + " is NULL";
                primaryKeyValue[pCP.m_DBName] = 'null';
            } else
            {
                temp = this.GetFormatedTextValue(pItem, pCP, true);
                if (temp == "NULL")
                {
                    temp = pCP.m_DBName + " is NULL";
                    primaryKeyValue[pCP.m_DBName] = 'null';
                } else
                {
                    primaryKeyValue[pCP.m_DBName] = this.getValue(row, pCP.m_DBName);
                    temp = pCP.m_DBName + " = " + temp;
                }
            } // 1

            WhereClause.push(temp);


        }


        if (WhereClause.length == 0)
        {
            throw new Exception("没有用于构建where子句的字段");
        }
        sql.where = WhereClause;
        sql.primaryKey = primaryKeyValue;

        return sql;

    },


    /**
     * 使用主键及其值，构建出一个where子句，此函数用在insert 和update时，对clob字段进行操作时构建where条件
     *
     * @param row
     * @return
     */
    assemblePrimaryKeyWhere: function (row) {
        var /*DsItem*/      pItem;
        var /*String*/      WhereClause = "";
        var /*String*/     temp = "";
        var /*ColumnProperty*/    pCP;
        var /*DsRow*/   pRow = this.m_Buffer[row];
        var i, n = this.columnCount;
        for (i = 0; i < n; i++)
        {
            pCP = this.getColumnProperty(i);
            if (!this.IsOneOfPrimaryKey(pCP.m_DBName)) continue;

            pItem = pRow.getAt(i);
            if (pItem == null) // 1
            {
                temp = pCP.m_DBName + " is NULL";
            } else
            {
                temp = this.GetFormatedTextValue(pItem, pCP, true);
                // 如果主键字段为空，那么可能的原因就是导入的新数据，主键字段的original值是没有，所以就取当前值
                if (temp == ("NULL")) temp = this.GetFormatedTextValue(pItem, pCP, false);
                if (temp == ("NULL"))
                {
                    temp = pCP.m_DBName + " is NULL";
                } else
                {
                    temp = pCP.m_DBName + " = " + temp;
                }
            } // 1

            if (WhereClause == (""))
            {
                WhereClause = temp;
            } else
            {
                WhereClause += " and " + temp;
            }

        }

        return WhereClause;
    },

    /**
     * 构造一个Insert语句
     * @param row
     * @return   [ ]
     */
    AsserbleSQLInsert: function (row) {
        var /*DsRow*/      pRow;
        var /*DsItem*/     pItem;
        var /*ColumnProperty*/    pCP;
        pRow = this.getRow(row);
        var i, n = this.columnCount;
        var sql;
        var ColClause = "";
        var ValueClause = "";

        // 因为oracle ,postgres 不支持 identity,所以要特别处理
        var isOracle = this.m_ds.getDialect().isOracle();
        var isPosgtres = this.m_ds.getDialect().isPostgres();
        var clobColumnList = [];
        //2018.06 增加：如果有lastmodifieddate字段存在，那么填上现在时间做为最后修改时间
        if (this.m_ds.columnExists('lastmodifieddate')) this.m_ds.setValue(row, "lastmodifieddate", new Date());
        for (i = 0; i < n; i++)
        {
            this.getColumnCount();
            pCP = this.getColumnProperty(i);
            if (pCP.m_Updatable)
            {
                pItem = pRow.getAt(i);
                if (pItem == null) continue; // 没有值 ,NULL
                if (pItem.m_ItemStatus == ItemStatus.isNew) continue;

                if (pCP.getObjType() != ObjectType.isColumn) continue; //  如果不是字段，那么忽略

                var value = this.GetFormatedTextValue(pItem, pCP);
                // 最大是4000 留一点余量
                if (pCP.m_DataType == Types.CLOB && isOracle && value.length() > 1500)
                {
                    value = pItem.m_CurrentValue.toString(); // 取没有格式化的数据
                    var zdp = {};
                    zdp.table = this.m_ds.m_UpdateTable;
                    zdp.col = pCP.m_DBName;
                    zdp.value = value;
                    clobColumnList.push(zdp);
                    value = "EMPTY_CLOB()";
                }

                if (ColClause == (""))
                {
                    ColClause = pCP.m_DBName;
                } else
                {
                    ColClause += "," + pCP.m_DBName;
                }
                if (ValueClause == (""))
                {
                    ValueClause = value;
                } else
                {
                    ValueClause += "," + value;
                }
            }

            else
            {
                // 如果是id字段，并且是oracle数据库，那么假定它是需要自增量的字段
                if (pCP.m_DBName.toLowerCase() == "id" && (isOracle || isPosgtres))
                {
                    if (ColClause == (""))
                    {
                        ColClause = pCP.m_DBName;
                    } else
                    {
                        ColClause += "," + pCP.m_DBName;
                    }

                    //TODO nextSequence
                    var newID = this.m_ds.nextSequence((pCP.m_DBTable + "." + pCP.m_DBName).toLowerCase());
                    // 回填到缓冲里
                    this.m_ds.setValue(row, pCP.m_Name, newID);// 这个保证了下面的pItem不为null
                    pItem = pRow.getAt(i);
                    pItem.m_CurrentValue = newID;
                    pItem.m_OriginalValue = newID; // 在assemblePrimaryKeyWhere时是要用原始数据
                    // ，所以也要设置值

                    if (ValueClause == (""))
                    {
                        ValueClause = '' + newID;
                    } else
                    {
                        ValueClause += "," + newID;
                    }
                }
            }

        }

        /*
         * 有可能是增加的用户列的值改变，所以行的状态是isNewModified ，单用户列是不保存到数据库中的 所以引起行的状态是
         * isNewModified 而 ColClause 为 "" , 这种情况下,不需要形成SQL语句
         */

        var ret = [];
        if (ColClause != "" || ValueClause == "")
        {
            sql = "Insert into " + this.m_ds.m_UpdateTable + " (" + ColClause + ") Values (" + ValueClause + ")";
            ret.push(sql);
        }

        // where 子句一定在在这里再处理，因为到这里，oracle里的 ID字段也已经有了值

        var clobColumnCount = clobColumnList.length;
        if (clobColumnCount > 0)
        {
            var WhereClause = this.assemblePrimaryKeyWhere(row);

            if (WhereClause == ("")) throw "没有用于构建where子句的字段";

            // clob 字段的处理
            for (var k = 0; k < clobColumnList.length; k++)
            {
                var zdp = clobColumnList[k];
                zdp.where = WhereClause;
                zdp.action = "updateblob";
                ret.push(zdp);
            }
        }

        return ret;
    },


    AsserbleCommandInsert: function (row) {
        var /*DsRow*/      pRow;
        var /*DsItem*/     pItem;
        var /*ColumnProperty*/    pCP;
        pRow = this.getRow(row);
        var i, n = this.columnCount;
        var sql = {command: 'insert', dataSourceName: this.m_ds.dataSourceName, table: this.m_ds.m_UpdateTable};


        var ValueClause = {};


        // 因为oracle ,postgres 不支持 identity,所以要特别处理
        var isOracle = this.m_ds.getDialect().isOracle();
        var isPosgtres = this.m_ds.getDialect().isPostgres();
        var clobColumnList = [];
        //2018.06 增加：如果有lastmodifieddate字段存在，那么填上现在时间做为最后修改时间
        if (this.m_ds.columnExists('lastmodifieddate')) this.m_ds.setValue(row, "lastmodifieddate", new Date());
        for (i = 0; i < n; i++)
        {
            this.getColumnCount();
            pCP = this.getColumnProperty(i);
            if (pCP.m_Updatable)
            {
                pItem = pRow.getAt(i);
                if (pItem == null) continue; // 没有值 ,NULL
                if (pItem.m_ItemStatus == ItemStatus.isNew) continue;

                if (pCP.getObjType() != ObjectType.isColumn) continue; //  如果不是字段，那么忽略

                var value = this.GetFormatedTextValue(pItem, pCP);
                // 最大是4000 留一点余量
                if (pCP.m_DataType == Types.CLOB && isOracle && value.length() > 1500)
                {
                    value = pItem.m_CurrentValue.toString(); // 取没有格式化的数据
                    var zdp = {};

                    zdp[pCP.m_DBName] = value;
                    clobColumnList.push({command: 'updateblob', table: this.m_ds.m_UpdateTable, value: zdp});
                    value = "EMPTY_CLOB()";
                }

                ValueClause[pCP.m_DBName] = value;

            }
            else
            {
                // 如果是id字段，并且是oracle数据库，那么假定它是需要自增量的字段
                if (pCP.m_DBName.toLowerCase() == "id" && (isOracle || isPosgtres))
                {

                    //TODO nextSequence
                    var newID = this.m_ds.nextSequence((pCP.m_DBTable + "." + pCP.m_DBName).toLowerCase());
                    // 回填到缓冲里
                    this.m_ds.setValue(row, pCP.m_Name, newID);// 这个保证了下面的pItem不为null
                    pItem = pRow.getAt(i);
                    pItem.m_CurrentValue = newID;
                    pItem.m_OriginalValue = newID; // 在assemblePrimaryKeyWhere时是要用原始数据
                    // ，所以也要设置值

                    ValueClause['id'] = newID;

                }
            }

        }

        /*
         * 有可能是增加的用户列的值改变，所以行的状态是isNewModified ，单用户列是不保存到数据库中的 所以引起行的状态是
         * isNewModified 而 ColClause 为 "" , 这种情况下,不需要形成SQL语句
         */

        var ret = [];
        if (Object.getOwnPropertyNames(ValueClause).length > 0)
        {
            sql.value = ValueClause;
            ret.push(sql);
        }


        // where 子句一定在在这里再处理，因为到这里，oracle里的 ID字段也已经有了值

        var clobColumnCount = clobColumnList.length;
        if (clobColumnCount > 0)
        {
            var WhereClause = this.assemblePrimaryKeyWhere(row);

            if (WhereClause == ("")) throw "没有用于构建where子句的字段";

            // clob 字段的处理
            for (var k = 0; k < clobColumnList.length; k++)
            {
                var zdp = clobColumnList[k];
                zdp.where = [WhereClause];  //这里的where是一个长句，但是其它的都是数组，所以转成数组
                ret.push(zdp);
            }
        }

        return ret;
    },


    /**
     * 构造一个Update语句
     * @param row
     * @return
     */


    AsserbleSQLUpdate: function (row) {
        var /*DsRow*/    pRow = this.getRow(row);
        var /*DsItem*/        pItem;
        var /*ColumnProperty*/       pCP;
        var i, n = this.columnCount;
        var sql;
        var SetClause = "";
        var WhereClause = "";
        var temp;
        var clobColumnList = []
        var isOracle = this.m_ds.getDialect().isOracle();

        //2018.06 增加：如果有lastmodifieddate字段存在，那么填上现在时间做为最后修改时间
        if (this.m_ds.columnExists('lastmodifieddate')) this.m_ds.setValue(row, "lastmodifieddate", new Date());

        for (i = 0; i < n; i++)
        {
            pCP = this.getColumnProperty(i);
            pItem = pRow.getAt(i);

            if (pCP.m_Updatable)
            {
                // 构造 set 子句
                if (pCP.m_Updatable && pItem != null) // 允许更新,并且不为空,那么
                {

                    if (pItem.m_ItemStatus == ItemStatus.isDataModified)
                    // 如果数据有修改,那么就需要构造set
                    {
                        var value = this.GetFormatedTextValue(pItem, pCP);
                        // 最大是4000 留一点余量 ,但注意，如果有多个clob ,应该是多个clob
                        // 累计不超过4000,所以这时简化了处理
                        if (pCP.m_DataType == Types.CLOB && isOracle && value.length() > 1000)
                        {
                            value = '' + pItem.m_CurrentValue // 取没有格式化的数据
                            var zdp = {}
                            zdp.table = m_ds.m_UpdateTable;
                            zdp.col = pCP.m_DBName;
                            zdp.value = value;
                            clobColumnList.push(zdp);
                            value = "EMPTY_CLOB()";
                        }

                        temp = pCP.m_DBName + " = " + value;
                        if (SetClause == (""))
                        {
                            SetClause = temp;
                        } else
                        {
                            SetClause += "," + temp;
                        }

                    }
                }
            }

            // 构造 Where子句
            if (pCP.m_UpdateWhereClause)
            {
                if (pItem == null)
                {
                    temp = pCP.m_DBName + " is NULL";
                } else
                {
                    temp = this.GetFormatedTextValue(pItem, pCP, true);
                    if (temp == (""))
                    {
                        temp = pCP.m_DBName + " is NULL";
                    } else
                    {
                        temp = pCP.m_DBName + " = " + temp;
                    }
                }

                if (WhereClause == (""))
                {
                    WhereClause = temp;
                } else
                {
                    WhereClause += " and " + temp;
                }
            }
        }

        var ret = [];
        if (WhereClause == ("")) throw ("没有用于构建where子句的字段");

        if (!SetClause == (""))
        {
            sql = "Update " + this.m_ds.m_UpdateTable + " Set " + SetClause;
            if (!WhereClause == ("")) sql += "  Where (" + WhereClause + ")";
            ret.push(sql);
        }

        for (var k = 0;
             k < clobColumnList.length;
             k++
        )
        {
            var zdp = clobColumnList[k];
            zdp.where = WhereClause;
            zdp.command = "updateblob";
            ret.push(zdp);
        }

        return ret;

    },


    AsserbleCommandUpdate: function (row) {
        var /*DsRow*/    pRow = this.getRow(row);
        var /*DsItem*/        pItem;
        var /*ColumnProperty*/       pCP;
        var i, n = this.columnCount;
        var sql = {command: 'update', dataSourceName: this.m_ds.dataSourceName, table: this.m_ds.m_UpdateTable};

        var SetClause = {};
        var OldClause = {};
        var primaryKeyValue = {};
        var WhereClause = [];
        var temp;
        var clobColumnList = []
        var isOracle = this.m_ds.getDialect().isOracle();

        //2018.06 增加：如果有lastmodifieddate字段存在，那么填上现在时间做为最后修改时间
        if (this.m_ds.columnExists('lastmodifieddate')) this.m_ds.setValue(row, "lastmodifieddate", new Date());

        for (i = 0; i < n; i++)
        {
            pCP = this.getColumnProperty(i);
            pItem = pRow.getAt(i);

            if (pCP.m_Updatable)
            {
                // 构造 set 子句
                if (pCP.m_Updatable && pItem != null) // 允许更新,并且不为空,那么
                {

                    if (pItem.m_ItemStatus == ItemStatus.isDataModified)
                    // 如果数据有修改,那么就需要构造set
                    {
                        var value = this.GetFormatedTextValue(pItem, pCP);
                        var oldValue = this.GetFormatedTextValue(pItem, pCP, true);
                        // 最大是4000 留一点余量 ,但注意，如果有多个clob ,应该是多个clob
                        // 累计不超过4000,所以这时简化了处理
                        if (pCP.m_DataType == Types.CLOB && isOracle && value.length() > 1000)
                        {
                            value = '' + pItem.m_CurrentValue // 取没有格式化的数据


                            clobColumnList.push({
                                command: 'updateblob',
                                table: this.m_ds.m_UpdateTable,
                                col: pCP.m_DBName,
                                value: value
                            });

                            value = "EMPTY_CLOB()";
                        }

                        SetClause [pCP.m_DBName] = value;
                        OldClause [pCP.m_DBName] = oldValue;


                    }
                }
            }

            // 构造 Where子句
            if (pCP.m_UpdateWhereClause)
            {
                if (pItem == null)
                {
                    temp = pCP.m_DBName + " is NULL";
                    primaryKeyValue[pCP.m_DBName] = 'null';
                } else
                {
                    temp = this.GetFormatedTextValue(pItem, pCP, true);
                    if (temp == (""))
                    {
                        temp = pCP.m_DBName + " is NULL";
                        primaryKeyValue[pCP.m_DBName] = 'null';
                    } else
                    {
                        primaryKeyValue[pCP.m_DBName] = this.getValue(row, pCP.m_DBName);
                        temp = pCP.m_DBName + " = " + temp;
                    }
                }


                WhereClause.push(temp);

            }
        }

        var ret = [];
        if (WhereClause.length == 0) throw ("没有用于构建where子句的字段");

        if (Object.getOwnPropertyNames(SetClause).length > 0)
        {
            sql.value = SetClause;
            sql.oldValue = OldClause; //旧的值也带上
            sql.where = WhereClause;
            sql.primaryKey = primaryKeyValue;
            ret.push(sql);
        }

        for (var k = 0;
             k < clobColumnList.length;
             k++
        )
        {

            zdp.where = WhereClause;
            ret.push(zdp);
        }

        return ret;

    },


    changeType: function (value, toType) {
        try
        {
            return ObjectTool.changeType(value, toType);
        } catch (e)
        {
            this.m_ds.error(e);
            return null;
        }
    },

    // 返回-1 表示数据需要排在前面
    //所以当升序时， -1表示小于从而数据排在前面，而在降序时，-1表示大于从而排在前面
    //当需要按多个字段排序时，如果某个字段中两个数据都是null则继续判断下一个字段
    //如果所有字段都是null ,那么保持现有顺序不变
    compare: function (o1, o2) {
        return 1;

    },

    /**
     * 比较两个值的大小， 切记，必须保证 v1 , v2 不能为null
     *
     * @param v1
     * @param v2
     * @return   1 表示 v1>v2 , 0 表示相等  -1 表示小于
     */

    ObjectCompare: function (v1, v2) {
        // 该函数不是完整的比较对象大小的函数，因为在buffer中，只存在String , Timestamp 和数字型，所以
        // 仅仅对String 做了明示的比较， 其它都转换成double 进行比较

        if (Util.isString(v1))
        {
            return v1.localeCompare(v2);
        }

        if (Util.isDate(v1))
        {
            var t1 = v1.getTime();
            var t2 = v2.getTime();
            if (t1 == t2) return 0;
            return t1 > t2 ? 1 : -1;

        }

        if (!isNaN(v1))
        {

            if (v1 == v2) return 0;
            return (v1 > v2) ? 1 : -1;
        }

        if (v1 == v2) return 0;
        return v1 > v2 ? 1 : -1;

    },

    sort: function () {
        //Collections.sort(m_Buffer, this);

    },

    saveAsTxtString: function (withHead, splitChar, colNames) {
        return this.saveAsTxt(null, withHead, splitChar, colNames);
    },

    saveAsTxt: function (fileName, withHead, splitChar, colNames) {

        if (splitChar == undefined) splitChar = '\t';
        if (colNames == undefined) colNames = "*";

        var data = [];

        var row, col;
        var rc = this.getRowCount();
        var cc = this.getColumnCount();

        var cols = [];
        if (colNames == '*')
        {
            for (var ci = 0; ci < cc; ci++)
            {
                cols.push(ci);
            }
        } else
        {
            var t = colNames.split(",");
            for (var ci = 0; ci < t.length; ci++)
            {
                var col = t[ci];
                if (col.trim() == '') continue;
                var colIndex = this.m_ds.col2Index(col);
                if (colIndex < 0) continue;
                cols.push(colIndex);

            }
        }

        cc = cols.length;
        var colIndex2ColName = {};

        if (withHead)
        {
            var rowData = [];
            for (var col = 0; col < cc; col++)
            {
                var cp = this.m_ds.getColumnProperty(cols[col]);

                var title = cp.getColumnLabel();
                var name = cp.getName();

                colIndex2ColName['c' + cols[col]] = name;
                rowData.push(title)
            }
            data.push(rowData.join(splitChar) + '\n');
        }


        var v;
        for (row = 0; row < rc; row++)
        {
            var rowData = [];
            for (col = 0; col < cc; col++)
            {

                v = this.m_ds.EM.fire("getExportValue", [row, colIndex2ColName['c' + cols[col]]], null); //导出值可以定制
                if (v == null) v = this.getValue(row, cols[col]);

                if (v == null)
                {
                    rowData.push('');
                } else
                {
                    try
                    {
                        var s = ObjectTool.changeType(v, UniformDataType.$String);
                        //2019.05 回车符引起格式混乱，所以去掉回车
                        s = s.replace(/\r/g, '');
                        s = s.replace(/\n/g, '');
                        rowData.push(s);
                    } catch (e)
                    {
                        rowData.push('');
                    }

                }

            }
            data.push(rowData.join(splitChar) + '\n');
        }


        if (fileName != null)
        {

            saveAs(
                new Blob(
                    data
                    , {type: "text/plain;charset=UTF-8"}
                )
                , fileName
            );
        } else
        {
            return data;
        }
    },


    saveAsDBF: function (file) {

    },


    importTxtFile: function (file, skipRow) {

    },


    importDBFFile: function (file, colMap) {

    },


    saveAsXML: function (file) {

    },


    ImportXMLFile: function (file, charsetName) {

    },


    importXMLSource: function (xmlSource) {

    },


    InsureColumnValueValid: function (col) {
        var /*ColumnProperty*/      pCPP = this.getColumnProperty(col);
        var row;
        var rc;

        if (pCP.m_ObjType == ObjectType.isUserComputer)
        {
            rc = this.getRowCount();
            for (row = 0; row < rc; row++)
            {
                var /*DsRow*/            dr = this.m_Buffer[row]
                var /*DsItem*/           item = dr.getAt(col);

                if (item == null)
                {
                    item = this.NEW_ITEM(item, dr, col);
                    item.m_ItemStatus = ItemStatus.isInvalid;
                }

                // 如果该计算列的值是有效的,那么直接跳过
                if (item.m_ItemStatus == ItemStatus.isAvailable) continue;
                // 计算它的值
                InternalRecalculateComputer(item, pCP.m_Formula, pCP.m_DependMap, row);

            }
        }

        // 如果是聚合列
        /*
         if (pCP.m_ObjType == ObjectType.isAggregateComputer)
         {
         var  GroupNo = pCP.m_AggregateDefine.m_GroupNo;
         if (GroupNo < 0 || GroupNo > (int)(m_ds.m_VectorGroups.size() - 1)) return;
         rc = getRowCount();
         for (row = 0; row < rc; row++)
         {
         Point
         range = AggregateRange(row, GroupNo);

         // 如果不是累计求和，那么该组中地聚合列地值应该相同，所以转向取改组的第一行的值
         if (pCP.m_AggregateDefine.m_AggregateType != AggregateType.AT_CUMULATIVE) row = range.x;

         DsRow_
         dr = (DsRow_)
         m_Buffer.get(row);
         DsItem
         item = dr.getAt(col);

         if (item == null)
         {
         item = NEW_ITEM(item, dr, col);
         item.m_ItemStatus = ItemStatus.isInvalid;
         }

         // 如果该聚合列的值是有效的,那么直接跳过
         if (item.m_ItemStatus == ItemStatus.isAvailable) continue;

         // 需要计算值
         // 评估该计算列表达式的值
         Object
         v = null;
         v = Eval(pCP.m_Formula, pCP.m_DependMap, row, pCP.m_AggregateDefine, range);
         // 修改值和状态
         item.m_ItemStatus = ItemStatus.isAvailable;
         // 注意：是复制值，不是指向
         item.m_CurrentValue = v;

         }
         */


    },


    setBalloonTip: function (row, col, type, info) {

        if (row > this.size() - 1) return;
        if (row < 0) return;

        var /*DsRow*/    rowData = this.getRow(row);
        rowData.setItemBalloonTip(col, type, info);
        this.m_ds.getEvent().fireItemBalloonTipChanged(row, col, type, info);
    },

    getBalloonTip: function (row, col, type) {

        if (row > this.size() - 1) return null;
        if (row < 0) return null;

        var /*DsRow*/  rowData = this.getRow(row);
        return rowData.getItemBalloonTip(col, type);
    }

});

export default DataBuffer;
