/**
 * Created by 三宝爹 on 2018/10/6.
 *
 * worksheet中与打印输出相关的操作
 */


import PrintRange from '../PrintRange.js';
import PageSplitInfo from '../PageSplitInfo.js';
import UnitConv from '../../util/UnitConv.js';
import GraphicsLogic from '../../gdi/GraphicsLogic.js';
import Rectangle from '../../gdi/Rectangle.js';
import Util from '../../util/Util.js';
import Tools from '../../util/Tools.js';


//sleep 以释放对CPU的占用
/*
  在 $exportAsPrintCommand 需要对每一页生成打印命令，然后用websocket.send 将打印命令发送给打印助手

  但是，生成打印命令是很占用CPU的，而websocket.send是异步的 ， 这个结果就是必须等所有页面都生成了打印命令后，CPU得到
  释放，此时 websocket.send 才真正开始工作 。这个与设想的不一样，设想的是生成一页打印命令，就发送一页打印命令到打印助手，
  避免等待。

  要解决这个问题，就必须在生成一页打印命令后， 稍稍地休息一下，让CPU缓一缓，让websocket认为现在不忙，马上发送。
  于是，就需要在计算好一页后，在发送前， 小小地sleep 一下。 参看[标记20191224]

  值得庆幸的是， $exportAsPrintCommand 是整个打印输出执行序列中的最后一个函数，没有后续操作，这就无需担心它整成异步执行
  会带来执行顺序的混乱。

  2020.08.20 当需要循环多次打印时，异步就带来了麻烦

 */

var sleep = function (delay) {
    var p = new Promise(function (resolve, reject) { //做一些异步操作
        setTimeout(function () {
            resolve();
        }, delay);
    });
    return p;
};


//测试sleep的代码
async function test_sleep()
{
    console.log("test1");
    await sleep(2000);
    console.log("test2");
}

//test_sleep();


var ISheetPrint = {

    getCurrentPrintingPage: function () {
        return this.m_currentPrintingPage;
    },


    setCurrentPrintingPage: function (currentPrintingPage) {
        this.m_currentPrintingPage = currentPrintingPage;
    },

    getCurrentPrintingPageCount: function () {
        return this.m_currentPrintingPageCount;
    },


    setCurrentPrintingPageCount: function (currentPrintingPageCount) {
        this.m_currentPrintingPageCount = currentPrintingPageCount;
    },


    getPrintScale()
    {
        return this.m_printScale;
    },


    /**
     * 准备打印输出时，使用的打印比例，
     * getPrintScale 是返回打印比例设置，
     * 本函数是返回真正的打印比例，当打印比例设置为0时，自动计算适应列宽时的打印比例
     * @returns {*}
     */
    getPrintScaleForPrint: function () {
        if (this.m_printScale > 0) return this.m_printScale;

        var cc = this.columnCount;
        var totalWidth = 0;
        for (var i = 0; i < cc; i++)
        {
            totalWidth += this.CPM.getColumnWidth(i);
        }
        //得到纸张的大小，除去了margin
        var rc = this.getPrintAreaSize();
        var pageWidth = rc.width;
        return Math.floor(pageWidth / totalWidth * 100.0);

    },


    setPrintScale: function (scale) {
        if (scale < 0) return;
        if (scale > 400) return;
        this.m_printScale = scale;

    },

    clearPageRange: function () {
        this.pageRange.clear();
    },

    // 打印分页
    /**
     *
     * @param pageWidth  是纸张去除边界留白后的可打印区域根据打印比例转换后的尺寸，不是纸张的原始尺寸
     * @param pageHeight
     */
    preparePageRange: function (pageWidth, pageHeight) {

        var headerHeight = 0;//this.getPageHeader().getHeight();
        var footerHeight = 0;//this.getPageFooter().getHeight();

        //对页眉及页脚的支持，在分页时，要扣去这两个高度，缺省情况下，它们高度为0
        pageHeight = pageHeight - headerHeight - footerHeight;

        this.clearPageRange();
        var /*ArrayList<PageSplitInfo> */splitColumn = [];
        var /*ArrayList<PageSplitInfo> */splitRow = []; // 保存的是一页的第一个行/列的号码.


        this.splitRowToPrint(pageHeight, splitRow);
        this.splitColumnToPrint(pageWidth, splitColumn);

        var cn = splitColumn.length;
        var rn = splitRow.length;
        if (cn * rn == 0) return this.getPageRange();
        var m, n;
        var RC = this.getRowCount();
        var CC = this.getColumnCount();


        for (m = 0; m < rn; m++)
            for (n = 0; n < cn; n++)
            {

                var row2, col2;
                if (m == rn - 1)
                {
                    row2 = RC;
                } else
                {

                    if (splitRow.get(m + 1).offset > 0)
                    {
                        row2 = splitRow.get(m + 1).index;
                    } else
                    {
                        row2 = splitRow.get(m + 1).index - 1;
                    }
                }

                if (n == cn - 1)
                {
                    col2 = CC;
                } else
                {
                    if (splitColumn.get(n + 1).offset > 0)
                    {
                        col2 = splitColumn.get(n + 1).index;
                    } else
                    {
                        col2 = splitColumn.get(n + 1).index - 1;
                    }
                }

                var range = new PrintRange(splitRow.get(m), splitColumn.get(n), row2, col2);
                this.pageRange.push(range);
            }

        return this.getPageRange();
    },


    /**
     * 列分页,按页宽，对列进行分页
     *
     *  TODO 存在但不常见的的问题：
     * 问题1 对列进行分页时，是按各列的宽度进行计算的，不考虑列合并的情况
     * 因为可能存在犬牙交错式的合并，所以不考虑列合并的情况。如果出现合并列被分到两页中，只能考虑重新设计表单，避免此情况发生
     *
     * 问题2：如果一列的宽度就超出了一页，那么不会把这列打印到两页中
     *
     * @param pageWidth
     * @param pArray
     */
    splitColumnToPrint: function (pageWidth, /*ArrayList<PageSplitInfo> */ pArray) {

        if (this.getRowCount() == 0) return;
        if (this.getColumnCount() == 0) return;

        var validWidth = pageWidth;


        var FixedColumnCount = this.CPM.getFixedColumnCount();
        var ColumnCount = this.getColumnCount();
        pArray.push(new PageSplitInfo(0, 0, 0));
        var FCW = this.CPM.getFixedColumnWidth();
        var widthSum = FCW;
        var isFirstPrintColumn = true;
        for (var col = FixedColumnCount; col < ColumnCount; col++)
        {
            var w = this.CPM.getColumnWidth(col);
            widthSum += w;
            if (widthSum > validWidth)// 如果整列的宽度加上超过了纸张的有效宽度
            {
                // 取得本页打印的起始点
                var begin = pArray.get(pArray.length - 1);

                // 如果col不是本页打印的第一列
                if (col != begin.index)
                {
                    // 计算留白
                    begin.blank = w - (widthSum - validWidth);

                    // 那么就把本行分到下一页中去打印
                    begin = new PageSplitInfo(col, 0, 0);
                    pArray.push(begin);
                    // 这样实际上又加了一页，begin指向的还是最新的
                }

                while (true)
                {
                    var subOffset = begin.offset; // 取得本页在行内的偏移坐标

                    // 除去固定列后，留下的有效宽度为 th
                    var th = validWidth - FCW;
                    if (th + subOffset < w) // 可供使用的宽度＋偏移量如果还比这个列的列宽小，那么要分页
                    {
                        // 这种情况下，当前的页是不需要留白的

                        // 新分页的开始坐标仍是第row行，但偏移量是现在的偏移量＋可用的高度
                        begin = new PageSplitInfo(col, th + subOffset, 0);
                        pArray.push(begin);
                    } else
                    // 如果第row行的最底边能在本页打印完，那么显然需要看下一行的情况
                    // 而此时的高度合计应该是固定行的高度＋本行在本页中需要打印的高度（即本行总高度－已经打印的部分）
                    {
                        widthSum = FCW + (w - subOffset);
                        break;
                    }
                }
            }
        }

    },

    /**
     * 行分页
     *
     * 不考虑行合并的情况
     * 不考虑一行高度超过一页的情况
     *
     * @param pageHeight
     * @param pArray
     */
    splitRowToPrint: function (pageHeight, /*ArrayList <PageSplitInfo> */pArray) {

        console.info('splitRowToPrint');
        if (this.getRowCount() == 0) return;
        if (this.getColumnCount() == 0) return;


        var FixedRowCount = this.RPM.getFixedRowCount();
        var RowCount = this.RPM.getRowCount();
        pArray.push(new PageSplitInfo(0, 0, 0));
        var FRH = this.RPM.getFixedRowHeight();//固定区域的高度

        var ignoreHeight = this.RPM.getRowY(this.ignoreBeforeRowWhenPrinting);

        FRH = FRH - ignoreHeight;//2019.12.17 打印忽略行之前的高度要去掉
        var HeightSum = FRH;

        var isFirstPrintRow = true;

        for (var row = FixedRowCount; row < RowCount; row++)
        {
            var h = this.RPM.getRowHeight(row);
            HeightSum += h;

            var mrdsn = this.RPM.getMultiRowDataSourceName(row);

            var maxDBRowCount = 0;
            var ds = null;
            if (mrdsn != '')
            {
                ds = this.Book.getDataSource(mrdsn).dataStore;
                maxDBRowCount = ds.rowCount;
            }

            if (HeightSum > pageHeight || maxDBRowCount > 1)// 如果整行的高度加上超过了纸张的有效高度
            {

                // 取得本页打印的起始点
                var begin = pArray.get(pArray.length - 1);

                // 如果这行就是单行:1 不绑定到数据源，2绑定到单行数据源，3 绑定到只有一行数据的多行数据源

                if (maxDBRowCount <= 1 && HeightSum > pageHeight)
                {

                    // 如果row不是本页打印的第一行
                    if (row != begin.index)
                    {
                        // 计算留白
                        begin.blank = h - (HeightSum - pageHeight);

                        // 那么就把本行分到下一页中去打印
                        begin = new PageSplitInfo(row, 0, 0);
                        pArray.push(begin);
                        // 这样实际上又加了一页，begin指向的还是最新的
                    }

                    while (true)
                    {
                        var subOffset = begin.offset; // 取得本页在行内的偏移坐标

                        // 除去固定行后，留下的有效高度为 th
                        var th = pageHeight - FRH;
                        if (th + subOffset < h) // 可拱使用的高度＋偏移量如果还比这个行的行高小，那么要分页
                        {
                            // 这种情况下，当前的页是不需要留白的

                            // 新分页的开始坐标仍是第row行，但偏移量是现在的偏移量＋可用的高度
                            begin = new PageSplitInfo(row, th + subOffset, 0);
                            pArray.push(begin);
                        } else
                        // 如果第row行的最底边能在本页打印完，那么显然需要看下一行的情况
                        // 而此时的高度合计应该是固定行的高度＋本行在本页中需要打印的高度（即本行总高度－已经打印的部分）
                        {
                            HeightSum = FRH + (h - subOffset);
                            break;
                        }
                    }
                } else
                {
                    // 到这里来，说明第row行是绑定到多行数据源的复合行，且有多行数据
                    // 为了简化处理，假设多行绑定且有多行数据的情况下，每一个子数据行必须能在一页中打印得下。
                    // 即假设 getDBRowHeight < validHeight - FRH;
                    // 不然就无法保证数据行的完整可见性，这种极端情况不处理，
                    // 如果出现，只能想办法避免。
                    var dbh = this.RPM.getOneDataRowHeightOfDataSource(mrdsn);// 得到单个数据行的高度

                    var gh = 0; // this.RPM.getGroupRowHeight(row);// 分类汇总行的高度

                    HeightSum -= h; // 先把高度回滚掉;

                    //把每行及分类汇总行的高度拉平到一个数组中


                    var rc = ds.getRowCount();
                    var heightList = [];


                    //为什么这么整，每个子行高度不都是一样的吗？
                    //不排除以后，各个明细行高可以不一样，这里先预留
                    // 只要修改 heightList.push(各子行的高度); 后面的分页计算就不用改了

                    for (var di = 0; di < rc; di++)
                    {
                        heightList.push(dbh);

                    }

                    var listCount = heightList.length;

                    var offset = 0;
                    for (var k = 0; k < listCount; k++)
                    {
                        var dbr = heightList.get(k);
                        HeightSum += dbr;

                        if (HeightSum > pageHeight)
                        {// 如果加上一个子数据行高度后超过了打印区域，那么新分一页
                            begin.blank = dbr - (HeightSum - pageHeight); // 计算留白
                            // 新的分页点
                            begin = new PageSplitInfo(row, offset, 0);
                            pArray.push(begin);
                            // 新分一页后，注意，要加上一个子数据行高度。因为第K个子行需要打印在下一页中，它的高度
                            // 需要算在下一页中。
                            HeightSum = FRH + dbr;
                        }
                        offset += dbr;
                    }
                    //2020.10.10 到了这里， 整个多行明细都已经分页完成了，那么 row不能是再从 row+1继续下面的循环，而是要跳过整个
                    // mrdsn 占据的行数
                    let lastSameBindRow = this.RPM.findLastRowWhichBindToMultiRowDataSource(mrdsn);

                    row= lastSameBindRow ;

                }
            }

        }

    },


    getPageRange: function () {
        return this.pageRange;
    },

    //因为设置时，需要用mm , 而在分页计算时，要用pt
    getPageConfig: function (unit) {
        if (unit == undefined) unit = "pt";

        if (this.pageConfig == undefined) this.setPageConfig();
        if (unit == 'mm') return this.pageConfig;
        var ret = {
            pageWidth: UnitConv.mm2pt(this.pageConfig.pageWidth),
            pageHeight: UnitConv.mm2pt(this.pageConfig.pageHeight),
            leftMargin: UnitConv.mm2pt(this.pageConfig.leftMargin),
            topMargin: UnitConv.mm2pt(this.pageConfig.topMargin),
            rightMargin: UnitConv.mm2pt(this.pageConfig.rightMargin),
            bottomMargin: UnitConv.mm2pt(this.pageConfig.bottomMargin),
            paper: this.pageConfig.paper,

            printDir: this.pageConfig.printDir,
            pageHeaderConfig: this.pageConfig.pageHeaderConfig, //不需要进行转换，就用px
            pageFooterConfig: this.pageConfig.pageFooterConfig

        };
        return ret;
    },

    /**
     * 设置页面大小及边距 ，单位是 mm
     * @param pageWidth
     * @param pageHeight
     * @param leftMargin
     * @param topMargin
     * @param rightMargin
     * @param bottomMargin
     */
    setPageConfig: function (paper, pageWidth, pageHeight, leftMargin, topMargin, rightMargin, bottomMargin,
                             printDir, pageHeaderConfig, pageFooterConfig, printScale) {

        if (pageWidth === undefined) pageWidth = 210;
        if (pageHeight === undefined) pageHeight = 297;
        if (leftMargin === undefined) leftMargin = 20;
        if (topMargin === undefined) topMargin = 20;
        if (rightMargin === undefined) rightMargin = 20;
        if (bottomMargin === undefined) bottomMargin = 20;
        if (paper == undefined) paper = "210*297";
        if (printScale == undefined) printScale = 0;

        if (printDir == undefined) printDir = 1; //纵向

        if (pageHeaderConfig == undefined) pageHeaderConfig = {define: "", height: 20};
        if (pageFooterConfig == undefined)
        {
            pageFooterConfig = {
                define: '="第 "+pageIndex()+" 页，共 "+pageCount()+" 页"',
                height: 20
            };
        }


        //把页眉，页脚的设置放入到页眉页脚中
        if (pageHeaderConfig != null)
        {
            this.pageHeader.Define = pageHeaderConfig.define;
            this.pageHeader.height = pageHeaderConfig.height; //扩展了一个height 属性， 因为这个虚拟单元格不在Sheet网格中
        }
        if (pageFooterConfig != null)
        {
            this.pageFooter.Define = pageFooterConfig.define;
            this.pageFooter.height = pageFooterConfig.height; //扩展了一个height 属性， 因为这个虚拟单元格不在Sheet网格中
        }


        this.setPrintScale(printScale); //为什么要要单独设置一下， 因为之前打印比例就是单独设置的，只是后来把打印比例也放到pageConfig中去了

        this.pageConfig = {
            pageWidth: pageWidth,
            pageHeight: pageHeight,
            leftMargin: leftMargin,
            topMargin: topMargin,
            rightMargin: rightMargin,
            bottomMargin: bottomMargin,
            paper: paper,

            printDir: printDir,
            pageHeaderConfig: pageHeaderConfig,
            pageFooterConfig: pageFooterConfig,
            printScale: printScale
        };

    },


    /**
     *
     *
     * @param printCmd  打印命令，包含打印设备名称，页面设置等信息
     * @param websocket
     * @returns {Promise}
     */
    exportAsPrintCommand: function (printCmd, websocket) {

        //导出前，需要确保如果单元格是以图片方式显示的，图片必须先准备好

        var promiseAll = [];

        //检查所有单元格，看看它们有没有需要校验图片是否合法的
        var RC = this.rowCount;
        var CC = this.columnCount;
        for (var row = 0; row < RC; row++)
        {
            for (var col = 0; col < CC; col++)
            {
                var cell = this.$cells(row, col);
                if (cell == null) continue;
                var viewAs = cell.viewAs;
                if (!['html', 'image'].contains(viewAs)) continue;
                var promise = null;
                if (viewAs == 'html') promise = cell.promiseInnerImageValid();
                if (viewAs == 'image') promise = cell.imagePromise();
                if (promise != null)
                {
                    var v = cell.getValue();
                    if (v == null) continue;
                    if (v == '') continue;
                    //TODO 对于明细行中有图片，理论上，要把每个都确认已经加载
                    // 但是对于常规应用来说，打印几页时，相应图片基本上都差不多了
                    // 对于很多页的，那么也不太会加上很多图片
                    // 所以下面的 continue 不是太大问题。
                    //如果出现明细行中图片无法输出，那么通常就可能是这个continue 忽略了它们的加载
                    if (Util.isArray(v)) continue;

                    var str = v;
                    //得到单元格的坐标，true表示是打印
                    var rc = this.View.getShowRectangleOfCell(true, row, col);
                    promiseAll.push(promise(str, rc));
                }

            }
        }

        let that = this;

        return new Promise(function (resolve, reject) {

            //如果有图片需要确保有效的，那么，必须是它们全部有效后，才继续
            if (promiseAll.length > 0)
            {
                Promise.all(promiseAll)
                    .then(function (result) {
                        that.$exportAsPrintCommand(printCmd, websocket);
                        resolve({});
                    });

            } else
            {
                that.$exportAsPrintCommand(printCmd, websocket);
                resolve({});
            }

        });


    },

    $exportAsPrintCommand: async function (printCmd, websocket) {
        var pageConfig = this.getPageConfig('pt'); //确保纸张设置不为null
        var prc = this.getPrintAreaSize();  //已经按打印方向进行了转换
        var printScale = this.getPrintScaleForPrint();

        //将纸张可打印大小按打印比例反向进行缩放，用于分页计算
        //比如如果是缩小打印，相当于把纸张放大了。
        var pageWidth = Math.floor(prc.width / printScale * 100);
        var pageHeight = Math.floor(prc.height / printScale * 100);

        //   分页
        var pageRange; //分页信息
        //   分页,注意不需要按打印方向做互换，因为 prc 已经处理过了
        pageRange = this.preparePageRange(pageWidth, pageHeight);


        var pn = pageRange.length;

        //到这里已经计算出需要的页数，就可以与打印助手通信了，让它初始化界面

        var printTaskID = Tools.newGUID(); //新建一个预览窗口，给它一个标记，后期好继续接收消息

        printCmd.id = printTaskID;
        printCmd.pageConfig = pageConfig;
        printCmd.pageCount = pn; //页数
        websocket.send(JSON.stringify(printCmd));//打印助手收到此消息，就会新创建一个打印预览任务
        await sleep(100);
        //到了这里，打印助手应该已经打开一个新的窗口（如果是预览的话）
        //但是没有内容输出，下面每整理一页的内容，就发送一页的内容到打印助手，而不是全部整理好之后再一次性发送给
        //打印助手，因为内存会溢出。


        this.setCurrentPrintingPageCount(pn);
        console.info(" total page: " + pn);
        for (let i = 0; i < pn; i++)
        {
            if (this.printTaskIsCanceled) //打印被终止
            {
                delete this['printTaskIsCanceled']; //删除这个标记
                break;
            }
            //输出到逻辑绘图接口
            //休息一下，让下面的 showInfoPane 有机会立即见效
            await sleep(1);
            showInfoPane("*正在输出第 " + (i + 1) + "/ " + pn + " 页...");
            var g = new GraphicsLogic();
            this.exportOnePageAsPrintCommand(
                g,
                pageRange.get(i),
                i,
                pageConfig.leftMargin,
                pageConfig.topMargin,
                printScale);

            var printOnePageCmd = {action: 'onePage', id: printTaskID, pageIndex: i, pageCommand: g.command};

            //[标记20191224]

            //上面的高强度计算后，休息一下，释放一下CPU的占用
            // 时间太短了不行， 大长了体验不好
            if (i < 2)
            {
                //前两页，等久一点，确保它能先输出
                await sleep(100);
            } else if (i < 10)
            {
                await sleep(50);
            } else
            {
                await sleep(10);
            }


            /*
            WebSocket 中的send( ) 方法是异步的：提供的数据会在客户端排队，而函数则立即返回。在传输大文件时，
            不要因为回调已经执行，就错误地以为数据已经发送出去了，数据很可能还在排队。要监控在浏览器中排队的数据量，
            可以查询套接字的bufferedAmount 属性
             */
            websocket.send(JSON.stringify(printOnePageCmd));//打印助手收到此消息，就会新创建一个打印预览任务


            console.info("websocket.bufferedAmount=" + websocket.bufferedAmount);


        }


    },


    /**
     * 在指定的图形设备上，画出一页的内容
     * @param graphics
     * @param range
     * @param pageIndex
     * @param leftMargin
     * @param topMargin
     * @param scale
     */
    exportOnePageAsPrintCommand: function (graphics, /*PrintRange*/range, pageIndex, leftMargin, topMargin, scale) {


        this.setCurrentPrintingPage(pageIndex);

        var x = Math.max(0, this.CPM.getColumnX(range.startCol) - this.CPM.getFixedColumnWidth());
        var y = Math.max(0, this.RPM.getRowY(range.startRow) + range.rowOffset - this.RPM.getFixedRowHeight());
        //设置偏移量
        this.setXPrintOffset(x);
        this.setYPrintOffset(y);

        graphics.translate(leftMargin, topMargin);
        var f = scale * 1.0 / 100;
        graphics.scale(f, f);

        var headerHeight = this.pageHeader.height || this.pageHeader.getPropertyValue("font-size", 12) / 3 * 4;
        var footerHeight = this.pageFooter.height || this.pageFooter.getPropertyValue("font-size", 12) / 3 * 4;

        var client = this.getPrintAreaSize();


        client.width = Math.floor(client.width * 1.0 / scale * 100);
        client.height = Math.floor(client.height * 1.0 / scale * 100);


        //主体的高度不要减去页眉和页脚，因为在分页时，是不去掉的，
        //打印时，保持一至， 页眉页脚向上及向下挤一下


        //页眉
        if (headerHeight > 0)
        {
            graphics.translate(0, -headerHeight);
            var headerRect = new Rectangle(0, 0, client.width, headerHeight);

            // 裁剪掉无法完整打印的部分
            graphics.clip(headerRect);
            graphics.setInvalidateRect(headerRect);

            this.View.onPrintPageHeader(graphics, headerRect);

            graphics.translate(0, headerHeight);
        }

        this.View.onPrint(graphics, client, range, pageIndex);

        //页脚
        if (footerHeight > 0)
        {

            //再往下移到脚注区
            graphics.translate(0, client.height);

            var footerRect = new Rectangle(0, 0, client.width, footerHeight);

            // 裁剪掉无法完整打印的部分

            graphics.clip(footerRect);
            graphics.setInvalidateRect(footerRect);
            this.View.onPrintPageFooter(graphics, footerRect);


        }


    },

    /**
     * 得到打印区域 ， 纸张大小减去边距 ，注意要根据打印方向的不同，做不同的处理
     * @returns {Rectangle}
     */
    getPrintAreaSize: function () {
        var page = this.getPageConfig();

        if (page.printDir == 1)
        {
            return new Rectangle(0, 0,
                page.pageWidth - page.leftMargin - page.rightMargin,
                page.pageHeight - page.topMargin - page.bottomMargin
            );
        } else
        {

            return new Rectangle(0, 0,
                page.pageHeight - page.leftMargin - page.rightMargin,
                page.pageWidth - page.topMargin - page.bottomMargin
            );
        }

    },

    loadPrintConfig: function (printConfigSaveToKey) {
        //触发事件，允许从外部加载打印的设置

        var t = this.Book.EM.fire("loadPrintConfig", [this, printConfigSaveToKey], "");
        if (t == null) t = "{}";
        if (t == '') t = "{}";


        t = JSON.parse(t);
        //即使t 不是有效设置，也要调用下面的setPageConfig以加载默认的设置，不然会沿用上一次的设置
        this.setPageConfig(t.paper,
            t.pageWidth, t.pageHeight, t.leftMargin, t.topMargin, t.rightMargin, t.bottomMargin,
            t.printDir, t.pageHeaderConfig, t.pageFooterConfig, t.printScale );

        if( t.printerDeviceName)  this.printerDeviceName=t.printerDeviceName;

    },


    /**
     * 显示打印机选择，页面设置对话框
     * 注意，打印机设备的清单是以参数形式传进来的。打印设备清单，是通过打印助手来获取的，
     * 在网页中，是无法获取的。
     * @param deviceList  打印设备清单
     * @param previewBeforePrint  是打印还是预览
     * @param printCallback
     */
    showPageSetupDialog: function (deviceList, previewBeforePrint, printCallback, printConfigSaveToKey) {

        // 从外部加载打印的设置
        this.loadPrintConfig(printConfigSaveToKey);

        var page = this.getPageConfig("mm"); //设置时用毫米

        //打印对话框的内容是纯手写的，请谨慎修改

        var html = ` <style>
                            .roundInput { border-radius: 5px; border:1px solid dodgerblue; background-color: white;}
                             .printSettingPanel TD {padding:6px;}
                             
                             .tab-panel-print { height:260px; background-color: #f7f7f7; 
                                                padding:10px; border-bottom-left-radius: 5px;
                             border-bottom-right-radius: 5px;}
                             .H30 td { height:50px;}
                             
                             .tabbed_top.skin-asbestos ul li.active,
                             .tabbed_top.skin-asbestos ul li.active:before,
                             .tabbed_top.skin-asbestos ul li.active:after {
                                            background-color: #f7f7f7;
                                            color:black;
                                }
                              </style> 
                            `;

        html = html +
            `                                                     
                   
          <div id=printTabContainer class='  tabbed_top  round  skin-asbestos' style="padding-top:1px;
             background-color:#dedede; border-top-left-radius: 5px; border-top-right-radius: 5px;
             -background:-webkit-gradient(linear, 0% 0%, 0% 100%,from(#ffffff), to(#efefef));">
                <ul>
                    <li id="panel_option">&nbsp;&nbsp;选项</li>
                    <li id="panel_footer">&nbsp;&nbsp;页脚</li>
                    <li id="panel_header">&nbsp;&nbsp;页眉</li>
                    <li id="panel_page">&nbsp;&nbsp;页面设置</li>
                    <li id="panel_printer" class="active">&nbsp;&nbsp;打印设置</li>
                </ul>
            </div>
            
                              
    <div  id= "print_container_panel_printer"   class="tab-panel-print  ">

    <table border="0" class="printSettingPanel H30">

        <tr>
            <td>
                <div style="width:70px; text-align:right;">打&nbsp;&nbsp;印&nbsp;&nbsp;机：</div>
            </td>
            <td colspan="4">
                <select id="deviceName" style="width:470px;" class="roundInput">   
                    ${deviceList}
                </select></td>
        </tr>
 

        <tr>
            <td align="right">打印范围：</td>
            <td colspan="2" style="padding-top:4px;">
                从第<input type="tel" style="text-align:center" id="fromPage" value="1"
                         class="roundInput" size="5">
                页到第<input type="tel" style="text-align:center" id="toPage" value="9999"
                          class="roundInput" size="5"> 页; &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;打印：
                          <label><input type="radio" class="pagerange" name="pageRange" value="1">奇数页</label>&nbsp;&nbsp;&nbsp;
                          <label><input type="radio" class="pagerange" name="pageRange" value="2">偶数页</label>&nbsp;&nbsp;&nbsp;
                          <label><input type="radio" class="pagerange" name="pageRange" value="3" checked>所有页</label>
                           
            </td>
        </tr>
     
        <tr>
            <td align="right">打印份数：</td>
            <td colspan="2" style="padding-top:4px;">
                <input type="tel" id="copy" style="text-align:center" value="1"
                       class="roundInput" size="5"> 份
            </td>
        </tr>

        <tr>
            <td align="right">打印预览：</td>
            <td colspan="2" style="padding-top:4px;">
            
                <label>
                    <input type="checkbox"  id="previewBeforePrint"  style="font-size:16px;"  > 打印前先预览
                </label>
             
            </td>
        </tr>
    </table>


    </div>
    `;

        html = html + `    

     <div  id= "print_container_panel_page"  class="tab-panel-print  " style=" display:none; ">
   
    <table border="0" class="printSettingPanel  ">

        <tr>
            <td align="right"><div style="width:70px;">纸张大小：</div></td>
            <td colspan="2">
                <select id="pageName" style="width:250px;" class="roundInput">
                    <option value="210*297">A4</option>
                    <option value="175*250">B5</option>
                    <option value="*">自定义</option>


                </select></td>
            <td style="width:30px;"></td>
            <td rowspan=8 style="  padding:5px;   border:1px solid lightgrey ; background-color:#efefef;">

                <div style="text-align:center; color:gray;">页&nbsp;&nbsp;边&nbsp;&nbsp;距：(单位：毫米)<br><br></div>
                <table border="0" style="height:190px;">
                    <tr>
                        <td></td>
                        <td>上:<input type="tel" id="pageMarginTop" size="3"></td>
                        <td></td>
                    </tr>
                    <tr>
                        <td>左:<input type="tel" id="pageMarginLeft" size="3"></td>
                        <td>
                            <div id="pageDemo"
                                 style="  width:60px; height:80px; border:1px solid gray;
                                                     background-color: lightyellow;"></div>
                        </td>
                        <td>右:<input type="tel" id="pageMarginRight" size="3"></td>
                    </tr>
                    <tr>
                        <td></td>
                        <td>下:<input type="tel" id="pageMarginBottom" size="3"></td>
                        <td></td>
                    </tr>
                </table>


            </td>
        </tr>

        <tr>
            <td align="right">纸张宽度：</td>
            <td colspan="2" style="padding-top:4px;">
                <input type="tel" id="pageWidth" class="roundInput" size="5"> mm;
            </td>


        </tr>

            <tr>
            <td align="right">纸张高度：</td>
            <td colspan="2" style="padding-top:4px;">
                <input type="tel" id="pageHeight" class="roundInput" size="5"> mm
            </td>


        </tr>
        
        <tr>
            <td align="right">打印比例：</td>
            <td><select id="select_printScale" style="width:150px;">
                <option value="0">自动缩放以适应页宽</option>
                <option value="25">缩小到25%打印</option>
                <option value="50">缩小到50%打印</option>
                <option value="67">缩小到67%打印</option>
                <option value="75">缩小到75%打印</option>
                <option value="100">原始尺寸打印</option>
                <option value="125">放大到125%印</option>
                <option value="150">放大到150%印</option>
                <option value="200">放大到200%印</option>
                <option value="300">放大到300%印</option>
                <option value="400">放大到400%印</option>
                <option value="999">自定义缩放比例</option>

            </select></td>
            <td><input type="tel" class="roundInput" size=5 id="printScale">%</td>
        </tr>
        <tr>
            <td align="right">打印方向：</td>
            <td colspan="2">
                <label>
                    <input type="radio" id="print_dir_1" name="print_dir"> 纵向&nbsp;&nbsp;&nbsp;&nbsp;
                    <span style="color:green">▋</span></label>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;
                
            </td>
        </tr>
 
        <tr>
            <td align="right"> </td>
            <td colspan="2">
                <label>
                <label><input type="radio" id="print_dir_0" name="print_dir"> 横向&nbsp;&nbsp;&nbsp;&nbsp;
                    <span style="color:green">▅</span></label>
            </td>
        </tr>
        
       
    </table>
</div>                                    
                
      `;

        html = html + `
                
                
                
     <div  id= "print_container_panel_header"  class="tab-panel-print  " style=" display:none; ">
   
        <table border="0" class="printSettingPanel">

        <tr>
            <td align="right"><div style="width:70px;">页眉定义：</div></td>
            <td>
                <input type="text" id="headerDefine" style="width:470px;" class="roundInput"  >
              </td>
           </tr>
        <tr><td></td>
        <td>
          <ul>
                <li>页眉可以是立即值，也可以是公式，公式在打印时解析</li>
                <li>以=开头的表示用公式来定义页眉</li>
                
                <ul></td>
        </tr>
        
        
         <tr>
            <td align="right">页眉高度：</td>
            <td >
                <input type="tel" id="headerHeight"  size="5" class="roundInput"  > px 
            </td>
        </tr>
        
        </table>
        </div>
        
        <!--------------------------------------->
        
        
        
        <div  id= "print_container_panel_footer"  class="tab-panel-print  " style=" display:none; ">
   
        <table border="0" class="printSettingPanel">

        <tr>
            <td align="right"><div style="width:70px;">页脚定义：</div></td>
            
            <td>
                <input type="text" id="footerDefine" style="width:470px;" class="roundInput"  >
                </td>
           </tr>
        <tr><td></td>
        <td>   
                 <ul>
                <li>页脚可以是立即值，也可以是公式，公式在打印时解析</li>
                <li>以=开头的表示用公式来定义页脚</li>
                
                </ul>
            </td>
        </tr>
        
         <tr>
            <td align="right">页脚高度：</td>
            <td >
                <input type="tel" id="footerHeight"  size="5" class="roundInput"  > px
            </td>
        </tr>
        
        </table>
        </div>
        
        
           <div  id= "print_container_panel_option"  class="tab-panel-print  " style=" display:none; ">
   
        <table border="0" class="printSettingPanel">

          <tr>
            <td align="right">背景颜色：</td>
            <td colspan="2">
                <select id="bgcolorPrint" style="width:250px;" class="roundInput" disabled>
                    <option value="0">忽略所有背景色</option>
                    <option value="1">打印背景色</option>
                    <option value="2" selected >二值化，浅色忽略，深色打印</option>
                </select></td>
          </tr>
                
        
        </table>
        </div>
        
        
                
                `;


        function resetPageDemoSize()
        {
            if ($('#print_dir_1')[0].checked)
            {
                $('#pageDemo').css({'width': 60, 'height': 80});
            } else
            {
                $('#pageDemo').css({'width': 80, 'height': 50});
            }
        }

        let that = this;

        $.confirm({
            columnClass: 'small',
            boxWidth: '650px',
            useBootstrap: false,
            title: '打印',
            content: html,
            onOpen: function () {


                function panel_print_tab(tab)
                {
                    $(".tab-panel-print").each(function (i, obj) {
                        $(obj).css('display', 'none');
                    });

                    $("#print_container_" + tab.id).css('display', '');

                }

                function initTabs(container, callback)
                {
                    var tabs = container.querySelectorAll('li');

                    for (var i = 0, len = tabs.length; i < len; i++)
                    {

                        tabs[i].addEventListener('click', function () {
                            if (this.classList.contains('active'))
                            {
                                return;
                            }
                            var parent = this.parentNode, innerTabs = parent.querySelectorAll('li');
                            for (var index = 0, iLen = innerTabs.length; index < iLen; index++)
                            {

                                innerTabs[index].classList.remove('active');
                            }

                            this.classList.add('active');
                            callback(this);
                        });


                    }

                }

                initTabs($("#printTabContainer")[0], panel_print_tab);


                if (previewBeforePrint) $('#previewBeforePrint')[0].checked = true;

                if (that.printerDeviceName != '')
                {
                    $('#deviceName').val(that.printerDeviceName); //打印机名称
                }

                $('#pageName').val(page.paper);//纸张名称
                $('#pageWidth').val(page.pageWidth);
                $('#pageHeight').val(page.pageHeight);
                $('#pageMarginTop').val(page.topMargin);
                $('#pageMarginBottom').val(page.bottomMargin);
                $('#pageMarginLeft').val(page.leftMargin);
                $('#pageMarginRight').val(page.rightMargin);
                if (![0, 25, 50, 67, 75, 100, 125, 150, 200, 300, 400].contains(that.m_printScale))
                {
                    $('#select_printScale').val(999);
                    $('#printScale').val(that.m_printScale);
                } else
                {
                    $('#select_printScale').val(that.m_printScale);

                }

                $('#printScale').val(that.m_printScale);

                if (page.printDir == 1)
                {
                    $('#print_dir_1')[0].checked = true;
                } else
                {
                    $('#print_dir_0')[0].checked = true;
                }

                resetPageDemoSize();

                $('#print_dir_1').on("click", function () {
                    resetPageDemoSize();
                });
                $('#print_dir_0').on("click", function () {
                    resetPageDemoSize();
                });

                //页眉

                $('#headerDefine').val(that.pageHeader.define);
                $('#headerHeight').val(that.pageHeader.height || 20);
                //页眉

                $('#footerDefine').val(that.pageFooter.define);
                $('#footerHeight').val(that.pageFooter.height || 20);


                //事件注册
                $('#pageName').on("change", function (e) {
                    var v = $('#pageName').val();
                    if (v == '*')
                    {
                        $('#pageWidth')[0].disabled = false;
                        $('#pageHeight')[0].disabled = false;
                        $('#pageWidth').css("background-color", "#ffffff");
                        $('#pageHeight').css("background-color", "#ffffff");

                    } else
                    {
                        $('#pageWidth')[0].disabled = true;
                        $('#pageHeight')[0].disabled = true;
                        $('#pageWidth').css("background-color", "#efefef");
                        $('#pageHeight').css("background-color", "#efefef");


                        var v2 = v.split('\*');
                        $('#pageWidth').val(v2[0]);
                        $('#pageHeight').val(v2[1]);

                    }
                });


                //缩放比例
                $('#select_printScale').on("change", function (e) {
                    var v = $('#select_printScale').val();
                    if (v == '999')
                    {
                        $('#printScale')[0].disabled = false;
                        $('#printScale').css("background-color", "#ffffff");
                    } else
                    {
                        $('#printScale')[0].disabled = true;
                        $('#printScale').css("background-color", "#efefef");
                        $('#printScale').val(v);
                        ;
                    }
                });

                $('#pageName').trigger("change");
                $('#select_printScale').trigger("change");


            },
            buttons: {
                confirm: {
                    text: "&nbsp;&nbsp;&nbsp;确&nbsp;&nbsp;&nbsp;定&nbsp;&nbsp;&nbsp;",
                    btnClass: "btn-success",
                    action: function () {

                        var paper = $('#pageName').val();
                        var pageWidth = $('#pageWidth').val();
                        var pageHeight = $('#pageHeight').val();

                        var topMargin = $('#pageMarginTop').val();
                        var bottomMargin = $('#pageMarginBottom').val();
                        var leftMargin = $('#pageMarginLeft').val();
                        var rightMargin = $('#pageMarginRight').val();

                        var printScale = $('#printScale').val();
                        var printDir = $('#print_dir_1')[0].checked ? 1 : 0;

                        var pageHeaderConfig = {
                            define: $('#headerDefine').val(),
                            height: $('#headerHeight').val()
                        };
                        var pageFooterConfig = {
                            define: $('#footerDefine').val(),
                            height: $('#footerHeight').val()
                        };


                        that.setPrintScale(printScale);
                        that.setPageConfig(paper, pageWidth, pageHeight,
                            leftMargin, topMargin, rightMargin, bottomMargin,
                            printDir, pageHeaderConfig, pageFooterConfig ,printScale );

                        //保存设置
                        var printConfig = that.getPageConfig("mm");
                        printConfig.printerDeviceName =  $('#deviceName').val(); //打印机名称也保存一下
                        printConfig.printScale = printScale;
                        that.Book.EM.fire("savePrintConfig", [that, printConfigSaveToKey, JSON.stringify(printConfig)]);

                        //设置完成后的回调

                        if (printCallback)
                        {
                            //页面设置与打印选项是不同的设置 ， 打印选项是不需要保存的

                            that.printerDeviceName = $('#deviceName').val();  //选择的打印机名称 ，需要保存一下
                            //因为可能接下来打印，都使用它

                            var printOption = {
                                deviceName: $('#deviceName').val(),
                                fromPage: $('#fromPage').val(),
                                toPage: $('#toPage').val(),
                                sheetName: that.name,
                                pageRange: $('.pagerange').filter(':checked').val(),
                                copy: $('#copy').val(),

                                previewBeforePrint: $('#previewBeforePrint')[0].checked
                            };

                            printCallback(printOption, printConfig);
                        }

                    }
                }
                ,
                cancel: {
                    text: "&nbsp;取&nbsp;&nbsp;消&nbsp;",
                    btnClass: "btn-warning",
                    action: function () {
                    }
                }
            }
        });


    }
    ,

    /**
     * 通过打印助手来打印
     * @param url
     * @param needShowPageSetupDialog  是否显示打印设置对话框，本参数暂时未生效
     * @param previewBeforePrint 打印前是否预览 ， 这个做为设置对话框打开时，是否预览的预设值
     * @param setupCallback [可选] 如果显示打印设置对话框，那么点确认后，执行的回调
     * @param printConfigSaveToKey [可选]打印设置保存的key，如果不提供直接使用本Sheet的GUID
     * @param forceUseOption [可选] 强制使用本选项进行打印，仅在 needShowPageSetupDialog=false时有用
     * @param printOverCallback [可选] 打印完成后的回调
     * @param printAgentNotFoundCallback [可选] 打印助手无法连接时的回调
     */
    printByAgent: function (url, needShowPageSetupDialog, previewBeforePrint, setupCallback,
                            printConfigSaveToKey, forceUseOption, printOverCallback, printAgentNotFoundCallback) {

        var sheet = this;

        if(! sheet.isPrintable())
        {
            toastr.error("抱歉，禁止打印");
            return;
        }


        let canPrint = this.Book.EM.fire("canSheetPrint", [this], true);
        if( canPrint==undefined) canPrint=true;
        if( !canPrint) return ;

        console.info( "打印页地址："+window.location.href);
        this.Book.EM.fire("beforeSheetPrint", [this]);

        if (printConfigSaveToKey == undefined) printConfigSaveToKey = sheet.guid;

        this.printAgentURL = url;

        showInfoPane("*连接打印助手...");

        var websocket;
        try
        {
            websocket = new WebSocket(url);
        } catch (err)
        {
            hideInfoPane();
            if (printAgentNotFoundCallback == undefined)
            {
                $.alert("无法连接打印助手，请检查打印助手是否已启动");
            } else
            {
                printAgentNotFoundCallback();
            }
            return
        }

        hideInfoPane();
        websocket.binaryType = 'blob';
        websocket.onopen = function (ev) {
            //先获取打印机设备清单

            var cmd = {action: 'getPrintServiceNameList'};
            websocket.send(JSON.stringify(cmd));

        };
        websocket.onerror = function (ev) {
            $.alert("无法连接打印助手，请检查打印助手是否已启动");
        };
        websocket.onclose = function (ev) {
            console.log("断开与打印助手的连接");
        };
        websocket.onmessage = function (ev) {
            console.dir(ev);
            var cmd = ev.data;
            cmd = JSON.parse(cmd);
            if (cmd.action == 'getPrintServiceNameList')
            {
                var serviceList = cmd.data;
                var options = [];
                for (var i = 0; i < serviceList.length; i++)
                {
                    options.push(`<option value='${serviceList[i]}'>${serviceList[i]}</option>`);
                }

                //注意，exportAsPrintCommand返回的是一个Promise对象，用于异步执行


                if (needShowPageSetupDialog)
                {
                    sheet.showPageSetupDialog(options.join(''), previewBeforePrint,

                        //2020.08.19 增加打印设置后的可以指定另外的回调操作
                        // 这个用处是在当前Sheet中可以做打印设置，但是并不是为自已做，而是为另外的Sheet做打印设置
                        //当不指定时，则使用默认的输出打印操作
                        setupCallback == undefined ?
                            function (printOption /*打印机，份数等*/,
                                      printConfig/*边距，缩放等*/) {
                                showInfoPane("*正在准备打印...");

                                let action = printOption.previewBeforePrint ? "preview" : "print";

                                var printCmd = {
                                    name: sheet.name,
                                    action: action,
                                    printOption: printOption
                                };

                                sheet.exportAsPrintCommand(printCmd, websocket).then(function (data) {
                                    hideInfoPane();
                                    //到这里，要打印的内容输出到打印助手完成了

                                });
                            } : setupCallback,
                        //可能指定了打印设置需要保存到哪里
                        printConfigSaveToKey
                    );
                } else  //不需要显示打印设置 ，使用当前的页面设置及打印设备
                {
                    var printOption = forceUseOption;

                    if (printOption == undefined)
                    {
                        printOption = {
                            deviceName: that.printerDeviceName,
                            fromPage: 1,
                            toPage: 9999,
                            pageRange:3,
                            copy: 1,
                            sheetName: sheet.name,
                            previewBeforePrint: previewBeforePrint
                        };
                    }

                    showInfoPane("*正在准备打印...");
                    let action = previewBeforePrint ? "preview" : "print";
                    //2020.08.20如果强制提供打印设置，那么用它
                    if (forceUseOption != undefined)
                    {
                        action = forceUseOption.previewBeforePrint ? "preview" : "print";
                    }


                    var printCmd = {
                        name: sheet.name,
                        action: action,
                        printOption: printOption
                    };

                    sheet.exportAsPrintCommand(printCmd, websocket).then(function (data) {
                        hideInfoPane();

                    });

                }

            }

            if (cmd.action == 'printover')
            {
                websocket.close(); //打印助手已经完全接收了打印命令，那么连接可以断开了
                hideInfoPane();
                toastr.info("打印任务已发送至打印机。");
                //2020.08.20增加回调
                if (printOverCallback) printOverCallback();
                //不是很严谨，有可能并没有真正打印，最好的是在打印助手中点击打印后，再发送一个消息， 这里接收到确信打印了
                //再触发事件，但是这样就需要维持websocket连接，所以算了，简化处理好了
                sheet.Book.EM.fire("afterSheetPrint", [this], true);
            }

            if (cmd.action == 'printCancel')
            {
                sheet.printTaskIsCanceled = true; //
                websocket.close(); //打印助手已经完全接收了打印命令，那么连接可以断开了
                hideInfoPane();
                //toastr.error("打印已被中止。");
            }

        };


    }


};

export default ISheetPrint;
