package websocketRPC;

import org.json.JSONObject;
import util.FF;
import util.NamedThreadFactory;
import webApp.App;

import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;
import javax.websocket.*;
import java.io.IOException;
import java.net.URI;
import java.security.SecureRandom;
import java.security.cert.X509Certificate;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

import static org.apache.tomcat.websocket.Constants.SSL_CONTEXT_PROPERTY;


/**
 * 在 onOpen中，加上每60秒ping一次
 * 在 close中停止 ，实际上是在 WSRPC.removeClient中调用本对象的 shutdownPing()
 */
public class WSRPCClient extends Endpoint
{

    public ScheduledExecutorService timer = Executors.newScheduledThreadPool(1, new NamedThreadFactory( App.appName+"-WSRPCClient"));

    long lastActionTime = System.currentTimeMillis();

    AtomicLong seqNumGenerator = new AtomicLong(0);
    ConcurrentHashMap<Long, RPCContext> no2ctx = new ConcurrentHashMap<>();
    Session session;

    int messageCount=0;
    URI endpointURI = null;
    private StringBuffer msgBuffer = new StringBuffer();
    public long  clientNo ; //标识自已是第几号客户服务提供者

    public WSRPCClient(URI endpointURI , long clientNo)
    {

        this.endpointURI = endpointURI;
        this.clientNo= clientNo;
      //  FF.log("新建WSRPCClient，no="+clientNo+" url:"+endpointURI);


    }

    public void reset()
    {
        messageCount=0;
    }

    public void shutdownPingTask()
    {
        if (!timer.isShutdown()) timer.shutdown(); //ping 任务终止
    }

    @Override
    public void onClose(Session session, CloseReason closeReason)
    {


        this.session = null;
        FF.log(" websocket client [no:"+this.clientNo+"] id=" + session.getId() + " closed");
        WSRPC.removeClient(this);
    }

    @Override
    public void onError(Session session, Throwable throwable)
    {
        FF.log(" websocket client  [no:"+this.clientNo+"]  id=" + session.getId() + " error: " + throwable.getMessage());
        onClose(session, null);
    }

    @Override
    public void onOpen(Session session, EndpointConfig endpointConfig)
    {
        this.session = session;
        FF.log("websocket client  [no:"+this.clientNo+"]  id=" + session.getId());

        // 开启60秒一次的ping
        timer.scheduleWithFixedDelay(new Runnable()
        {
            @Override
            public void run()
            {
                //  if( needPing()) ping();
                try
                {
                    sendMessage("ping");
                } catch (Exception e)
                {

                }
            }
        }, 60, 60, TimeUnit.SECONDS);


        session.addMessageHandler(new MessageHandler.Partial<String>()
        {
            @Override
            public void onMessage(String message, boolean isLast)
            {
                msgBuffer.append(message);
                if (isLast)
                {

                    onWholeMessage(msgBuffer.toString());
                    msgBuffer = new StringBuffer();

                }
            }
        });

    }

    //如果最后一次调用，在60秒以内，那么不需要检查一下
    public boolean connectionNeedValidCheck()
    {
        long now = System.currentTimeMillis();

        if (now - lastActionTime > 60 * 1000) return true;
        return false;

    }


    public JSONObject connectionValidCheck()
    {

        JSONObject js;

        //整这么个特别的名称 ，是防止与调用的参数名冲突 [标记20190612-1]
        JSONObject param = new JSONObject().put("${rpcRouteRemoteAddr}$", App.IPADDRESS);//调用者的IP地址放进来
        JSONObject j = new JSONObject();
        j.put("className", "websocketRPC.ConnectionValidCheck");
        j.put("methodName", "connectionValidCheck");

        j.put("param", param);

        JSONObject ret = call(j, 300);

       // FF.log("WSRCClient  Ping " + ret.toString());
        return ret;

    }

    //@OnMessage
    public void onWholeMessage(String message)
    {
        {

            if (message.equals("pong")) return; //WSRPCServer.java中会返回 pong ,直接忽略

            // FF.log("Websocket  back message: "+ message);
            JSONObject js = new JSONObject(message);

             messageCount++;
            long no = js.getLong(WSRPC.KEY_WebsocketRPCSequenceNo, 0);
          //  FF.log("onWholeMessage [msgCount="+messageCount+"]  收到序号："+ clientNo+":"+ no+"调用的返回数据："+js.toString());
            RPCContext ctx = no2ctx.get(no);

            if (ctx == null)
            {
                FF.log(App.appName + "  序号为" + clientNo+":"+no + "的请求已经因为等待超时无放弃了等待，返回值无法返回给调用方");
            }
            else
            {
                no2ctx.remove(no);
                ctx.response = js;
                ctx.semp.release();//释放许可证
            }

        }
    }

    public void sendMessage(String message) throws IOException
    {

        session.getBasicRemote().sendText(message);

    }


    /**
     * 注意， 本函数 ，可能被调用者使用，也可能本自已的 ping使用，所以需要加上 synchronized处理
     *
     * @param js
     * @param timeOutSecond
     * @return
     */
    public JSONObject call(JSONObject js, int timeOutMilliSecond)
    {


        long startTime = System.currentTimeMillis();

        //申请一个唯一编号
        long no = this.getNextSequentNumber();
        RPCContext ctx = new RPCContext();

        //把编号与ctx对应起来
        no2ctx.put(no, ctx);

        js.put(WSRPC.KEY_WebsocketRPCSequenceNo, no);

       // FF.log("websocket  call  " + js.toString());

        boolean isTimeout=false;
        try
        {


            ctx.semp.acquire();//获取信号量，视为申请一个许可，申请后，许可证没有了，让标记1 处阻塞


            this.sendMessage(js.toString());

            //上面写数据后，服务器收到数据，然后反馈 ，这个过程是异步的，
            //因此，客户端此时必须阻塞起来，等服务器回馈。

            //服务器处理消息后，会把no标记也带上，返回回来，这样用它来追踪客户，通知客户消息已经回馈回来了
            // no2ctx [no]可以定位到客户的semp标记，回馈事件中，将它释放，于是下面的 tryAcquire得以解除阻塞并返回true


            //FF.log(new Date()+" 等待回应："+no);
            //标记1  ，阻塞，直到收回复的消息后，解除
            if (ctx.semp.tryAcquire(timeOutMilliSecond, TimeUnit.MILLISECONDS))
            {
                lastActionTime = System.currentTimeMillis(); // 记录下最后一次成功调用的时间
               // FF.log("序号:"+clientNo+":"+ no+" 正常返回："+ctx.response);
                return ctx.response;
            }

            isTimeout=true;
            FF.log( "websocket  call  "+ js.toString()+" 序号:"+ clientNo+":"+no +" 超时 ,耗时： " + ( System.currentTimeMillis() - startTime)+" ms \n"+
            "失败的调用是："+ js.toString() +"\n"+
             "移除： 序号:"+ clientNo+":"+no);
            return new JSONObject().put("success", false)
                    .put("timeout",true)  //2020.03.05 增加表明是调用超时
                    .put("message",     "调用超时:" +js.toString()+", timeout="+ timeOutMilliSecond+" 毫秒");



        } catch (Exception e)
        {
            return new JSONObject().put("success", false).put("message", e.getMessage());

        } finally
        {

            long d2 = System.currentTimeMillis();
            //   FF.log( "websocket  call  "+ js.toString()+"耗时：" + ( d2-d1)+" ms ");


            //本次RPC调用完成 ， 删除信号
            //FF.log(new Date()+"  移除："+no);
            no2ctx.remove(no);
        }
    }


    public long getNextSequentNumber()
    {
        return seqNumGenerator.getAndAdd(1);
    }


    //https://www.programcreek.com/java-api-examples/index.php?api=javax.websocket.ClientEndpointConfig
    public boolean start() throws DeploymentException, IOException
    {

        WebSocketContainer container = ContainerProvider.getWebSocketContainer();

        try
        {


            // Create a trust manager that does not validate certificate chains
            TrustManager[] trustAllCerts = new TrustManager[]{new X509TrustManager()
            {
                public X509Certificate[] getAcceptedIssuers()
                {
                    return null;
                }

                public void checkClientTrusted(X509Certificate[] certs, String authType)
                {
                }

                public void checkServerTrusted(X509Certificate[] certs, String authType)
                {
                }
            }};

            // Install the all-trusting trust manager

            SSLContext sc = SSLContext.getInstance("TLS");
            sc.init(null, trustAllCerts, new SecureRandom());


            ClientEndpointConfig config = ClientEndpointConfig.Builder.create().build();

            config.getUserProperties().put(SSL_CONTEXT_PROPERTY, sc);


            Session session = container.connectToServer(this, config, endpointURI);

           // FF.log(" websocket   MaxTextMessageBufferSize = " + session.getMaxTextMessageBufferSize());
           // FF.log(" websocket   getMaxIdleTimeout = " + session.getMaxIdleTimeout());


            return session.isOpen();


        } catch (Exception e)
        {
            FF.log(e);

            return false;
        }

    }


    public void close()
    {
        try
        {

            WSRPC.removeClient(this);

            this.session.close();
        } catch (Exception e)
        {

        }
    }

}
