package websocketRPC;

import app.User;
import com.netflix.appinfo.InstanceInfo;
import com.netflix.discovery.EurekaClient;
import config.CacheConfig;
import http.OkHttpUtil;
import org.apache.commons.pool2.ObjectPool;
import org.apache.commons.pool2.impl.GenericObjectPool;
import org.apache.commons.pool2.impl.GenericObjectPoolConfig;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.MDC;
import rpc.RPC;
import util.EmulateHttpServletRequest;
import util.FF;
import webApp.App;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.net.URI;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;


public class WSRPC extends RPC
{

    //按连接地址创建一个客户端池
    private static ConcurrentHashMap<String, ObjectPool<WSRPCClient>> rpcClientMap = new ConcurrentHashMap<String, ObjectPool<WSRPCClient>>();
    private static ConcurrentHashMap<Integer, WSRPCServer> rpcServerMap = new ConcurrentHashMap<Integer, WSRPCServer>();

    private static String HomeURL = "";
    private static EurekaClient eurekaClient = null; //用来查询其它服务
    private static String virtualHostName = ""; // 所在应用提供的eureka服务的名称
    private static String hostIPAddress = ""; //所在应用的地址

    public static String KEY_WebsocketRPCSequenceNo = "WebsocketRPCSequenceNo";

    public static String  KEY_ServerUnreachable ="serverUnreachable";

    public static EmulateHttpServletRequest EHR = new EmulateHttpServletRequest();

    public static void init(String homeURL, EurekaClient client, String vipName, String hostIP)
    {
        HomeURL = homeURL;
        eurekaClient = client;
        //当需要调用其它服务时，可能需要把自身的信息传递过去做权限控制
        virtualHostName = vipName;
        hostIPAddress = hostIP;

    }

    private static ArrayList<HashMap<String,String>> getAllRPCServerInfoForService(String serviceName) throws Exception
    {
        ArrayList<HashMap<String,String>> ret = new ArrayList<HashMap<String,String>>();
        String vipAddress = serviceName; //需要调用的服务的名称

        List<InstanceInfo> serverInfoList = eurekaClient.getInstancesByVipAddress(vipAddress, false); //非加密方式

        for (InstanceInfo nextServerInfo : serverInfoList)
        {
            //检查服务的状态是不是有效的
            String ip = nextServerInfo.getIPAddr();
            int port = nextServerInfo.getPort();
            //   String url = nextServerInfo.getHomePageUrl();
            //   url = "ws" + url.substring(4);

            Map map = nextServerInfo.getMetadata();

            //2020.03.06 改用内部局域网地址
            String local_ipaddress = (String) map.get("local_ipaddress");
            String local_port = (String) map.get("local_port");
            String contextPath = (String) map.get("contextPath");

            String serviceTarget = FF.getStringFromMap(map, "APP_SERVICE_SHARDING", "");

            //直接使用 ws协议，不需要管wss ，因为在内网地址中，都是以http协议发布的， 对外的地址，如果是https
            // 证书都是通过nginx 处理的
            String url = "ws://" + local_ipaddress + ":" + local_port+contextPath;  //contextPath 前面自带/，或直接就是空字符串，所以 url最后是不带/的
            HashMap<String,String> one= new HashMap<String,String>();
            one.put("url", url);
            one.put("sharding",serviceTarget);

            ret.add(one);
        }
        return ret;
    }


    /**
     * 得到一个可用的netth rpc 服务的地址
     * @param serviceName
     * @param sharding  服务隔离限制
     * @param loopUntilURL  如果在循环查找服务时，一直无法满足sharding，如果碰到这个服务，那么就不要再找了，就它吧
     * @return
     * @throws Exception
     */
    public static String getRPCServerInfoForService(String serviceName,String sharding ,String loopUntilURL ) throws Exception
    {
        //FF.log("getRPCServerInfoForService  " + serviceName+" sharding="+ sharding );

        try
        {
            String vipAddress = serviceName; //需要调用的服务的名称

            InstanceInfo nextServerInfo = null;

            nextServerInfo = eurekaClient.getNextServerFromEureka(vipAddress, false); //非加密方式
            Map map = nextServerInfo.getMetadata();
            //2020.03.06 改用内部局域网地址
            String local_ipaddress = (String) map.get("local_ipaddress");
            String local_port = (String) map.get("local_port");
            String contextPath = (String) map.get("contextPath");

            if( local_ipaddress==null)
            {
                FF.log("getRPCServerInfoForService  异常，无法获取 metaData ");
                String ip = nextServerInfo.getIPAddr();
                int port = nextServerInfo.getPort();

                local_ipaddress= ip;
                local_port=""+port;
                contextPath= "/"+serviceName;
                if( serviceName.equals("app"))  contextPath="";
            }

            //直接使用 ws协议，不需要管wss ，因为在内网地址中，都是以http协议发布的， 对外的地址，如果是https
            // 证书都是通过nginx 处理的
            String url = "ws://" + local_ipaddress + ":" + local_port + contextPath;  //contextPath 前面自带/，或直接就是空字符串，所以 url最后是不带/的

           // FF.log("  getRPCServerInfoForService   url="+ url);

            String serviceTarget = FF.getStringFromMap(map, "APP_SERVICE_SHARDING", "");

            //如果当前的服务就满足了服务隔离，那么返回
            if (serviceTarget.equalsIgnoreCase(sharding)) return url;
            //到了这里， 就是不满足服务隔离要求
            if( url.equals(loopUntilURL)) return url;

            //下面进入递归了。当第一次本函数执行时，  loopUntilURL =='' , 进入递归时， 把当前的 url 代入做为终止递归的条件。
            //如果是多次进入递归，那么loopUntilURL !=''  ，那么仍是以它做为终止递归的条件。
            // 就个逻辑就是：我第一次找到了一个服务地址，它不符合要求，然后一直往下找，找了一圈，可能仍找不到满足要求的，
            // 那就算了吧，就把第一次找到的地址返回回去吧。
            return  getRPCServerInfoForService(serviceName, sharding, loopUntilURL.isEmpty()?url:loopUntilURL);


        }catch(Exception e)
        {
            FF.log( "getRPCServerInfoForService 异常：" +FF.exceptionMessage(e));

        }

        return null;
    }


    public static void startServer()
    {
        //不需要做任何处理，tomcat启动时，自动启动 WSRPCServer
    }

    public static JSONObject dispatch(String serviceName, String className, String methodName, JSONObject param,
                                      HttpServletRequest req, HttpServletResponse res)
    {
        return dispatch(serviceName, className, methodName, param, req, res, false, 1200);
    }

    public static JSONObject dispatch(String serviceName, String className, String methodName, JSONObject param,
                                      HttpServletRequest req, HttpServletResponse res, boolean broadcast)
    {
        return dispatch(serviceName, className, methodName, param, req, res, broadcast, 1200);
    }

    public static JSONObject dispatch(String serviceName, String className, String methodName, JSONObject param,
                                      HttpServletRequest req, HttpServletResponse res, boolean broadcast, int timeoutSecond )
    {
        return dispatch(serviceName,className, methodName,param,req,res,broadcast,timeoutSecond,"");
    }



    public static JSONObject dispatch(String serviceName, String className, String methodName, JSONObject param,
                                      HttpServletRequest req, HttpServletResponse res, boolean broadcast, int timeoutSecond ,String sharding)
    {


            // 如果是前端调用后端 ，那么param中可能存在currentGUID
        //如果是后端调用后端 ，或脚本中远程调用，要能 不存在currentGUID
        if( param.getMap().containsKey("currentGUID" ) )
        {
            MDC.put("currentGUID",  param.getString("currentGUID",""));
        }

        if( req==null)  req= EHR;
        try
        {
            if (broadcast)
            {
                //2020.03.07重大改进， 广播时，所有的返回值都放入到返回值数组中
                ArrayList<HashMap<String,String>> serverList = getAllRPCServerInfoForService(serviceName);
                JSONArray rets= new JSONArray();
                for (HashMap<String,String> one : serverList)
                {

                    String serverInfo=one.get("url");
                    String $sharding=one.get("sharding");

                    JSONObject obj=dispatchToService(serverInfo, className, methodName, param, req, res, timeoutSecond);
                    obj.put("$serviceInfo",serverInfo); //把地址也放入到返回值 中，注意变量名前加上$，避免覆盖
                    obj.put("$sharding",$sharding); //把服务分片也放入到返回值 中，注意变量名前加上$，避免覆盖

                    rets.put(obj);

                }
                return new JSONObject().put("success", true).put("result",rets);
            }
            else
            {
                //2020.04.04 修正，如果服务入是自已，那么不要远程调用，而是直接调用
                if( serviceName.equalsIgnoreCase( App.appName) || serviceName.equalsIgnoreCase(""))
                {
                    return  WSRPCServer.RPC(className,methodName,param, req,res);
                }

                HashSet<String>  unableConnectedServerInfo= null;

                /*
                重大改进：当一个服务的一个副本无法提供服务时， dispatchToService返回 ServerUnreachable  ，
                此时需要再循环获取下一个副本，直到服务可达
                如果循环下去，又回到了起点（ lastServerInfo.equals( serverInfo)  ,那么所有的副本都不可用，退出吧

                为什么会出现这种情况呢，因为Eureka服务信息的延时，比如一个服务重启了， 同一个服务，可能出现2个注册信息，一个是上次残留的
                不可达的，一个是当前新的，那么就会出现这种情况。

                本函数还有一个问题：必须保存服务至少有一个副本是可达的。这个检测放在外面。如果每次调用都检测，那么影响性能，所以通常是对一些
                核心操作先做可达验证，再到本函数中做调用

                 */
                while(true)
                {
                    String serverInfo = getRPCServerInfoForService(serviceName, sharding, "");
                    if (serverInfo == null)    return new JSONObject().put("success", false).put("message", "无法定位" + serviceName + "服务所在地址");
                   // FF.log( serverInfo);
                    JSONObject ret= dispatchToService(serverInfo, className, methodName, param, req, res, timeoutSecond);
                    if( ! ret.getBoolean(KEY_ServerUnreachable,false)) return ret;

                    FF.log( serverInfo+" 不可达，尝试"+serviceName+"的下一个副本");

                    if( unableConnectedServerInfo==null)  unableConnectedServerInfo=new HashSet();
                    if( unableConnectedServerInfo.contains(serverInfo))
                    {  //轮询后，发现获取的服务地址 ，已经在不可达地址清单中，那么肯定是没有服务可用了
                        FF.log("没有可达的服务副本，放弃");
                        return ret;
                    }
                    unableConnectedServerInfo.add( serverInfo) ;

                }
            }
        } catch (Exception e)
        {
            return null;
        }
    }


    /**
     *
     * @param serverInfo 必须是  ws://IP:port/contextPath  的方式
     * @param className
     * @param methodName
     * @param param
     * @param req
     * @param res
     * @param timeoutSecond
     * @return
     */
    public static JSONObject dispatchToService(String serverInfo,
                                                String className, String methodName, JSONObject param,
                                                HttpServletRequest req, HttpServletResponse res, int timeoutSecond)
    {

        if(   CacheConfig.get("/系统配置/RPC方案", "HTTP").equals("HTTP"))
        {
           // FF.log("rpc with okhttp "+ className+"."+ methodName);
            return OkHttpUtil.dispatchToService( req,res,className,methodName,param, serverInfo ,timeoutSecond);
        }


        long startTime = System.currentTimeMillis();

        WSRPCClient client = null;


        int tryTimes = 0;
        while (tryTimes++ < 1000) //可能 池中所有对象都过期了
        {
            client = findRPCClient(serverInfo);
            //无法连接，退出循环
            if (client == null) break;
            // 连接新鲜有效，不需要ping测试
            if (!client.connectionNeedValidCheck()) break;

            //  FF.log("连接需要ping测试一下有效性");
            JSONObject pingResult = client.connectionValidCheck();
            if (pingResult.getBoolean("success", true))
            {
                FF.log(" 连接有效，使用它,测试连接耗时：" + pingResult.getInt( "",0));
                break;
            }
            else
            {
                FF.log("连接异常:" + pingResult.getString("message", ""));
                FF.log("序号："+client.clientNo+ "的连接被抛弃，重新申请一个连接");

                client.close();
            }


        }


        if (client == null)
        {
            return new JSONObject().put("success", false).put("serverUnreachable",true).put("message", "WSRPC.dispatch失败 ，无法连接" + serverInfo.toString());
        }

        if (tryTimes > 1) FF.log("尝试申请" + tryTimes + "次后，成功获取了一个websocket连接，下面使用它进行RPC通信");


        long  connectionValidCheckCost = System.currentTimeMillis() - startTime;


        //整这么个特别的名称 ，是防止与调用的参数名冲突 [标记20190612-1]
        param.put("${rpcRouteRemoteAddr}$", App.IPADDRESS);//调用者的IP地址放进来
        JSONObject j = new JSONObject();
        j.put("className", className);
        j.put("methodName", methodName);
        j.put("param", param);

        //websocket的长连接，是服务器到服务器的长连接，它并不是给某一个用户用的，它是复用的，所以不能在连接时传入用户信息，
        //必须是在每次调用时，传入用户信息。这里的 request应该是客户端请求服务器A，然后服务器A把request 传入到本函数，所以它里面是用用户token 的
        // 取出来，直接放到参数中
        //[标记20201118]
        String tokenId = FF.getTokenIdFromRequest(req);
        j.put(FF.ENCRYPTED_TOKENID, FF.encryptString(tokenId));


        JSONObject ret = client.call(j, timeoutSecond*1000);

        ret.put("connectionValidCheckCost" ,connectionValidCheckCost);
        ret.put("rpcTotalCost" , System.currentTimeMillis()- startTime);

        //客户连接用完后，释放回到池中
        returnObject(client);

        return ret;

    }

    private static void returnObject(WSRPCClient client)
    {
        try
        {

            String ws = client.endpointURI.toString();
            ObjectPool<WSRPCClient> pool = rpcClientMap.get(ws);
            if (pool == null) return;
            pool.returnObject(client);
        } catch (Exception e)
        {
            FF.log(e);
        }
    }

    /**
     * 根据服务器地址和端口号，看看有没有连接上，如果连接上了，就返回连接上的Client对象
     * 如果没有，就创建一个连接 ，等它连接成功后，返回
     *
     * @param serverInfo
     * @return
     */
    private static WSRPCClient findRPCClient(String serverInfo)
    {
        try
        {


            String ws = serverInfo + "/websocket/RPC?client=" + App.appName;
            ObjectPool<WSRPCClient> pool = null;
            if (rpcClientMap.containsKey(ws))
            {
                pool = rpcClientMap.get(ws);
            }
            else
            {

                int maxTotal = FF.getVMProperty("APP_WSCLIENT_MAXTOTAL", 500);
                int maxIdle = FF.getVMProperty("APP_WSCLIENT_MAXIDLE", 20);
                int minIdle = FF.getVMProperty("APP_WSCLIENT_MINIDLE", 1);
                int maxWaitMillis = FF.getVMProperty("APP_WSCLIENT_MAXWAITMILLIS", 30 * 1000);

                WSRPCClientFactory factory = new WSRPCClientFactory(new URI(ws));
                GenericObjectPoolConfig config = new GenericObjectPoolConfig();
                config.setMaxTotal(maxTotal);
                config.setMaxIdle(maxIdle);
                config.setMinIdle(minIdle);
                config.setMaxWaitMillis(maxWaitMillis);

                pool = new GenericObjectPool(factory, config);

                rpcClientMap.put(ws, pool);
            }

            WSRPCClient ret = pool.borrowObject();
            ret.reset();//复位一些信息
            if (ret.session.isOpen()) return ret;

        } catch (Exception e)
        {
            FF.log("findRPCClient获取可用的"+serverInfo+"服务连接时异常："+  e.getMessage());
        }
        return null;


    }

    /**
     * 当连接断开时，移除连接,注意是从池中移掉它
     *
     * @param client
     */
    public static void removeClient(WSRPCClient client)
    {

        client.shutdownPingTask();

        String ws = client.endpointURI.toString();

        ObjectPool<WSRPCClient> pool = rpcClientMap.get(ws);
        if (pool == null) return;

        try
        {
            pool.invalidateObject(client); //让池中的该对象失效

        } catch (Exception e)
        {

        }

    }


    public static void shutdown()
    {
        Iterator<ObjectPool<WSRPCClient>> it = rpcClientMap.values().iterator();
        while (it.hasNext())
        {
            ObjectPool<WSRPCClient> pool = it.next();
            try
            {
                pool.clear();
                pool.close();
            } catch (Exception e)
            {

            }
        }

        rpcClientMap.clear();


    }

    public static String status()
    {
        StringBuffer sb = new StringBuffer(1024);

        Iterator<String> it = rpcClientMap.keySet().iterator();
        while (it.hasNext())
        {
            String key = it.next();
            ObjectPool<WSRPCClient> pool = rpcClientMap.get(key);

            sb.append(key + "  active: " + pool.getNumActive() + "  idle:" + pool.getNumIdle() + "  \n");

        }

        return sb.toString();
    }
}