package util;

import com.aliyuncs.DefaultAcsClient;
import com.aliyuncs.IAcsClient;
import com.aliyuncs.dysmsapi.model.v20170525.SendSmsRequest;
import com.aliyuncs.dysmsapi.model.v20170525.SendSmsResponse;
import com.aliyuncs.exceptions.ClientException;
import com.aliyuncs.profile.DefaultProfile;
import com.aliyuncs.profile.IClientProfile;

import java.util.HashMap;
import java.util.Map;

/**
 * 阿里云短信发送工具类（经过验证的版本）
 * 支持代理和非代理两种模式
 */
public class AliyunSmsSender {

    private static String regionId = "cn-hangzhou";

    // 使用ThreadLocal存储原始代理设置，避免影响其他HTTP请求
    private static final ThreadLocal<Map<String, String>> originalProxySettings = new ThreadLocal<>();

    private final String accessKeyId;
    private final String accessKeySecret;

    private final Map<String, Object> proxyConfig;
    private final boolean useProxy;

    private final String customEndpoint; // 自定义endpoint
    private final boolean useCustomEndpoint;

    /**
     * 构造函数 - 不使用代理
     */
    public AliyunSmsSender(String accessKeyId, String accessKeySecret) {
        this(accessKeyId, accessKeySecret, null,null);
    }

    /**
     * 构造函数 - 使用代理
     */
    public AliyunSmsSender(String accessKeyId, String accessKeySecret, String regionId,
                           Map<String, Object> proxyConfig) {
        this(accessKeyId, accessKeySecret, regionId, null, proxyConfig);
    }

    /**
     * 构造函数 - 使用自定义endpoint
     */
    public AliyunSmsSender(String accessKeyId, String accessKeySecret,
                           String customEndpoint) {
        this(accessKeyId, accessKeySecret, "cn-hangzhou", customEndpoint, null);
    }

    /**
     * 完整构造函数
     */
    public AliyunSmsSender(String accessKeyId, String accessKeySecret, String regionId,
                           String customEndpoint, Map<String, Object> proxyConfig) {
        this.accessKeyId = accessKeyId;
        this.accessKeySecret = accessKeySecret;
        this.regionId = regionId != null ? regionId : "cn-hangzhou";
        this.customEndpoint = customEndpoint;
        this.useCustomEndpoint = customEndpoint != null && !customEndpoint.trim().isEmpty();
        this.useProxy = proxyConfig != null && !proxyConfig.isEmpty();
        this.proxyConfig = proxyConfig;
    }

    /**
     * 发送短信
     */
    public SendSmsResponse sendSms(String phoneNumbers, String signName,
                                   String templateCode, String templateParam) {
        // 保存原始代理设置
        if (useProxy) {
            saveOriginalProxySettings();
            setupProxy();
        }

        try {
            // 创建客户端（每次请求都新建，避免代理设置冲突）
            IAcsClient client = createAcsClient();

            SendSmsRequest request = new SendSmsRequest();
            request.setPhoneNumbers(phoneNumbers);
            request.setSignName(signName);
            request.setTemplateCode(templateCode);

            if (templateParam != null) {
                request.setTemplateParam(templateParam);
            }

            SendSmsResponse response = client.getAcsResponse(request);
            return response;

        } catch (ClientException e) {
            throw new RuntimeException("短信发送失败 - 错误码: " + e.getErrCode() + ", 错误信息: " + e.getMessage(), e);
        } finally {
            // 恢复原始代理设置
            if (useProxy) {
                restoreOriginalProxySettings();
            }
        }
    }

    /**
     * 创建阿里云客户端
     */
    private IAcsClient createAcsClient() throws ClientException {
        IClientProfile profile = DefaultProfile.getProfile(regionId, accessKeyId, accessKeySecret);

        // 设置自定义endpoint
        if (useCustomEndpoint) {
            // 移除http://或https://前缀（如果有）
            String endpoint = customEndpoint.replaceFirst("^(http|https)://", "");
            DefaultProfile.addEndpoint(regionId, regionId, "Dysmsapi", endpoint);
        }

        return new DefaultAcsClient(profile);
    }

    /**
     * 保存原始代理设置
     */
    private void saveOriginalProxySettings() {
        Map<String, String> original = new HashMap<>();
        original.put("http.proxyHost", System.getProperty("http.proxyHost"));
        original.put("http.proxyPort", System.getProperty("http.proxyPort"));
        original.put("https.proxyHost", System.getProperty("https.proxyHost"));
        original.put("https.proxyPort", System.getProperty("https.proxyPort"));
        original.put("http.proxyUser", System.getProperty("http.proxyUser"));
        original.put("http.proxyPassword", System.getProperty("http.proxyPassword"));

        originalProxySettings.set(original);
    }

    /**
     * 设置代理
     */
    private void setupProxy() {
        String proxyHost = (String) proxyConfig.get("host");
        Integer proxyPort = (Integer) proxyConfig.get("port");

        if (proxyHost != null && proxyPort != null) {
            System.setProperty("http.proxyHost", proxyHost);
            System.setProperty("http.proxyPort", String.valueOf(proxyPort));
            System.setProperty("https.proxyHost", proxyHost);
            System.setProperty("https.proxyPort", String.valueOf(proxyPort));

            // 处理代理认证
            String username = (String) proxyConfig.get("username");
            String password = (String) proxyConfig.get("password");

            if (username != null && password != null) {
                System.setProperty("http.proxyUser", username);
                System.setProperty("http.proxyPassword", password);
                System.setProperty("https.proxyUser", username);
                System.setProperty("https.proxyPassword", password);
            }
        }
    }

    /**
     * 恢复原始代理设置
     */
    private void restoreOriginalProxySettings() {
        Map<String, String> original = originalProxySettings.get();
        if (original != null) {
            for (Map.Entry<String, String> entry : original.entrySet()) {
                if (entry.getValue() != null) {
                    System.setProperty(entry.getKey(), entry.getValue());
                } else {
                    System.clearProperty(entry.getKey());
                }
            }
            originalProxySettings.remove();
        }
    }

    /**
     * 批量发送短信
     */
    public SendSmsResponse sendBatchSms(String[] phoneNumbers, String signName,
                                        String templateCode, String templateParam) {
        String phones = String.join(",", phoneNumbers);
        return sendSms(phones, signName, templateCode, templateParam);
    }

    /**
     * 检查是否使用代理
     */
    public boolean isUsingProxy() {
        return useProxy;
    }

    /**
     * 获取代理配置信息
     */
    public Map<String, Object> getProxyConfig() {
        return proxyConfig != null ? new HashMap<>(proxyConfig) : new HashMap<>();
    }

    /**
     * 创建代理配置的便捷方法
     */
    public static Map<String, Object> createProxyConfig(String host, int port) {
        Map<String, Object> config = new HashMap<>();
        config.put("host", host);
        config.put("port", port);
        return config;
    }

    public static Map<String, Object> createProxyConfig(String host, int port,
                                                        String username, String password) {
        Map<String, Object> config = new HashMap<>();
        config.put("host", host);
        config.put("port", port);
        config.put("username", username);
        config.put("password", password);
        return config;
    }

    public static void main(String[] args) {
        // 阿里云配置
        String accessKeyId = "LTAI5t9WXNZ6V1fZrsyavwgA";
        String accessKeySecret = "awdL7LDZYpiR6ToV5ybXMhv4vijX5J";


        try {
            // 示例1：不使用代理
            System.out.println("=== 示例1：不使用代理 ===");
            AliyunSmsSender sender1 = new AliyunSmsSender(accessKeyId, accessKeySecret);

            SendSmsResponse response1 = sender1.sendSms(
                    "13588867195",
                    "杭州九曲科技",
                    "SMS_489715109",
                    "{eventtime:\"2025.08.27 23:31:32\" , address:\"实验南楼\" , device:\"烟雾报警器123-232334\" , alarm:\"xxxxx\" , name:\"张三\", contact:\"13567891234\"}"
            );

            System.out.println("发送结果: " + response1.getCode());
            System.out.println("请求ID: " + response1.getRequestId());
            System.out.println("业务ID: " + response1.getBizId());

            // 示例2：使用代理
            System.out.println("\n=== 示例2：使用代理 ===");
            Map<String, Object> proxyConfig = new HashMap<>();
            proxyConfig.put("host", "192.168.1.200");
            proxyConfig.put("port", 3128);

            AliyunSmsSender sender2 = new AliyunSmsSender(accessKeyId, accessKeySecret, null,proxyConfig);

            SendSmsResponse response2 = sender2.sendSms(
                    "13800138000",
                    "你的签名",
                    "SMS_123456789",
                    "{\"code\":\"654321\"}"
            );

            System.out.println("发送结果: " + response2.getCode());
            System.out.println("请求ID: " + response2.getRequestId());

        } catch (Exception e) {
            System.err.println("发送失败: " + e.getMessage());
            e.printStackTrace();
        }
    }
}
