package nettyrpc.client;

import io.netty.bootstrap.Bootstrap;
import io.netty.buffer.ByteBuf;
import io.netty.channel.*;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.SocketChannel;
import io.netty.channel.socket.nio.NioSocketChannel;
import io.netty.handler.codec.LengthFieldBasedFrameDecoder;
import io.netty.handler.timeout.IdleStateHandler;
import nettyrpc.common.CustomHeartbeatHandler;
import nettyrpc.manage.RPCServerInfo;
import org.json.JSONObject;
import util.FF;
import webApp.App;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;


public class Client
{
    private NioEventLoopGroup workGroup;
    private Channel channel;
    private Bootstrap bootstrap;

    public static final  int  IDLE=10;

    AtomicLong seqNumGenerator = new AtomicLong(0);

    ConcurrentHashMap<Long, RPCContext> no2ctx = new ConcurrentHashMap<>();


    public RPCServerInfo serverInfo;

    public Client(RPCServerInfo serverInfo, int threadCount)
    {
        this.serverInfo = serverInfo;

        workGroup = new NioEventLoopGroup(threadCount);

    }


    //下面的main及 testSendData
    public static void main(String[] args) throws Exception
    {
        Client client = new Client(new RPCServerInfo("127.0.0.1", 12345), 10);
        client.start();

        client.testSendData();


    }

    public void testSendData() throws Exception
    {
        long d1 = System.currentTimeMillis();

        int threadNum = 1;
        final int requestNum = 1 * 1000;
        Thread[] threads = new Thread[threadNum];

        for (int i = 0; i < threadNum; i++)
        {
            final int ti = i;
            threads[i] = new Thread(new Runnable()
            {
                @Override
                public void run()
                {


                    for (int j = 0; j < requestNum; j++)
                    {
                        String content = "消息client msg " + ti + " ," + j;
                        FF.log( call(content, 2));
                    }

                }
            });
            threads[i].start();
        }


        for (int i = 0; i < threads.length; i++)
        {
            threads[i].join();
        }

        long d2 = System.currentTimeMillis();

        System.out.println(threadNum * requestNum / (d2 - d1) * 1000.0);

    }

    public String call(String content, int timeOutSecond)
    {

        //申请一个唯一编号
        long no = this.getNextSequentNumber();
        RPCContext ctx = new RPCContext();

        //把编号与ctx对应起来
        no2ctx.put(no, ctx);

        try
        {
            long d1 = System.currentTimeMillis();


            byte[] bs=content.getBytes("UTF-8");
            ByteBuf buf = channel.alloc().buffer(5 + 8+bs.length);
            buf.writeInt(5 + 8 + bs.length);
            buf.writeByte(CustomHeartbeatHandler.CUSTOM_MSG);

            buf.writeLong(no);

            buf.writeBytes(bs);
            ctx.semp.acquire();//获取信号量，视为申请一个许可，申请后，许可证没有了，让标记1 处阻塞


            channel.writeAndFlush(buf);
            //上面往channel中写数据后，服务器收到数据，并回馈，这个过程是异步的，
            //因此，此时必须阻塞起来，等服务器回馈。
            // 消息内容前4位是表明消息数据的长度，第5位是标记位，标记是一个ping,pong 消息，还是常规有用的消息
            // 接下来8位（字节)是本消息的唯一标记。
            //服务器处理消息后，会把这个8位长的标记放在相同的位置上，返回回来，这样用它来追踪客户，通知客户消息已经回馈回来了
            // no2ctx [no]可以定位到客户的semp标记，回馈事件中，将它释放，于是下面的 tryAcquire得以解除阻塞并返回true

            //FF.log(new Date()+" 等待回应："+no);
            //标记1  ，阻塞，直到收回复的消息后，解除
            if (ctx.semp.tryAcquire(timeOutSecond, TimeUnit.SECONDS))
            {
                return ctx.response;
            }

            return new JSONObject().put("success", false).put("message", "调用超时").toString();


        } catch (Exception e)
        {
            return new JSONObject().put("success", false).put("message", e.getMessage()).toString();

        } finally
        {
            //本次RPC调用完成 ， 删除信号
            //FF.log(new Date()+"  移除："+no);
            no2ctx.remove(no);
        }
    }


    public long getNextSequentNumber()
    {
        return seqNumGenerator.getAndAdd(1);
    }


    public boolean start()
    {
        try
        {
            bootstrap = new Bootstrap();
            bootstrap
                    .group(workGroup)
                    .channel(NioSocketChannel.class)
                    .handler(new ChannelInitializer<SocketChannel>()
                    {
                        protected void initChannel(SocketChannel socketChannel) throws Exception
                        {
                            ChannelPipeline p = socketChannel.pipeline();
                            p.addLast(new IdleStateHandler(0, 0, IDLE));
                            p.addLast(new LengthFieldBasedFrameDecoder(2 * 1024 * 1024, 0, 4, -4, 0));
                            p.addLast(new ClientHandler(Client.this));
                        }
                    });
            return doConnect();

        } catch (Exception e)
        {
            FF.log(e);
            return false;
        }
    }

    public boolean doConnect()
    {

        try
        {
            Semaphore semp = new Semaphore(1);
            semp.acquire();//将许可用掉


            ChannelFuture future = bootstrap.connect(serverInfo.host, serverInfo.port);

            future.addListener(new ChannelFutureListener()
            {
                public void operationComplete(ChannelFuture futureListener) throws Exception
                {
                    if (futureListener.isSuccess())
                    {
                        channel = futureListener.channel();
                        System.out.println(App.appName+" Connect to "+ serverInfo.toString()+" successfully!");
                        //放回许可，程序跳到标记1
                        semp.release(1);//
                }
                    else
                    {

                        FF.log(App.appName+"无法连接到" + serverInfo.toString());
                    }
                }
            });


            //标记1  再次申请许可，直到10秒超时
            if (semp.tryAcquire(10, TimeUnit.SECONDS))
            {
                //到这里就成功连接了
                return true;
            }
            else
            {
                FF.log(App.appName+"连接"+serverInfo.toString()+"超时，放弃");
                return false;
            }

        } catch (Exception e)
        {
            return false;
        }
    }

    public void close()
    {
        channel.close();
    }
}
