package mq_zbus;

import io.zbus.mq.MqClient;
import io.zbus.transport.DataHandler;
import io.zbus.transport.ErrorHandler;
import io.zbus.transport.Message;
import org.apache.commons.pool2.ObjectPool;
import org.apache.commons.pool2.impl.GenericObjectPool;
import org.apache.commons.pool2.impl.GenericObjectPoolConfig;

import util.FF;
import webApp.App;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;


/**
 * zbus 与rocketMQ的类比
 * <p>
 * rocketMQ用  customGroup 来注册同一组客户 ,消息发布者不需要加以区分，它用topic 来区分不同的消息组
 * <p>
 * zbus 用 mq 名称来实现类似 topic 的功能 ， 用 channel 来实现 customerGroup 的功能
 */
public class MessageProducer
{


    static int MaxCacheSize = 50000; //最多堆积消息条数
    static ConcurrentLinkedQueue<Message> messageCache = new ConcurrentLinkedQueue<Message>();


    static volatile boolean exitSend = false;
    static SimpleDateFormat SDF = new SimpleDateFormat("yyyy.MM.dd HH:mm:ss SSS");

    static ConcurrentHashMap<String, ObjectPool<MqClient>> mqPool = null;


    public static String topic_log = "appLog";

    public static String tag_log = "appLog";


    public static Thread quickSendMessageThread;

    public static int maxTotal, maxIdle, minIdle, maxWaitMillis;

    public static void init(int maxTotal_, int maxIdle_, int minIdle_, int maxWaitMillis_)
    {

        mqPool = new ConcurrentHashMap<String, ObjectPool<MqClient>>();


        maxTotal = maxTotal_;
        maxIdle = maxIdle_;
        minIdle = minIdle_;
        maxWaitMillis = maxWaitMillis_;

        quickSendMessageThread = new Thread()
        {


            @Override
            public void run()
            {
                while (true)
                {

                    /*为什么不用LinkedBlockingQueue
                        由于LinkedBlockingQueue   ，take方法在队列空的时候会阻塞，直到有队列成员被放进来。
                        所以当没有消息时，它一直阻塞，就无法中断它，当然解决办法也是有的，发送一个特定内容的消息
                        让它退出阻塞，并判断消息是不是某特定消息，是，就终止线程。
                        */
                    while (messageCache.isEmpty())
                    {
                        if (exitSend) return;
                        try
                        {
                            FF.sleep(20);
                        } catch (Exception e)
                        {
                        }
                    }


                    //poll并不阻塞队列，当队列为空时，它直接返回null ,所以msg是可能为null的，所以前面要处理一下，避免没有消息时进入下面的处理
                    Message msg = messageCache.poll();
                    //如果没有消息了，并且需要结束，那么退出吧
                    if (msg == null && exitSend) return;

                    if (msg == null) continue;

                    String topic = msg.getHeader("topic");

                    ObjectPool<MqClient> pool = getPool(topic);
                    MqClient producer = null;


                    //为什么要循环3次，因为下面的发布者池中获取的对象，它可能还没有正常连接到RocketMQ，导致发送失败
                    //所以在发送失败后，等10秒，再试，试3次，（30秒）还不能发送成功，就放弃
                    for (int i = 0; i < 3; i++)
                    {
                        try
                        {

                            while (producer == null)
                            {
                                producer = pool.borrowObject();


                                if (producer == null)
                                {
                                    int waitCount = 0;
                                    while (waitCount++ < 100)
                                    {
                                        FF.sleep(100);
                                        if (exitSend) return;
                                    }
                                }

                                if (exitSend) return;

                            }

                            //如果消息发送者超过3个，那么显示一下提示信息
                            //
                            //发送消息
                            producer.invoke(msg);
                            //到这里就是成功了，如果成功了， 那么把发送者返回到池中备用
                            pool.returnObject(producer);

                            break;

                        } catch (Exception ex)
                        {
                            FF.log("zbus 生产者异常：" + ex.getMessage());
                            //发送异常，那么可能是生产者异常，那么把它置为非法，进入下一次循环
                            //@off
                            try { pool.invalidateObject(producer); }  catch (Exception ex2)   {   }
                            //@on

                            //等10秒 , 可能等的过程中被通知停止，那么就停止吧
                            int waitCount = 0;
                            while (waitCount++ < 100)
                            {
                                FF.delay(100);
                                if (exitSend) return;
                            }
                        }
                    }

                }
            }

        };
        quickSendMessageThread.setName("ZBUS-QUICK-SEND-线程-"+ App.appName);
        quickSendMessageThread.start();
    }

    public static ObjectPool<MqClient> getPool(String topic)
    {
        if (mqPool.containsKey(topic)) return mqPool.get(topic);

        MessageProducerFactory factory = new MessageProducerFactory(topic);

        GenericObjectPoolConfig config = new GenericObjectPoolConfig();
        config.setMaxTotal(maxTotal);
        config.setMaxIdle(maxIdle);
        config.setMinIdle(minIdle);
        config.setMaxWaitMillis(maxWaitMillis);

        ObjectPool<MqClient> pool = new GenericObjectPool(factory, config);
        mqPool.put(topic, pool);
        return pool;

    }

    public static void destroy()
    {
        try
        {
            exitSend = true;//通知快速发送线程可以结束了

            for (ObjectPool<MqClient> pool : mqPool.values())
            {
                pool.clear();
                pool.close();
            }

            mqPool.clear();
            messageCache.clear();

        } catch (Exception e)
        {

        }
    }


    //发送日志信息
    public static boolean sendLog(String keys, String msgbody)
    {
        return quickSend(topic_log, tag_log, keys, msgbody);
    }


    /**
     * 同步发送消息
     *
     * @param topic
     * @param tags
     * @param keys
     * @param msgbody
     * @return
     */
    public static Message syncSend(String topic, String tags, String keys, String msgbody)
    {

        ObjectPool<MqClient> pool = getPool(topic);
        if (pool == null) return null;

        MqClient producer = null;
        try
        {
            producer = pool.borrowObject();
            if (producer == null) return null;
            Message msg = new Message();
            msg.setHeader("cmd", "pub");  //Publish
            msg.setHeader("mq", topic); //zbus的需要
            msg.setHeader("topic", topic);//程序的需要
            msg.setHeader("tag", tags);
            msg.setHeader("keys", keys);

            msg.setBody(msgbody);    //set business data in body


            msg.setHeader("vipname", App.getInstance().getVirtualHostName());
            return producer.invoke(msg);

        } catch (Exception ex)
        {

            try
            {
                pool.invalidateObject(producer);
            } catch (Exception er)
            {
            }

            return null;

        } finally
        {
            if (producer != null)
            {
                try
                {
                    pool.returnObject(producer);
                } catch (Exception ex)
                {

                }
            }
        }

    }

    /**
     * 异步发送消息
     *
     * @param topic
     * @param tags
     * @param keys
     * @param msgbody
     * @param dataHandler
     */

    public static void asyncSend(String topic, String tags, String keys, String msgbody, DataHandler<Message> dataHandler)
    {

        ObjectPool<MqClient> pool = getPool(topic);
        if (pool == null) return;

        MqClient producer = null;
        try
        {
            producer = pool.borrowObject();
            if (producer == null) return;
            Message msg = new Message();
            msg.setHeader("cmd", "pub");  //Publish
            msg.setHeader("mq", topic); //zbus的需要
            msg.setHeader("topic", topic); //程序的需要
            msg.setHeader("tag", tags);
            msg.setHeader("keys", keys);

            msg.setBody(msgbody);    //set business data in body

            msg.setHeader("vipname", App.getInstance().getVirtualHostName());
            producer.invoke(msg, dataHandler);

        } catch (Exception ex)
        {

            try
            {
                pool.invalidateObject(producer);
            } catch (Exception er)
            {
            }


        } finally
        {
            if (producer != null)
            {
                try
                {
                    pool.returnObject(producer);
                } catch (Exception ex)
                {

                }
            }
        }

    }

    /**
     * 快速发送消息
     *
     * @param topic
     * @param tags
     * @param keys
     * @param msgbody
     */
    public static boolean quickSend(String topic, String tags, String keys, String msgbody)
    {

        if (msgbody == null) return false;
        if (msgbody.isEmpty()) return false;


        try
        {

            //当太多快速消息积累无法发送时，丢掉开始的那些
            while (messageCache.size() > MaxCacheSize)
            {

                System.out.println("oneWay 方式的消息堆积过多无法发送，丢弃所有未发送的消息");
                messageCache.clear();
            }

            Message msg = new Message();
            msg.setHeader("cmd", "pub");  //Publish
            msg.setHeader("mq", topic);
            msg.setHeader("topic", topic);
            msg.setHeader("tag", tags);
            msg.setHeader("keys", keys);
            msg.setBody(msgbody);    //set business data in body
            msg.setHeader("vipname", App.getInstance().getVirtualHostName());

            msg.setHeader("address", App.IPADDRESS_AND_PORT);

            msg.setHeader("time", SDF.format(new Date()));

            messageCache.add(msg);
            return true;

        } catch (Exception ex)
        {


            return false;
        }
    }

}
