package http;


import okhttp3.*;
import org.apache.http.HttpEntity;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.TrustStrategy;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.ssl.SSLContextBuilder;
import org.apache.http.util.EntityUtils;
import org.json.JSONObject;
import util.FF;
import util.RPCRouter;
import webApp.App;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.BufferedInputStream;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.zip.GZIPInputStream;

import static websocketRPC.WSRPC.KEY_ServerUnreachable;

public class OkHttpUtil
{

    private static int defaultTimeoutSeconds = 600;

    public static ConcurrentHashMap<Integer, OkHttpClient> okHttpClientMap = new ConcurrentHashMap<>();

    static
    {
        // getOkHttpClient(defaultTimeoutSeconds);
        Logger.getLogger(OkHttpClient.class.getName()).setLevel(Level.FINE);

    }

    public static OkHttpClient getOkHttpClient()
    {
        return getOkHttpClient(defaultTimeoutSeconds);
    }

    public static OkHttpClient getOkHttpClient(int timeoutSeconds)
    {

      //  FF.log("OKHTTP getClient timeout="+ timeoutSeconds+"   cache size: "+ okHttpClientMap.size());
        if (okHttpClientMap.containsKey(timeoutSeconds))
        {
         //   FF.log("直接从缓存中返回 okhttpclient");
            return okHttpClientMap.get(timeoutSeconds);
        }
        try
        {
            //
            FF.log("新建  okhttpClient  timeout=" +timeoutSeconds );
            OkHttpClient okHttpClient = OkHttpConfig.newOkHttpClient(timeoutSeconds);
            okHttpClientMap.put(timeoutSeconds, okHttpClient);
            return okHttpClient;

        } catch (Exception e)
        {
            FF.log(FF.exceptionMessage(e));
            return null;
        }
    }

    /**
     * get
     *
     * @param url     请求的url
     * @param queries 请求的参数，在浏览器？后面的数据，没有可以传null
     */
    public static String get(String url, Map<String, String> queries)
    {
        String responseBody = "";
        StringBuffer sb = new StringBuffer(url);
        if (queries != null && queries.keySet().size() > 0)
        {
            boolean firstFlag = true;
            Iterator iterator = queries.entrySet().iterator();
            while (iterator.hasNext())
            {
                Map.Entry entry = (Map.Entry<String, String>) iterator.next();
                if (firstFlag)
                {
                    sb.append("?" + entry.getKey() + "=" + entry.getValue());
                    firstFlag = false;
                }
                else
                {
                    sb.append("&" + entry.getKey() + "=" + entry.getValue());
                }
            }
        }
        Request request = new Request.Builder()
                .url(sb.toString())
                .build();
        OkHttpClient okHttpClient = getOkHttpClient(defaultTimeoutSeconds);
        try (Response response = okHttpClient.newCall(request).execute();)
        {


            int status = response.code();
            if (response.isSuccessful())
            {
                return response.body().string();
            }
        } catch (Exception e)
        {
            FF.log("okhttp3 put error >> " + FF.exceptionMessage(e));
        }
        return responseBody;
    }

    /**
     * post
     *
     * @param url    请求的url
     * @param params post form 提交的参数
     * @return
     */
    public static String post(String url, Map<String, String> params)
    {
        String responseBody = "";
        FormBody.Builder builder = new FormBody.Builder();
        //添加参数
        if (params != null && params.keySet().size() > 0)
        {
            for (String key : params.keySet())
            {
                builder.add(key, params.get(key));
            }
        }
        Request request = new Request.Builder()
                .url(url)
                .post(builder.build())
                .build();

        OkHttpClient okHttpClient = getOkHttpClient(defaultTimeoutSeconds);
        try (Response response = okHttpClient.newCall(request).execute();)
        {
            int status = response.code();
            if (response.isSuccessful())
            {
                return response.body().string();
            }
        } catch (Exception e)
        {
            FF.log("okhttp3 post error >> " + FF.exceptionMessage(e));
        }
        return responseBody;
    }

    /**
     * get
     *
     * @param url     请求的url
     * @param queries 请求的参数，在浏览器？后面的数据，没有可以传null
     * @return
     */
    public static String getForHeader(String url, Map<String, String> queries)
    {
        String responseBody = "";
        StringBuffer sb = new StringBuffer(url);
        if (queries != null && queries.keySet().size() > 0)
        {
            boolean firstFlag = true;
            Iterator iterator = queries.entrySet().iterator();
            while (iterator.hasNext())
            {
                Map.Entry entry = (Map.Entry<String, String>) iterator.next();
                if (firstFlag)
                {
                    sb.append("?" + entry.getKey() + "=" + entry.getValue());
                    firstFlag = false;
                }
                else
                {
                    sb.append("&" + entry.getKey() + "=" + entry.getValue());
                }
            }
        }
        Request request = new Request.Builder()
                .addHeader("key", "value")
                .url(sb.toString())
                .build();
        OkHttpClient okHttpClient = getOkHttpClient(defaultTimeoutSeconds);
        try (Response response = okHttpClient.newCall(request).execute();)
        {
            int status = response.code();
            if (response.isSuccessful())
            {
                return response.body().string();
            }
        } catch (Exception e)
        {
            FF.log("okhttp3 put error >> " + FF.exceptionMessage(e));
        }
        return responseBody;
    }

    /**
     * Post请求发送JSON数据....{"name":"zhangsan","pwd":"123456"}
     * 参数一：请求Url
     * 参数二：请求的JSON
     * 参数三：请求回调
     */
    public static String post (String url, String jsonParams)
    {
        String responseBody = "";
        RequestBody requestBody = RequestBody.create(MediaType.parse("application/json; charset=utf-8"), jsonParams);
        Request request = new Request.Builder()
                .url(url)
                .post(requestBody)
                .build();
        OkHttpClient okHttpClient = getOkHttpClient(defaultTimeoutSeconds);
        try (Response response = okHttpClient.newCall(request).execute();)
        {
            int status = response.code();
            if (response.isSuccessful())
            {
                return response.body().string();
            }
        } catch (Exception e)
        {
            FF.log("okhttp3 post error >> " + FF.exceptionMessage(e));
        }
        return responseBody;
    }

    /**
     * Post请求发送xml数据....
     * 参数一：请求Url
     * 参数二：请求的xmlString
     * 参数三：请求回调
     */
    public static String postXmlParams(String url, String xml)
    {
        String responseBody = "";
        RequestBody requestBody = RequestBody.create(MediaType.parse("application/xml; charset=utf-8"), xml);
        Request request = new Request.Builder()
                .url(url)
                .post(requestBody)
                .build();
        OkHttpClient okHttpClient = getOkHttpClient(defaultTimeoutSeconds);
        try (Response response = okHttpClient.newCall(request).execute();)
        {
            int status = response.code();
            if (response.isSuccessful())
            {
                return response.body().string();
            }
        } catch (Exception e)
        {
            FF.log("okhttp3 post error >> " + FF.exceptionMessage(e));
        }
        return responseBody;
    }


    public static JSONObject dispatchToService(HttpServletRequest req, HttpServletResponse res,
                                               String className, String methodName,
                                               JSONObject param, String serviceHomePageURL, int timeoutSeconds)
    {


        JSONObject ret = null;


        // 创建链接
        OkHttpClient okHttpClient = getOkHttpClient(timeoutSeconds);  // 按超时时间获取连接对象


        String url = serviceHomePageURL + "/RPCRouter";
        url = FF.replaceAll(url, "ws://", "http://");

        //  FF.log("okhttp   "+ url);

        JSONObject p = new JSONObject();
        p.put("classname", className);
        p.put("methodname", methodName);
        p.put("parameter", param);


        MediaType mediaType = MediaType.parse("application/json");
        RequestBody body = RequestBody.create(mediaType, p.toString());
        Request.Builder reqBuild = new Request.Builder();
        reqBuild.url(url)
                .method("POST", body)
                .addHeader("Content-Type", "application/json;charset=UTF-8");

        //把用户信息传过去，注意是用cookie
        if (req != null)
        {
            String tokenId = FF.getTokenIdFromRequest(req);
            reqBuild.addHeader(FF.ENCRYPTED_TOKENID, FF.encryptString(tokenId));
            reqBuild.addHeader("rpcRouteRemoteAddr", req.getRemoteAddr()); //把调用者的IP地址也传过去
        }
        else
        {
            reqBuild.addHeader(FF.ENCRYPTED_TOKENID, "server");
            reqBuild.addHeader("rpcRouteRemoteAddr", App.IPADDRESS); //把调用者的IP地址也传过去
        }


        Request request = reqBuild.build();

        try (Response response = okHttpClient.newCall(request).execute())
        {

            if (response.isSuccessful())
            {
                ret = new JSONObject(response.body().string());
            }
            else
            {
                FF.log(url + "参数：" + param.toString() + "执行返回失败");
                ret = new JSONObject().put("success", false).put("message", url + "执行失败，状态码 " + response.code());
            }
        } catch (Exception e)
        {
            FF.log("异常：" + e.getClass().getName() + "   " + FF.exceptionMessage(e));
            ret = new JSONObject().put("success", false).put("message", e.getMessage());
            if (e instanceof java.net.ConnectException || e instanceof java.net.NoRouteToHostException)
            {
                //服务不可用 ， 让 WSRPC重新试下一个服务
                FF.log(serviceHomePageURL + " " + FF.exceptionMessage(e));
                ret.put(KEY_ServerUnreachable, true);
            }
        }

        return ret;
    }
}

/*

修改okhttp3 最大并发数方法
源码中okhttp3.Dispatcher这个类里面提供了

maxRequests = 64: 最大并发请求数为64
maxRequestsPerHost = 5: 每个主机最大请求数为5
1
2
并且对外提供了相应的public的set方法
调用如：

OkHttpClient().dispatcher().setMaxRequestsPerHost(8);

 */