package dfs;

import config.CacheConfig;
import http.OkHttpUtil;
import io.minio.MinioClient;

import io.minio.Result;
import io.minio.ServerSideEncryption;
import io.minio.errors.InvalidEndpointException;
import io.minio.errors.InvalidPortException;
import io.minio.messages.Item;
import okhttp3.OkHttpClient;
import org.json.JSONArray;
import org.json.JSONObject;
import util.DirCopyUtil;
import util.FF;
import util.URLUtil;
import webApp.App;

import java.io.File;
import java.net.URLEncoder;
import java.util.Date;
import java.util.Map;

/**
 * https://repo1.maven.org/maven2/io/minio/minio/    minio 的仓库
 *
 * https://github.com/minio/minio-java/tree/release/examples
 * <p>
 * 其中
 * minioXXXX-all.jar 是完整包，需要复制到 formengine 及app的 lib 目录
 * minioxxxx.jar 需要复制到shared/lib 目录
 */
public class MinioFileServer implements IFileServer
{

    int serverIndex;

    public MinioFileServer(int serverIndex)
    {
        this.serverIndex = serverIndex;
    }

    @Override
    public String getBucketName()
    {


        String bucketName = CacheConfig.get("/文件服务/文件服务器." + serverIndex + "/minio_bucket", "default");
        return bucketName;
    }

    @Override
    public int getImageMaxWidthOrHeight()
    {
        String t = CacheConfig.get("/文件服务/文件服务器." + serverIndex + "/imageMaxWidthOrHeight", "4096");
        int ret = FF.String2Int(t);
        if (ret == 0) ret = 4096;
        return ret;
    }

    @Override
    public JSONArray getFileList(String path)
    {
        String endpoint = CacheConfig.get("/文件服务/文件服务器." + serverIndex + "/minio_address", "");
        String accessKey = CacheConfig.get("/文件服务/文件服务器." + serverIndex + "/minio_access_key", "");
        String secretKey = CacheConfig.get("/文件服务/文件服务器." + serverIndex + "/minio_access_secret", "");

        String bucketName = getBucketName();

        JSONArray ret = new JSONArray();
        if( path.startsWith("/")) path=path.substring(1);
        String fullPath=  path +"/";
        if( fullPath.equals("/")) fullPath="";

        try
        {


            // 初始化MinIO客户端对象
            MinioClient minioClient = getMinioClient(endpoint, accessKey, secretKey);

            // 指定要查看的目录路径


            // 调用listObjects()方法获取目录中的文件列表
            Iterable<Result<Item>> results = minioClient.listObjects(bucketName, fullPath, false);

            // 打印每个文件的名称
            for (Result<Item> result : results) {
                Item item = result.get();

                String extName = item.objectName();

                if (extName.indexOf(".") > 0)
                {
                    extName = extName.substring(extName.lastIndexOf(".") + 1);
                }
                else
                {
                    extName = "";
                }

                if (item.isDir()) extName = "目录";

                String name= item.objectName();
                if(name.endsWith("/"))  name= name.substring(0, name.length()-1);
                //if( item.isDir())
                {
                    int p= name.lastIndexOf("/");
                    if( p>0) name= name.substring(p+1);
                }

                JSONObject one = new JSONObject();
                one.put("parentPath", fullPath);
                one.put("name", name);
                one.put("type", extName );
                one.put("fullName", item.objectName());
                one.put("isDir", item.isDir());

                if( item.isDir())
                {
                    one.put("fileSize", "");
                    one.put("lastModified", "");

                }else {
                    one.put("fileSize",InnerFileServer.fileSize( item.objectSize() ));
                    one.put("lastModified", FF.get_yyyyMMdd_HHmm_FormatedDate(  item.lastModified()  ));
                    one.put("lastModifiedDate",   item.lastModified() );


                } ;

                FF.log(one.toString());
                ret.put(one);

            }
        } catch (Exception e) {
            System.err.print(e.getMessage());
        }

        return ret;

    }

    @Override
    public String mkdir(String path, String subPath)
    {
        return "暂不支持";
    }

    @Override
    public String rename(String objectName, String newName)
    {
        return "暂不支持";
    }

    @Override
    public boolean objectExists(String objectName)
    {
        String endpoint = CacheConfig.get("/文件服务/文件服务器." + serverIndex + "/minio_address", "");
        String accessKey = CacheConfig.get("/文件服务/文件服务器." + serverIndex + "/minio_access_key", "");
        String secretKey = CacheConfig.get("/文件服务/文件服务器." + serverIndex + "/minio_access_secret", "");

        String bucketName = getBucketName();
        try
        {

            MinioClient minioClient = getMinioClient(endpoint, accessKey, secretKey);
            // Check whether the object exists using statObject().  If the object is not found,
            // statObject() throws an exception.  It means that the object exists when statObject()
            // execution is successful.
            minioClient.statObject(bucketName, objectName);
            return true;
        } catch (Throwable e)
        {
            return false;
        }
    }

    /**
     * 更好的办法就是把 MinioClient 创建单例
     * @param endpoint
     * @param accessKey
     * @param secretKey
     * @return
     * @throws Exception
     */
    private MinioClient getMinioClient(String endpoint ,String accessKey, String secretKey) throws  Exception
    {
        int port =0; // 表示 endpoint中包含了 port，不要需要再指定port
        boolean secure = endpoint == null || !endpoint.startsWith("http://");
        OkHttpClient httpClient  =OkHttpUtil.getOkHttpClient();
        String region=null;
        //2024.01.20 重大问题： 如果不指定 httpClient,那么 MinitClient自已就会创建一个,那当短时间上传大量文件时，就会造成 okhttp connectinPool 大量创建
        //而造成 oom
        MinioClient minioClient = new MinioClient(endpoint,port, accessKey, secretKey,region, secure, httpClient);
        return minioClient;

    }

    @Override
    public JSONObject upload(String objectName, String showName, String file) throws Exception
    {
        String bucketName = getBucketName();
        JSONObject ret = new JSONObject().put("success", true);

        if (objectName.startsWith("/")) objectName = objectName.substring(1);

        String endpoint = CacheConfig.get("/文件服务/文件服务器." + serverIndex + "/minio_address", "");
        String accessKey = CacheConfig.get("/文件服务/文件服务器." + serverIndex + "/minio_access_key", "");
        String secretKey = CacheConfig.get("/文件服务/文件服务器." + serverIndex + "/minio_access_secret", "");


        try
        {

            MinioClient minioClient = getMinioClient(endpoint, accessKey, secretKey);

            boolean found = minioClient.bucketExists(bucketName);
            if (!found)
            {
                minioClient.makeBucket(bucketName);
                minioClient.setBucketPolicy(bucketName, "readwrite");
            }

            //https://docs.aws.amazon.com/zh_cn/AmazonS3/latest/dev/BucketRestrictions.html

            if ( objectExists( objectName) )  //如果文件已存在，那么自动改名
            {
                String name0 = objectName;
                String extName = "";
                if (name0.indexOf(".") > 0)
                {
                    int p = name0.lastIndexOf(".");
                    name0 = name0.substring(0, p);
                    extName = objectName.substring(p);
                }
                int i = 0;
                while (true)
                {
                    i++;
                    objectName = name0 + "-" + i + extName;
                    if (!objectExists(objectName))
                    {

                        break;
                    }
                    else
                    {
                        FF.log(    objectName + "已存在， 系统自动重新生成一个文件名，避免覆盖");
                    }
                }

            }


            // Upload 'my-filename' as object 'my-objectname' in 'my-bucketname'.
            minioClient.putObject(bucketName, objectName, file, (Long) null, (Map) null, (ServerSideEncryption) null, (String) null);

            System.out.println("my-filename is uploaded to my-objectname successfully");
        } catch (Throwable e)
        {

            FF.log(e);
            ret.put("success", false);
            ret.put("message", e.getMessage());

        }

        String dn= getDomain();
        String url= dn;

        if( !url.isEmpty())
        {
            if (!url.endsWith("/")) url = url + "/";
            url += FF.pathJoin(bucketName, objectName);
            url = URLUtil.URLEncoder(url);

        }else
        {
            File f = new File(file);
            String shortName = f.getName();
            if (showName != null && !showName.isEmpty()) shortName = showName;
            shortName = URLEncoder.encode(shortName, "UTF-8");
            url = "fileServer?bucketName=" + bucketName + "&objectName=" + objectName + "&fileName=" + shortName + "&serverIndex=" + serverIndex + "&action=view";
        }

        ret.put("url", url);
        //下面的参数是为了着返回值兼容文件上传的返回结果
        ret.put("url_img", url);
        ret.put("serverindex", this.serverIndex);
        ret.put("bucketname", this.getBucketName());
        ret.put("objectname", objectName);
        ret.put("filename", showName);

        return ret;
    }

    @Override
    /**
     * 把 文件下载到本地文件系统中， 1小时后删除
     */
    public String getDownloadToLocalFileName(String objectName)
    {
        String bucketName = getBucketName();
        String t = objectName;
        int p = t.lastIndexOf("/");
        if (p > 0) t = t.substring(p + 1);

        String toFile = App.FileRoot + "/FileServer/autoDeleteAfterOneHour/" + t; // 去掉子目录 ，放到autoDeleteAfterOneHour 便于删除维护
        return toFile;
    }


    /**
     * 根据  bucketName ,  objectName 定位文件名
     *
     * @param bucketName
     * @param objectName
     * @return
     * @throws Exception
     */
    public String download(String objectName) throws Exception
    {
        String bucketName = getBucketName();
        if (objectName.startsWith("/")) objectName = objectName.substring(1);

        String endpoint = CacheConfig.get("/文件服务/文件服务器." + serverIndex + "/minio_address", "");
        String accessKey = CacheConfig.get("/文件服务/文件服务器." + serverIndex + "/minio_access_key", "");
        String secretKey = CacheConfig.get("/文件服务/文件服务器." + serverIndex + "/minio_access_secret", "");

        String toFile = getDownloadToLocalFileName(objectName);
        FF.makeSurePathOfFileExists(toFile); //确保文件目录一定存在
        //如果有缓存的文件，那么不需要再下载了 ，之所以可以这么处理， 是因为，这个上传的文件，不存在更改的，只会是删除，重新上传
        //所以如果本地有对应的下载好的缓存文件，那么不存在与OSS上不一至的情况。
        if (FF.fileExists(toFile)) return toFile;

        try
        {
            /* play.minio.io for test and development. */
            MinioClient minioClient = getMinioClient(endpoint, accessKey, secretKey);


            // Check whether the object exists using statObject().  If the object is not found,
            // statObject() throws an exception.  It means that the object exists when statObject()
            // execution is successful.
            minioClient.statObject(bucketName, objectName);
            // Download 'my-objectname' from 'my-bucketname' to 'my-filename'
            minioClient.getObject(bucketName, objectName, toFile);
            return toFile;
        } catch (Exception e)
        {
            FF.log(e);
            throw e;

        }
    }

    @Override
    public boolean delete(String objectName)
    {
        String bucketName = getBucketName();
        if (objectName.startsWith("/")) objectName = objectName.substring(1);

        String endpoint = CacheConfig.get("/文件服务/文件服务器." + serverIndex + "/minio_address", "");
        String accessKey = CacheConfig.get("/文件服务/文件服务器." + serverIndex + "/minio_access_key", "");
        String secretKey = CacheConfig.get("/文件服务/文件服务器." + serverIndex + "/minio_access_secret", "");

        try
        {
            /* play.minio.io for test and development. */
            MinioClient minioClient = getMinioClient(endpoint, accessKey, secretKey);


            // Check whether the object exists using statObject().  If the object is not found,
            // statObject() throws an exception.  It means that the object exists when statObject()
            // execution is successful.
            minioClient.statObject(bucketName, objectName);
            // Download 'my-objectname' from 'my-bucketname' to 'my-filename'
            minioClient.removeObject(bucketName, objectName);
            return true;
        } catch (Exception e)
        {
            FF.log(e);

            return false;
        }

    }

    public String getDomain()
    {
        String d = CacheConfig.get("/文件服务/文件服务器." + serverIndex + "/minio_dn", "");
        d= d.trim();

        return d;
    }
}
