package dfs;

import http.OkHttpUtil;
import io.minio.MinioClient;
import io.minio.Result;
import io.minio.messages.Item;
import okhttp3.OkHttpClient;

import java.io.ByteArrayInputStream;
import java.io.InputStream;

public class MinioExample
{
    // MinIO服务器配置
    private static final String ENDPOINT = "http://localhost:8002";
    private static final String ACCESS_KEY = "root";
    private static final String SECRET_KEY = "Abc123";

    // 存储桶和对象名称
    private static final String BUCKET_NAME = "temp";
    private static final String OBJECT_NAME = "example.txt";
    private static final String FILE_CONTENT = "Hello, this is a test file content.";

    private static MinioClient getMinioClient(String endpoint ,String accessKey, String secretKey) throws  Exception
    {
        int port =0; // 表示 endpoint中包含了 port，不要需要再指定port
        boolean secure = endpoint == null || !endpoint.startsWith("http://");
        OkHttpClient httpClient  = OkHttpUtil.getOkHttpClient();
        String region=null;
        //2024.01.20 重大问题： 如果不指定 httpClient,那么 MinitClient自已就会创建一个,那当短时间上传大量文件时，就会造成 okhttp connectinPool 大量创建
        //而造成 oom
        MinioClient minioClient = new MinioClient(endpoint,port, accessKey, secretKey,region, secure, httpClient);
        return minioClient;

    }

    public static void main(String[] args) {
        try {
            // 1. 创建MinioClient实例
            MinioClient minioClient = getMinioClient(
                    ENDPOINT,
                    ACCESS_KEY,
                    SECRET_KEY
            );

            // 2. 检查存储桶是否存在，不存在则创建
            if (!minioClient.bucketExists(BUCKET_NAME)) {
                minioClient.makeBucket(BUCKET_NAME);
                System.out.println("创建存储桶: " + BUCKET_NAME);
            }

            // 3. 上传文件
            uploadFile(minioClient);

            // 4. 列出存储桶内容
            listBucketContents(minioClient);

            // 5. 下载文件
            downloadFile(minioClient);

            // 6. 删除文件
            deleteFile(minioClient);

        } catch (Exception e) {
            System.err.println("发生错误: " + e.getMessage());
            e.printStackTrace();
        }
    }

    // 上传文件方法
    private static void uploadFile(MinioClient minioClient) throws Exception {
        // 创建输入流
        InputStream stream = new ByteArrayInputStream(FILE_CONTENT.getBytes("UTF-8"));

        // 上传对象
        minioClient.putObject(
                BUCKET_NAME,
                OBJECT_NAME,
                stream,
                "text/plain"
        );

        stream.close();
        System.out.println("文件上传成功: " + OBJECT_NAME);
    }

    // 列出存储桶内容方法
    private static void listBucketContents(MinioClient minioClient) throws Exception {
        System.out.println("存储桶 " + BUCKET_NAME + " 中的内容:");

        // 列出所有对象
        Iterable<Result<Item>> results = minioClient.listObjects(BUCKET_NAME);
        for (Result<Item> result : results) {
            Item item = result.get();
            System.out.println(" - " + item.objectName() + " (大小: " + item.objectSize() + " bytes)");
        }
    }

    // 下载文件方法
    private static void downloadFile(MinioClient minioClient) throws Exception {
        // 获取对象
        InputStream stream = minioClient.getObject(BUCKET_NAME, OBJECT_NAME);

        // 读取内容
        byte[] buf = new byte[1024];
        int bytesRead;
        System.out.println("文件内容:");
        while ((bytesRead = stream.read(buf, 0, buf.length)) != -1) {
            System.out.print(new String(buf, 0, bytesRead, "UTF-8"));
        }
        System.out.println();

        stream.close();
    }

    // 删除文件方法
    private static void deleteFile(MinioClient minioClient) throws Exception {
        minioClient.removeObject(BUCKET_NAME, OBJECT_NAME);
        System.out.println("文件删除成功: " + OBJECT_NAME);
    }
}