package dfs;

import java.io.*;
import java.net.HttpURLConnection;
import java.net.URL;

public class FileDownloader {
    private int connectTimeout = 100000;
    private int readTimeout = 100000;
    private int bufferSize = 1024;

    // 可选：自定义配置方法
    public void setConnectTimeout(int timeout) {
        this.connectTimeout = timeout;
    }

    public void setReadTimeout(int timeout) {
        this.readTimeout = timeout;
    }

    public void setBufferSize(int size) {
        this.bufferSize = size;
    }

    public void download(String urlStr, String localFilePath) throws IOException {
        // 校验协议类型
        if (!urlStr.startsWith("http://") && !urlStr.startsWith("https://")) {
            throw new IllegalArgumentException("Only HTTP and HTTPS protocols are supported");
        }

        URL url = new URL(urlStr);
        HttpURLConnection connection = (HttpURLConnection) url.openConnection();
        configureConnection(connection);

        try {
            validateResponse(connection);
            saveToFile(connection, localFilePath);
        } finally {
            connection.disconnect();
        }
    }

    private void configureConnection(HttpURLConnection connection) throws IOException {
        connection.setRequestMethod("GET");
        connection.setConnectTimeout(connectTimeout);
        connection.setReadTimeout(readTimeout);
    }

    private void validateResponse(HttpURLConnection connection) throws IOException {
        int responseCode = connection.getResponseCode();
        if (responseCode != HttpURLConnection.HTTP_OK) {
            throw new IOException("HTTP request failed with code: " + responseCode);
        }
    }

    private void saveToFile(HttpURLConnection connection, String filePath) throws IOException {
        File targetFile = new File(filePath);
        validateFile(targetFile);
        createParentDirs(targetFile);

        try (InputStream in = new BufferedInputStream(connection.getInputStream());
             OutputStream out = new BufferedOutputStream(new FileOutputStream(targetFile))) {

            byte[] buffer = new byte[bufferSize];
            int bytesRead;
            while ((bytesRead = in.read(buffer)) != -1) {
                out.write(buffer, 0, bytesRead);
            }
            out.flush();
        }
    }

    private void validateFile(File file) throws IOException {
        if (file.exists() && file.isDirectory()) {
            throw new IOException("Target path is a directory: " + file.getAbsolutePath());
        }
    }

    private void createParentDirs(File file) throws IOException {
        File parent = file.getParentFile();
        if (parent != null && !parent.exists() && !parent.mkdirs()) {
            throw new IOException("Failed to create directory: " + parent.getAbsolutePath());
        }
    }

    // 使用示例
    public static void main(String[] args) {
        FileDownloader downloader = new FileDownloader();
        try
        {
            downloader.download(
                    "https://demo.wiccloud.cn/fileServer?objectName=2025.02.12%2F67ab740599325a98d060c02b.png&serverIndex=1&fileName=67ab740599325a98d060c02b.png&action=view",
                    "/Users/zengjun/Downloads/aaa/a.png"
            );
            System.out.println("File downloaded successfully!");
        } catch (IOException e)
        {
            System.err.println("Download failed: " + e.getMessage());
        }
    }
}

