/*
 * $Id: ae19c37e97db5f39a33b959d1b6b1ccb5d797c60 $
 *
 * This file is part of the iText (R) project.
 * Copyright (c) 2007-2015 iText Group NV
 * Authors: Bruno Lowagie et al.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License version 3
 * as published by the Free Software Foundation with the addition of the
 * following permission added to Section 15 as permitted in Section 7(a):
 * FOR ANY PART OF THE COVERED WORK IN WHICH THE COPYRIGHT IS OWNED BY
 * ITEXT GROUP. ITEXT GROUP DISCLAIMS THE WARRANTY OF NON INFRINGEMENT
 * OF THIRD PARTY RIGHTS
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Affero General Public License for more details.
 * You should have received a copy of the GNU Affero General Public License
 * along with this program; if not, see http://www.gnu.org/licenses or write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA, 02110-1301 USA, or download the license from the following URL:
 * http://itextpdf.com/terms-of-use/
 *
 * The interactive user interfaces in modified source and object code versions
 * of this program must display Appropriate Legal Notices, as required under
 * Section 5 of the GNU Affero General Public License.
 *
 * In accordance with Section 7(b) of the GNU Affero General Public License,
 * a covered work must retain the producer line in every PDF that is created
 * or manipulated using iText.
 *
 * You can be released from the requirements of the license by purchasing
 * a commercial license. Buying such a license is mandatory as soon as you
 * develop commercial activities involving the iText software without
 * disclosing the source code of your own applications.
 * These activities include: offering paid services to customers as an ASP,
 * serving PDFs on the fly in a web application, shipping iText with a closed
 * source product.
 *
 * For more information, please contact iText Software Corp. at this
 * address: sales@itextpdf.com
 */
package com.itextpdf.rups.view.itext;

import java.util.Observable;
import java.util.Observer;

import javax.swing.JTable;
import javax.swing.event.ListSelectionEvent;
import javax.swing.table.TableColumn;

import com.itextpdf.rups.controller.PdfReaderController;
import com.itextpdf.rups.model.IndirectObjectFactory;
import com.itextpdf.rups.model.ObjectLoader;
import com.itextpdf.rups.view.models.JTableAutoModel;
import com.itextpdf.rups.view.models.JTableAutoModelInterface;
import com.itextpdf.text.pdf.PdfNull;
import com.itextpdf.text.pdf.PdfObject;

/**
 * A JTable that shows the indirect objects of a PDF xref table.
 */
public class XRefTable extends JTable implements JTableAutoModelInterface, Observer {

	/** The factory that can produce all the indirect objects. */
	protected IndirectObjectFactory objects;
	/** The renderer that will render an object when selected in the table. */
	protected PdfReaderController controller;
	
	/** Creates a JTable visualizing xref table. */
	public XRefTable(PdfReaderController controller) {
		super();
		this.controller = controller;
	}
	
	/**
	 * @see java.util.Observer#update(java.util.Observable, java.lang.Object)
	 */
	public void update(Observable observable, Object obj) {
		if (obj == null) {
			objects = null;
			repaint();
			return;
		}
		if (observable instanceof PdfReaderController
				&& obj instanceof ObjectLoader) {
			ObjectLoader loader = (ObjectLoader)obj;
			objects = loader.getObjects();
			setModel(new JTableAutoModel(this));
			TableColumn col= getColumnModel().getColumn(0);
			col.setPreferredWidth(5);
		}
	}
	
	/**
	 * @see javax.swing.JTable#getColumnCount()
	 */
	public int getColumnCount() {
		return 2;
	}
	
	/**
	 * @see javax.swing.JTable#getRowCount()
	 */
	public int getRowCount() {
		if (objects == null) return 0;
		return objects.size();
	}

    /**
     * @see javax.swing.JTable#getValueAt(int, int)
     */
    public Object getValueAt(int rowIndex, int columnIndex) {
		switch (columnIndex) {
		case 0:
			return getObjectReferenceByRow(rowIndex);
		case 1:
			return getObjectDescriptionByRow(rowIndex);
		default:
			return null;
		}
	}
	
	/**
	 * Gets the reference number of an indirect object
	 * based on the row index.
	 * @param rowIndex	a row number
	 * @return	a reference number
	 */
	protected int getObjectReferenceByRow(int rowIndex) {
		return objects.getRefByIndex(rowIndex);
	}
	
	/**
	 * Gets the object that is shown in a row.
	 * @param rowIndex	the row number containing the object
	 * @return	a PDF object
	 */
	protected String getObjectDescriptionByRow(int rowIndex) {
		PdfObject object = objects.getObjectByIndex(rowIndex);
		if (object instanceof PdfNull)
			return "Indirect object";
		return object.toString();
	}
	
	/**
	 * @see javax.swing.JTable#getColumnName(int)
	 */
	public String getColumnName(int columnIndex) {
		switch (columnIndex) {
		case 0:
			return "Number";
		case 1:
			return "Object";
		default:
			return null;
		}
	}
	
	/**
	 * Gets the object that is shown in a row.
	 * @param rowIndex	the row number containing the object
	 * @return	a PDF object
	 */
	protected PdfObject getObjectByRow(int rowIndex) {
		return objects.loadObjectByReference(getObjectReferenceByRow(rowIndex));
	}
	
	/**
	 * Selects a row containing information about an indirect object.
	 * @param ref	the reference number of the indirect object
	 */
	public void selectRowByReference(int ref) {
		int row = objects.getIndexByRef(ref);
		setRowSelectionInterval(row, row);
		scrollRectToVisible(getCellRect(row, 1, true));
		valueChanged(null);
	}

	/**
	 * @see javax.swing.JTable#valueChanged(javax.swing.event.ListSelectionEvent)
	 */
	@Override
	public void valueChanged(ListSelectionEvent evt) {
		if (evt != null)
			super.valueChanged(evt);
		if (controller != null && objects != null) {
			controller.render(getObjectByRow(this.getSelectedRow()));
			controller.selectNode(getObjectReferenceByRow(getSelectedRow()));
		}
	}
	
	/** A serial version UID. */
	private static final long serialVersionUID = -382184619041375537L;

}